/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.acl.AclBinding;
import org.apache.kafka.common.acl.AclBindingFilter;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.UnknownServerException;
import org.apache.kafka.common.metadata.AccessControlEntryRecord;
import org.apache.kafka.common.metadata.RemoveAccessControlEntryRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.metadata.authorizer.ClusterMetadataAuthorizer;
import org.apache.kafka.metadata.authorizer.ConfluentStandardAcl;
import org.apache.kafka.metadata.authorizer.StandardAclWithId;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.server.authorizer.AclCreateResult;
import org.apache.kafka.server.authorizer.AclDeleteResult;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.mutable.BoundedList;
import org.apache.kafka.server.mutable.BoundedListTooLongException;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.apache.kafka.timeline.TimelineHashSet;
import org.slf4j.Logger;

public class AclControlManager {
    private final TimelineHashMap<Uuid, ConfluentStandardAcl> idToAcl;
    private final TimelineHashSet<ConfluentStandardAcl> existingAcls;
    private final TimelineHashMap<Uuid, TimelineHashSet<Uuid>> linkIdToAcls;
    private final Optional<ClusterMetadataAuthorizer> authorizer;
    private final Function<Uuid, Boolean> isValidClusterLink;
    private final Logger log;
    private final SnapshotRegistry snapshotRegistry;

    AclControlManager(LogContext logContext, SnapshotRegistry snapshotRegistry, Optional<ClusterMetadataAuthorizer> authorizer, Function<Uuid, Boolean> validLinkIdChecker) {
        this.idToAcl = new TimelineHashMap(snapshotRegistry, 0);
        this.existingAcls = new TimelineHashSet(snapshotRegistry, 0);
        this.linkIdToAcls = new TimelineHashMap(snapshotRegistry, 0);
        this.authorizer = authorizer;
        this.isValidClusterLink = validLinkIdChecker;
        this.log = logContext.logger(AclControlManager.class);
        this.snapshotRegistry = snapshotRegistry;
    }

    ControllerResult<List<AclCreateResult>> createAcls(List<AclBinding> acls) {
        ArrayList<AclCreateResult> results = new ArrayList<AclCreateResult>(acls.size());
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        for (AclBinding acl : acls) {
            try {
                AclControlManager.validateNewAcl(acl);
                for (Uuid linkId : acl.entry().clusterLinkIds()) {
                    if (linkId.equals((Object)Uuid.ZERO_UUID)) {
                        throw new InvalidRequestException("Invalid link ID " + linkId + " provided to create the ACL. Provide a valid link id or create an ACL without one.");
                    }
                    if (this.isValidClusterLink.apply(linkId).booleanValue()) continue;
                    throw new InvalidRequestException("Unknown link ID " + linkId + " provided to create ACL");
                }
            }
            catch (Throwable t) {
                ApiException e = t instanceof ApiException ? (ApiException)t : new UnknownServerException("Unknown error while trying to create ACL", t);
                results.add(new AclCreateResult(e));
                continue;
            }
            List<ConfluentStandardAcl> confluentStandardAclList = ConfluentStandardAcl.fromAclBinding(acl);
            for (ConfluentStandardAcl confluentStandardAcl : confluentStandardAclList) {
                if (!this.existingAcls.add((Object)confluentStandardAcl)) continue;
                StandardAclWithId standardAclWithId = new StandardAclWithId(this.newAclId(), confluentStandardAcl);
                this.idToAcl.put((Object)standardAclWithId.id(), (Object)confluentStandardAcl);
                records.add(new ApiMessageAndVersion((ApiMessage)standardAclWithId.toRecord(), 0));
            }
            results.add(AclCreateResult.SUCCESS);
        }
        return new ControllerResult<List<AclCreateResult>>((List<ApiMessageAndVersion>)records, results, true);
    }

    Uuid newAclId() {
        Uuid uuid;
        while (this.idToAcl.containsKey((Object)(uuid = Uuid.randomUuid()))) {
        }
        return uuid;
    }

    static void validateNewAcl(AclBinding binding) {
        switch (binding.pattern().resourceType()) {
            case UNKNOWN: 
            case ANY: {
                throw new InvalidRequestException("Invalid resourceType " + binding.pattern().resourceType());
            }
        }
        switch (binding.pattern().patternType()) {
            case LITERAL: 
            case PREFIXED: {
                break;
            }
            default: {
                throw new InvalidRequestException("Invalid patternType " + binding.pattern().patternType());
            }
        }
        switch (binding.entry().operation()) {
            case UNKNOWN: 
            case ANY: {
                throw new InvalidRequestException("Invalid operation " + binding.entry().operation());
            }
        }
        switch (binding.entry().permissionType()) {
            case DENY: 
            case ALLOW: {
                break;
            }
            default: {
                throw new InvalidRequestException("Invalid permissionType " + binding.entry().permissionType());
            }
        }
        if (binding.pattern().name() == null || binding.pattern().name().isEmpty()) {
            throw new InvalidRequestException("Resource name should not be empty");
        }
    }

    ControllerResult<List<AclDeleteResult>> deleteAcls(List<AclBindingFilter> filters) {
        ArrayList<AclDeleteResult> results = new ArrayList<AclDeleteResult>();
        HashSet<ApiMessageAndVersion> records = new HashSet<ApiMessageAndVersion>();
        for (AclBindingFilter filter : filters) {
            try {
                AclControlManager.validateFilter(filter);
                AclDeleteResult result = this.deleteAclsForFilter(filter, records);
                results.add(result);
            }
            catch (Throwable e) {
                results.add(new AclDeleteResult(ApiError.fromThrowable((Throwable)e).exception()));
            }
        }
        return ControllerResult.atomicOf(new ArrayList<ApiMessageAndVersion>(records), results);
    }

    AclDeleteResult deleteAclsForFilter(AclBindingFilter filter, Set<ApiMessageAndVersion> records) {
        ArrayList<AclDeleteResult.AclBindingDeleteResult> deleted = new ArrayList<AclDeleteResult.AclBindingDeleteResult>();
        for (Map.Entry entry : this.idToAcl.entrySet()) {
            Uuid id = (Uuid)entry.getKey();
            ConfluentStandardAcl acl = (ConfluentStandardAcl)entry.getValue();
            AclBinding binding = acl.toBinding();
            if (!filter.matches(binding)) continue;
            deleted.add(new AclDeleteResult.AclBindingDeleteResult(binding));
            records.add(new ApiMessageAndVersion((ApiMessage)new RemoveAccessControlEntryRecord().setId(id), 0));
            if (records.size() <= 10000) continue;
            throw new BoundedListTooLongException("Cannot remove more than 10000 acls in a single delete operation.");
        }
        return new AclDeleteResult(deleted);
    }

    static void validateFilter(AclBindingFilter filter) {
        if (filter.patternFilter().isUnknown()) {
            throw new InvalidRequestException("Unknown patternFilter.");
        }
        if (filter.entryFilter().isUnknown()) {
            throw new InvalidRequestException("Unknown entryFilter.");
        }
    }

    public void replay(AccessControlEntryRecord record, Optional<OffsetAndEpoch> snapshotId) {
        StandardAclWithId aclWithId = StandardAclWithId.fromRecord(record);
        this.idToAcl.put((Object)aclWithId.id(), (Object)aclWithId.acl());
        this.existingAcls.add((Object)aclWithId.acl());
        if (aclWithId.acl().hasLinkId()) {
            Uuid linkId = aclWithId.acl().clusterLinkId().get();
            TimelineHashSet aclsForLink = (TimelineHashSet)this.linkIdToAcls.computeIfAbsent((Object)linkId, __ -> new TimelineHashSet(this.snapshotRegistry, 0));
            aclsForLink.add((Object)aclWithId.id());
        }
        if (!snapshotId.isPresent()) {
            this.authorizer.ifPresent(a -> a.addAcl(aclWithId.id(), aclWithId.acl()));
        }
    }

    public void replay(RemoveAccessControlEntryRecord record, Optional<OffsetAndEpoch> snapshotId) {
        ConfluentStandardAcl acl = (ConfluentStandardAcl)this.idToAcl.remove((Object)record.id());
        if (acl == null) {
            throw new RuntimeException("Unable to replay " + record + ": no acl with that ID found.");
        }
        if (!this.existingAcls.remove((Object)acl)) {
            throw new RuntimeException("Unable to replay " + record + " for " + acl + ": acl not found in existingAcls.");
        }
        if (acl.hasLinkId()) {
            Uuid linkId = acl.clusterLinkId().get();
            this.linkIdToAcls.compute((Object)linkId, (id, aclsForLink) -> {
                if (aclsForLink == null || !aclsForLink.remove((Object)record.id())) {
                    throw new RuntimeException("Unable to replay " + record + " for " + acl + ": acl not found in ACLs with link id");
                }
                if (aclsForLink.isEmpty()) {
                    return null;
                }
                return aclsForLink;
            });
        }
        if (!snapshotId.isPresent()) {
            this.authorizer.ifPresent(a -> a.removeAcl(record.id()));
        }
    }

    Map<Uuid, ConfluentStandardAcl> idToAcl() {
        return Collections.unmodifiableMap(this.idToAcl);
    }

    void unlinkAcls(Uuid linkId, Optional<OffsetAndEpoch> snapshotId) {
        if (linkId.equals((Object)Uuid.ZERO_UUID)) {
            throw new IllegalStateException("Trying to unlink ACLs for invalid link id");
        }
        Set toRemoveLinkAcls = (Set)this.linkIdToAcls.remove((Object)linkId);
        if (toRemoveLinkAcls == null) {
            return;
        }
        HashSet removedAcls = new HashSet();
        HashMap addedAcls = new HashMap();
        toRemoveLinkAcls.forEach(aclId -> {
            ConfluentStandardAcl acl = (ConfluentStandardAcl)this.idToAcl.remove(aclId);
            if (acl == null || !this.existingAcls.remove((Object)acl)) {
                this.log.error("Found non-existent ACL trying to clear link ID " + linkId);
            } else {
                ConfluentStandardAcl localAcl = acl.toLocalAcl();
                if (this.existingAcls.add((Object)localAcl)) {
                    this.idToAcl.put(aclId, (Object)localAcl);
                    addedAcls.put(aclId, localAcl);
                } else {
                    removedAcls.add(aclId);
                }
            }
        });
        this.log.info("Removed " + removedAcls.size() + "acls and added " + addedAcls.size() + "local acls after unlinking cluster link " + linkId);
        if (!snapshotId.isPresent()) {
            this.authorizer.ifPresent(authz -> {
                removedAcls.forEach(authz::removeAcl);
                addedAcls.forEach(authz::addAcl);
            });
        }
    }

    boolean validateAclsInCache(Set<ConfluentStandardAcl> acls) {
        if (this.idToAcl.size() != this.existingAcls.size()) {
            return false;
        }
        return this.existingAcls.equals(acls);
    }
}

