/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.tokenapi.jwt;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import io.confluent.tokenapi.jwt.JwsProvider;
import io.confluent.tokenapi.jwt.JwtConfig;
import java.security.Principal;
import java.security.PublicKey;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import org.apache.kafka.common.Configurable;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.NumericDate;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.lang.JoseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwtProvider
implements Configurable {
    private static final Logger log = LoggerFactory.getLogger(JwtProvider.class);
    private JwsProvider jws;
    public String issuer;
    private long maxTokenLifetime;
    private LoadingCache<String, JwtConsumer> jwtConsumerCache;
    private JwtConsumer jwtReader;

    public void configure(Map<String, ?> props) {
        JwtConfig config = new JwtConfig(props);
        this.issuer = config.getString("token.issuer");
        this.maxTokenLifetime = config.getLong("token.max.lifetime.ms") / 1000L;
        this.initializeJwtConsumerCache(100L);
        this.jws = new JwsProvider();
        this.jws.configure(props);
        this.jwtReader = new JwtConsumerBuilder().setSkipSignatureVerification().setDisableRequireSignature().setSkipAllValidators().build();
    }

    private void initializeJwtConsumerCache(long maxSize) {
        CacheLoader<String, JwtConsumer> loader = new CacheLoader<String, JwtConsumer>(){

            public JwtConsumer load(String aud) {
                return JwtProvider.this.newJwtConsumer(aud);
            }
        };
        this.jwtConsumerCache = CacheBuilder.newBuilder().maximumSize(maxSize).build((CacheLoader)loader);
    }

    public PublicKey getPublicKey() {
        return this.jws.getPublicKey();
    }

    private JwtClaims newJwtToken(Principal requester, String ... audience) {
        return this.newJwtToken(null, requester, audience);
    }

    private JwtClaims newJwtToken(String orgResourceId, Principal requester, String ... audience) {
        JwtClaims claims = new JwtClaims();
        claims.setGeneratedJwtId();
        claims.setIssuer(this.issuer);
        claims.setSubject(requester.getName());
        claims.setExpirationTimeMinutesInTheFuture((float)(this.maxTokenLifetime / 60L));
        claims.setIssuedAtToNow();
        claims.setNotBeforeMinutesInThePast(1.0f);
        if (orgResourceId != null) {
            claims.setClaim("orgResourceId", (Object)orgResourceId);
        }
        if (audience.length > 0) {
            claims.setAudience(audience);
        }
        claims.setClaim("azp", (Object)requester.getName());
        claims.setNumericDateClaim("auth_time", NumericDate.now());
        return claims;
    }

    private JwtClaims newJwtToken(String orgResourceId, Principal requester, Map<String, Object> customClaims, long expirySec, String ... audience) {
        JwtClaims claims = new JwtClaims();
        if (Objects.nonNull(customClaims) && !customClaims.isEmpty()) {
            customClaims.forEach((arg_0, arg_1) -> ((JwtClaims)claims).setClaim(arg_0, arg_1));
        }
        claims.setGeneratedJwtId();
        claims.setIssuer(this.issuer);
        claims.setSubject(requester.getName());
        claims.setExpirationTimeMinutesInTheFuture((float)expirySec / 60.0f);
        claims.setIssuedAtToNow();
        claims.setNotBeforeMinutesInThePast(1.0f);
        if (audience.length > 0) {
            claims.setAudience(audience);
        } else {
            claims.unsetClaim("aud");
        }
        claims.setClaim("azp", (Object)requester.getName());
        claims.setNumericDateClaim("auth_time", NumericDate.now());
        return claims;
    }

    public String newJwsToken(Principal principal, String ... audience) throws JoseException {
        return this.signClaims(this.newJwtToken(principal, audience));
    }

    public String newJwsToken(Principal principal, Map<String, Object> customClaims, long expirySec, String ... audience) throws JoseException {
        return this.signClaims(this.newJwtToken(null, principal, customClaims, expirySec, audience));
    }

    public String newJwsToken(String orgResourceId, Principal principal, String ... audience) throws JoseException {
        return this.signClaims(this.newJwtToken(orgResourceId, principal, audience));
    }

    public String refreshTokenWithoutValidation(Principal requester, String jwt, String ... audience) throws JoseException {
        try {
            JwtClaims currClaims = this.jwtReader.processToClaims(jwt);
            Map customClaims = currClaims.hasClaim("iss") && currClaims.getIssuer().equals(this.issuer) ? currClaims.getClaimsMap() : new HashMap();
            long extendabilitySec = this.getExtendabilityWhenTokenRefreshWithoutValidation(currClaims);
            log.debug("Refreshing bearer token with extendability value as (in secs) {}, for principal: {}", (Object)extendabilitySec, (Object)requester.getName());
            return this.newJwsToken(requester, customClaims, extendabilitySec, audience);
        }
        catch (MalformedClaimException | InvalidJwtException e) {
            throw new JoseException("Unable to refresh token without validation", e);
        }
    }

    private long getExtendabilityWhenTokenRefreshWithoutValidation(JwtClaims currClaims) {
        long extendabilitySec = this.maxTokenLifetime;
        if (currClaims.hasClaim("mex")) {
            long mexSec = (Long)currClaims.getClaimsMap().get("mex");
            long currentTimeSec = System.currentTimeMillis() / 1000L;
            extendabilitySec = Math.min(Math.max(mexSec - currentTimeSec, 0L), this.maxTokenLifetime);
        }
        return extendabilitySec;
    }

    public String refreshToken(Principal requester, String jwt, Map<String, Object> customClaims, long expirySec) throws JoseException {
        return this.refreshToken(requester, jwt, customClaims, expirySec, new String[0]);
    }

    public String refreshToken(Principal requester, String jwt, Map<String, Object> customClaims, long expirySec, String ... audience) throws JoseException {
        try {
            JwtConsumer jwtConsumer = (JwtConsumer)this.jwtConsumerCache.get((Object)requester.getName());
            JwtClaims claims = jwtConsumer.processToClaims(jwt);
            if (Objects.nonNull(customClaims) && !customClaims.isEmpty()) {
                customClaims.forEach((arg_0, arg_1) -> ((JwtClaims)claims).setClaim(arg_0, arg_1));
            }
            claims.setExpirationTimeMinutesInTheFuture((float)expirySec / 60.0f);
            claims.setIssuedAtToNow();
            claims.setClaim("azp", (Object)requester.getName());
            if (audience.length > 0) {
                claims.setAudience(audience);
            }
            return this.signClaims(claims);
        }
        catch (ExecutionException | InvalidJwtException e) {
            throw new JoseException("Unable to refresh invalid token", e);
        }
    }

    public String refreshToken(Principal requester, String jwt) throws JoseException {
        try {
            JwtConsumer jwtConsumer = (JwtConsumer)this.jwtConsumerCache.get((Object)requester.getName());
            JwtClaims claims = jwtConsumer.processToClaims(jwt);
            claims.setExpirationTimeMinutesInTheFuture((float)(this.maxTokenLifetime / 60L));
            claims.setIssuedAtToNow();
            claims.setClaim("azp", (Object)requester.getName());
            return this.signClaims(claims);
        }
        catch (ExecutionException | InvalidJwtException e) {
            throw new JoseException("Unable to refresh invalid token", e);
        }
    }

    public String sanitizeToken(Principal requester, String jwt, Collection<String> sanitizeClaims) throws JoseException {
        try {
            JwtConsumer jwtConsumer = (JwtConsumer)this.jwtConsumerCache.get((Object)requester.getName());
            JwtClaims claims = jwtConsumer.processToClaims(jwt);
            if (Objects.nonNull(sanitizeClaims) && !sanitizeClaims.isEmpty()) {
                sanitizeClaims.forEach(arg_0 -> ((JwtClaims)claims).unsetClaim(arg_0));
            }
            return this.signClaims(claims);
        }
        catch (Exception e) {
            throw new JoseException("Unable to sanitize token", (Throwable)e);
        }
    }

    public long tokenLifetime() {
        return this.maxTokenLifetime;
    }

    private String signClaims(JwtClaims claims) throws JoseException {
        return this.jws.signClaims(claims);
    }

    private JwtConsumer newJwtConsumer(String aud) {
        return new JwtConsumerBuilder().setRequireJwtId().setExpectedIssuer(true, this.issuer).setSkipDefaultAudienceValidation().setRequireSubject().setRequireExpirationTime().setAllowedClockSkewInSeconds(30).setVerificationKeyResolver(this.jws.jwksResolver).build();
    }

    public JwtConsumer getJwtReader() {
        return this.jwtReader;
    }
}

