/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.tokenapi.services;

import io.confluent.common.security.auth.JwtPrincipal;
import io.confluent.tokenapi.entities.AuthenticationResponse;
import io.confluent.tokenapi.entities.RefreshTokenRequest;
import io.confluent.tokenapi.entities.SanitizeTokenRequest;
import io.confluent.tokenapi.exceptions.AuthenticationTokenException;
import io.confluent.tokenapi.jwt.JwtProvider;
import java.security.Principal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.eclipse.jetty.http.DateGenerator;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.lang.JoseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TokenService {
    private static final Logger log = LoggerFactory.getLogger(TokenService.class);
    private final JwtProvider jwtProvider;
    public static final String TOKEN_MAX_EXP_CLAIM_NAME = "mex";
    public static final String C3_TOKEN_MAX_LIFE_HEADER = "X-C3-Token-Max-Life";
    public static final long MAX_C3_TOKEN_MAX_LIFE_HEADER_VALUE = 86400000L;

    public TokenService(JwtProvider jwtProvider) {
        this.jwtProvider = jwtProvider;
    }

    public String issueToken(Principal principal, Map<String, Object> customClaims, List<String> aud) {
        return this.issueToken(principal, customClaims, this.jwtProvider.tokenLifetime(), aud);
    }

    public String issueToken(Principal principal, Map<String, Object> customClaims, long expirySec, List<String> aud) {
        try {
            if (principal instanceof JwtPrincipal) {
                JwtPrincipal jwtPrincipal = (JwtPrincipal)principal;
                return this.jwtProvider.refreshTokenWithoutValidation((Principal)jwtPrincipal, jwtPrincipal.getJwt(), aud.toArray(new String[0]));
            }
            return this.jwtProvider.newJwsToken(principal, customClaims, expirySec, aud.toArray(new String[0]));
        }
        catch (JoseException e) {
            throw new AuthenticationTokenException("Authentication Token generation failed", e);
        }
    }

    public String refreshToken(Principal principal, RefreshTokenRequest refreshRequest) {
        try {
            return this.jwtProvider.refreshToken(principal, refreshRequest.getAccessToken());
        }
        catch (JoseException e) {
            throw new AuthenticationTokenException("Token refresh failed. ", e);
        }
    }

    public String refreshToken(Principal principal, RefreshTokenRequest refreshRequest, Map<String, Object> customClaims, long expirySec) {
        try {
            return this.jwtProvider.refreshToken(principal, refreshRequest.getAccessToken(), customClaims, expirySec);
        }
        catch (JoseException e) {
            throw new AuthenticationTokenException("Token refresh failed. ", e);
        }
    }

    public String sanitizeToken(Principal principal, SanitizeTokenRequest sanitizeTokenRequest) {
        try {
            return this.jwtProvider.sanitizeToken(principal, sanitizeTokenRequest.accessToken(), sanitizeTokenRequest.sanitizeClaims());
        }
        catch (JoseException e) {
            log.error("Token sanitize failed due to problem in auth token.", (Throwable)e);
            throw new AuthenticationTokenException("Token sanitize failed.", e);
        }
    }

    public String getCookieHeader(String token) {
        return this.getCookieHeader(token, this.jwtProvider.tokenLifetime());
    }

    public String getCookieHeader(String token, long tokenLifetime) {
        String expires = tokenLifetime <= 0L ? DateGenerator.formatCookieDate((long)0L).trim() : DateGenerator.formatCookieDate((long)(System.currentTimeMillis() + 1000L * tokenLifetime));
        token = Objects.isNull(token) ? "" : token;
        return "auth_token=" + token + "; HttpOnly; Secure; Path=/; Expires=" + expires + "; SameSite=Lax; Max-Age=" + tokenLifetime;
    }

    public AuthenticationResponse getAuthenticationResponse(String token) {
        return this.getAuthenticationResponse(token, this.jwtProvider.tokenLifetime());
    }

    public AuthenticationResponse getAuthenticationResponse(String token, long tokenLifetime) {
        return new AuthenticationResponse(token, "Bearer", tokenLifetime);
    }

    public long getTokenLifeTime(String token, long currentTimeMillis) {
        try {
            JwtClaims jwtClaims = this.jwtProvider.getJwtReader().processToClaims(token);
            return jwtClaims.getExpirationTime().getValue() - currentTimeMillis / 1000L;
        }
        catch (MalformedClaimException | InvalidJwtException e) {
            throw new AuthenticationTokenException("Authentication Token generation failed", e);
        }
    }

    public Map<String, Object> customClaimsForTokenLifetime(List<String> mexValue, long currentTimeMillis) {
        HashMap<String, Object> customClaims = new HashMap<String, Object>();
        long extendability = Long.parseLong(mexValue.get(0));
        extendability = Math.min(Math.max(0L, extendability), 86400000L);
        log.debug("Setting C3_TOKEN_MAX_LIFE value as {}", (Object)extendability);
        long mexEpoch = (currentTimeMillis + extendability) / 1000L;
        customClaims.put(TOKEN_MAX_EXP_CLAIM_NAME, mexEpoch);
        return customClaims;
    }
}

