/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.validator;

import io.confluent.kafka.schemaregistry.ParsedSchema;
import io.confluent.kafka.schemaregistry.client.SchemaRegistryClient;
import io.confluent.kafka.schemaregistry.client.rest.exceptions.RestClientException;
import io.confluent.kafka.schemaregistry.validator.LruSchemaRegistryClient;
import io.confluent.kafka.schemaregistry.validator.RecordSchemaValidatorConfig;
import io.confluent.kafka.serializers.subject.strategy.SubjectNameStrategy;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.utils.AdaptiveSampler;
import org.apache.kafka.common.utils.LogAction;
import org.apache.kafka.common.utils.NoOpSampler;
import org.apache.kafka.common.utils.TimeBasedSampler;
import org.apache.kafka.server.interceptor.RecordInterceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecordSchemaValidator
implements RecordInterceptor {
    protected static final byte MAGIC_BYTE = 0;
    protected SchemaRegistryClient schemaRegistry;
    protected SubjectNameStrategy keyStrategy;
    protected SubjectNameStrategy valueStrategy;
    protected Logger log = LoggerFactory.getLogger(RecordSchemaValidator.class);
    protected boolean validateKey;
    protected boolean validateValue;
    protected final AtomicReference<Sensor> allRecordsSensor = new AtomicReference();
    protected final AtomicReference<Sensor> rejectedRecordsSensor = new AtomicReference();
    protected final AtomicReference<Sensor> connectionErrorsSensor = new AtomicReference();
    protected final AtomicReference<Sensor> clientErrorsSensor = new AtomicReference();
    protected final AtomicReference<Sensor> serverErrorsSensor = new AtomicReference();
    private TimeBasedSampler loggingSampler = new AllSampler();

    public RecordSchemaValidator() {
    }

    public RecordSchemaValidator(SchemaRegistryClient schemaRegistry, Logger log) {
        this.schemaRegistry = schemaRegistry;
        this.log = log;
    }

    public void configure(Map<String, ?> configs) {
        RecordSchemaValidatorConfig config = new RecordSchemaValidatorConfig(configs);
        if (this.schemaRegistry == null) {
            this.schemaRegistry = new LruSchemaRegistryClient(config.getSchemaRegistryUrls(), config.getMaxCacheSize(), config.getMaxRetries(), config.getRetriesWaitMs(), config.originalsWithPrefix("confluent."), null, config.getMaxIdQueryRange(), config.getBadIdCacheTtl(), config.getBadSchemaCacheTtl());
        }
        this.keyStrategy = config.keySubjectNameStrategy();
        this.valueStrategy = config.valueSubjectNameStrategy();
        this.validateKey = config.validateKey();
        this.validateValue = config.validateValue();
        if (!this.validateKey && !this.validateValue) {
            throw new IllegalArgumentException("Neither key or value validation is enabled");
        }
        this.loggingSampler = RecordSchemaValidator.perMinuteSampler(config.getSamplesPerMin());
    }

    public RecordInterceptor.RecordInterceptorResponse onAppend(TopicPartition tp, Record record) {
        RecordSchemaValidator.incrementSensor(this.allRecordsSensor);
        RecordInterceptor.RecordInterceptorResponse response = this.getRecordInterceptorResponse(tp, record);
        if (response != RecordInterceptor.RecordInterceptorResponse.ACCEPT) {
            RecordSchemaValidator.incrementSensor(this.rejectedRecordsSensor);
        }
        return response;
    }

    private RecordInterceptor.RecordInterceptorResponse getRecordInterceptorResponse(TopicPartition tp, Record record) {
        try {
            if (this.validateKey && !this.validate(this.keyStrategy, tp, record.key(), true)) {
                return RecordInterceptor.RecordInterceptorResponse.REJECT;
            }
            if (this.validateValue && !this.validate(this.valueStrategy, tp, record.value(), false)) {
                return RecordInterceptor.RecordInterceptorResponse.REJECT;
            }
            return RecordInterceptor.RecordInterceptorResponse.ACCEPT;
        }
        catch (IOException e) {
            RecordSchemaValidator.incrementSensor(this.connectionErrorsSensor);
            if (this.loggingSampler.maybeSample(System.nanoTime()).shouldLog()) {
                this.log.error("Invalid record due to exception", (Throwable)e);
            }
            return RecordInterceptor.RecordInterceptorResponse.REJECT;
        }
        catch (Exception e) {
            if (this.loggingSampler.maybeSample(System.nanoTime()).shouldLog()) {
                this.log.error("Invalid record due to exception", (Throwable)e);
            }
            return RecordInterceptor.RecordInterceptorResponse.REJECT;
        }
    }

    private boolean validate(SubjectNameStrategy strategy, TopicPartition tp, ByteBuffer payload, boolean isKey) throws IOException {
        block9: {
            try {
                if (strategy == null) {
                    return true;
                }
                int id = this.getSchemaId(payload);
                if (id <= 0) {
                    return id == 0;
                }
                return this.validateSchemaID(strategy, tp, id, isKey);
            }
            catch (RestClientException e) {
                LogAction action = this.loggingSampler.maybeSample(System.nanoTime());
                int status = e.getStatus();
                if (status >= 500) {
                    RecordSchemaValidator.incrementSensor(this.serverErrorsSensor);
                    if (action.shouldLog()) {
                        this.log.warn("Invalid record due to REST server error", (Throwable)e);
                    }
                } else {
                    RecordSchemaValidator.incrementSensor(this.clientErrorsSensor);
                    if (action.shouldLog()) {
                        this.log.info("Invalid record due to REST client error", (Throwable)e);
                    }
                }
            }
            catch (BufferUnderflowException e) {
                LogAction action = this.loggingSampler.maybeSample(System.nanoTime());
                if (!action.shouldLog()) break block9;
                this.log.info("Invalid record due to insufficient bytes to read an integer for schema ID");
            }
        }
        return false;
    }

    protected int getSchemaId(ByteBuffer payload) {
        if (payload == null) {
            return 0;
        }
        ByteBuffer buffer = RecordSchemaValidator.getByteBuffer(payload);
        if (buffer == null) {
            if (this.loggingSampler.maybeSample(System.nanoTime()).shouldLog()) {
                this.log.info("Invalid record due to missing magic byte");
            }
            return Integer.MIN_VALUE;
        }
        return buffer.getInt();
    }

    protected boolean validateSchemaID(SubjectNameStrategy strategy, TopicPartition tp, int id, boolean isKey) throws IOException, RestClientException {
        int savedId;
        boolean validated;
        String subject = null;
        if (!strategy.usesSchema()) {
            subject = strategy.subjectName(tp.topic(), isKey, null);
        }
        ParsedSchema schema = this.schemaRegistry.getSchemaBySubjectAndId(subject, id);
        if (subject == null) {
            subject = strategy.subjectName(tp.topic(), isKey, schema);
        }
        boolean bl = validated = id == (savedId = this.schemaRegistry.getId(subject, schema));
        if (!validated && this.loggingSampler.maybeSample(System.nanoTime()).shouldLog()) {
            this.log.info("Invalid record due to id {} not matching existing id {}", (Object)id, (Object)savedId);
        }
        return validated;
    }

    private static ByteBuffer getByteBuffer(ByteBuffer payload) {
        if (payload.get() != 0) {
            return null;
        }
        return payload;
    }

    private static void incrementSensor(AtomicReference<Sensor> atomicSensor) {
        Sensor sensor = atomicSensor.get();
        if (sensor != null) {
            sensor.record();
        }
    }

    public void close() {
        if (this.schemaRegistry != null) {
            this.schemaRegistry.reset();
            try {
                this.schemaRegistry.close();
            }
            catch (IOException e) {
                throw new RuntimeException("Exception while closing schema registry client", e);
            }
        }
    }

    private static TimeBasedSampler perMinuteSampler(long samplesPerMin) {
        if (samplesPerMin < 0L) {
            return new NoOpSampler();
        }
        if (samplesPerMin == 0L) {
            return new AllSampler();
        }
        return new ConcurrentSampler((TimeBasedSampler)new AdaptiveSampler(true, samplesPerMin));
    }

    private static class ConcurrentSampler
    implements TimeBasedSampler {
        private final TimeBasedSampler sampler;

        public ConcurrentSampler(TimeBasedSampler sampler) {
            this.sampler = sampler;
        }

        public synchronized LogAction maybeSample(long requestTimeNanos) {
            return this.sampler.maybeSample(requestTimeNanos);
        }
    }

    private static class AllSampler
    implements TimeBasedSampler {
        private AllSampler() {
        }

        public LogAction maybeSample(long requestTimeNanos) {
            return LogAction.LOGGED;
        }
    }
}

