/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.model;

import com.linkedin.cruisecontrol.monitor.sampling.aggregator.AggregatedMetricValues;
import com.linkedin.kafka.cruisecontrol.analyzer.AnalyzerUtils;
import com.linkedin.kafka.cruisecontrol.analyzer.BalancingConstraint;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityInfo;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.exception.OptimizationFailureException;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.Capacity;
import com.linkedin.kafka.cruisecontrol.model.Cell;
import com.linkedin.kafka.cruisecontrol.model.ClusterModelHelper;
import com.linkedin.kafka.cruisecontrol.model.ClusterModelStats;
import com.linkedin.kafka.cruisecontrol.model.Disk;
import com.linkedin.kafka.cruisecontrol.model.Host;
import com.linkedin.kafka.cruisecontrol.model.Load;
import com.linkedin.kafka.cruisecontrol.model.Partition;
import com.linkedin.kafka.cruisecontrol.model.Rack;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.model.ReplicaPlacementInfo;
import com.linkedin.kafka.cruisecontrol.model.ResourceStats;
import com.linkedin.kafka.cruisecontrol.model.Tenant;
import com.linkedin.kafka.cruisecontrol.model.Utilization;
import com.linkedin.kafka.cruisecontrol.monitor.BrokerStats;
import com.linkedin.kafka.cruisecontrol.monitor.ModelGeneration;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.common.CellLoad;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.message.DescribeCellsResponseData;
import org.apache.kafka.common.message.DescribeTenantsResponseData;
import org.apache.kafka.metadata.TopicPlacement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterModel
implements Serializable,
ResourceStats {
    private static final Logger LOG = LoggerFactory.getLogger(ClusterModel.class);
    private static final long serialVersionUID = -6840253566423285966L;
    private static final Broker GENESIS_BROKER = Broker.createGenesisBroker();
    private static final double UNAVAILABLE_BROKER_UTILIZATION = 1.0;
    private final ModelGeneration generation;
    private final Map<String, Rack> racksById;
    private final Map<Integer, Rack> brokerIdToRack;
    private final Map<Integer, Cell> cellIdToCell;
    private final Cell deadBrokersCell;
    private final Map<String, Tenant> tenantsById;
    private final Map<TopicPartition, Partition> partitionsByTopicPartition;
    private final Set<Replica> selfHealingEligibleReplicas;
    private final SortedSet<Broker> newBrokers;
    private final SortedSet<Broker> brokersWithBadDisks;
    private final Set<Broker> aliveBrokers;
    private final SortedSet<Broker> deadBrokers;
    private final Set<Broker> ignoredBrokers;
    private final SortedSet<Broker> brokers;
    private final Set<TopicPartition> reassigningPartitions;
    private final double monitoredPartitionsRatio;
    private final Capacity capacity;
    private final Load load;
    private final Utilization utilization;
    private int maxReplicationFactor;
    private final Map<String, Integer> replicationFactorByTopic;
    private final Map<Integer, Load> potentialLeadershipLoadByBrokerId;
    private final Map<Integer, Utilization> potentialLeadershipUtilizationByBrokerId;
    private int unknownHostId;
    private final Map<Integer, String> capacityEstimationInfoByBrokerId;
    private Optional<Map<String, TopicPlacement>> topicPlacements;
    private Set<Integer> activeBrokerReplicaExclusions = new HashSet<Integer>();
    private boolean isCellEnabled;

    public ClusterModel(ModelGeneration generation, double monitoredPartitionsRatio) {
        this.generation = generation;
        this.racksById = new HashMap<String, Rack>();
        this.brokerIdToRack = new HashMap<Integer, Rack>();
        this.cellIdToCell = new HashMap<Integer, Cell>();
        this.deadBrokersCell = Cell.createDeadCell();
        this.tenantsById = new HashMap<String, Tenant>();
        this.partitionsByTopicPartition = new HashMap<TopicPartition, Partition>();
        this.selfHealingEligibleReplicas = new HashSet<Replica>();
        this.newBrokers = new TreeSet<Broker>();
        this.brokersWithBadDisks = new TreeSet<Broker>();
        this.aliveBrokers = new HashSet<Broker>();
        this.brokers = new TreeSet<Broker>();
        this.deadBrokers = new TreeSet<Broker>();
        this.ignoredBrokers = new HashSet<Broker>();
        this.load = new Load();
        this.utilization = Utilization.from(this.load);
        this.capacity = Capacity.create();
        this.maxReplicationFactor = 1;
        this.replicationFactorByTopic = new HashMap<String, Integer>();
        this.potentialLeadershipLoadByBrokerId = new HashMap<Integer, Load>();
        this.potentialLeadershipUtilizationByBrokerId = new HashMap<Integer, Utilization>();
        this.monitoredPartitionsRatio = monitoredPartitionsRatio;
        this.unknownHostId = 0;
        this.capacityEstimationInfoByBrokerId = new HashMap<Integer, String>();
        this.topicPlacements = Optional.empty();
        this.reassigningPartitions = new HashSet<TopicPartition>();
    }

    public ModelGeneration generation() {
        return this.generation;
    }

    public double monitoredPartitionsRatio() {
        return this.monitoredPartitionsRatio;
    }

    public ClusterModelStats getClusterStats(BalancingConstraint balancingConstraint) {
        return new ClusterModelStats().populate(this, balancingConstraint);
    }

    public Set<String> aliveRackIds() {
        return this.racksById.values().stream().filter(Rack::isRackAlive).map(Rack::id).collect(Collectors.toSet());
    }

    public Rack rack(String rackId) {
        return this.racksById.get(rackId);
    }

    public Cell cell(int cellId) {
        return Cell.isDeadCellId(cellId) ? this.deadBrokersCell : this.cellIdToCell.get(cellId);
    }

    public Optional<Tenant> tenant(String tenantId) {
        return Optional.ofNullable(this.tenantsById.get(tenantId));
    }

    public Collection<Cell> cells() {
        return Collections.unmodifiableCollection(this.cellIdToCell.values());
    }

    public void isCellEnabled(boolean isCellEnabled) {
        this.isCellEnabled = isCellEnabled;
    }

    public boolean isCellEnabled() {
        return this.isCellEnabled;
    }

    public boolean skipCellBalancing() {
        return !this.isCellEnabled || this.cellIdToCell.size() <= 1;
    }

    public Map<TopicPartition, List<ReplicaPlacementInfo>> getReplicaDistribution() {
        HashMap<TopicPartition, List<ReplicaPlacementInfo>> replicaDistribution = new HashMap<TopicPartition, List<ReplicaPlacementInfo>>(this.partitionsByTopicPartition.size());
        for (Map.Entry<TopicPartition, Partition> entry : this.partitionsByTopicPartition.entrySet()) {
            TopicPartition tp = entry.getKey();
            Partition partition = entry.getValue();
            List replicaPlacementInfos = partition.replicas().stream().map(r -> r.disk() == null ? new ReplicaPlacementInfo(r.broker().id()) : new ReplicaPlacementInfo(r.broker().id(), r.disk().logDir())).collect(Collectors.toList());
            replicaDistribution.put(tp, replicaPlacementInfos);
        }
        return replicaDistribution;
    }

    public Map<TopicPartition, ReplicaPlacementInfo> getLeaderDistribution() {
        HashMap<TopicPartition, ReplicaPlacementInfo> leaders = new HashMap<TopicPartition, ReplicaPlacementInfo>(this.partitionsByTopicPartition.size());
        for (Map.Entry<TopicPartition, Partition> entry : this.partitionsByTopicPartition.entrySet()) {
            Replica leaderReplica = entry.getValue().leader();
            if (leaderReplica.disk() == null) {
                leaders.put(entry.getKey(), new ReplicaPlacementInfo(leaderReplica.broker().id()));
                continue;
            }
            leaders.put(entry.getKey(), new ReplicaPlacementInfo(leaderReplica.broker().id(), leaderReplica.disk().logDir()));
        }
        return leaders;
    }

    public Map<TopicPartition, List<ReplicaPlacementInfo>> getObserverDistribution() {
        HashMap<TopicPartition, List<ReplicaPlacementInfo>> observerDistribution = new HashMap<TopicPartition, List<ReplicaPlacementInfo>>(this.partitionsByTopicPartition.size());
        for (Map.Entry<TopicPartition, Partition> entry : this.partitionsByTopicPartition.entrySet()) {
            TopicPartition tp = entry.getKey();
            Partition partition = entry.getValue();
            List replicaPlacementInfos = partition.replicas().stream().filter(Replica::isObserver).map(r -> r.disk() == null ? new ReplicaPlacementInfo(r.broker().id()) : new ReplicaPlacementInfo(r.broker().id(), r.disk().logDir())).collect(Collectors.toList());
            observerDistribution.put(tp, replicaPlacementInfos);
        }
        return observerDistribution;
    }

    public Set<Replica> selfHealingEligibleReplicas() {
        return this.selfHealingEligibleReplicas;
    }

    @Override
    public Load load() {
        return this.load;
    }

    @Override
    public Utilization utilization() {
        return this.utilization;
    }

    public Load potentialLeadershipLoadFor(Integer brokerId) {
        return this.potentialLeadershipLoadByBrokerId.get(brokerId);
    }

    public Utilization potentialLeadershipUtilizationFor(Integer brokerId) {
        return this.potentialLeadershipUtilizationByBrokerId.get(brokerId);
    }

    public int maxReplicationFactor() {
        return this.maxReplicationFactor;
    }

    public Map<String, Integer> replicationFactorByTopic() {
        return this.replicationFactorByTopic;
    }

    public Partition partition(TopicPartition tp) {
        return this.partitionsByTopicPartition.get(tp);
    }

    Map<TopicPartition, Partition> partitionsByTopicPartition() {
        return this.partitionsByTopicPartition;
    }

    public SortedMap<String, List<Partition>> getPartitionsByTopic() {
        TreeMap<String, List<Partition>> partitionsByTopic = new TreeMap<String, List<Partition>>();
        for (String string : this.topics()) {
            partitionsByTopic.put(string, new ArrayList());
        }
        for (Map.Entry entry : this.partitionsByTopicPartition.entrySet()) {
            ((List)partitionsByTopic.get(((TopicPartition)entry.getKey()).topic())).add(entry.getValue());
        }
        return partitionsByTopic;
    }

    public Set<Replica> leaderReplicas() {
        return this.partitionsByTopicPartition.values().stream().map(Partition::leader).collect(Collectors.toSet());
    }

    private void updateBrokerLists(Broker broker) {
        switch (broker.strategy()) {
            case DEAD: {
                this.aliveBrokers.remove(broker);
                this.deadBrokers.add(broker);
                this.brokersWithBadDisks.remove(broker);
                this.ignoredBrokers.remove(broker);
                break;
            }
            case NEW: {
                this.newBrokers.add(broker);
            }
            case ALIVE: {
                this.aliveBrokers.add(broker);
                this.deadBrokers.remove(broker);
                this.brokersWithBadDisks.remove(broker);
                this.ignoredBrokers.remove(broker);
                break;
            }
            case IGNORE: {
                this.deadBrokers.remove(broker);
                this.brokersWithBadDisks.remove(broker);
                this.ignoredBrokers.add(broker);
                this.aliveBrokers.add(broker);
                break;
            }
            case BAD_DISKS: {
                this.aliveBrokers.add(broker);
                this.deadBrokers.remove(broker);
                this.brokersWithBadDisks.add(broker);
                break;
            }
            default: {
                throw new IllegalArgumentException("Illegal broker strategy " + (Object)((Object)broker.strategy()) + " is provided.");
            }
        }
    }

    public void relocateReplica(TopicPartition tp, int brokerId, String destinationLogdir) {
        Replica replicaToMove = this.partitionsByTopicPartition.get(tp).replica(brokerId);
        replicaToMove.broker().moveReplicaBetweenDisks(tp, replicaToMove.disk().logDir(), destinationLogdir);
    }

    public void relocateReplica(TopicPartition tp, int sourceBrokerId, int destinationBrokerId) {
        Replica replica = this.removeReplica(sourceBrokerId, tp);
        if (replica == null) {
            throw new IllegalArgumentException("Replica is not in the cluster.");
        }
        Broker destinationBroker = this.broker(destinationBrokerId);
        replica.setBroker(destinationBroker);
        replica.broker().rack().addReplica(replica);
        replica.broker().cell().addReplica(replica);
        this.load.addLoad(replica.load());
        this.utilization.addLoad(replica.broker().strategy(), replica.load());
        Load leaderLoad = this.partition(tp).leader().load();
        this.potentialLeadershipLoadByBrokerId.get(destinationBrokerId).addLoad(leaderLoad);
        this.potentialLeadershipUtilizationByBrokerId.get(destinationBrokerId).addLoad(destinationBroker.strategy(), leaderLoad);
    }

    public boolean relocateLeadership(TopicPartition tp, int sourceBrokerId, int destinationBrokerId) {
        Replica sourceReplica = this.partitionsByTopicPartition.get(tp).replica(sourceBrokerId);
        if (!sourceReplica.isLeader()) {
            return false;
        }
        Replica destinationReplica = this.partitionsByTopicPartition.get(tp).replica(destinationBrokerId);
        if (destinationReplica.isLeader()) {
            throw new IllegalArgumentException("Cannot relocate leadership of partition " + tp + "from broker " + sourceBrokerId + " to broker " + destinationBrokerId + " because the destination replica is a leader.");
        }
        Broker sourceBroker = this.broker(sourceBrokerId);
        Broker destinationBroker = this.broker(destinationBrokerId);
        AggregatedMetricValues leadershipLoadDelta = this.calculateLeadershipLoadDelta(tp, sourceBroker, destinationBroker, Resource.CPU, Resource.NW_OUT, Resource.PRODUCE_IN, Resource.CONSUME_OUT);
        Rack sourceRack = sourceBroker.rack();
        sourceRack.makeFollower(sourceBrokerId, tp, leadershipLoadDelta);
        Cell sourceCell = sourceBroker.cell();
        sourceCell.makeFollower(sourceBroker.strategy(), leadershipLoadDelta);
        this.utilization.subtractLoad(sourceBroker.strategy(), leadershipLoadDelta);
        Rack destinationRack = destinationBroker.rack();
        destinationRack.makeLeader(destinationBrokerId, tp, leadershipLoadDelta);
        Cell destinationCell = destinationBroker.cell();
        destinationCell.makeLeader(destinationBroker.strategy(), leadershipLoadDelta);
        this.utilization.addLoad(destinationBroker.strategy(), leadershipLoadDelta);
        Partition partition = this.partitionsByTopicPartition.get(tp);
        partition.relocateLeadership(destinationReplica);
        return true;
    }

    private AggregatedMetricValues calculateLeadershipLoadDelta(TopicPartition tp, Broker sourceBroker, Broker destinationBroker, Resource ... resources) {
        AggregatedMetricValues leadershipLoadDelta = new AggregatedMetricValues();
        Replica sourceReplica = sourceBroker.replica(tp);
        Replica destinationReplica = destinationBroker.replica(tp);
        for (Resource resource : resources) {
            AggregatedMetricValues sourceReplicaMetricValues = sourceReplica.load().loadFor(resource, true);
            leadershipLoadDelta.add(sourceReplicaMetricValues);
            AggregatedMetricValues destinationReplicaMetricValues = destinationReplica.load().loadFor(resource, true);
            leadershipLoadDelta.subtract(destinationReplicaMetricValues);
        }
        return leadershipLoadDelta;
    }

    public boolean changeObservership(TopicPartition tp, int replicaId) {
        Replica replica = this.partitionsByTopicPartition.get(tp).replica(replicaId);
        boolean toBeObserver = !replica.isObserver();
        replica.setObservership(toBeObserver);
        return toBeObserver;
    }

    public Set<Broker> aliveBrokers() {
        return this.aliveBrokers;
    }

    public SortedSet<Broker> eligibleSourceBrokers() {
        TreeSet<Broker> allBrokers = new TreeSet<Broker>(this.brokers);
        allBrokers.removeAll(this.ignoredBrokers);
        return allBrokers;
    }

    public SortedSet<Broker> eligibleDestinationBrokers() {
        TreeSet<Broker> allBrokers = new TreeSet<Broker>(this.brokers);
        allBrokers.removeAll(this.ignoredBrokers);
        allBrokers.removeAll(this.deadBrokers);
        return allBrokers;
    }

    public SortedSet<Broker> eligibleSourceAndDestinationBrokers() {
        SortedSet<Broker> result = this.eligibleDestinationBrokers();
        result.retainAll(this.eligibleSourceBrokers());
        return result;
    }

    public SortedSet<Broker> eligibleSourceOrDestinationBrokers() {
        SortedSet<Broker> result = this.eligibleSourceBrokers();
        result.addAll(this.eligibleDestinationBrokers());
        return result;
    }

    public SortedSet<Broker> deadBrokers() {
        return new TreeSet<Broker>(this.deadBrokers);
    }

    public Set<Broker> ignoredBrokers() {
        return Collections.unmodifiableSet(this.ignoredBrokers);
    }

    public SortedSet<Broker> brokenBrokers() {
        TreeSet<Broker> brokenBrokers = new TreeSet<Broker>(this.deadBrokers);
        brokenBrokers.addAll(this.brokersWithBadDisks());
        return Collections.unmodifiableSortedSet(brokenBrokers);
    }

    public Map<Integer, String> capacityEstimationInfoByBrokerId() {
        return Collections.unmodifiableMap(this.capacityEstimationInfoByBrokerId);
    }

    public SortedSet<Broker> newBrokers() {
        return this.newBrokers;
    }

    public SortedSet<Broker> brokersWithBadDisks() {
        return Collections.unmodifiableSortedSet(this.brokersWithBadDisks);
    }

    public Set<Broker> brokersHavingOfflineReplicasOnBadDisks() {
        HashSet<Broker> brokersWithOfflineReplicasOnBadDisks = new HashSet<Broker>();
        for (Broker brokerWithBadDisks : this.brokersWithBadDisks) {
            if (brokerWithBadDisks.currentOfflineReplicas().isEmpty()) continue;
            brokersWithOfflineReplicasOnBadDisks.add(brokerWithBadDisks);
        }
        return brokersWithOfflineReplicasOnBadDisks;
    }

    public Set<Broker> aliveBrokersMatchingAttributes(Map<String, String> attributes) {
        return this.aliveBrokers().stream().filter(broker -> broker.attributes().equals(attributes)).collect(Collectors.toSet());
    }

    public boolean clusterHasEligibleDestinationBrokers() {
        return this.racksById.values().stream().anyMatch(Rack::isEligibleDestination);
    }

    public Replica removeReplica(int brokerId, TopicPartition tp) {
        for (Rack rack : this.racksById.values()) {
            Replica removedReplica = rack.removeReplica(brokerId, tp);
            if (removedReplica == null) continue;
            if (removedReplica.broker().id() != brokerId) {
                throw new IllegalStateException(String.format("Removed replica %s is not on the broker %d", removedReplica, brokerId));
            }
            this.load.subtractLoad(removedReplica.load());
            this.utilization.subtractLoad(removedReplica.broker().strategy(), removedReplica.load());
            removedReplica.broker().cell().removeReplica(removedReplica);
            Load leaderLoad = this.partition(tp).leader().load();
            this.potentialLeadershipLoadByBrokerId.get(brokerId).subtractLoad(leaderLoad);
            this.potentialLeadershipUtilizationByBrokerId.get(brokerId).subtractLoad(removedReplica.broker().strategy(), leaderLoad);
            return removedReplica;
        }
        return null;
    }

    public SortedSet<Broker> allBrokers() {
        return new TreeSet<Broker>(this.brokers);
    }

    public Set<Broker> brokersEligibleForRebalancing() {
        return this.brokers.stream().filter(broker -> broker.isEligibleSource() || broker.isEligibleDestination()).collect(Collectors.toSet());
    }

    public SortedSet<Broker> allBrokersWithStateOtherThan(Broker.Strategy stateToSkip) {
        return this.brokers.stream().filter(broker -> !broker.strategy().equals((Object)stateToSkip)).collect(Collectors.toCollection(TreeSet::new));
    }

    public Set<Broker> brokersOfStatesMatchingAttributes(Collection<Broker> brokerPool, EnumSet<Broker.Strategy> states, Map<String, String> attributes) {
        return brokerPool.stream().filter(broker -> states.contains((Object)broker.strategy()) && broker.attributes().equals(attributes)).collect(Collectors.toSet());
    }

    public Set<Broker> brokersNotOfStatesMatchingAttributes(Collection<Broker> brokerPool, EnumSet<Broker.Strategy> states, Map<String, String> attributes) {
        return brokerPool.stream().filter(broker -> !states.contains((Object)broker.strategy()) && broker.attributes().equals(attributes)).collect(Collectors.toSet());
    }

    public Broker broker(int brokerId) {
        Rack rack = this.brokerIdToRack.get(brokerId);
        return rack == null ? null : rack.broker(brokerId);
    }

    public void trackSortedReplicas(String sortName, Function<Replica, Double> scoreFunction) {
        this.trackSortedReplicas(sortName, null, scoreFunction);
    }

    public void trackSortedReplicas(String sortName, Function<Replica, Integer> priorityFunc, Function<Replica, Double> scoreFunc) {
        this.trackSortedReplicas(sortName, null, priorityFunc, scoreFunc);
    }

    public void trackSortedReplicas(String sortName, Function<Replica, Boolean> selectionFunc, Function<Replica, Integer> priorityFunc, Function<Replica, Double> scoreFunc) {
        this.trackSortedReplicas(this.brokers, sortName, selectionFunc, priorityFunc, scoreFunc);
    }

    public void trackSortedReplicas(Collection<Broker> brokersToTrack, String sortName, Function<Replica, Boolean> selectionFunc, Function<Replica, Integer> priorityFunc, Function<Replica, Double> scoreFunc) {
        brokersToTrack.forEach(b -> b.trackSortedReplicas(sortName, selectionFunc, priorityFunc, scoreFunc));
    }

    public void untrackSortedReplicas(String sortName) {
        this.brokers.forEach(b -> b.untrackSortedReplicas(sortName));
    }

    public int numTopicReplicas(String topic) {
        int numTopicReplicas = 0;
        for (Rack rack : this.racksById.values()) {
            numTopicReplicas += rack.numTopicReplicas(topic);
        }
        return numTopicReplicas;
    }

    public int numTopicReplicasOnEligibleSourceBrokers(String topic) {
        return this.eligibleSourceBrokers().stream().mapToInt(b -> b.replicasOfTopicInBroker(topic).size()).sum();
    }

    public int numLeaderReplicas() {
        return this.partitionsByTopicPartition.size();
    }

    public int numLeaderReplicasOnEligibleSourceBrokers() {
        return this.eligibleSourceBrokers().stream().mapToInt(Broker::numLeaderReplicas).sum();
    }

    public int numReplicas() {
        return this.partitionsByTopicPartition.values().stream().mapToInt(p -> p.replicas().size()).sum();
    }

    public int numReplicasOnEligibleSourceBrokers() {
        return this.eligibleSourceBrokers().stream().mapToInt(broker -> broker.replicas().size()).sum();
    }

    public Set<String> topics() {
        HashSet<String> topics = new HashSet<String>();
        for (Rack rack : this.racksById.values()) {
            topics.addAll(rack.topics());
        }
        return topics;
    }

    public Map<Integer, Cell> cellsById() {
        return Collections.unmodifiableMap(this.cellIdToCell);
    }

    public Map<String, Tenant> tenantsById() {
        return Collections.unmodifiableMap(this.tenantsById);
    }

    @Override
    public Capacity capacity() {
        return this.capacity;
    }

    public void setReplicaLoad(String rackId, int brokerId, TopicPartition tp, AggregatedMetricValues metricValues, List<Long> windows) {
        Broker broker = this.broker(brokerId);
        if (!broker.replica(tp).load().isEmpty()) {
            throw new IllegalStateException(String.format("The load for %s on broker %d, rack %s already has metric values.", tp, brokerId, rackId));
        }
        Rack rack = this.rack(rackId);
        rack.setReplicaLoad(brokerId, tp, metricValues, windows);
        broker.cell().replicaLoad(broker, metricValues, windows);
        this.load.addMetricValues(metricValues, windows);
        this.utilization.addMetricValues(broker.strategy(), metricValues, windows);
        Replica leader = this.partition(tp).leader();
        if (leader != null && leader.broker().id() == brokerId) {
            for (Replica replica : this.partition(tp).replicas()) {
                int replicaBrokerId = replica.broker().id();
                this.potentialLeadershipLoadByBrokerId.get(replicaBrokerId).addMetricValues(metricValues, windows);
                this.potentialLeadershipUtilizationByBrokerId.get(replicaBrokerId).addMetricValues(replica.broker().strategy(), metricValues, windows);
            }
        }
    }

    public void handleDeadBroker(String rackId, int brokerId, BrokerCapacityInfo brokerCapacityInfo) {
        this.createRackIfAbsent(rackId);
        if (this.broker(brokerId) == null) {
            this.createBroker(rackId, this.deadBrokersCell.id(), String.format("UNKNOWN_HOST-%d", this.unknownHostId++), brokerId, brokerCapacityInfo, false, Broker.Strategy.DEAD);
        }
    }

    public Replica createReplica(String rackId, int brokerId, TopicPartition tp, int index, boolean isLeader) {
        return this.createReplica(rackId, brokerId, tp, index, isLeader, !this.broker(brokerId).isAlive(), null, false, false);
    }

    public Replica createReplica(String rackId, int brokerId, TopicPartition tp, int index, boolean isLeader, boolean isOffline, String logdir, boolean isFuture, boolean isObserver) {
        Replica replica;
        Broker broker = this.broker(brokerId);
        if (!isFuture) {
            Disk disk = null;
            if (logdir != null && (disk = broker.disk(logdir)) == null) {
                if (isOffline) {
                    disk = broker.addDeadDisk(logdir);
                } else {
                    throw new IllegalStateException("Missing disk information for disk " + logdir + " on broker " + this);
                }
            }
            replica = new Replica(tp, broker, isLeader, isOffline, disk, isObserver);
        } else {
            replica = new Replica(tp, GENESIS_BROKER, false);
            replica.setBroker(broker);
        }
        this.rack(rackId).addReplica(replica);
        if (!this.partitionsByTopicPartition.containsKey(tp)) {
            this.partitionsByTopicPartition.put(tp, new Partition(tp));
            this.replicationFactorByTopic.putIfAbsent(tp.topic(), 1);
        }
        if (broker.currentOfflineReplicas().contains(replica)) {
            this.selfHealingEligibleReplicas.add(replica);
            this.partitionsByTopicPartition.get(replica.topicPartition()).addBadDiskBroker(broker);
        }
        Partition partition = this.partitionsByTopicPartition.get(tp);
        if (replica.isLeader()) {
            partition.addLeader(replica, index);
            return replica;
        }
        partition.addFollower(replica, index);
        Replica leaderReplica = this.partition(tp).leader();
        if (leaderReplica != null) {
            Load leaderLoad = leaderReplica.load();
            this.potentialLeadershipLoadByBrokerId.get(brokerId).addMetricValues(leaderLoad.loadByWindows(), leaderLoad.windows());
            this.potentialLeadershipUtilizationByBrokerId.get(brokerId).addMetricValues(broker.strategy(), leaderLoad.loadByWindows(), leaderLoad.windows());
        }
        int replicationFactor = Math.max(this.replicationFactorByTopic.get(tp.topic()), partition.followers().size() + 1);
        this.replicationFactorByTopic.put(tp.topic(), replicationFactor);
        this.maxReplicationFactor = Math.max(this.maxReplicationFactor, replicationFactor);
        return replica;
    }

    public void deleteReplica(TopicPartition topicPartition, int brokerId) {
        int currentReplicaCount = this.partitionsByTopicPartition.get(topicPartition).replicas().size();
        if (currentReplicaCount < 2) {
            throw new IllegalStateException(String.format("Unable to delete replica for topic partition %s since it only has %d replicas.", topicPartition, currentReplicaCount));
        }
        this.removeReplica(brokerId, topicPartition);
        Partition partition = this.partitionsByTopicPartition.get(topicPartition);
        partition.deleteReplica(brokerId);
        this.replicationFactorByTopic.put(topicPartition.topic(), partition.replicas().size());
    }

    public void refreshClusterMaxReplicationFactor() {
        this.maxReplicationFactor = this.replicationFactorByTopic.values().stream().max(Integer::compareTo).orElse(0);
    }

    public Broker createBroker(String rackId, int cellId, String host, int brokerId, BrokerCapacityInfo brokerCapacityInfo, boolean populateReplicaPlacementInfo, Broker.Strategy strategy) {
        this.potentialLeadershipLoadByBrokerId.putIfAbsent(brokerId, new Load());
        this.potentialLeadershipUtilizationByBrokerId.putIfAbsent(brokerId, Utilization.from(this.potentialLeadershipLoadByBrokerId.get(brokerId)));
        Rack rack = this.rack(rackId);
        this.brokerIdToRack.put(brokerId, rack);
        Cell cell = this.cell(cellId);
        if (brokerCapacityInfo.isEstimated()) {
            this.capacityEstimationInfoByBrokerId.put(brokerId, brokerCapacityInfo.estimationInfo());
        }
        Broker broker = rack.createBroker(brokerId, host, cell, brokerCapacityInfo, populateReplicaPlacementInfo, strategy);
        this.aliveBrokers.add(broker);
        this.brokers.add(broker);
        if (broker.isAlive()) {
            this.capacity.addCapacity(Capacity.from(broker, brokerCapacityInfo));
        }
        this.updateBrokerLists(broker);
        return broker;
    }

    public Broker createBroker(String rackId, String host, int brokerId, BrokerCapacityInfo brokerCapacityInfo, boolean populateReplicaPlacementInfo, Broker.Strategy strategy) {
        return this.createBroker(rackId, -1, host, brokerId, brokerCapacityInfo, populateReplicaPlacementInfo, strategy);
    }

    public Rack createRackIfAbsent(String rackId) {
        if (!this.racksById.containsKey(rackId)) {
            Rack rack = new Rack(rackId);
            this.racksById.put(rackId, rack);
        }
        return this.racksById.get(rackId);
    }

    public Cell createCellIfAbsent(Optional<Integer> cellId) {
        int cellIdToUse = cellId.orElse(-1);
        if (!this.cellIdToCell.containsKey(cellIdToUse)) {
            Cell cell = new Cell(cellIdToUse);
            this.cellIdToCell.put(cellIdToUse, cell);
        }
        return this.cellIdToCell.get(cellIdToUse);
    }

    public Cell createCell(DescribeCellsResponseData.Cell cellResponseData) {
        if (cellResponseData == null) {
            LOG.info("Create a cell with default cell id and unknown state since no cell metadata is available.");
            return this.createCellIfAbsent(Optional.empty());
        }
        int cellIdToUse = cellResponseData.cellId();
        if (!this.cellIdToCell.containsKey(cellIdToUse)) {
            Cell cell = new Cell(cellResponseData);
            this.cellIdToCell.put(cellIdToUse, cell);
        }
        return this.cellIdToCell.get(cellIdToUse);
    }

    public void createTenant(DescribeTenantsResponseData.TenantDescription tenantDescription) {
        this.tenantsById.put(tenantDescription.tenantId(), new Tenant(tenantDescription));
    }

    public void addCell(Cell cell) {
        if (!this.cellIdToCell.containsKey(cell.id())) {
            this.cellIdToCell.put(cell.id(), cell);
        }
    }

    public boolean createOrDeleteReplicas(Map<Short, Set<String>> topicsByReplicationFactor, Map<String, List<Integer>> brokersByRack) throws OptimizationFailureException {
        boolean needToRefreshClusterMaxReplicationFactor = false;
        boolean replicasAltered = false;
        SortedMap<String, List<Partition>> partitionsByTopic = this.getPartitionsByTopic();
        for (Map.Entry<Short, Set<String>> entry : topicsByReplicationFactor.entrySet()) {
            short replicationFactor = entry.getKey();
            Set<String> topics = entry.getValue();
            for (String topic : topics) {
                for (Partition partition : (List)partitionsByTopic.get(topic)) {
                    if (partition.replicas().size() == replicationFactor) continue;
                    if (partition.replicas().size() < replicationFactor) {
                        this.addReplicasUpToReplicationFactor(partition, replicationFactor, brokersByRack);
                    } else {
                        Set newReplicas = partition.replicas().stream().filter(r -> r.broker().strategy() == Broker.Strategy.IGNORE).map(r -> r.broker().id()).collect(Collectors.toSet());
                        newReplicas.add(partition.leader().broker().id());
                        if (newReplicas.size() > replicationFactor) {
                            throw new OptimizationFailureException(String.format("Required RF for partition %s is %d, but we currently have %d replicas which can't be removed.", partition.topicPartition().toString(), replicationFactor, newReplicas.size()));
                        }
                        ArrayList<Replica> replicas = new ArrayList<Replica>(partition.replicas());
                        for (Replica replica : replicas) {
                            int brokerId = replica.broker().id();
                            if (newReplicas.contains(brokerId)) continue;
                            if (newReplicas.size() < replicationFactor) {
                                newReplicas.add(brokerId);
                                continue;
                            }
                            this.deleteReplica(partition.topicPartition(), brokerId);
                            needToRefreshClusterMaxReplicationFactor = true;
                        }
                    }
                    replicasAltered = true;
                }
            }
        }
        if (needToRefreshClusterMaxReplicationFactor) {
            this.refreshClusterMaxReplicationFactor();
        }
        return replicasAltered;
    }

    public boolean updateReplicationFactor(Map<Short, Set<String>> topicsByReplicationFactor, Set<Integer> brokersExcludedForReplicaMovement) throws OptimizationFailureException {
        Map<Integer, String> includedBrokerIdToRack = this.brokerIdToRack.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((Rack)entry.getValue()).id()));
        includedBrokerIdToRack.keySet().removeAll(brokersExcludedForReplicaMovement);
        Map<String, List<Integer>> rackToBrokerId = includedBrokerIdToRack.entrySet().stream().collect(Collectors.groupingBy(Map.Entry::getValue, Collectors.mapping(Map.Entry::getKey, Collectors.toList())));
        return this.createOrDeleteReplicas(topicsByReplicationFactor, rackToBrokerId);
    }

    public List<Broker> brokersUnderThreshold(Collection<Broker> brokers, Resource resource, double utilizationThreshold) {
        CapacityLimitProvider provider = broker -> ClusterModelHelper.resourceStatsFor(broker, resource).capacity().totalCapacityFor(resource) * utilizationThreshold;
        return this.brokersUnderCapacityLimit(brokers, resource, provider);
    }

    public List<Broker> brokersUnderCapacityLimit(Collection<Broker> brokers, Resource resource, CapacityLimitProvider capacityLimitProvider) {
        ArrayList<Broker> brokersUnderThreshold = new ArrayList<Broker>();
        for (Broker broker : brokers) {
            double capacityLimit = capacityLimitProvider.capacityLimit(broker);
            double utilization = ClusterModelHelper.resourceStatsFor(broker, resource).load().expectedUtilizationFor(resource);
            if (utilization >= capacityLimit) continue;
            brokersUnderThreshold.add(broker);
        }
        return brokersUnderThreshold;
    }

    public List<Broker> brokersOverThreshold(Collection<Broker> originalBrokers, Resource resource, double utilizationThreshold) {
        ArrayList<Broker> brokersOverThreshold = new ArrayList<Broker>();
        for (Broker broker : originalBrokers) {
            ResourceStats resourceStats = ClusterModelHelper.resourceStatsFor(broker, resource);
            double capacityLimit = resourceStats.capacity().totalCapacityFor(resource) * utilizationThreshold;
            double utilization = resourceStats.load().expectedUtilizationFor(resource);
            if (utilization <= capacityLimit) continue;
            brokersOverThreshold.add(broker);
        }
        return brokersOverThreshold;
    }

    public void sanityCheck() {
        HashMap<String, Integer> errorMsgAndNumWindows = new HashMap<String, Integer>();
        int expectedNumWindows = this.load.numWindows();
        for (Map.Entry<Integer, Load> entry : this.potentialLeadershipLoadByBrokerId.entrySet()) {
            int n = entry.getKey();
            Load load = entry.getValue();
            if (load.numWindows() == expectedNumWindows || this.broker(n).replicas().size() == 0) continue;
            errorMsgAndNumWindows.put(String.format("Leadership(%d)", n), load.numWindows());
        }
        for (Rack rack : this.racksById.values()) {
            if (rack.load().numWindows() != expectedNumWindows && rack.replicas().size() != 0) {
                errorMsgAndNumWindows.put(String.format("Rack(%s)", rack.id()), rack.load().numWindows());
            }
            for (Host host : rack.hosts()) {
                if (host.load().numWindows() != expectedNumWindows && host.replicas().size() != 0) {
                    errorMsgAndNumWindows.put(String.format("Host(%s)", host.name()), host.load().numWindows());
                }
                for (Broker broker : rack.brokers()) {
                    if (broker.load().numWindows() != expectedNumWindows && broker.replicas().size() != 0) {
                        errorMsgAndNumWindows.put(String.format("Broker(%d)", broker.id()), broker.load().numWindows());
                    }
                    for (Replica replica : broker.replicas()) {
                        if (replica.load().numWindows() == expectedNumWindows) continue;
                        errorMsgAndNumWindows.put(String.format("Replica(%s-%d)", replica.topicPartition(), broker.id()), replica.load().numWindows());
                    }
                }
            }
        }
        StringBuilder exceptionMsg = new StringBuilder();
        for (Map.Entry entry : errorMsgAndNumWindows.entrySet()) {
            exceptionMsg.append(String.format("[%s: %d]%n", entry.getKey(), entry.getValue()));
        }
        if (exceptionMsg.length() > 0) {
            throw new IllegalArgumentException(String.format("Loads must have all have %d windows. Following loads violate this constraint with specified number of windows: %s", expectedNumWindows, exceptionMsg));
        }
        String string = "Inconsistent load distribution.";
        for (Broker broker : this.allBrokers()) {
            for (Resource resource : Resource.cachedValues()) {
                double sumOfReplicaUtilization = 0.0;
                for (Replica replica : broker.replicas()) {
                    sumOfReplicaUtilization += replica.load().expectedUtilizationFor(resource);
                }
                double brokerUtilization = broker.load().expectedUtilizationFor(resource);
                if (AnalyzerUtils.compare(sumOfReplicaUtilization, brokerUtilization, resource) == 0) continue;
                throw new IllegalArgumentException(String.format("%s Broker utilization for %s is different from the total replica utilization in the broker with id: %d. Sum of the replica utilization: %f, broker utilization: %f", new Object[]{string, resource, broker.id(), sumOfReplicaUtilization, brokerUtilization}));
            }
        }
        HashMap<Resource, Double> hashMap = new HashMap<Resource, Double>(Resource.cachedValues().size());
        for (Rack rack : this.racksById.values()) {
            HashMap<Resource, Double> sumOfHostUtilizationByResource = new HashMap<Resource, Double>(Resource.cachedValues().size());
            for (Host host : rack.hosts()) {
                for (Resource resource : Resource.cachedValues()) {
                    double sumOfBrokerUtilization = 0.0;
                    for (Broker broker : host.brokers()) {
                        sumOfBrokerUtilization += broker.load().expectedUtilizationFor(resource);
                    }
                    double hostUtilization = host.load().expectedUtilizationFor(resource);
                    if (AnalyzerUtils.compare(sumOfBrokerUtilization, hostUtilization, resource) != 0) {
                        throw new IllegalArgumentException(String.format("%s Host utilization for %s is different from the total broker utilization in the host : %s. Sum of the broker utilization: %f, host utilization: %f", new Object[]{string, resource, host.name(), sumOfBrokerUtilization, hostUtilization}));
                    }
                    sumOfHostUtilizationByResource.compute(resource, (k, v) -> (v == null ? 0.0 : v) + hostUtilization);
                }
            }
            for (Map.Entry entry : sumOfHostUtilizationByResource.entrySet()) {
                Resource resource = (Resource)((Object)entry.getKey());
                double sumOfHostsUtil = (Double)entry.getValue();
                double rackUtilization = rack.load().expectedUtilizationFor(resource);
                if (AnalyzerUtils.compare(rackUtilization, sumOfHostsUtil, resource) != 0) {
                    throw new IllegalArgumentException(String.format("%s Rack utilization for %s is different from the total host utilization in rack : %s. Sum of the host utilization: %f, rack utilization: %f", new Object[]{string, resource, rack.id(), sumOfHostsUtil, rackUtilization}));
                }
                hashMap.compute(resource, (k, v) -> (v == null ? 0.0 : v) + sumOfHostsUtil);
            }
        }
        for (Map.Entry entry : hashMap.entrySet()) {
            Resource resource;
            resource = (Resource)((Object)entry.getKey());
            double sumOfRackUtil = (Double)entry.getValue();
            double clusterUtilization = this.load.expectedUtilizationFor(resource);
            if (AnalyzerUtils.compare(this.load.expectedUtilizationFor(resource), sumOfRackUtil, resource) == 0) continue;
            throw new IllegalArgumentException(String.format("%s Cluster utilization for %s is different from the total rack utilization in the cluster. Sum of the rack utilization: %f, cluster utilization: %f", new Object[]{string, resource, sumOfRackUtil, clusterUtilization}));
        }
        for (Broker broker : this.allBrokers()) {
            double sumOfLeaderOfReplicaUtilization = 0.0;
            for (Replica replica : broker.replicas()) {
                sumOfLeaderOfReplicaUtilization += this.partition(replica.topicPartition()).leader().load().expectedUtilizationFor(Resource.NW_OUT);
            }
            double d = this.potentialLeadershipLoadByBrokerId.get(broker.id()).expectedUtilizationFor(Resource.NW_OUT);
            if (AnalyzerUtils.compare(sumOfLeaderOfReplicaUtilization, d, Resource.NW_OUT) != 0) {
                throw new IllegalArgumentException(String.format("%s Leadership utilization for %s is different from the total utilization leader of replicas in the broker with id: %d. Expected: %f Received: %f", new Object[]{string, Resource.NW_OUT, broker.id(), sumOfLeaderOfReplicaUtilization, d}));
            }
            for (Resource resource : Resource.cachedValues()) {
                double cachedLoad;
                double leaderSum;
                if (resource == Resource.CPU || AnalyzerUtils.compare(leaderSum = broker.leaderReplicas().stream().mapToDouble(r -> r.load().expectedUtilizationFor(resource)).sum(), cachedLoad = broker.leadershipLoadForNwResources().expectedUtilizationFor(resource), resource) == 0) continue;
                throw new IllegalArgumentException(String.format("%s Leadership load for resource %s is %f but recomputed sum is %f", new Object[]{string, resource, cachedLoad, leaderSum}));
            }
        }
    }

    public BrokerStats brokerStats(KafkaCruiseControlConfig config) {
        BrokerStats brokerStats = new BrokerStats(config);
        this.allBrokers().forEach(broker -> {
            double leaderBytesInRate = broker.leadershipLoadForNwResources().expectedUtilizationFor(Resource.NW_IN);
            double cpuUsagePercent = broker.load().expectedUtilizationFor(Resource.CPU) / broker.capacity().totalCapacityFor(Resource.CPU);
            brokerStats.addSingleBrokerStats(broker.host().name(), broker.id(), broker.strategy(), broker.replicas().isEmpty() ? 0.0 : broker.load().expectedUtilizationFor(Resource.DISK), cpuUsagePercent, leaderBytesInRate, broker.load().expectedUtilizationFor(Resource.NW_IN) - leaderBytesInRate, broker.load().expectedUtilizationFor(Resource.NW_OUT), this.potentialLeadershipLoadFor(broker.id()).expectedUtilizationFor(Resource.NW_OUT), broker.replicas().size(), broker.leaderReplicas().size(), this.capacityEstimationInfoByBrokerId.get(broker.id()) != null, broker.capacity().totalCapacityFor(Resource.DISK), broker.diskStats());
        });
        return brokerStats;
    }

    public void writeTo(OutputStream out) throws IOException {
        String cluster = String.format("<Cluster maxPartitionReplicationFactor=\"%d\">%n", this.maxReplicationFactor);
        out.write(cluster.getBytes(StandardCharsets.UTF_8));
        for (Rack rack : this.racksById.values()) {
            rack.writeTo(out);
        }
        out.write("</Cluster>".getBytes(StandardCharsets.UTF_8));
    }

    public Optional<TopicPlacement> getTopicPlacement(String topic) {
        return this.topicPlacements.filter(stringTopicPlacementMap -> stringTopicPlacementMap.containsKey(topic)).map(stringTopicPlacementMap -> (TopicPlacement)stringTopicPlacementMap.get(topic));
    }

    public void setTopicPlacements(Map<String, TopicPlacement> topicPlacements) {
        this.topicPlacements = Optional.ofNullable(topicPlacements);
    }

    public void setReplicaExclusions(Set<Integer> brokerIds) {
        this.activeBrokerReplicaExclusions = Collections.unmodifiableSet(new HashSet<Integer>(brokerIds));
    }

    public Set<Integer> excludedBrokers() {
        return this.activeBrokerReplicaExclusions;
    }

    public void addReassigningPartition(TopicPartition tp) {
        this.reassigningPartitions.add(tp);
    }

    public boolean hasReassigningPartitions() {
        return !this.reassigningPartitions.isEmpty();
    }

    public Set<TopicPartition> reassigningPartitions() {
        return new HashSet<TopicPartition>(this.reassigningPartitions);
    }

    public String toString() {
        return String.format("ClusterModel[brokerCount=%d,partitionCount=%d,aliveBrokerCount=%d]", this.brokers.size(), this.partitionsByTopicPartition.size(), this.aliveBrokers.size());
    }

    private void addReplicasUpToReplicationFactor(Partition partition, Short desiredReplicationFactor, Map<String, List<Integer>> brokersByRack) {
        Integer brokerId;
        Cell partitionCell = partition.leader().broker().cell();
        brokersByRack = brokersByRack.entrySet().stream().map(entry -> new AbstractMap.SimpleEntry(entry.getKey(), ((List)entry.getValue()).stream().filter(partitionCell::containsBroker).collect(Collectors.toList()))).filter(entry -> !((List)entry.getValue()).isEmpty()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        ArrayList<String> racks = new ArrayList<String>(brokersByRack.keySet());
        int[] cursors = new int[racks.size()];
        int rackCursor = 0;
        ArrayList<Integer> newReplicas = new ArrayList<Integer>();
        HashSet<String> currentOccupiedRack = new HashSet<String>();
        TopicPartition tp = partition.topicPartition();
        for (Replica replica : partition.replicas()) {
            brokerId = replica.broker().id();
            newReplicas.add(brokerId);
            currentOccupiedRack.add(this.brokerIdToRack.get(brokerId).id());
        }
        while (newReplicas.size() < desiredReplicationFactor) {
            String rack = (String)racks.get(rackCursor);
            if (!currentOccupiedRack.contains(rack) || currentOccupiedRack.size() == racks.size()) {
                int cursor = cursors[rackCursor];
                brokerId = brokersByRack.get(rack).get(cursor);
                if (!newReplicas.contains(brokerId)) {
                    newReplicas.add(brokersByRack.get(rack).get(cursor));
                    Load newReplicaLoad = partition.estimatedNewReplicaLoad();
                    this.createReplica(rack, brokerId, tp, partition.replicas().size(), false, false, null, true, false);
                    this.setReplicaLoad(rack, brokerId, tp, newReplicaLoad.loadByWindows(), newReplicaLoad.windows());
                    currentOccupiedRack.add(rack);
                }
                cursors[rackCursor] = (cursor + 1) % brokersByRack.get(rack).size();
            }
            rackCursor = (rackCursor + 1) % racks.size();
        }
    }

    public List<CellLoad> cellLoadStats(Long maxReplicasPerBroker, List<Integer> cellIds) {
        List<Integer> cellIdsToUse = cellIds.isEmpty() ? this.cellIdToCell.keySet() : cellIds;
        return cellIdsToUse.stream().map(this.cellIdToCell::get).map(cell -> new CellLoad(cell.id(), this.cellUtilRatio((Cell)cell, maxReplicasPerBroker))).collect(Collectors.toList());
    }

    double cellUtilRatio(Cell cell, Long maxReplicasPerBroker) {
        Collection<Broker> brokers = cell.brokers();
        if (brokers.isEmpty()) {
            return 1.0;
        }
        double totalCellUtil = 0.0;
        for (Broker broker : brokers) {
            if (!broker.isAlive()) {
                totalCellUtil += 1.0;
                continue;
            }
            if (!broker.isEligibleSource() || !broker.isEligibleDestination()) {
                totalCellUtil += 1.0;
                continue;
            }
            totalCellUtil += ClusterModel.compositeBrokerUtilRatio(broker, maxReplicasPerBroker);
        }
        return totalCellUtil / (double)brokers.size();
    }

    private static double compositeBrokerUtilRatio(Broker broker, Long maxReplicasPerBroker) {
        ArrayList<Double> ratios = new ArrayList<Double>();
        Arrays.asList(Resource.CPU, Resource.NW_IN, Resource.NW_OUT).forEach(resource -> ratios.add(ClusterModel.usageRatio(broker, resource)));
        ratios.add((double)broker.replicas().size() / ((double)maxReplicasPerBroker.longValue() * 1.0));
        return ratios.stream().max(Double::compare).orElse(0.0);
    }

    private static double usageRatio(Broker broker, Resource resource) {
        return broker.load().expectedUtilizationFor(resource) / broker.capacity().totalCapacityFor(resource);
    }

    public List<CellLoad> cellLoadStats(Long maxReplicasPerBroker) {
        return this.cellLoadStats(maxReplicasPerBroker, Collections.emptyList());
    }

    public double expectedUtilizationInEligibleSourceBrokersFor(Resource resource) {
        return this.utilization.eligibleSourceUtilization().map(load -> load.expectedUtilizationFor(resource)).orElse(0.0);
    }

    public double eligibleDestinationCapacityFor(Resource resource) {
        return this.capacity().eligibleDestinationCapacityFor(resource);
    }

    @FunctionalInterface
    public static interface ThresholdProvider {
        public double threshold(Broker var1);
    }

    @FunctionalInterface
    public static interface CapacityLimitProvider {
        public double capacityLimit(Broker var1);
    }

    public static class NonExistentBrokerException
    extends Exception {
        public NonExistentBrokerException(String msg) {
            super(msg);
        }
    }
}

