/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.monitor.sampling;

import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityConfigResolver;
import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityInfo;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.metricsreporter.exception.UnknownVersionException;
import com.linkedin.kafka.cruisecontrol.metricsreporter.metric.CruiseControlMetric;
import com.linkedin.kafka.cruisecontrol.metricsreporter.metric.RawMetricType;
import com.linkedin.kafka.cruisecontrol.monitor.MonitorUtils;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.MetricSampler;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.SamplingUtils;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.BrokerLoad;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.BrokerMetricSample;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.PartitionMetricSample;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.ReplicaMetricSample;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CruiseControlMetricsProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(CruiseControlMetricsProcessor.class);
    private static final long INIT_METRIC_MAX_TIMESTAMP = -1L;
    private static final long INIT_METRIC_MIN_TIMESTAMP = Long.MAX_VALUE;
    private static final int MAX_PARTITION_ERROR_LOGS = 10;
    private static final int MAX_REPLICA_ERROR_LOGS = 10;
    private final Map<Integer, BrokerLoad> brokerLoad = new HashMap<Integer, BrokerLoad>();
    private final Map<Integer, Short> cachedNumCoresByBroker = new HashMap<Integer, Short>();
    private final BrokerCapacityConfigResolver brokerCapacityConfigResolver;
    private final boolean allowCpuCapacityEstimation;
    private long minMetricTimestamp;
    private long maxMetricTimestamp;
    private final double requestContributionWeight;
    private final double bytesContributionWeight;

    public CruiseControlMetricsProcessor(BrokerCapacityConfigResolver brokerCapacityConfigResolver, KafkaCruiseControlConfig config) {
        this.brokerCapacityConfigResolver = brokerCapacityConfigResolver;
        this.allowCpuCapacityEstimation = config.getBoolean("sampling.allow.cpu.capacity.estimation");
        this.requestContributionWeight = config.getDouble("request.cpu.contribution.weight");
        this.bytesContributionWeight = config.getDouble("bytes.cpu.contribution.weight");
        this.minMetricTimestamp = Long.MAX_VALUE;
        this.maxMetricTimestamp = -1L;
    }

    public void addMetric(CruiseControlMetric metric) {
        int brokerId = metric.brokerId();
        LOG.trace("Adding cruise control metric {}", (Object)metric);
        this.minMetricTimestamp = Math.min(metric.time(), this.minMetricTimestamp);
        this.maxMetricTimestamp = Math.max(metric.time(), this.maxMetricTimestamp);
        this.brokerLoad.compute(brokerId, (bid, load) -> {
            BrokerLoad brokerLoad = load == null ? new BrokerLoad() : load;
            brokerLoad.recordMetric(metric);
            return brokerLoad;
        });
    }

    BrokerLoad brokerLoad(int brokerId) {
        return this.brokerLoad.get(brokerId);
    }

    private void updateCachedNumCoresByBroker(Cluster cluster) {
        for (int brokerId : this.brokerLoad.keySet()) {
            Node node = cluster.nodeById(brokerId);
            if (node == null) continue;
            this.cachedNumCoresByBroker.computeIfAbsent(brokerId, bid -> {
                BrokerCapacityInfo capacity = this.brokerCapacityConfigResolver.capacityForBroker(MonitorUtils.getRackHandleNull(node), node.host(), (int)bid);
                return !this.allowCpuCapacityEstimation && capacity.isEstimated() ? null : Short.valueOf(capacity.numCpuCores());
            });
        }
    }

    private void updateDiskCapacityByBroker(Cluster cluster) {
        for (Map.Entry<Integer, BrokerLoad> entry : this.brokerLoad.entrySet()) {
            Integer brokerId = entry.getKey();
            BrokerLoad brokerLoad = entry.getValue();
            Node node = cluster.nodeById(brokerId.intValue());
            if (node == null || !brokerLoad.brokerMetricAvailable(RawMetricType.BROKER_DISK_CAPACITY)) continue;
            double brokerDiskCapacityInMiB = brokerLoad.brokerMetric(RawMetricType.BROKER_DISK_CAPACITY) / 1048576.0;
            this.brokerCapacityConfigResolver.updateDiskCapacityForBroker(MonitorUtils.getRackHandleNull(node), node.host(), brokerId, brokerDiskCapacityInMiB);
        }
    }

    Map<Integer, Short> cachedNumCoresByBroker() {
        return this.cachedNumCoresByBroker;
    }

    private void prepareReplicaMetrics(Cluster cluster, long time) {
        this.setLeaderReplicationBytesOut(cluster, time);
        this.setFollowerReplicationBytesIn(cluster, time);
    }

    void setLeaderReplicationBytesOut(Cluster cluster, long time) {
        for (Node node : cluster.nodes()) {
            BrokerLoad brokerLoad = this.brokerLoad.get(node.id());
            if (brokerLoad == null) {
                LOG.debug("Skipping broker {} as broker load is not present.", (Object)node);
                continue;
            }
            HashMap<String, Integer> leaderTopicsWithRf = new HashMap<String, Integer>();
            for (PartitionInfo partitionInfo : cluster.partitionsForNode(node.id())) {
                if (leaderTopicsWithRf.containsKey(partitionInfo.topic())) continue;
                leaderTopicsWithRf.put(partitionInfo.topic(), partitionInfo.replicas().length);
            }
            leaderTopicsWithRf.forEach((topic, rf) -> {
                if (brokerLoad.topicMetricsAvailable((String)topic)) {
                    double topicBytesIn = brokerLoad.topicMetrics((String)topic, RawMetricType.TOPIC_BYTES_IN, false);
                    double replicationBytesOut = topicBytesIn * (double)(rf - 1);
                    brokerLoad.setTopicMetrics((String)topic, RawMetricType.TOPIC_REPLICATION_BYTES_OUT, replicationBytesOut, time);
                } else {
                    LOG.debug("Skipping setting replication bytes out for topic {} as topic metrics are not available.", topic);
                }
            });
        }
    }

    void setFollowerReplicationBytesIn(Cluster cluster, long time) {
        Map<Integer, List<PartitionInfo>> followerPartitionsForNodes = this.followerPartitionsForNodes(cluster);
        for (Node node : cluster.nodes()) {
            BrokerLoad brokerLoad = this.brokerLoad.get(node.id());
            if (brokerLoad == null) {
                LOG.debug("Skipping broker {} as broker load is not present.", (Object)node);
                continue;
            }
            HashMap<String, Double> topicReplicationBytesIn = new HashMap<String, Double>();
            List<PartitionInfo> followerPartitions = followerPartitionsForNodes.get(node.id());
            Map<String, List<PartitionInfo>> followersByLeaders = followerPartitions.stream().filter(partitionInfo -> partitionInfo.leader() != null && !partitionInfo.leader().isEmpty()).collect(Collectors.groupingBy(p -> p.leader().id() + p.topic()));
            for (List<PartitionInfo> followers : followersByLeaders.values()) {
                PartitionInfo partitionInfo2 = followers.get(0);
                int leaderId = partitionInfo2.leader().id();
                BrokerLoad leaderBrokerLoad = this.brokerLoad.get(leaderId);
                if (leaderBrokerLoad == null) {
                    LOG.debug("Skipping partition {} as its leader broker load is not present.", (Object)partitionInfo2);
                    continue;
                }
                if (!leaderBrokerLoad.topicMetricsAvailable(partitionInfo2.topic())) {
                    LOG.debug("Skipping partition {} as its metrics are not present at the leader broker.", (Object)partitionInfo2);
                    continue;
                }
                double replicationBytesIn2 = leaderBrokerLoad.topicMetrics(partitionInfo2.topic(), RawMetricType.TOPIC_BYTES_IN, false);
                if (Double.isNaN(replicationBytesIn2)) continue;
                long numLeaders = cluster.partitionsForNode(leaderId).stream().filter(p -> p.topic().equals(partitionInfo2.topic())).count();
                if (numLeaders == 0L) {
                    LOG.warn("We have leader broker load but no leaders on the broker {} for topic {}. Ignoring replication bytes in metric.", (Object)leaderId, (Object)partitionInfo2.topic());
                    continue;
                }
                replicationBytesIn2 /= (double)numLeaders;
                topicReplicationBytesIn.merge(partitionInfo2.topic(), replicationBytesIn2 *= (double)followers.size(), Double::sum);
            }
            topicReplicationBytesIn.forEach((topic, replicationBytesIn) -> {
                if (brokerLoad.topicMetricsAvailable((String)topic)) {
                    brokerLoad.setTopicMetrics((String)topic, RawMetricType.TOPIC_REPLICATION_BYTES_IN, (double)replicationBytesIn, time);
                } else {
                    LOG.debug("Skipping setting replication bytes in for topic {} as topic metrics are not available.", topic);
                }
            });
        }
    }

    Map<Integer, List<PartitionInfo>> followerPartitionsForNodes(Cluster cluster) {
        HashMap<Integer, List<PartitionInfo>> followerPartitionsByNode = new HashMap<Integer, List<PartitionInfo>>();
        cluster.topics().stream().flatMap(topic -> cluster.partitionsForTopic(topic).stream()).filter(partitionInfo -> partitionInfo.leader() != null && !partitionInfo.leader().isEmpty()).forEach(partitionInfo -> Arrays.stream(partitionInfo.replicas()).filter(replica -> replica != null && !replica.isEmpty()).filter(replica -> !replica.equals((Object)partitionInfo.leader())).forEach(replica -> followerPartitionsByNode.computeIfAbsent(replica.id(), ArrayList::new).add(partitionInfo)));
        cluster.nodes().forEach(node -> followerPartitionsByNode.computeIfAbsent(node.id(), ignored -> Collections.emptyList()));
        return followerPartitionsByNode;
    }

    public MetricSampler.Samples process(Cluster cluster, Set<PartitionInfo> partitions) {
        this.updateCachedNumCoresByBroker(cluster);
        this.updateDiskCapacityByBroker(cluster);
        this.prepareReplicaMetrics(cluster, this.maxMetricTimestamp);
        this.brokerLoad.forEach((broker, load) -> load.prepareBrokerMetrics(cluster, (int)broker, this.maxMetricTimestamp));
        HashSet<PartitionInfo> validPartitions = new HashSet<PartitionInfo>();
        int skippedReplicas = 0;
        int skippedPartitions = 0;
        for (PartitionInfo topicPartition : partitions) {
            Node leader;
            if (SamplingUtils.skipBuildingMetricSample(topicPartition, leader = topicPartition.leader(), this.brokerLoad, this.cachedNumCoresByBroker)) {
                ++skippedPartitions;
                continue;
            }
            validPartitions.add(topicPartition);
        }
        int numBrokers = cluster.nodes().size();
        HashMap<Integer, Map<String, Integer>> leaderDistribution = new HashMap<Integer, Map<String, Integer>>(numBrokers);
        HashMap<Integer, Map<String, Integer>> followerDistribution = new HashMap<Integer, Map<String, Integer>>(numBrokers);
        SamplingUtils.populateReplicaDistribution(cluster, leaderDistribution, followerDistribution);
        HashSet<ReplicaMetricSample> replicaMetricSamples = new HashSet<ReplicaMetricSample>();
        HashSet<PartitionMetricSample> partitionMetricSamples = new HashSet<PartitionMetricSample>();
        HashSet<BrokerMetricSample> brokerMetricSamples = new HashSet<BrokerMetricSample>();
        int skippedBroker = this.addBrokerMetricSamples(cluster, this.brokerLoad, this.minMetricTimestamp, this.maxMetricTimestamp, brokerMetricSamples);
        LOG.info("Generated {}{} replica metrics samples, {}{} partition metric samples and {}{} broker metric samples from timestamp {} to timestamp {}.", new Object[]{replicaMetricSamples.size(), skippedReplicas > 0 ? "(" + (skippedReplicas += this.addReplicaMetricSamples(validPartitions, replicaMetricSamples, leaderDistribution, followerDistribution, this.minMetricTimestamp, this.maxMetricTimestamp)) + " skipped)" : "", partitionMetricSamples.size(), skippedPartitions > 0 ? "(" + (skippedPartitions += this.addPartitionMetricSamples(validPartitions, partitionMetricSamples, leaderDistribution, this.minMetricTimestamp, this.maxMetricTimestamp)) + " skipped)" : "", brokerMetricSamples.size(), skippedBroker > 0 ? "(" + skippedBroker + " skipped)" : "", this.minMetricTimestamp, this.maxMetricTimestamp});
        return new MetricSampler.Samples(replicaMetricSamples, partitionMetricSamples, brokerMetricSamples);
    }

    public void clear() {
        this.brokerLoad.clear();
        this.maxMetricTimestamp = -1L;
        this.minMetricTimestamp = Long.MAX_VALUE;
    }

    private int addReplicaMetricSamples(Set<PartitionInfo> topicPartitions, Set<ReplicaMetricSample> replicaMetricSamples, Map<Integer, Map<String, Integer>> leaderDistribution, Map<Integer, Map<String, Integer>> followerDistribution, long minMetricTimestamp, long maxMetricTimestamp) {
        int skippedReplicas = 0;
        int loggedReplicaErrors = 0;
        for (PartitionInfo topicPartition : topicPartitions) {
            for (Node replica : topicPartition.replicas()) {
                try {
                    if (SamplingUtils.skipBuildingMetricSample(topicPartition, replica, this.brokerLoad, this.cachedNumCoresByBroker)) {
                        ++skippedReplicas;
                        continue;
                    }
                    boolean isLeader = replica.id() == topicPartition.leader().id();
                    Map<Integer, Map<String, Integer>> distribution = isLeader ? leaderDistribution : followerDistribution;
                    Map topicReplicaCount = distribution.getOrDefault(replica.id(), Collections.emptyMap());
                    Integer replicaCount = (Integer)topicReplicaCount.get(topicPartition.topic());
                    if (replicaCount == null) continue;
                    ReplicaMetricSample sample = SamplingUtils.buildReplicaMetricSample(topicPartition, replica, replicaCount, this.brokerLoad, minMetricTimestamp, maxMetricTimestamp, this.cachedNumCoresByBroker, this.requestContributionWeight, this.bytesContributionWeight);
                    LOG.trace("Added replica metrics sample for {}-{}, replica: {}.", new Object[]{topicPartition.topic(), topicPartition.partition(), replica});
                    replicaMetricSamples.add(sample);
                }
                catch (Exception e) {
                    if (loggedReplicaErrors < 10) {
                        LOG.error("Error building replica metric sample for {}-{}, replica: {}. Error: {}", new Object[]{topicPartition.topic(), topicPartition.partition(), replica, e.getMessage(), e});
                        if (++loggedReplicaErrors == 10) {
                            LOG.info("Already logged {} errors. Switching to trace level logging now.", (Object)10);
                        }
                    } else {
                        LOG.trace("Error building replica metric sample for {}-{}, replica: {}. Error: {}", new Object[]{topicPartition.topic(), topicPartition.partition(), replica, e.getMessage(), e});
                    }
                    ++skippedReplicas;
                }
            }
        }
        return skippedReplicas;
    }

    private int addPartitionMetricSamples(Set<PartitionInfo> topicPartitions, Set<PartitionMetricSample> partitionMetricSamples, Map<Integer, Map<String, Integer>> leaderDistribution, long minMetricTimestamp, long maxMetricTimestamp) {
        int skippedPartition = 0;
        int loggedPartitionErrors = 0;
        for (PartitionInfo topicPartition : topicPartitions) {
            try {
                PartitionMetricSample sample = SamplingUtils.buildPartitionMetricSample(leaderDistribution, topicPartition, this.brokerLoad, minMetricTimestamp, maxMetricTimestamp);
                LOG.trace("Added partition metrics sample for {}.", (Object)topicPartition);
                partitionMetricSamples.add(sample);
            }
            catch (Exception e) {
                if (loggedPartitionErrors < 10) {
                    LOG.warn("Error building partition metric sample for {}.", (Object)topicPartition, (Object)e);
                    ++loggedPartitionErrors;
                } else {
                    LOG.trace("Error building partition metric sample for {}.", (Object)topicPartition, (Object)e);
                }
                ++skippedPartition;
            }
        }
        return skippedPartition;
    }

    private int addBrokerMetricSamples(Cluster cluster, Map<Integer, BrokerLoad> brokerLoad, long minMetricTimestamp, long maxMetricTimestamp, Set<BrokerMetricSample> brokerMetricSamples) {
        int skippedBroker = 0;
        for (Node node : cluster.nodes()) {
            try {
                BrokerMetricSample sample = SamplingUtils.buildBrokerMetricSample(node, brokerLoad, minMetricTimestamp, maxMetricTimestamp);
                if (sample != null) {
                    LOG.trace("Added broker metric sample for broker {}.", (Object)node.id());
                    brokerMetricSamples.add(sample);
                    continue;
                }
                ++skippedBroker;
            }
            catch (UnknownVersionException e) {
                LOG.warn("Unrecognized serde version detected during broker metric sampling.", (Throwable)e);
                ++skippedBroker;
            }
            catch (Exception e) {
                LOG.warn("Error building broker metric sample for {}.", (Object)node.id(), (Object)e);
                ++skippedBroker;
            }
        }
        return skippedBroker;
    }
}

