/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.controlcenter.client;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.confluent.controlcenter.ClusterRegistryConfig;
import io.confluent.controlcenter.ControlCenterConfig;
import io.confluent.controlcenter.annotation.Warning;
import io.confluent.controlcenter.client.MetadataServiceClient;
import io.confluent.controlcenter.errors.UpdateException;
import io.confluent.controlcenter.httpclient.BasicHttpCredential;
import io.confluent.controlcenter.httpclient.BearerTokenHttpCredential;
import io.confluent.controlcenter.httpclient.Client;
import io.confluent.controlcenter.httpclient.HttpCredential;
import io.confluent.controlcenter.kafka.ClusterManagementModule;
import io.confluent.controlcenter.servicehealthcheck.ServiceHealthCheckModule;
import io.confluent.controlcenter.servicehealthcheck.SingleServiceHealthCheck;
import io.confluent.controlcenter.validation.ClusterInfoValidator;
import io.confluent.rbacapi.entities.ClusterInfo;
import io.confluent.rbacapi.utils.ClusterType;
import io.confluent.rbacapi.validation.v1.V1ValidationUtil;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.validation.ConstraintViolationException;
import javax.ws.rs.WebApplicationException;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.MimeTypes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class ClusterRegistryClient
extends MetadataServiceClient {
    private static final Logger log = LoggerFactory.getLogger(ClusterRegistryClient.class);
    private static final Set<Integer> successStatusCodes = ImmutableSet.of((Object)204);
    private static final String KAFKA_CLUSTER = "kafka-cluster";
    private static final V1ValidationUtil validator = new V1ValidationUtil();
    public static final String CLUSTER_REGISTRY_FEATURE_KEY = "cluster.registry.crud.api.1.enabled";
    private final HttpCredential controlCenterCredential;
    private final AtomicBoolean controlCenterStarted;
    private final ClusterInfoValidator clusterInfoValidator;

    @Inject
    public ClusterRegistryClient(ControlCenterConfig controlCenterConfig, ClusterRegistryConfig registryConfig, ObjectMapper objectMapper, @ServiceHealthCheckModule.MetadataServiceHealthCheck SingleServiceHealthCheck mdsHealthCheck, @ClusterManagementModule.ControlCenterStarted AtomicBoolean controlCenterStarted, ClusterInfoValidator clusterInfoValidator) {
        super(controlCenterConfig, registryConfig, objectMapper, mdsHealthCheck);
        this.controlCenterCredential = new BasicHttpCredential(registryConfig.getControlCenterUsername(), registryConfig.getControlCenterPassword());
        this.controlCenterStarted = controlCenterStarted;
        this.clusterInfoValidator = clusterInfoValidator;
    }

    @VisibleForTesting
    ClusterRegistryClient(AtomicBoolean controlCenterStarted) {
        super(null, null, null, null);
        this.controlCenterCredential = null;
        this.controlCenterStarted = controlCenterStarted;
        this.clusterInfoValidator = new ClusterInfoValidator(null);
    }

    @Warning(reason="This method uses the Control Center credential to call Cluster Registry. Use this method only if user credential is unavailable")
    public List<ClusterInfo> getClusters(ClusterType clusterType) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        return this.getClustersOfType(clusterType, this.controlCenterCredential);
    }

    public List<ClusterInfo> getClusters(ClusterType clusterType, String token) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        Preconditions.checkNotNull((Object)token);
        return this.getClustersOfType(clusterType, new BearerTokenHttpCredential(token));
    }

    public void addOrUpdateClusters(List<ClusterInfo> clusterInfos, String token) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        Preconditions.checkNotNull(clusterInfos);
        Preconditions.checkNotNull((Object)token);
        Preconditions.checkArgument((clusterInfos.size() > 0 ? 1 : 0) != 0);
        this.makeRequestWithRetries(baseUrl -> this.getClient().makeRequestWithContent(baseUrl + "/security/1.0/registry/clusters", HttpMethod.POST, new BearerTokenHttpCredential(token), MimeTypes.Type.APPLICATION_JSON, clusterInfos, new TypeReference<Void>(){}, successStatusCodes));
    }

    public void addOrUpdateCluster(ClusterInfo clusterInfo, String token) {
        this.addOrUpdateClusters((List<ClusterInfo>)ImmutableList.of((Object)clusterInfo), token);
    }

    public void deleteCluster(ClusterInfo clusterInfo, String token) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        Preconditions.checkNotNull((Object)clusterInfo);
        Preconditions.checkNotNull((Object)token);
        this.makeRequestWithRetries(baseUrl -> this.getClient().makeRequestNoBody(baseUrl + "/security/1.0/registry/clusters/" + Client.urlEncode(clusterInfo.getClusterName()), HttpMethod.DELETE, new BearerTokenHttpCredential(token), new TypeReference<Void>(){}, successStatusCodes));
    }

    public void updateCluster(ClusterInfo oldClusterInfo, ClusterInfo updatedClusterInfo, String token) throws UpdateException {
        Preconditions.checkNotNull((Object)oldClusterInfo);
        Preconditions.checkNotNull((Object)updatedClusterInfo);
        Preconditions.checkArgument((boolean)this.isClusterRegistryFeatureEnabled());
        if (!Objects.deepEquals(oldClusterInfo, updatedClusterInfo)) {
            try {
                this.getValidator().validate(updatedClusterInfo);
            }
            catch (ConstraintViolationException ex) {
                throw new UpdateException((Exception)((Object)ex));
            }
            List<ClusterInfo> existingClusters = this.getClusters(ClusterType.KAFKA_CLUSTER);
            if (existingClusters.stream().anyMatch(clusterInfo -> !ClusterRegistryClient.getKafkaClusterId(clusterInfo).equals(ClusterRegistryClient.getKafkaClusterId(updatedClusterInfo)) && clusterInfo.getClusterName().equals(updatedClusterInfo.getClusterName()))) {
                throw new UpdateException("Cluster with same name already exists");
            }
            if (ClusterRegistryClient.requiresDeleteBeforeUpdate(oldClusterInfo, updatedClusterInfo)) {
                this.deleteCluster(oldClusterInfo, token);
            }
            try {
                this.addOrUpdateCluster(updatedClusterInfo, token);
            }
            catch (Exception ex) {
                this.addOrUpdateCluster(oldClusterInfo, token);
                log.warn("Failed to update cluster name", (Throwable)ex);
                throw new UpdateException("Failed to update cluster name");
            }
        }
    }

    public boolean ready() {
        return this.controlCenterStarted.get();
    }

    public List<ClusterInfo> getClustersOfType(ClusterType clusterType, HttpCredential credential) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        Preconditions.checkArgument((boolean)this.isClusterRegistryFeatureEnabled());
        return this.makeRequestWithRetries(baseUrl -> this.getClient().makeRequestNoBody(baseUrl + "/security/1.0/registry/clusters?clusterType=" + clusterType, HttpMethod.GET, credential, new TypeReference<List<ClusterInfo>>(){}));
    }

    public ClusterInfo getClusterByName(String clusterName, String token) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        Preconditions.checkNotNull((Object)clusterName);
        Preconditions.checkNotNull((Object)token);
        return this.makeRequestWithRetries(baseUrl -> this.getClient().makeRequestNoBody(baseUrl + "/security/1.0/registry/clusters?clusterName=" + Client.urlEncode(clusterName), HttpMethod.GET, new BearerTokenHttpCredential(token), new TypeReference<ClusterInfo>(){}));
    }

    public ClusterInfo getKafkaClusterByID(String clusterID, String token) {
        Preconditions.checkArgument((boolean)this.controlCenterStarted.get(), (Object)"Client cannot be used before Control center application is ready");
        Preconditions.checkNotNull((Object)clusterID);
        Preconditions.checkNotNull((Object)token);
        List<ClusterInfo> clusterInfoList = this.getClusters(ClusterType.KAFKA_CLUSTER, token);
        Optional<ClusterInfo> clusterInfo = clusterInfoList.stream().filter(cluster -> clusterID.equals(cluster.getScope().clusters().get(KAFKA_CLUSTER))).findFirst();
        return clusterInfo.orElse(null);
    }

    public boolean isClusterRegistryFeatureEnabled() {
        try {
            return this.getMetadataServiceFeatures().getFeatures().getOrDefault(CLUSTER_REGISTRY_FEATURE_KEY, false);
        }
        catch (WebApplicationException e) {
            log.error("Cannot check if Cluster Registry enabled: failed to fetch MDS features. please check if MDS cluster is up and running properly");
            if (log.isDebugEnabled()) {
                log.error("Exception", (Throwable)e);
            }
            return false;
        }
    }

    @VisibleForTesting
    static boolean requiresDeleteBeforeUpdate(ClusterInfo oldClusterInfo, ClusterInfo updatedClusterInfo) {
        return !oldClusterInfo.getClusterName().equals(updatedClusterInfo.getClusterName()) || !Objects.deepEquals(oldClusterInfo.getScope(), updatedClusterInfo.getScope());
    }

    static String getKafkaClusterId(ClusterInfo clusterInfo) {
        Preconditions.checkNotNull((Object)clusterInfo);
        return (String)clusterInfo.getScope().clusters().get(KAFKA_CLUSTER);
    }

    @VisibleForTesting
    public ClusterInfoValidator getValidator() {
        return this.clusterInfoValidator;
    }
}

