/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.azure.core.credential.TokenCredential;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.http.rest.Response;
import com.azure.core.util.Context;
import com.azure.core.util.polling.LongRunningOperationStatus;
import com.azure.core.util.polling.PollResponse;
import com.azure.core.util.polling.SyncPoller;
import com.azure.identity.ClientSecretCredential;
import com.azure.identity.ClientSecretCredentialBuilder;
import com.azure.identity.DefaultAzureCredential;
import com.azure.identity.DefaultAzureCredentialBuilder;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.BlobServiceClientBuilder;
import com.azure.storage.blob.models.AccessTier;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobHttpHeaders;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.BlobListDetails;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.BlobRequestConditions;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.blob.models.ParallelTransferOptions;
import com.azure.storage.blob.options.BlobBeginCopyOptions;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.AzureBlockBlobAutoAbortingInputStream;
import kafka.tier.store.AzureBlockBlobTierObjectStoreConfig;
import kafka.tier.store.BucketHealthResult;
import kafka.tier.store.TierObjectAttribute;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.TierObjectStoreUtils;
import kafka.tier.store.VersionInformation;
import kafka.utils.CoreUtils;
import org.apache.kafka.common.utils.ByteBufferInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzureBlockBlobTierObjectStore
implements TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(AzureBlockBlobTierObjectStore.class);
    private final BlobServiceClient blobServiceClient;
    private final BlobContainerClient blobContainerClient;
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final String container;
    private final String prefix;
    private final int drainThreshold;
    private static final long DEFAULT_TIMEOUT_IN_SECS = 30L;

    public AzureBlockBlobTierObjectStore(AzureBlockBlobTierObjectStoreConfig config) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.container = config.container;
        this.prefix = config.azureBlobPrefix;
        this.drainThreshold = config.drainThreshold;
        this.blobServiceClient = AzureBlockBlobTierObjectStore.createServiceClient(config);
        this.blobContainerClient = AzureBlockBlobTierObjectStore.createContainerClient(this.blobServiceClient, config);
    }

    public AzureBlockBlobTierObjectStore(AzureBlockBlobTierObjectStoreConfig config, BlobServiceClient blobServiceClient, BlobContainerClient blobContainerClient) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.container = config.container;
        this.prefix = config.azureBlobPrefix;
        this.drainThreshold = config.drainThreshold;
        this.blobServiceClient = blobServiceClient;
        this.blobContainerClient = blobContainerClient;
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.AzureBlockBlob;
    }

    @Override
    public Map<String, List<VersionInformation>> listObject(String keyPrefix, boolean getVersionInfo) {
        HashMap<String, List<VersionInformation>> results = new HashMap<String, List<VersionInformation>>();
        Duration timeout = Duration.ofSeconds(30L);
        try {
            ListBlobsOptions options = new ListBlobsOptions().setPrefix(keyPrefix).setDetails(new BlobListDetails().setRetrieveVersions(getVersionInfo));
            PagedIterable blobItems = this.blobContainerClient.listBlobs(options, timeout);
            for (BlobItem blobItem : blobItems) {
                results.putIfAbsent(blobItem.getName(), new ArrayList());
                if (!getVersionInfo) continue;
                ((List)results.get(blobItem.getName())).add(new VersionInformation(blobItem.getVersionId()));
            }
            if (log.isDebugEnabled()) {
                StringBuilder allBlobs = new StringBuilder();
                results.forEach((key, versions) -> allBlobs.append("[").append((String)key).append("->").append(Arrays.toString(versions.toArray())).append("] "));
                log.debug("TierObjectStore listObjects versions: " + getVersionInfo + " prefix: " + keyPrefix + " " + allBlobs);
            }
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to list objects with keyPrefix: %s, getVersionInfo: %b", keyPrefix, getVersionInfo), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to list objects with keyPrefix: %s, getVersionInfo: %b", keyPrefix, getVersionInfo), e);
        }
        return results;
    }

    @Override
    public TierObjectStoreResponse getObject(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType, Integer byteOffsetStart, Integer byteOffsetEnd, VersionInformation versionInformation) {
        InputStream inputStream;
        String key = this.keyPath(objectMetadata, fileType);
        BlobClient blob = versionInformation != null ? this.blobContainerClient.getBlobVersionClient(key, versionInformation.getVersionId()) : this.blobContainerClient.getBlobClient(key);
        AzureBlockBlobTierObjectStore.checkOffsets(byteOffsetStart, byteOffsetEnd);
        log.debug("Fetching object from {}/{}, with range of {} to {}", new Object[]{this.container, key, byteOffsetStart, byteOffsetEnd});
        long byteOffsetStartLong = byteOffsetStart == null ? 0L : byteOffsetStart.longValue();
        BlobRange range = byteOffsetEnd != null ? new BlobRange(byteOffsetStartLong, Long.valueOf(byteOffsetEnd.longValue() - byteOffsetStartLong)) : new BlobRange(byteOffsetStartLong);
        try {
            inputStream = this.getInputStreamFromBlobClient(blob, range);
        }
        catch (BlobStorageException e) {
            if (e.getErrorCode().equals((Object)BlobErrorCode.BLOB_NOT_FOUND)) {
                throw new TierObjectStoreFatalException(String.format("Failed to fetch object from %s, metadata: %s type: %s range %s-%s.Object not found.", new Object[]{key, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
            }
            throw new TierObjectStoreRetriableException(String.format("Failed to fetch object from %s, metadata: %s type: %s range %s-%s", new Object[]{key, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException(String.format("Runtime exception when fetching object from %s, metadata: %s type: %s range %s-%s", new Object[]{key, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when fetching object from %s, metadata: %s type: %s range %s-%s", new Object[]{key, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        long streamSize = byteOffsetEnd == null ? Long.MAX_VALUE : byteOffsetEnd.longValue() - byteOffsetStartLong;
        return new AzureBlockBlobTierObjectStoreResponse(inputStream, this.drainThreshold, streamSize);
    }

    private static void checkOffsets(Integer byteOffsetStart, Integer byteOffsetEnd) {
        if (byteOffsetStart != null && byteOffsetEnd != null && byteOffsetStart > byteOffsetEnd) {
            throw new IllegalStateException(String.format("Invalid range of byteOffsetStart=%d and byteOffsetEnd=%d", byteOffsetStart, byteOffsetEnd));
        }
        if (byteOffsetStart == null && byteOffsetEnd != null) {
            throw new IllegalStateException(String.format("Cannot specify a byteOffsetEnd=%d without specifying a byteOffsetStart", byteOffsetEnd));
        }
    }

    InputStream getInputStreamFromBlobClient(BlobClient blob, BlobRange range) {
        return blob.getBlockBlobClient().openInputStream(range, new BlobRequestConditions());
    }

    @Override
    public TierObjectStore.OpaqueData prepPutSegment() throws TierObjectStoreRetriableException, IOException {
        return TierObjectStore.OpaqueData.ZEROED;
    }

    @Override
    public void putSegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<File> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData);
            producerStateSnapshotData.ifPresent(file -> this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, (File)file));
            transactionIndexData.ifPresent(byteBuffer -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, (ByteBuffer)byteBuffer));
            epochState.ifPresent(byteBuffer -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, (ByteBuffer)byteBuffer));
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException("Failed to upload segment " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
    }

    @Override
    public void putInMemorySegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<ByteBuffer> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData);
            producerStateSnapshotData.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, (ByteBuffer)buf));
            transactionIndexData.ifPresent(byteBuffer -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, (ByteBuffer)byteBuffer));
            epochState.ifPresent(byteBuffer -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, (ByteBuffer)byteBuffer));
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException("Failed to upload segment " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
    }

    @Override
    public void putObject(TierObjectStore.ObjectStoreMetadata objectMetadata, File file, TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, fileType), metadata, file);
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
    }

    @Override
    public void putBuffer(TierObjectStore.ObjectStoreMetadata objectMetadata, ByteBuffer buffer, TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putBuf(this.keyPath(objectMetadata, fileType), metadata, buffer);
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, buffer %s, type %s", new Object[]{objectMetadata, buffer, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, buffer %s, type %s", new Object[]{objectMetadata, buffer, fileType}), e);
        }
    }

    @Override
    public void restoreObjectByCopy(TierObjectStore.ObjectMetadata objectMetadata, String key, VersionInformation lastLiveVersion) {
        String lastLiveVersionId = lastLiveVersion.getVersionId();
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            BlobClient blobSource = this.blobContainerClient.getBlobVersionClient(key, lastLiveVersionId);
            String copySource = blobSource.getBlobUrl();
            log.debug(String.format("Azure restore key: %s lastLiveVersionId: %s copySource: %s", key, lastLiveVersionId, copySource));
            BlobClient blobDest = this.blobContainerClient.getBlobClient(key);
            BlobBeginCopyOptions options = new BlobBeginCopyOptions(copySource).setMetadata(metadata).setTier(AccessTier.HOT);
            SyncPoller syncPoller = blobDest.beginCopy(options);
            PollResponse pollResponse = syncPoller.waitUntil(LongRunningOperationStatus.SUCCESSFULLY_COMPLETED);
            log.debug(String.format("Azure restore key: %s response status: %s", key, pollResponse.getStatus()));
        }
        catch (BlobStorageException e) {
            if (e.getErrorCode().equals((Object)BlobErrorCode.BLOB_NOT_FOUND)) {
                throw new TierObjectStoreFatalException(String.format("Failed to restore object %s (version: %s) as blob not found", key, lastLiveVersionId), e);
            }
            throw new TierObjectStoreRetriableException(String.format("Failed to restore object %s (version: %s)", key, lastLiveVersionId), e);
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to restore object %s (version: %s)", key, lastLiveVersionId), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when restoring object %s (version: %s)", key, lastLiveVersionId), e);
        }
    }

    @Override
    public void deleteSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        for (TierObjectStore.FileType type : TierObjectStore.FileType.values()) {
            String key = this.keyPath(objectMetadata, type);
            try {
                BlobClient blob = this.blobContainerClient.getBlobClient(key);
                log.debug("Deleting " + key);
                blob.delete();
            }
            catch (BlobStorageException be) {
                if (be.getErrorCode().equals((Object)BlobErrorCode.BLOB_NOT_FOUND)) continue;
                throw new TierObjectStoreRetriableException("Failed to delete file " + key, be);
            }
            catch (RuntimeException e) {
                throw new TierObjectStoreRetriableException("Failed to delete file " + key, e);
            }
            catch (Exception e) {
                throw new TierObjectStoreFatalException("Unknown exception when deleting segment " + objectMetadata, e);
            }
        }
    }

    @Override
    public void deleteVersions(List<TierObjectStore.KeyAndVersion> keys) {
        Duration timeout = Duration.ofSeconds(30L);
        for (TierObjectStore.KeyAndVersion key : keys) {
            BlobClient blobClient = key.versionId() == null ? this.blobContainerClient.getBlobClient(key.key()) : this.blobContainerClient.getBlobVersionClient(key.key(), key.versionId());
            try {
                Response response = blobClient.deleteWithResponse(null, null, timeout, Context.NONE);
                log.debug("TierObjectStore delete request for " + key + " statusCode: " + response.getStatusCode());
            }
            catch (RuntimeException e) {
                throw new TierObjectStoreRetriableException("Failed to delete versioned objects: " + key, e);
            }
            catch (Exception e) {
                throw new TierObjectStoreFatalException("Unknown exception when deleting versioned objects: " + key, e);
            }
        }
    }

    @Override
    public TierObjectAttribute objectExists(TierObjectStore.ObjectMetadata objectMetadata, TierObjectStore.FileType type) throws IOException, TierObjectStoreRetriableException {
        String key = this.keyPath(objectMetadata, type);
        TierObjectAttribute result = new TierObjectAttribute(false);
        try {
            BlobClient client = this.blobContainerClient.getBlobClient(key);
            if (client.exists().booleanValue()) {
                result.size = client.getProperties().getBlobSize();
                result.exist = true;
            }
        }
        catch (RuntimeException e) {
            throw new TierObjectStoreRetriableException("Failed to check for object attributes with metadata " + objectMetadata + " @ " + key, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception while checking for object attributes with metadata " + objectMetadata + " @ " + key, e);
        }
        return result;
    }

    @Override
    public BucketHealthResult checkBucketHealth() {
        try {
            ByteBuffer payload = TierObjectStoreUtils.timeHealthPayload();
            TierObjectStore.HealthMetadata metadata = new TierObjectStore.HealthMetadata(this.clusterIdOpt, this.brokerIdOpt);
            String key = metadata.toPath(this.prefix, TierObjectStore.FileType.HEALTH_CHECK);
            this.putBuf(key, metadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt), payload);
            try (InputStream inputStream = this.getObject(metadata, TierObjectStore.FileType.HEALTH_CHECK).getInputStream();){
                int read;
                while ((read = inputStream.read()) > 0) {
                    log.trace("Bucket probe read {} bytes", (Object)read);
                }
            }
            BlobClient blob = this.blobContainerClient.getBlobClient(key);
            log.debug("Deleting " + key);
            blob.delete();
            return BucketHealthResult.HEALTHY;
        }
        catch (Exception e) {
            log.error("Bucket health checker returned unclassified error", (Throwable)e);
            return BucketHealthResult.UNCLASSIFIED;
        }
    }

    @Override
    public void close() {
    }

    private void putFile(String key, Map<String, String> metadata, File file) {
        BlobClient blobClient = this.blobContainerClient.getBlobClient(key);
        blobClient.uploadFromFile(file.getPath(), new ParallelTransferOptions(), new BlobHttpHeaders(), metadata, AccessTier.HOT, new BlobRequestConditions(), null);
    }

    public void putBuf(String key, Map<String, String> metadata, ByteBuffer buf) {
        BlobClient blobClient = this.blobContainerClient.getBlobClient(key);
        byte[] md5 = CoreUtils.md5hash(buf);
        blobClient.getBlockBlobClient().uploadWithResponse((InputStream)new BufferedInputStream((InputStream)new ByteBufferInputStream(buf.duplicate())), (long)(buf.limit() - buf.position()), new BlobHttpHeaders().setContentMd5(md5), metadata, AccessTier.HOT, md5, new BlobRequestConditions(), null, null);
    }

    private static BlobServiceClient createServiceClient(AzureBlockBlobTierObjectStoreConfig config) {
        BlobServiceClient blobServiceClient;
        if (config.azureCredentialsConfig.isPresent()) {
            AzureBlockBlobTierObjectStoreConfig.AzureCredentialsConfig azureCredentialsConfig = config.azureCredentialsConfig.get();
            if (azureCredentialsConfig.connectionStringAuthMethod().booleanValue()) {
                blobServiceClient = new BlobServiceClientBuilder().connectionString(azureCredentialsConfig.connectionString()).buildClient();
            } else {
                ClientSecretCredential credential = ((ClientSecretCredentialBuilder)((ClientSecretCredentialBuilder)new ClientSecretCredentialBuilder().clientId(azureCredentialsConfig.azureClientId())).tenantId(azureCredentialsConfig.azureTenantId())).clientSecret(azureCredentialsConfig.azureClientSecret()).build();
                blobServiceClient = new BlobServiceClientBuilder().endpoint(config.endpoint.get()).credential((TokenCredential)credential).buildClient();
            }
        } else {
            DefaultAzureCredential credential = new DefaultAzureCredentialBuilder().build();
            blobServiceClient = new BlobServiceClientBuilder().endpoint(config.endpoint.get()).credential((TokenCredential)credential).buildClient();
        }
        return blobServiceClient;
    }

    private static BlobContainerClient createContainerClient(BlobServiceClient blobServiceClient, AzureBlockBlobTierObjectStoreConfig config) {
        BlobContainerClient blobContainerClient = blobServiceClient.getBlobContainerClient(config.container);
        if (!blobContainerClient.exists()) {
            throw new TierObjectStoreFatalException("Container " + config.container + " does not exist or could not be found");
        }
        return blobContainerClient;
    }

    private String keyPath(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType) {
        return objectMetadata.toPath(this.prefix, fileType);
    }

    private static class AzureBlockBlobTierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final InputStream inputStream;

        AzureBlockBlobTierObjectStoreResponse(InputStream inputStream, int drainThreshold, long streamSize) {
            this.inputStream = new AzureBlockBlobAutoAbortingInputStream(inputStream, drainThreshold, streamSize);
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }
}

