/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.api.server;

import com.google.common.collect.ImmutableList;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.confluent.ksql.api.auth.AuthenticationPlugin;
import io.confluent.ksql.api.server.KsqlApiException;
import io.confluent.ksql.api.server.LoggingRateLimiter;
import io.confluent.ksql.api.server.PushQueryHolder;
import io.confluent.ksql.api.server.ServerVerticle;
import io.confluent.ksql.api.spi.Endpoints;
import io.confluent.ksql.api.util.ApiServerUtils;
import io.confluent.ksql.internal.PullQueryExecutorMetrics;
import io.confluent.ksql.rest.Errors;
import io.confluent.ksql.rest.entity.PushQueryId;
import io.confluent.ksql.rest.server.KsqlRestConfig;
import io.confluent.ksql.rest.server.state.ServerState;
import io.confluent.ksql.security.KsqlSecurityExtension;
import io.confluent.ksql.util.FileWatcher;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.VertxCompletableFuture;
import io.netty.handler.ssl.OpenSsl;
import io.vertx.core.Handler;
import io.vertx.core.Verticle;
import io.vertx.core.Vertx;
import io.vertx.core.WorkerExecutor;
import io.vertx.core.http.ClientAuth;
import io.vertx.core.http.HttpConnection;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.impl.ConcurrentHashSet;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Server {
    private static final Logger log = LoggerFactory.getLogger(Server.class);
    private final Vertx vertx;
    private final KsqlRestConfig config;
    private final Endpoints endpoints;
    private final Map<PushQueryId, PushQueryHolder> queries = new ConcurrentHashMap<PushQueryId, PushQueryHolder>();
    private final Set<HttpConnection> connections = new ConcurrentHashSet();
    private final int maxPushQueryCount;
    private final Set<String> deploymentIds = new HashSet<String>();
    private final KsqlSecurityExtension securityExtension;
    private final Optional<AuthenticationPlugin> authenticationPlugin;
    private final ServerState serverState;
    private final List<URI> listeners = new ArrayList<URI>();
    private final List<URI> proxyProtocolListeners = new ArrayList<URI>();
    private final Optional<PullQueryExecutorMetrics> pullQueryMetrics;
    private URI internalListener;
    private WorkerExecutor workerExecutor;
    private FileWatcher fileWatcher;

    @SuppressFBWarnings(value={"EI_EXPOSE_REP2"})
    public Server(Vertx vertx, KsqlRestConfig config, Endpoints endpoints, KsqlSecurityExtension securityExtension, Optional<AuthenticationPlugin> authenticationPlugin, ServerState serverState, Optional<PullQueryExecutorMetrics> pullQueryMetrics) {
        this.vertx = Objects.requireNonNull(vertx);
        this.config = Objects.requireNonNull(config);
        this.endpoints = Objects.requireNonNull(endpoints);
        this.securityExtension = Objects.requireNonNull(securityExtension);
        this.authenticationPlugin = Objects.requireNonNull(authenticationPlugin);
        this.serverState = Objects.requireNonNull(serverState);
        this.maxPushQueryCount = config.getInt("ksql.max.push.queries");
        this.pullQueryMetrics = Objects.requireNonNull(pullQueryMetrics, "pullQueryMetrics");
        if (!OpenSsl.isAvailable()) {
            log.warn("OpenSSL does not appear to be installed. ksqlDB will fall back to using the JDK TLS implementation. OpenSSL is recommended for better performance.");
        }
    }

    public synchronized void start() {
        if (!this.deploymentIds.isEmpty()) {
            throw new IllegalStateException("Already started");
        }
        int idleConnectionTimeoutSeconds = this.config.getInt("ksql.idle.connection.timeout.seconds");
        this.workerExecutor = this.vertx.createSharedWorkerExecutor("ksql-workers", this.config.getInt("ksql.worker.pool.size").intValue());
        LoggingRateLimiter loggingRateLimiter = new LoggingRateLimiter(this.config);
        this.configureTlsCertReload(this.config);
        List<URI> listenUris = ApiServerUtils.parseListeners(this.config);
        List<URI> proxyProtocolListenUris = ApiServerUtils.parseProxyProtocolListeners(this.config);
        HashSet<URI> proxyProtocolListenUriSet = new HashSet<URI>(proxyProtocolListenUris);
        Optional<URI> internalListenUri = Server.parseInternalListener(this.config, listenUris);
        ArrayList<URI> allListenUris = new ArrayList<URI>(listenUris);
        internalListenUri.ifPresent(allListenUris::add);
        int instances = this.config.getInt("ksql.verticle.instances");
        log.debug("Deploying " + instances + " instances of server verticle");
        ArrayList<CompletableFuture> deployFutures = new ArrayList<CompletableFuture>();
        ConcurrentHashMap<URI, URI> uris = new ConcurrentHashMap<URI, URI>();
        for (URI uRI : allListenUris) {
            Optional<Boolean> isInternalListener = internalListenUri.map(uri -> uri.equals(uRI));
            boolean isProxyProtocolListener = proxyProtocolListenUriSet.contains(uRI);
            int i = 0;
            while (i < instances) {
                VertxCompletableFuture vcf = new VertxCompletableFuture();
                ServerVerticle serverVerticle = new ServerVerticle(this.endpoints, Server.createHttpServerOptions(this.config, uRI.getHost(), uRI.getPort(), uRI.getScheme().equalsIgnoreCase("https"), isInternalListener.orElse(false), isProxyProtocolListener, idleConnectionTimeoutSeconds), this, isInternalListener, loggingRateLimiter);
                this.vertx.deployVerticle((Verticle)serverVerticle, (Handler)vcf);
                int index = i++;
                CompletableFuture deployFuture = vcf.thenApply(s -> {
                    if (index == 0) {
                        try {
                            URI uriWithPort = new URI(uRI.getScheme(), null, uRI.getHost(), serverVerticle.actualPort(), null, null, null);
                            uris.put(uRI, uriWithPort);
                        }
                        catch (URISyntaxException e) {
                            throw new KsqlException((Throwable)e);
                        }
                    }
                    return s;
                });
                deployFutures.add(deployFuture);
            }
        }
        CompletableFuture<Void> allDeployFuture = CompletableFuture.allOf(deployFutures.toArray(new CompletableFuture[0]));
        try {
            allDeployFuture.get();
            for (CompletableFuture deployFuture : deployFutures) {
                this.deploymentIds.add((String)deployFuture.get());
            }
        }
        catch (Exception exception) {
            throw new KsqlException("Failed to start API server", (Throwable)exception);
        }
        this.initListeners(listenUris, proxyProtocolListenUris, internalListenUri, uris);
        log.info("API server started");
    }

    private void initListeners(List<URI> listenUris, List<URI> proxyProtocolListenUris, Optional<URI> internalListenUri, Map<URI, URI> uris) {
        for (URI uri : listenUris) {
            this.listeners.add(uris.get(uri));
        }
        for (URI uri : proxyProtocolListenUris) {
            this.proxyProtocolListeners.add(uris.get(uri));
        }
        if (internalListenUri.isPresent()) {
            this.internalListener = uris.get(internalListenUri.get());
        }
    }

    public synchronized void stop() {
        if (this.deploymentIds.isEmpty()) {
            throw new IllegalStateException("Not started");
        }
        if (this.workerExecutor != null) {
            this.workerExecutor.close();
        }
        if (this.fileWatcher != null) {
            this.fileWatcher.shutdown();
        }
        ArrayList<VertxCompletableFuture> undeployFutures = new ArrayList<VertxCompletableFuture>();
        for (String deploymentID : this.deploymentIds) {
            VertxCompletableFuture future = new VertxCompletableFuture();
            this.vertx.undeploy(deploymentID, (Handler)future);
            undeployFutures.add(future);
        }
        try {
            CompletableFuture.allOf(undeployFutures.toArray(new CompletableFuture[0])).get();
        }
        catch (Exception e) {
            throw new KsqlException("Failure in stopping API server", (Throwable)e);
        }
        this.deploymentIds.clear();
        this.listeners.clear();
        this.proxyProtocolListeners.clear();
        log.info("API server stopped");
    }

    public synchronized void restart() {
        log.info("Restarting server");
        this.stop();
        this.start();
    }

    public WorkerExecutor getWorkerExecutor() {
        return this.workerExecutor;
    }

    synchronized void registerQuery(PushQueryHolder query) throws KsqlApiException {
        Objects.requireNonNull(query);
        if (this.queries.size() == this.maxPushQueryCount) {
            throw new KsqlApiException("Maximum number of push queries exceeded", Errors.ERROR_CODE_MAX_PUSH_QUERIES_EXCEEDED);
        }
        if (this.queries.putIfAbsent(query.getId(), query) != null) {
            throw new IllegalStateException("Glitch in the matrix");
        }
    }

    Optional<PushQueryHolder> removeQuery(PushQueryId queryId) {
        return Optional.ofNullable(this.queries.remove(queryId));
    }

    public Set<PushQueryId> getQueryIDs() {
        return new HashSet<PushQueryId>(this.queries.keySet());
    }

    KsqlSecurityExtension getSecurityExtension() {
        return this.securityExtension;
    }

    public Optional<AuthenticationPlugin> getAuthenticationPlugin() {
        return this.authenticationPlugin;
    }

    ServerState getServerState() {
        return this.serverState;
    }

    public KsqlRestConfig getConfig() {
        return this.config;
    }

    void registerQueryConnection(HttpConnection connection) {
        this.connections.add(Objects.requireNonNull(connection));
    }

    void removeQueryConnection(HttpConnection connection) {
        this.connections.remove(Objects.requireNonNull(connection));
    }

    public int queryConnectionCount() {
        return this.connections.size();
    }

    public synchronized List<URI> getListeners() {
        return ImmutableList.copyOf(this.listeners);
    }

    public synchronized Optional<URI> getInternalListener() {
        return Optional.ofNullable(this.internalListener);
    }

    public synchronized List<URI> getProxyProtocolListeners() {
        return ImmutableList.copyOf(this.proxyProtocolListeners);
    }

    private void configureTlsCertReload(KsqlRestConfig config) {
        if (config.getBoolean("ssl.keystore.reload").booleanValue()) {
            Path watchLocation = !config.getString("ssl.keystore.watch.location").isEmpty() ? Paths.get(config.getString("ssl.keystore.watch.location"), new String[0]) : Paths.get(config.getString("ssl.keystore.location"), new String[0]);
            try {
                this.fileWatcher = new FileWatcher(watchLocation, this::restart);
                this.fileWatcher.start();
                log.info("Enabled SSL cert auto reload for: " + watchLocation);
            }
            catch (IOException e) {
                log.error("Failed to enable SSL cert auto reload", (Throwable)e);
            }
        }
    }

    private static HttpServerOptions createHttpServerOptions(KsqlRestConfig ksqlRestConfig, String host, int port, boolean tls, boolean isInternalListener, boolean isProxyProtocolListener, int idleTimeoutSeconds) {
        HttpServerOptions options = new HttpServerOptions().setHost(host).setPort(port).setReuseAddress(true).setReusePort(true).setIdleTimeout(idleTimeoutSeconds).setIdleTimeoutUnit(TimeUnit.SECONDS).setPerMessageWebSocketCompressionSupported(true).setPerFrameWebSocketCompressionSupported(true).setUseProxyProtocol(isProxyProtocolListener);
        if (tls) {
            String ksConfigName = isInternalListener ? "ksql.ssl.keystore.alias.internal" : "ksql.ssl.keystore.alias.external";
            ClientAuth clientAuth = isInternalListener ? ksqlRestConfig.getClientAuthInternal() : ksqlRestConfig.getClientAuth();
            String alias = ksqlRestConfig.getString(ksConfigName);
            ApiServerUtils.setTlsOptions(ksqlRestConfig, options, alias, clientAuth);
        }
        return options;
    }

    private static Optional<URI> parseInternalListener(KsqlRestConfig config, List<URI> listenUris) {
        if (config.getString("ksql.internal.listener") == null) {
            return Optional.empty();
        }
        URI uri = ApiServerUtils.parseListenerStrings(config, (List<String>)ImmutableList.of((Object)config.getString("ksql.internal.listener"))).get(0);
        if (listenUris.contains(uri)) {
            return Optional.empty();
        }
        return Optional.of(uri);
    }
}

