/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.rest.server.computation;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.ksql.KsqlExecutionContext;
import io.confluent.ksql.config.SessionConfig;
import io.confluent.ksql.engine.KsqlEngine;
import io.confluent.ksql.engine.KsqlPlan;
import io.confluent.ksql.exception.ExceptionUtil;
import io.confluent.ksql.parser.KsqlParser;
import io.confluent.ksql.parser.tree.AlterSystemProperty;
import io.confluent.ksql.parser.tree.CreateAsSelect;
import io.confluent.ksql.parser.tree.ExecutableDdlStatement;
import io.confluent.ksql.parser.tree.InsertInto;
import io.confluent.ksql.parser.tree.PauseQuery;
import io.confluent.ksql.parser.tree.ResumeQuery;
import io.confluent.ksql.parser.tree.TerminateQuery;
import io.confluent.ksql.planner.plan.ConfiguredKsqlPlan;
import io.confluent.ksql.query.QueryId;
import io.confluent.ksql.query.id.SpecificQueryIdGenerator;
import io.confluent.ksql.rest.entity.CommandId;
import io.confluent.ksql.rest.entity.CommandStatus;
import io.confluent.ksql.rest.server.StatementParser;
import io.confluent.ksql.rest.server.computation.Command;
import io.confluent.ksql.rest.server.computation.CommandStatusFuture;
import io.confluent.ksql.rest.server.computation.InternalTopicSerdes;
import io.confluent.ksql.rest.server.computation.QueuedCommand;
import io.confluent.ksql.services.ServiceContext;
import io.confluent.ksql.util.KsqlConfig;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.QueryMetadata;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.kafka.common.serialization.Deserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InteractiveStatementExecutor {
    private static final Logger log = LoggerFactory.getLogger(InteractiveStatementExecutor.class);
    private final ServiceContext serviceContext;
    private final KsqlExecutionContext ksqlEngine;
    private final StatementParser statementParser;
    private final SpecificQueryIdGenerator queryIdGenerator;
    private final Map<CommandId, CommandStatus> statusStore;
    private final Deserializer<Command> commandDeserializer;

    public InteractiveStatementExecutor(ServiceContext serviceContext, KsqlEngine ksqlEngine, SpecificQueryIdGenerator queryIdGenerator) {
        this(serviceContext, ksqlEngine, new StatementParser((KsqlExecutionContext)ksqlEngine), queryIdGenerator, InternalTopicSerdes.deserializer(Command.class));
    }

    @VisibleForTesting
    InteractiveStatementExecutor(ServiceContext serviceContext, KsqlEngine ksqlEngine, StatementParser statementParser, SpecificQueryIdGenerator queryIdGenerator, Deserializer<Command> commandDeserializer) {
        this.serviceContext = Objects.requireNonNull(serviceContext, "serviceContext");
        this.ksqlEngine = (KsqlExecutionContext)Objects.requireNonNull(ksqlEngine, "ksqlEngine");
        this.statementParser = Objects.requireNonNull(statementParser, "statementParser");
        this.queryIdGenerator = Objects.requireNonNull(queryIdGenerator, "queryIdGenerator");
        this.commandDeserializer = Objects.requireNonNull(commandDeserializer, "commandDeserializer");
        this.statusStore = new ConcurrentHashMap<CommandId, CommandStatus>();
    }

    KsqlExecutionContext getKsqlEngine() {
        return this.ksqlEngine;
    }

    void handleStatement(QueuedCommand queuedCommand) {
        this.throwIfNotConfigured();
        this.handleStatementWithTerminatedQueries(queuedCommand.getAndDeserializeCommand(this.commandDeserializer), queuedCommand.getAndDeserializeCommandId(), queuedCommand.getStatus(), Mode.EXECUTE, queuedCommand.getOffset(), false);
    }

    void handleRestore(QueuedCommand queuedCommand) {
        this.throwIfNotConfigured();
        this.handleStatementWithTerminatedQueries(queuedCommand.getAndDeserializeCommand(this.commandDeserializer), queuedCommand.getAndDeserializeCommandId(), queuedCommand.getStatus(), Mode.RESTORE, queuedCommand.getOffset(), true);
    }

    public Map<CommandId, CommandStatus> getStatuses() {
        return new HashMap<CommandId, CommandStatus>(this.statusStore);
    }

    public Optional<CommandStatus> getStatus(CommandId statementId) {
        return Optional.ofNullable(this.statusStore.get(statementId));
    }

    private void putStatus(CommandId commandId, Optional<CommandStatusFuture> commandStatusFuture, CommandStatus status) {
        this.statusStore.put(commandId, status);
        commandStatusFuture.ifPresent(s -> s.setStatus(status));
    }

    private void putFinalStatus(CommandId commandId, Optional<CommandStatusFuture> commandStatusFuture, CommandStatus status) {
        this.statusStore.put(commandId, status);
        commandStatusFuture.ifPresent(s -> s.setFinalStatus(status));
    }

    private void throwIfNotConfigured() {
        if (!this.ksqlEngine.getKsqlConfig().getKsqlStreamConfigProps().containsKey("application.server")) {
            throw new IllegalStateException("No initialized");
        }
    }

    private void handleStatementWithTerminatedQueries(Command command, CommandId commandId, Optional<CommandStatusFuture> commandStatusFuture, Mode mode, long offset, boolean restoreInProgress) {
        try {
            if (command.getPlan().isPresent()) {
                this.executePlan(command, commandId, commandStatusFuture, command.getPlan().get(), mode, offset, restoreInProgress);
                return;
            }
            String statementString = command.getStatement();
            this.putStatus(commandId, commandStatusFuture, new CommandStatus(CommandStatus.Status.PARSING, "Parsing statement"));
            KsqlParser.PreparedStatement statement = this.statementParser.parseSingleStatement(statementString);
            this.putStatus(commandId, commandStatusFuture, new CommandStatus(CommandStatus.Status.EXECUTING, "Executing statement"));
            this.executeStatement(statement, commandId, commandStatusFuture);
        }
        catch (KsqlException exception) {
            log.error("Failed to handle: " + command, (Throwable)exception);
            CommandStatus errorStatus = new CommandStatus(CommandStatus.Status.ERROR, ExceptionUtil.stackTraceToString((Exception)((Object)exception)));
            this.putStatus(commandId, commandStatusFuture, errorStatus);
            throw exception;
        }
    }

    private void executePlan(Command command, CommandId commandId, Optional<CommandStatusFuture> commandStatusFuture, KsqlPlan plan, Mode mode, long offset, boolean restoreInProgress) {
        KsqlConfig mergedConfig = this.buildMergedConfig(command);
        ConfiguredKsqlPlan configured = ConfiguredKsqlPlan.of((KsqlPlan)plan, (SessionConfig)SessionConfig.of((KsqlConfig)mergedConfig, command.getOverwriteProperties()));
        this.putStatus(commandId, commandStatusFuture, new CommandStatus(CommandStatus.Status.EXECUTING, "Executing statement"));
        KsqlExecutionContext.ExecuteResult result = this.ksqlEngine.execute(this.serviceContext, configured, restoreInProgress);
        this.queryIdGenerator.setNextId(offset + 1L);
        if (result.getQuery().isPresent() && mode == Mode.EXECUTE) {
            ((QueryMetadata)result.getQuery().get()).start();
        }
        String successMessage = this.getSuccessMessage(result);
        Optional<QueryId> queryId = result.getQuery().map(QueryMetadata::getQueryId);
        CommandStatus successStatus = new CommandStatus(CommandStatus.Status.SUCCESS, successMessage, queryId);
        this.putFinalStatus(commandId, commandStatusFuture, successStatus);
    }

    private String getSuccessMessage(KsqlExecutionContext.ExecuteResult result) {
        if (result.getCommandResult().isPresent()) {
            return (String)result.getCommandResult().get();
        }
        return "Created query with ID " + ((QueryMetadata)result.getQuery().get()).getQueryId();
    }

    private void executeStatement(KsqlParser.PreparedStatement<?> statement, CommandId commandId, Optional<CommandStatusFuture> commandStatusFuture) {
        if (statement.getStatement() instanceof PauseQuery) {
            this.pauseQuery(statement);
            String successMessage = "Query paused.";
            CommandStatus successStatus = new CommandStatus(CommandStatus.Status.SUCCESS, "Query paused.", Optional.empty());
            this.putFinalStatus(commandId, commandStatusFuture, successStatus);
        } else if (statement.getStatement() instanceof ResumeQuery) {
            this.resumeQuery(statement);
            String successMessage = "Query resumed.";
            CommandStatus successStatus = new CommandStatus(CommandStatus.Status.SUCCESS, "Query resumed.", Optional.empty());
            this.putFinalStatus(commandId, commandStatusFuture, successStatus);
        } else if (statement.getStatement() instanceof TerminateQuery) {
            this.terminateQuery(statement);
            String successMessage = "Query terminated.";
            CommandStatus successStatus = new CommandStatus(CommandStatus.Status.SUCCESS, "Query terminated.", Optional.empty());
            this.putFinalStatus(commandId, commandStatusFuture, successStatus);
        } else if (statement.getStatement() instanceof ExecutableDdlStatement) {
            InteractiveStatementExecutor.throwUnsupportedStatementError();
        } else if (statement.getStatement() instanceof CreateAsSelect) {
            InteractiveStatementExecutor.throwUnsupportedStatementError();
        } else if (statement.getStatement() instanceof InsertInto) {
            InteractiveStatementExecutor.throwUnsupportedStatementError();
        } else if (statement.getStatement() instanceof AlterSystemProperty) {
            KsqlParser.PreparedStatement<?> alterSystemQuery = statement;
            String propertyName = ((AlterSystemProperty)alterSystemQuery.getStatement()).getPropertyName();
            String propertyValue = ((AlterSystemProperty)alterSystemQuery.getStatement()).getPropertyValue();
            this.ksqlEngine.alterSystemProperty(propertyName, propertyValue);
            this.ksqlEngine.updateStreamsPropertiesAndRestartRuntime();
            String successMessage = String.format("System property %s was set to %s.", propertyName, propertyValue);
            CommandStatus successStatus = new CommandStatus(CommandStatus.Status.SUCCESS, successMessage, Optional.empty());
            this.putFinalStatus(commandId, commandStatusFuture, successStatus);
        } else {
            throw new KsqlException(String.format("Unexpected statement type: %s", statement.getClass().getName()));
        }
    }

    private KsqlConfig buildMergedConfig(Command command) {
        return this.ksqlEngine.getKsqlConfig().overrideBreakingConfigsWithOriginalValues(command.getOriginalProperties());
    }

    private void pauseQuery(KsqlParser.PreparedStatement<PauseQuery> pauseQuery) {
        Optional queryId = ((PauseQuery)pauseQuery.getStatement()).getQueryId();
        if (!queryId.isPresent()) {
            this.ksqlEngine.getPersistentQueries().forEach(QueryMetadata::pause);
            return;
        }
        Optional query = this.ksqlEngine.getPersistentQuery((QueryId)queryId.get());
        query.ifPresent(QueryMetadata::pause);
    }

    private void resumeQuery(KsqlParser.PreparedStatement<ResumeQuery> resumeQuery) {
        Optional queryId = ((ResumeQuery)resumeQuery.getStatement()).getQueryId();
        if (!queryId.isPresent()) {
            this.ksqlEngine.getPersistentQueries().forEach(QueryMetadata::resume);
            return;
        }
        Optional query = this.ksqlEngine.getPersistentQuery((QueryId)queryId.get());
        query.ifPresent(QueryMetadata::resume);
    }

    private void terminateQuery(KsqlParser.PreparedStatement<TerminateQuery> terminateQuery) {
        Optional queryId = ((TerminateQuery)terminateQuery.getStatement()).getQueryId();
        if (!queryId.isPresent()) {
            this.ksqlEngine.getPersistentQueries().forEach(QueryMetadata::close);
            return;
        }
        Optional query = this.ksqlEngine.getPersistentQuery((QueryId)queryId.get());
        query.ifPresent(QueryMetadata::close);
    }

    private static void throwUnsupportedStatementError() {
        throw new KsqlException("This version of ksqlDB does not support executing statements submitted prior to ksqlDB 0.8.0 or Confluent Platform ksqlDB 5.5. Please see the upgrading guide to upgrade.");
    }

    private static enum Mode {
        RESTORE,
        EXECUTE;

    }
}

