/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.multitenant.quota;

import io.confluent.kafka.multitenant.MultiTenantPrincipal;
import io.confluent.kafka.multitenant.TenantMetadata;
import io.confluent.kafka.multitenant.TestCluster;
import io.confluent.kafka.multitenant.quota.QuotaConfig;
import io.confluent.kafka.multitenant.quota.TenantQuotaCallback;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.server.quota.ClientQuotaClusterDescriber;
import org.apache.kafka.server.quota.ClientQuotaType;
import org.apache.kafka.server.quota.ZkClientQuotaClusterDescriber;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TenantQuotaCallbackTest {
    private static final double EPS = 1.0E-4;
    private static final Long MIN_BROKER_CONSUME_QUOTA = 20L;
    private static final Long MIN_BROKER_PRODUCE_QUOTA = 10L;
    private static final Long MAX_BROKER_CONSUME_QUOTA = 1200L;
    private static final Long MAX_BROKER_PRODUCE_QUOTA = 600L;
    private static final Double DEFAULT_CONTROLLER_QUOTA = 20.0;
    private static final Long TENANT_1_PRODUCE_BYTE_RATE = 1000L;
    private static final Long TENANT_1_CONSUME_BYTE_RATE = 2000L;
    private final int brokerId = 1;
    private TestCluster testCluster;
    private TenantQuotaCallback quotaCallback;

    @BeforeEach
    public void setUp() {
        this.quotaCallback = this.createCallbackWithConfigs(this.quotaCallbackProps());
    }

    private TenantQuotaCallback createCallbackWithConfigs(Map<String, Object> configs) {
        TenantQuotaCallback.closeAll();
        TenantQuotaCallback quotaCallback = new TenantQuotaCallback();
        quotaCallback.configure(configs);
        HashMap<String, QuotaConfig> tenantQuotas = new HashMap<String, QuotaConfig>();
        tenantQuotas.put("tenant1", this.quotaConfig(TENANT_1_PRODUCE_BYTE_RATE, TENANT_1_CONSUME_BYTE_RATE, 300.0));
        tenantQuotas.put("tenant2", this.quotaConfig(2000L, 3000L, 400.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        return quotaCallback;
    }

    @AfterEach
    public void tearDown() {
        this.quotaCallback.close();
    }

    @Test
    public void testTenantQuotaConstructor() {
        TenantQuotaCallback callback = new TenantQuotaCallback();
        callback.configure(Collections.singletonMap("broker.id", 0));
        TenantQuotaCallback tenantQuotaCallback = callback;
        tenantQuotaCallback.getClass();
        TenantQuotaCallback.TenantQuota unlimitedQuota = new TenantQuotaCallback.TenantQuota(tenantQuotaCallback, QuotaConfig.UNLIMITED_QUOTA);
        Assertions.assertFalse((boolean)unlimitedQuota.hasQuotaLimit(ClientQuotaType.PRODUCE));
        Assertions.assertFalse((boolean)unlimitedQuota.hasQuotaLimit(ClientQuotaType.FETCH));
        Assertions.assertFalse((boolean)unlimitedQuota.hasQuotaLimit(ClientQuotaType.REQUEST));
        Assertions.assertFalse((boolean)unlimitedQuota.hasQuotaLimit(ClientQuotaType.CONTROLLER_MUTATION));
        QuotaConfig defaultQuota = this.quotaConfig(102400L, 102400L, 50.0);
        Assertions.assertTrue((boolean)defaultQuota.hasQuotaLimit(ClientQuotaType.PRODUCE));
        Assertions.assertEquals((double)102400.0, (double)defaultQuota.quota(ClientQuotaType.PRODUCE), (double)1.0E-4, (String)"Unexpected PRODUCE quota");
        Assertions.assertTrue((boolean)defaultQuota.hasQuotaLimit(ClientQuotaType.FETCH));
        Assertions.assertEquals((double)102400.0, (double)defaultQuota.quota(ClientQuotaType.FETCH), (double)1.0E-4, (String)"Unexpected FETCH quota");
        Assertions.assertTrue((boolean)defaultQuota.hasQuotaLimit(ClientQuotaType.REQUEST));
        Assertions.assertEquals((double)50.0, (double)defaultQuota.quota(ClientQuotaType.REQUEST), (double)1.0E-4, (String)"Unexpected REQUEST quota");
        Assertions.assertFalse((boolean)defaultQuota.hasQuotaLimit(ClientQuotaType.CONTROLLER_MUTATION));
        Assertions.assertEquals((double)2.147483647E9, (double)defaultQuota.quota(ClientQuotaType.CONTROLLER_MUTATION), (double)1.0E-4, (String)"Unexpected CONTROLLER_MUTATION quota");
    }

    @Test
    public void testGetOrCreateTenantQuota() {
        TenantQuotaCallback newQuotaCallback = new TenantQuotaCallback();
        HashMap<String, Object> configs = new HashMap<String, Object>();
        configs.put("broker.id", String.valueOf(1));
        configs.put("confluent.quota.tenant.default.controller.mutation.rate", DEFAULT_CONTROLLER_QUOTA);
        newQuotaCallback.configure(configs);
        TenantQuotaCallback.TenantQuota tenantQuota = newQuotaCallback.getOrCreateTenantQuota("tenantA", this.quotaConfig(204800L, 204800L, 500.0), false);
        Assertions.assertTrue((boolean)tenantQuota.hasQuotaLimit(ClientQuotaType.PRODUCE));
        Assertions.assertEquals((double)1.048576E7, (double)tenantQuota.quotaLimit(ClientQuotaType.PRODUCE), (double)1.0E-4, (String)"Unexpected PRODUCE quota");
        Assertions.assertTrue((boolean)tenantQuota.hasQuotaLimit(ClientQuotaType.FETCH));
        Assertions.assertEquals((double)1.048576E7, (double)tenantQuota.quotaLimit(ClientQuotaType.FETCH), (double)1.0E-4, (String)"Unexpected FETCH quota");
        Assertions.assertTrue((boolean)tenantQuota.hasQuotaLimit(ClientQuotaType.REQUEST));
        Assertions.assertEquals((double)500.0, (double)tenantQuota.quotaLimit(ClientQuotaType.REQUEST), (double)1.0E-4, (String)"Unexpected REQUEST quota");
        Assertions.assertTrue((boolean)tenantQuota.hasQuotaLimit(ClientQuotaType.CONTROLLER_MUTATION));
        Assertions.assertEquals((double)DEFAULT_CONTROLLER_QUOTA, (double)tenantQuota.quotaLimit(ClientQuotaType.CONTROLLER_MUTATION), (double)1.0E-4, (String)"Unexpected CONTROLLER_MUTATION quota");
        this.createCluster(5);
        this.testCluster.setPartitionLeaders("tenantA_topic1", 0, 1, 1);
        newQuotaCallback.updateClusterMetadata((ClientQuotaClusterDescriber)new ZkClientQuotaClusterDescriber(this.testCluster.cluster()));
        TenantQuotaCallback.TenantQuota tenantQuota2 = newQuotaCallback.getOrCreateTenantQuota("tenantA", this.quotaConfig(404800L, 404800L, 600.0), false);
        Assertions.assertEquals((double)204800.0, (double)tenantQuota2.quotaLimit(ClientQuotaType.PRODUCE), (double)1.0E-4, (String)"Unexpected PRODUCE quota");
        Assertions.assertEquals((double)204800.0, (double)tenantQuota2.quotaLimit(ClientQuotaType.FETCH), (double)1.0E-4, (String)"Unexpected FETCH quota");
        Assertions.assertEquals((double)500.0, (double)tenantQuota2.quotaLimit(ClientQuotaType.REQUEST), (double)1.0E-4, (String)"Unexpected REQUEST quota");
        Assertions.assertEquals((double)DEFAULT_CONTROLLER_QUOTA, (double)tenantQuota.quotaLimit(ClientQuotaType.CONTROLLER_MUTATION), (double)1.0E-4, (String)"Unexpected CONTROLLER_MUTATION quota");
        TenantQuotaCallback.TenantQuota updatedQuota = newQuotaCallback.getOrCreateTenantQuota("tenantA", this.quotaConfig(404800L, 404800L, 600.0), true);
        Assertions.assertEquals((double)404800.0, (double)updatedQuota.quotaLimit(ClientQuotaType.PRODUCE), (double)1.0E-4, (String)"Unexpected PRODUCE quota");
        Assertions.assertEquals((double)404800.0, (double)updatedQuota.quotaLimit(ClientQuotaType.FETCH), (double)1.0E-4, (String)"Unexpected FETCH quota");
        Assertions.assertEquals((double)600.0, (double)updatedQuota.quotaLimit(ClientQuotaType.REQUEST), (double)1.0E-4, (String)"Unexpected REQUEST quota");
        Assertions.assertEquals((double)DEFAULT_CONTROLLER_QUOTA, (double)tenantQuota.quotaLimit(ClientQuotaType.CONTROLLER_MUTATION), (double)1.0E-4, (String)"Unexpected CONTROLLER_MUTATION quota");
    }

    @Test
    public void testTenantQuota() {
        this.createCluster(5);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        this.setPartitionLeaders("tenant1_topic1", 0, 5, 1);
        this.setPartitionLeaders("tenant1_topic2", 0, 5, 2);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant2_topic1", 0, 5, 1);
        this.setPartitionLeaders("tenant2_topic2", 0, 5, 2);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.deleteTopic("tenant1_topic2");
        this.verifyQuotas(principal, MAX_BROKER_PRODUCE_QUOTA.longValue(), MAX_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic3", 0, 5, 3);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic3", 1, 1, 1);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testTenantEqualQuotaDistribution() {
        this.createCluster(5);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        MultiTenantPrincipal principal2 = new MultiTenantPrincipal("userB", new TenantMetadata("tenant2", "tenant2_cluster_id"));
        this.setPartitionLeaders("tenant1_topic1", 0, 1, 1);
        this.setPartitionLeaders("tenant1_topic2", 0, 1000, 2);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant2_topic1", 0, 1000, 1);
        this.setPartitionLeaders("tenant2_topic2", 0, 1, 2);
        this.setPartitionLeaders("tenant2_topic3", 0, 1, 3);
        this.setPartitionLeaders("tenant2_topic4", 0, 1, 4);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.verifyQuotas(principal2, 500.0, 750.0, 400.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic3", 0, 10, 3);
        this.setPartitionLeaders("tenant1_topic4", 0, 100, 4);
        this.setPartitionLeaders("tenant1_topic5", 0, 500, 5);
        this.verifyQuotas(principal, 200.0, 400.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.deleteTopic("tenant1_topic2");
        this.verifyQuotas(principal, 250.0, 500.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testTenantQuotaUpdateSetsCorrectUpdateFlags() {
        int numBrokers = 10;
        String tenantName = "tenant1";
        String topic = "tenant1_topic1";
        HashMap<String, QuotaConfig> tenantQuotas = new HashMap<String, QuotaConfig>();
        tenantQuotas.put("tenant1", this.quotaConfig(3000L, 2000L, 300.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
        this.createCluster(10);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        for (int i = 1; i <= 10; ++i) {
            this.setPartitionLeaders("tenant1_topic1", i - 1, 1, i);
        }
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 300.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 200.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1", false);
        tenantQuotas.put("tenant1", this.quotaConfig(2000L, 4000L, 500.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 200.0, "tenant1", true);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 400.0, "tenant1", true);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 500.0, "tenant1", true);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1", false);
        tenantQuotas.put("tenant1", this.quotaConfig(1000L, 4000L, 500.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 100.0, "tenant1", true);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 400.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 500.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1", false);
        tenantQuotas.put("tenant1", this.quotaConfig(1000L, 2000L, 500.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 100.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 200.0, "tenant1", true);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 500.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1", false);
        tenantQuotas.put("tenant1", this.quotaConfig(1000L, 2000L, 200.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 100.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 200.0, "tenant1", false);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 200.0, "tenant1", true);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1", false);
        MultiTenantPrincipal principal2 = new MultiTenantPrincipal("userB", new TenantMetadata("tenant2", "tenant2_cluster_id"));
        tenantQuotas.put("tenant2", this.quotaConfig(500L, 500L, 500.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
        this.setPartitionLeaders("tenant2_topic1", 0, 1, 1);
        this.verifyQuotas(principal2, 500.0, 500.0, 500.0, DEFAULT_CONTROLLER_QUOTA);
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
        tenantQuotas.put("tenant2", this.quotaConfig(400L, 500L, 500.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        this.verifyQuotas(principal2, 400.0, 500.0, 500.0, DEFAULT_CONTROLLER_QUOTA);
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Assertions.assertFalse((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
    }

    @Test
    public void testDynamicQuotaDisabled() {
        String tenantName = "tenant1";
        this.createCluster(10);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata(tenantName, tenantName + "_cluster_id"));
        for (int i = 1; i <= 10; ++i) {
            this.setPartitionLeaders(tenantName + "_topic1", i - 1, 1, i);
        }
        HashMap<String, Long> quotas = new HashMap<String, Long>();
        quotas.put("PRODUCE", 200L);
        quotas.put("FETCH", 400L);
        Map<String, String> metricTags = Collections.singletonMap("tenant", "tenant1");
        Assertions.assertFalse((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, quotas)));
    }

    @Test
    public void testUpdateTenantQuotasCorrectlyParsesQuotas() {
        String tenantName = "tenant1";
        this.createCluster(10);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata(tenantName, tenantName + "_cluster_id"));
        for (int i = 1; i <= 10; ++i) {
            this.setPartitionLeaders(tenantName + "_topic1", i - 1, 1, i);
        }
        HashMap<String, QuotaConfig> tenantQuotas = new HashMap<String, QuotaConfig>();
        tenantQuotas.put(tenantName, this.quotaConfig(3000L, 2000L, 300.0));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
        HashMap<String, Long> quotas = new HashMap<String, Long>();
        quotas.put("PRODUCE", 200L);
        quotas.put("FETCH", 400L);
        Map<String, String> metricTags = Collections.singletonMap("tenant", "tenant1");
        this.quotaCallback.enableDynamicQuota(true);
        Assertions.assertTrue((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, quotas)));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 200.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 400.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, tenantName, false);
        Assertions.assertFalse((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, quotas)));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 200.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 400.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, tenantName, false);
        Assertions.assertTrue((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, Collections.singletonMap("PRODUCE", 100L))));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 100.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 200.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, tenantName, false);
        Assertions.assertTrue((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, Collections.singletonMap("FETCH", 300L))));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 300.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 300.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, tenantName, false);
        Assertions.assertTrue((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, Collections.singletonMap("REQUEST", 600L))));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 200.0, tenantName, true);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, tenantName, false);
        Assertions.assertFalse((boolean)this.quotaCallback.updateDynamicQuotas(Collections.singletonMap(metricTags, Collections.singletonMap("PRODUCE", 0L))));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 200.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 300.0, tenantName, false);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, tenantName, false);
    }

    @Test
    public void testSmallNumberOfPartitions() throws Exception {
        this.createCluster(5);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        this.setPartitionLeaders("tenant1_topic1", 0, 1, 1);
        this.verifyQuotas(principal, MAX_BROKER_PRODUCE_QUOTA.longValue(), MAX_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic2", 0, 1, 1);
        this.verifyQuotas(principal, MAX_BROKER_PRODUCE_QUOTA.longValue(), MAX_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic3", 0, 2, 2);
        this.verifyQuotas(principal, 500.0, 1000.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic4", 0, 6, 3);
        this.verifyQuotas(principal, 333.0, 666.0, 300.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testNoPartitions() throws Exception {
        this.createCluster(5);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        this.verifyQuotas(principal, MIN_BROKER_PRODUCE_QUOTA.longValue(), MIN_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant1_topic1", 0, 2, 1);
        this.verifyQuotas(principal, MAX_BROKER_PRODUCE_QUOTA.longValue(), MAX_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
        this.createCluster(5);
        this.verifyQuotas(principal, MIN_BROKER_PRODUCE_QUOTA.longValue(), MIN_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testNoClusterMetadata() {
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        this.verifyQuotas(principal, MIN_BROKER_PRODUCE_QUOTA.longValue(), MIN_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testUnlimitedTenantQuota() {
        this.createCluster(5);
        HashMap<String, QuotaConfig> tenantQuotas = new HashMap<String, QuotaConfig>();
        tenantQuotas.put("tenant1", this.quotaConfig(Long.MAX_VALUE, Long.MAX_VALUE, 2.147483647E9));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 2.147483647E9, null);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1");
        this.setPartitionLeaders("tenant1_topic1", 0, 5, 1);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 2.147483647E9, null);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1");
        this.setPartitionLeaders("tenant1_topic2", 0, 5, 2);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 2.147483647E9, null);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1");
        tenantQuotas.put("tenant1", this.quotaConfig(1000L, 2000L, 2.147483647E9));
        TenantQuotaCallback.updateQuotas(tenantQuotas, (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 500.0, "tenant1");
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 1000.0, "tenant1");
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 2.147483647E9, null);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant1");
    }

    @Test
    public void testDefaultTenantQuota() {
        this.createCluster(5);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant100", "tenant100_cluster_id"));
        this.setPartitionLeaders("tenant100_topic1", 0, 1, 1);
        this.setPartitionLeaders("tenant100_topic2", 0, 10, 2);
        this.setPartitionLeaders("tenant100_topic3", 0, 1, 3);
        this.setPartitionLeaders("tenant100_topic4", 0, 100, 4);
        this.setPartitionLeaders("tenant100_topic5", 0, 5, 5);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, 9.223372036854776E18, null);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, 2.147483647E9, null);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, DEFAULT_CONTROLLER_QUOTA, "tenant100");
        QuotaConfig defaultQuota = this.quotaConfig(1000L, 2000L, 10.0);
        TenantQuotaCallback.updateQuotas(Collections.emptyMap(), (QuotaConfig)defaultQuota);
        MultiTenantPrincipal principal2 = new MultiTenantPrincipal("userA", new TenantMetadata("tenant101", "tenant101_cluster_id"));
        this.setPartitionLeaders("tenant101_topic2", 0, 1, 2);
        this.setPartitionLeaders("tenant101_topic3", 0, 1, 3);
        this.setPartitionLeaders("tenant101_topic4", 0, 1, 4);
        this.setPartitionLeaders("tenant101_topic5", 0, 1, 5);
        this.verifyQuotas(principal2, MIN_BROKER_PRODUCE_QUOTA.longValue(), MIN_BROKER_CONSUME_QUOTA.longValue(), 10.0, DEFAULT_CONTROLLER_QUOTA);
        this.setPartitionLeaders("tenant101_topic1", 0, 1, 1);
        this.verifyQuotas(principal2, 200.0, 400.0, 10.0, DEFAULT_CONTROLLER_QUOTA);
        this.verifyQuotas(principal, 200.0, 400.0, 10.0, DEFAULT_CONTROLLER_QUOTA);
        QuotaConfig newDefaultQuota = this.quotaConfig(3000L, 6000L, 30.0);
        TenantQuotaCallback.updateQuotas(Collections.emptyMap(), (QuotaConfig)newDefaultQuota);
        this.quotaCallback.updateClusterMetadata((ClientQuotaClusterDescriber)new ZkClientQuotaClusterDescriber(this.testCluster.cluster()));
        this.verifyQuotas(principal2, 600.0, 1200.0, 30.0, DEFAULT_CONTROLLER_QUOTA);
        this.verifyQuotas(principal, 600.0, 1200.0, 30.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testNonTenantPrincipal() {
        this.createCluster(5);
        KafkaPrincipal principal = KafkaPrincipal.ANONYMOUS;
        for (ClientQuotaType quotaType : ClientQuotaType.values()) {
            this.verifyQuota(quotaType, principal, QuotaConfig.UNLIMITED_QUOTA.quota(quotaType), null);
        }
    }

    @Test
    public void testParentQuotaMetricTagsWithUserQuotasDisabled() {
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        for (ClientQuotaType quotaType : ClientQuotaType.values()) {
            Map metricTags = this.quotaCallback.quotaMetricTags(quotaType, (KafkaPrincipal)principal, "");
            Assertions.assertEquals(Collections.emptyMap(), (Object)this.quotaCallback.parentQuotaMetricTags(quotaType, metricTags));
        }
    }

    @Test
    public void testUnavailableBrokers() {
        this.createCluster(2);
        MultiTenantPrincipal principal = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        this.setPartitionLeaders("tenant1_topic1", 0, 1, 1);
        this.setPartitionLeaders("tenant1_topic2", 0, 1, 30);
        Assertions.assertNull((Object)this.testCluster.cluster().nodeById(30), (String)"Unavailable node created");
        Assertions.assertNull((Object)((PartitionInfo)this.testCluster.cluster().partitionsForTopic("tenant1_topic2").get(0)).leader(), (String)"Unavailable node created");
        this.quotaCallback.updateClusterMetadata((ClientQuotaClusterDescriber)new ZkClientQuotaClusterDescriber(this.testCluster.cluster()));
        this.verifyQuotas(principal, MAX_BROKER_PRODUCE_QUOTA.longValue(), MAX_BROKER_CONSUME_QUOTA.longValue(), 300.0, DEFAULT_CONTROLLER_QUOTA);
    }

    @Test
    public void testReconfigureControllerMutationQuota() {
        this.createCluster(5);
        MultiTenantPrincipal tenant100 = new MultiTenantPrincipal("userA", new TenantMetadata("tenant100", "tenant100_cluster_id"));
        MultiTenantPrincipal tenant200 = new MultiTenantPrincipal("userB", new TenantMetadata("tenant200", "tenant100_cluster_id"));
        MultiTenantPrincipal tenant300 = new MultiTenantPrincipal("userC", new TenantMetadata("tenant300", "tenant100_cluster_id"));
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)tenant100, DEFAULT_CONTROLLER_QUOTA, tenant100.tenantMetadata().tenantName);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)tenant200, DEFAULT_CONTROLLER_QUOTA, tenant200.tenantMetadata().tenantName);
        this.quotaCallback.reconfigure(Collections.singletonMap("confluent.quota.tenant.default.controller.mutation.rate", "10"));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.CONTROLLER_MUTATION));
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)tenant100, 10.0, tenant100.tenantMetadata().tenantName);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)tenant200, 10.0, tenant200.tenantMetadata().tenantName);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)tenant300, 10.0, tenant300.tenantMetadata().tenantName);
    }

    @Test
    public void testReconfigureQuotaMultiplier() {
        this.createCluster(5);
        MultiTenantPrincipal tenant1 = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        String tenant1Name = tenant1.tenantMetadata().tenantName;
        this.setPartitionLeaders("tenant1_topic1", 0, 2, 1);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, MAX_BROKER_PRODUCE_QUOTA.longValue(), tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, MAX_BROKER_CONSUME_QUOTA.longValue(), tenant1Name);
        this.setPartitionLeaders("tenant1_topic2", 0, 1, 2);
        this.setPartitionLeaders("tenant1_topic3", 0, 1, 3);
        this.setPartitionLeaders("tenant1_topic4", 0, 1, 4);
        this.setPartitionLeaders("tenant1_topic5", 0, 1, 5);
        long computedProduceByteRate = TENANT_1_PRODUCE_BYTE_RATE / 5L;
        long computedFetchByteRate = TENANT_1_CONSUME_BYTE_RATE / 5L;
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, computedProduceByteRate, tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, computedFetchByteRate, tenant1Name);
        double updatedProduceMultiplier = 2.0;
        double updatedFetchMultiplier = 3.0;
        this.quotaCallback.reconfigure(this.quotaMultiplierProps(updatedProduceMultiplier, updatedFetchMultiplier));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assertions.assertTrue((boolean)this.quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, (double)computedProduceByteRate * updatedProduceMultiplier, tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, (double)computedFetchByteRate * updatedFetchMultiplier, tenant1Name);
    }

    @Test
    public void testQuotaMultiplier() {
        Map<String, Object> quotaCallbackProps = this.quotaCallbackProps();
        double produceQuotaMultiplier = 2.0;
        double fetchQuotaMultiplier = 3.0;
        quotaCallbackProps.putAll(this.quotaMultiplierProps(produceQuotaMultiplier, fetchQuotaMultiplier));
        this.quotaCallback = this.createCallbackWithConfigs(quotaCallbackProps);
        this.createCluster(5);
        MultiTenantPrincipal tenant1 = new MultiTenantPrincipal("userA", new TenantMetadata("tenant1", "tenant1_cluster_id"));
        String tenant1Name = tenant1.tenantMetadata().tenantName;
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, MIN_BROKER_PRODUCE_QUOTA.longValue(), tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, MIN_BROKER_CONSUME_QUOTA.longValue(), tenant1Name);
        this.setPartitionLeaders("tenant1_topic1", 0, 2, 1);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, MAX_BROKER_PRODUCE_QUOTA.longValue(), tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, MAX_BROKER_CONSUME_QUOTA.longValue(), tenant1Name);
        this.setPartitionLeaders("tenant1_topic2", 0, 1, 2);
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, MAX_BROKER_PRODUCE_QUOTA.longValue(), tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, MAX_BROKER_CONSUME_QUOTA.longValue(), tenant1Name);
        this.setPartitionLeaders("tenant1_topic3", 0, 1, 3);
        this.setPartitionLeaders("tenant1_topic4", 0, 1, 4);
        this.setPartitionLeaders("tenant1_topic5", 0, 1, 5);
        long computedProduceByteRate = TENANT_1_PRODUCE_BYTE_RATE / 5L;
        long computedFetchByteRate = TENANT_1_CONSUME_BYTE_RATE / 5L;
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)tenant1, (double)computedProduceByteRate * produceQuotaMultiplier, tenant1Name);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)tenant1, (double)computedFetchByteRate * fetchQuotaMultiplier, tenant1Name);
    }

    @Test
    public void testDefaultControllerMutationQuota() {
        this.createCluster(5);
        TenantQuotaCallback.updateQuotas(Collections.singletonMap("tenantA", this.quotaConfig(10L, 20L, 30.0)), (QuotaConfig)QuotaConfig.UNLIMITED_QUOTA);
        Assertions.assertEquals((Double)DEFAULT_CONTROLLER_QUOTA, (Double)this.quotaCallback.quotaLimit(ClientQuotaType.CONTROLLER_MUTATION, Collections.singletonMap("tenant", "tenantA")));
        Assertions.assertEquals((Double)DEFAULT_CONTROLLER_QUOTA, (Double)this.quotaCallback.quotaLimit(ClientQuotaType.CONTROLLER_MUTATION, Collections.singletonMap("tenant", "tenantB")));
        Assertions.assertEquals((Double)2.147483647E9, (Double)this.quotaCallback.quotaLimit(ClientQuotaType.CONTROLLER_MUTATION, Collections.emptyMap()));
    }

    @Test
    public void testClusterLevelQuotaNoTenant() {
        TenantQuotaCallback callback = new TenantQuotaCallback();
        Assertions.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.PRODUCE), (Double)callback.clusterQuotaLimit(ClientQuotaType.PRODUCE, Collections.emptyMap()));
        Assertions.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.FETCH), (Double)callback.clusterQuotaLimit(ClientQuotaType.FETCH, Collections.emptyMap()));
        HashMap<String, String> metricTags = new HashMap<String, String>();
        metricTags.put("tenant", "");
        Assertions.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.PRODUCE), (Double)callback.clusterQuotaLimit(ClientQuotaType.PRODUCE, metricTags));
        Assertions.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.FETCH), (Double)callback.clusterQuotaLimit(ClientQuotaType.FETCH, metricTags));
        metricTags.put("tenant", "non-existent-tenant");
        Assertions.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.PRODUCE), (Double)callback.clusterQuotaLimit(ClientQuotaType.PRODUCE, metricTags));
        Assertions.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.FETCH), (Double)callback.clusterQuotaLimit(ClientQuotaType.FETCH, metricTags));
    }

    @Test
    public void testClusterLevelTenantQuota() {
        HashMap<String, String> metricTags = new HashMap<String, String>();
        metricTags.put("tenant", "tenant1");
        Assertions.assertEquals((double)TENANT_1_PRODUCE_BYTE_RATE.doubleValue(), (Double)this.quotaCallback.clusterQuotaLimit(ClientQuotaType.PRODUCE, metricTags));
        Assertions.assertEquals((double)TENANT_1_CONSUME_BYTE_RATE.doubleValue(), (Double)this.quotaCallback.clusterQuotaLimit(ClientQuotaType.FETCH, metricTags));
    }

    private Map<String, Object> quotaCallbackProps() {
        HashMap<String, Object> configs = new HashMap<String, Object>();
        configs.put("broker.id", String.valueOf(1));
        configs.put("confluent.quota.tenant.follower.broker.min.producer.rate", MIN_BROKER_PRODUCE_QUOTA.toString());
        configs.put("confluent.quota.tenant.broker.max.producer.rate", MAX_BROKER_PRODUCE_QUOTA.toString());
        configs.put("confluent.quota.tenant.follower.broker.min.consumer.rate", MIN_BROKER_CONSUME_QUOTA.toString());
        configs.put("confluent.quota.tenant.broker.max.consumer.rate", MAX_BROKER_CONSUME_QUOTA.toString());
        configs.put("confluent.quota.tenant.default.controller.mutation.rate", DEFAULT_CONTROLLER_QUOTA.toString());
        return configs;
    }

    private Map<String, Object> quotaMultiplierProps(double produceMultiplier, double fetchMultiplier) {
        HashMap<String, Object> configs = new HashMap<String, Object>();
        configs.put("confluent.quota.tenant.produce.multiplier", produceMultiplier);
        configs.put("confluent.quota.tenant.fetch.multiplier", fetchMultiplier);
        return configs;
    }

    private void createCluster(int numNodes) {
        this.testCluster = new TestCluster();
        for (int i = 1; i <= numNodes; ++i) {
            this.testCluster.addNode(i, "rack0");
        }
        Cluster cluster = this.testCluster.cluster();
        this.quotaCallback.updateClusterMetadata((ClientQuotaClusterDescriber)new ZkClientQuotaClusterDescriber(cluster));
        Assertions.assertEquals((Object)cluster, (Object)((ZkClientQuotaClusterDescriber)this.quotaCallback.cluster()).cluster());
    }

    private void verifyQuotas(MultiTenantPrincipal principal, double produceQuota, double fetchQuota, double requestQuota, double controllerQuota) {
        String tenant = principal.tenantMetadata().tenantName;
        this.verifyQuota(ClientQuotaType.PRODUCE, (KafkaPrincipal)principal, produceQuota, tenant);
        this.verifyQuota(ClientQuotaType.FETCH, (KafkaPrincipal)principal, fetchQuota, tenant);
        this.verifyQuota(ClientQuotaType.REQUEST, (KafkaPrincipal)principal, requestQuota, tenant);
        this.verifyQuota(ClientQuotaType.CONTROLLER_MUTATION, (KafkaPrincipal)principal, controllerQuota, tenant);
    }

    private void verifyQuota(ClientQuotaType type, KafkaPrincipal principal, double expectedValue, String expectedTenantTag) {
        this.verifyQuota(type, principal, expectedValue, expectedTenantTag, null);
    }

    private void verifyQuota(ClientQuotaType type, KafkaPrincipal principal, double expectedValue, String expectedTenantTag, Boolean expectQuotaResetRequired) {
        Map metricTags = this.quotaCallback.quotaMetricTags(type, principal, "some-client");
        if (expectedTenantTag != null) {
            Assertions.assertEquals(Collections.singletonMap("tenant", expectedTenantTag), (Object)metricTags);
        } else {
            Assertions.assertTrue((boolean)metricTags.isEmpty(), (String)("Unexpected tags " + metricTags));
        }
        Double quotaLimit = this.quotaCallback.quotaLimit(type, metricTags);
        Assertions.assertEquals((double)expectedValue, (double)quotaLimit, (double)1.0E-4, (String)("Unexpected quota of type " + type));
        if (expectQuotaResetRequired != null) {
            Assertions.assertEquals((Object)expectQuotaResetRequired, (Object)this.quotaCallback.quotaResetRequired(type));
        }
    }

    private void setPartitionLeaders(String topic, int firstPartition, int count, Integer leaderBrokerId) {
        this.testCluster.setPartitionLeaders(topic, firstPartition, count, leaderBrokerId);
        this.quotaCallback.updateClusterMetadata((ClientQuotaClusterDescriber)new ZkClientQuotaClusterDescriber(this.testCluster.cluster()));
    }

    private void deleteTopic(String topic) {
        this.testCluster.deleteTopic(topic);
        this.quotaCallback.updateClusterMetadata((ClientQuotaClusterDescriber)new ZkClientQuotaClusterDescriber(this.testCluster.cluster()));
    }

    private QuotaConfig quotaConfig(long producerByteRate, long consumerByteRate, double requestPercentage) {
        return new QuotaConfig(Long.valueOf(producerByteRate), Long.valueOf(consumerByteRate), Double.valueOf(requestPercentage), null, null, QuotaConfig.UNLIMITED_QUOTA);
    }
}

