/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafkarest.auth;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.kafka.server.plugins.auth.MultiTenantSaslConfigEntry;
import io.confluent.kafka.server.plugins.auth.MultiTenantSaslSecrets;
import io.confluent.kafka.server.plugins.auth.MultiTenantSaslSecretsStore;
import io.confluent.kafkarest.CeKafkaRestConfig;
import io.confluent.kafkarest.KafkaRestConfig;
import io.confluent.kafkarest.auth.AuthorizationHeader;
import io.confluent.kafkarest.auth.CloudPrincipal;
import io.confluent.kafkarest.auth.CloudSecurityContext;
import io.confluent.kafkarest.auth.IdentityPoolIdHeader;
import io.confluent.rest.entities.ErrorMessage;
import java.io.IOException;
import java.net.URI;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Priority;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import org.glassfish.jersey.server.ContainerRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Priority(value=1000)
public final class CloudExtensionsAuthFilter
implements ContainerRequestFilter {
    private static final Logger log = LoggerFactory.getLogger(CloudExtensionsAuthFilter.class);
    private static final String API_KEY_CLUSTER_MISMATCH_TMPL = "API key '%s' is not allowed to access provided cluster ID '%s'";
    public static final String REQUIRE_SECRETS_STORE_VALIDATION_CONFIG = "secrets.store.validation.required";
    public static final String VALIDATE_CLUSTER_ID_IN_URL_CONFIG = "validate.cluster.id.in.url";
    private static final Pattern CLUSTER_ID_PATTERN = Pattern.compile("^lkc-[a-z0-9]+");
    private final boolean requireSecretsStoreValidation;
    private final MultiTenantSaslSecretsStore secretsStore;
    private final boolean validateClusterIdInUrl;

    @Inject
    public CloudExtensionsAuthFilter(Provider<KafkaRestConfig> configProvider, Provider<MultiTenantSaslSecretsStore> secretsStoreProvider) {
        KafkaRestConfig config = (KafkaRestConfig)Objects.requireNonNull(configProvider.get());
        this.requireSecretsStoreValidation = CeKafkaRestConfig.getBooleanOrDefault(config, REQUIRE_SECRETS_STORE_VALIDATION_CONFIG, true);
        this.secretsStore = (MultiTenantSaslSecretsStore)secretsStoreProvider.get();
        if (this.secretsStore == null && this.requireSecretsStoreValidation) {
            log.error("Credentials store not available; validation for API key credentials cannot be performed!");
        }
        this.validateClusterIdInUrl = CeKafkaRestConfig.getBooleanOrDefault(config, VALIDATE_CLUSTER_ID_IN_URL_CONFIG, false);
    }

    public void filter(ContainerRequestContext requestContext) throws IOException {
        Response validationErrorResponse;
        String clusterIdFromPath;
        try {
            clusterIdFromPath = (String)Objects.requireNonNull(requestContext.getUriInfo().getPathParameters(true).getFirst((Object)"clusterId"));
        }
        catch (Exception e) {
            log.debug("Received exception while extracting cluster id from path params.", (Throwable)e);
            requestContext.abortWith(CloudExtensionsAuthFilter.errorResponseWithEntity(Response.Status.BAD_REQUEST.getStatusCode(), "Malformed url, undefined cluster id in request path."));
            return;
        }
        CloudPrincipal cloudPrincipal = CloudExtensionsAuthFilter.extractCredentials(clusterIdFromPath, requestContext);
        if (cloudPrincipal == null) {
            requestContext.abortWith(CloudExtensionsAuthFilter.errorResponseWithEntity(Response.Status.UNAUTHORIZED.getStatusCode(), "Authentication failed."));
            return;
        }
        if (this.validateClusterIdInUrl) {
            Optional<String> clusterId;
            try {
                clusterId = CloudExtensionsAuthFilter.extractClusterIdFromUriHost(requestContext.getUriInfo().getBaseUri());
            }
            catch (IllegalArgumentException e) {
                log.debug("Could not extract cluster id from request URI", (Throwable)e);
                requestContext.abortWith(CloudExtensionsAuthFilter.errorResponseWithEntity(Response.Status.BAD_REQUEST.getStatusCode(), "Invalid request url"));
                return;
            }
            if (clusterId.isPresent() && !cloudPrincipal.getClusterId().equals(clusterId.get())) {
                requestContext.abortWith(CloudExtensionsAuthFilter.errorResponseWithEntity(Response.Status.BAD_REQUEST.getStatusCode(), "Cluster id mismatched between host and path parameter"));
                return;
            }
        }
        if ((validationErrorResponse = this.validateCredentials(cloudPrincipal)) != null) {
            requestContext.abortWith(validationErrorResponse);
            return;
        }
        CloudExtensionsAuthFilter.setSecurityContext(cloudPrincipal, requestContext);
    }

    @VisibleForTesting
    static Optional<String> extractClusterIdFromUriHost(URI uri) throws IllegalArgumentException {
        String host = uri.getHost();
        if (host == null) {
            throw new IllegalArgumentException("Host is null in request URI");
        }
        if (host.startsWith("lkc-")) {
            Matcher matcher = CLUSTER_ID_PATTERN.matcher(host);
            if (matcher.find()) {
                String clusterId = matcher.group();
                return Optional.of(clusterId);
            }
            throw new IllegalArgumentException("Invalid cluster id in host: " + host);
        }
        return Optional.empty();
    }

    private static Response errorResponseWithEntity(int statusCode, String message) {
        ErrorMessage errorMessage = new ErrorMessage(statusCode, message);
        return Response.status((int)statusCode).entity((Object)errorMessage).build();
    }

    private static CloudPrincipal extractCredentials(String clusterId, ContainerRequestContext requestContext) {
        try {
            AuthorizationHeader authorizationHeader = AuthorizationHeader.parse((String)requestContext.getHeaders().getFirst((Object)"Authorization"));
            String identityPoolIdHeaderString = (String)requestContext.getHeaders().getFirst((Object)"Confluent-Identity-Pool-Id");
            Optional<IdentityPoolIdHeader> identityPoolIdHeader = identityPoolIdHeaderString == null ? Optional.empty() : Optional.of(IdentityPoolIdHeader.parse(((ContainerRequest)requestContext.getRequest()).getHeaderString("Confluent-Identity-Pool-Id")));
            return CloudPrincipal.create(clusterId, authorizationHeader, identityPoolIdHeader);
        }
        catch (Exception e) {
            log.debug("Received exception while extracting credentials from request.", (Throwable)e);
            return null;
        }
    }

    private Response validateCredentials(CloudPrincipal cloudPrincipal) {
        if (!AuthorizationHeader.Scheme.BASIC.equals((Object)cloudPrincipal.getScheme())) {
            log.debug("Validation for non-API key credentials will be handled as part of authentication.");
            return null;
        }
        if (this.secretsStore == null) {
            log.error("Credentials store not available; validation for API key credentials cannot be performed!");
            return this.requireSecretsStoreValidation ? Response.status((Response.Status)Response.Status.INTERNAL_SERVER_ERROR).build() : null;
        }
        MultiTenantSaslSecrets secrets = this.secretsStore.load();
        if (secrets == null) {
            log.error("Credential store still unavailable after server initialization!");
            return Response.status((Response.Status)Response.Status.INTERNAL_SERVER_ERROR).build();
        }
        String apiKey = cloudPrincipal.getName();
        MultiTenantSaslConfigEntry userInfo = (MultiTenantSaslConfigEntry)secrets.entries().get(apiKey);
        if (userInfo == null) {
            return Response.status((int)Response.Status.UNAUTHORIZED.toEnum().getStatusCode(), (String)"API-key based authentication failed.").build();
        }
        String providedClusterIdForApiKey = cloudPrincipal.getClusterId();
        String expectedClusterIdForApiKey = userInfo.logicalClusterId();
        if (providedClusterIdForApiKey == null || !providedClusterIdForApiKey.equals(expectedClusterIdForApiKey)) {
            String apiKeyClusterMismatchMessage = String.format(API_KEY_CLUSTER_MISMATCH_TMPL, apiKey, providedClusterIdForApiKey);
            log.debug(apiKeyClusterMismatchMessage);
            ErrorMessage apiKeyClusterMismatch = new ErrorMessage(Response.Status.FORBIDDEN.getStatusCode(), apiKeyClusterMismatchMessage);
            return Response.status((Response.Status)Response.Status.FORBIDDEN).entity((Object)apiKeyClusterMismatch).build();
        }
        return null;
    }

    private static void setSecurityContext(CloudPrincipal cloudPrincipal, ContainerRequestContext requestContext) {
        boolean isSecure = requestContext.getUriInfo().getRequestUri().toString().startsWith("https");
        requestContext.setSecurityContext((SecurityContext)new CloudSecurityContext(cloudPrincipal, isSecure));
    }
}

