/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafkarest.controllers;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import io.confluent.kafkarest.controllers.KafkaConsumerProvider;
import io.confluent.kafkarest.controllers.SimpleConsumeManager;
import io.confluent.kafkarest.controllers.SimpleConsumeModule;
import io.confluent.kafkarest.entities.v3.PartitionConsumeData;
import io.confluent.kafkarest.entities.v3.PartitionConsumeRecord;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.ws.rs.BadRequestException;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SimpleConsumeManagerImpl
implements SimpleConsumeManager {
    private static final Duration POLL_TIMEOUT = Duration.ofSeconds(1L);
    private static final int MAX_POLLS = 5;
    private static final Logger log = LoggerFactory.getLogger(SimpleConsumeManagerImpl.class);
    private final KafkaConsumerProvider consumerProvider;
    private final Executor simpleConsumeExecutor;

    @Inject
    public SimpleConsumeManagerImpl(KafkaConsumerProvider consumerProvider, @SimpleConsumeModule.SimpleConsumeExecutor ExecutorService simpleConsumeExecutor) {
        this.consumerProvider = Objects.requireNonNull(consumerProvider);
        this.simpleConsumeExecutor = Objects.requireNonNull(simpleConsumeExecutor);
    }

    @Override
    public CompletableFuture<PartitionConsumeData> consumeFromPartitionWithOffset(@Nonnull TopicPartition partition, @Nonnull Long offset, Integer maxPollRecords, Integer fetchMaxBytes) {
        log.debug("consumeFromPartitionWithOffset(start): {}", (Object)System.nanoTime());
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(maxPollRecords, fetchMaxBytes);
        return CompletableFuture.supplyAsync(() -> SimpleConsumeManagerImpl.doConsume(consumer, partition, offset), this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("consumeFromPartitionWithOffset(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }

    private KafkaConsumer<byte[], byte[]> getConsumerWithConfigs(Integer maxPollRecords, Integer fetchMaxBytes) {
        Properties consumerConfigOverride = new Properties();
        if (maxPollRecords != null) {
            consumerConfigOverride.put("max.poll.records", maxPollRecords);
        }
        if (fetchMaxBytes != null) {
            consumerConfigOverride.put("fetch.max.bytes", fetchMaxBytes);
        }
        try {
            return this.consumerProvider.getConsumer(consumerConfigOverride);
        }
        catch (ConfigException ce) {
            throw new BadRequestException(ce.getMessage());
        }
    }

    private <T> T handleConsumeResult(T result, Throwable error, KafkaConsumer<byte[], byte[]> consumer) {
        this.consumerProvider.releaseConsumer(consumer);
        if (error != null) {
            throw SimpleConsumeManagerImpl.processError(error);
        }
        return result;
    }

    @Override
    public CompletableFuture<PartitionConsumeData> consumeFromPartitionWithTimestamp(@Nonnull TopicPartition partition, @Nonnull Long timestamp, Integer maxPollRecords, Integer fetchMaxBytes) {
        log.debug("consumeFromPartitionWithTimestamp(start): {}", (Object)System.nanoTime());
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(maxPollRecords, fetchMaxBytes);
        return CompletableFuture.supplyAsync(() -> {
            Map<TopicPartition, Long> timestampMap = Collections.singletonMap(partition, timestamp);
            log.debug("offsetsForTimes(start): {}", (Object)System.nanoTime());
            Map offsetMap = consumer.offsetsForTimes(timestampMap);
            log.debug("offsetsForTimes(end): {}", (Object)System.nanoTime());
            OffsetAndTimestamp offsetData = (OffsetAndTimestamp)offsetMap.get(partition);
            if (offsetData == null) {
                Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
                Long offset = (Long)endOffsets.get(partition);
                return SimpleConsumeManagerImpl.doConsume(consumer, partition, offset);
            }
            return SimpleConsumeManagerImpl.doConsume(consumer, partition, offsetData.offset());
        }, this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("consumeFromPartitionWithTimestamp(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }

    private static PartitionConsumeData doConsume(KafkaConsumer<byte[], byte[]> consumer, TopicPartition partition, long offset) {
        List singlePartitionRecords = Collections.emptyList();
        Set<TopicPartition> singlePartitionSet = Collections.singleton(partition);
        Map<Integer, Long> offsets = Collections.singletonMap(partition.partition(), offset);
        log.debug("endOffsets(start): {}", (Object)System.nanoTime());
        Map endOffsets = consumer.endOffsets(singlePartitionSet);
        log.debug("endOffsets(end): {}", (Object)System.nanoTime());
        if (SimpleConsumeManagerImpl.canConsumeFromOffsets(offsets, endOffsets)) {
            consumer.assign(singlePartitionSet);
            consumer.seek(partition, offset);
            log.debug("poll(start): {}", (Object)System.nanoTime());
            singlePartitionRecords = consumer.poll(POLL_TIMEOUT).records(partition);
            log.debug("poll(end): {}", (Object)System.nanoTime());
        }
        return SimpleConsumeManagerImpl.getPartitionConsumeData(singlePartitionRecords, partition, offsets, endOffsets);
    }

    @VisibleForTesting
    static boolean canConsumeFromOffsets(Map<Integer, Long> offsets, Map<TopicPartition, Long> endOffsets) {
        if (offsets == null || offsets.isEmpty()) {
            return false;
        }
        for (Map.Entry<TopicPartition, Long> endOffset : endOffsets.entrySet()) {
            long offset;
            long partitionEndOffset = endOffset.getValue();
            if (partitionEndOffset < (offset = offsets.get(endOffset.getKey().partition()).longValue())) continue;
            return true;
        }
        return false;
    }

    private static PartitionConsumeData getPartitionConsumeData(List<ConsumerRecord<byte[], byte[]>> singlePartitionRecords, @Nonnull TopicPartition partition, Map<Integer, Long> offsets, @Nonnull Map<TopicPartition, Long> endOffsets) {
        int numRecords;
        int partitionId = partition.partition();
        long nextOffset = endOffsets.get(partition);
        if (offsets != null && offsets.containsKey(partitionId)) {
            nextOffset = Math.min(nextOffset, offsets.get(partitionId));
        }
        int n = numRecords = singlePartitionRecords != null ? singlePartitionRecords.size() : 0;
        if (numRecords > 0) {
            nextOffset = singlePartitionRecords.get(numRecords - 1).offset() + 1L;
        }
        return PartitionConsumeData.builder().setPartitionId(partitionId).setNextOffset(nextOffset).setRecords(PartitionConsumeRecord.listFromConsumerRecordList(singlePartitionRecords)).build();
    }

    private static RuntimeException processError(Throwable t) {
        if (t instanceof CompletionException) {
            t = t.getCause();
        }
        if (t instanceof IllegalArgumentException || t instanceof IllegalStateException) {
            return new BadRequestException(t.getMessage());
        }
        if (t instanceof TimeoutException) {
            return new io.confluent.kafkarest.exceptions.TimeoutException(t);
        }
        if (t instanceof RuntimeException) {
            return (RuntimeException)t;
        }
        return new RuntimeException(t);
    }

    @Override
    public CompletableFuture<List<PartitionConsumeData>> consumeFromMultiplePartitions(@Nonnull String topicName, int numPartitions, Map<Integer, Long> offsets, Boolean fromBeginning, Long timestamp, Integer maxPollRecords, Integer fetchMaxBytes, boolean guaranteeProgressForEachPartition) {
        log.debug("consumeFromMultiPartitions(start): {}", (Object)System.nanoTime());
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(maxPollRecords, fetchMaxBytes);
        List<TopicPartition> partitions = SimpleConsumeManagerImpl.getPartitions(topicName, numPartitions, offsets);
        Supplier<List> consumeLambda = guaranteeProgressForEachPartition ? () -> SimpleConsumeManagerImpl.doConsumeMultiGuaranteeProgress(consumer, partitions, offsets, fromBeginning, timestamp, maxPollRecords) : () -> SimpleConsumeManagerImpl.doConsumeMulti(consumer, partitions, offsets, fromBeginning, timestamp);
        return CompletableFuture.supplyAsync(consumeLambda, this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("consumeFromMultiPartitions(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }

    private static List<TopicPartition> getPartitions(String topicName, int numPartitions, Map<Integer, Long> offsets) {
        if (offsets == null || offsets.isEmpty()) {
            return IntStream.range(0, numPartitions).mapToObj(partitionId -> new TopicPartition(topicName, partitionId)).collect(Collectors.toList());
        }
        return offsets.keySet().stream().map(partitionId -> new TopicPartition(topicName, partitionId.intValue())).collect(Collectors.toList());
    }

    private static List<PartitionConsumeData> doConsumeMulti(KafkaConsumer<byte[], byte[]> consumer, List<TopicPartition> partitions, Map<Integer, Long> offsets, Boolean fromBeginning, Long timestamp) {
        log.debug("endOffsets(start): {}", (Object)System.nanoTime());
        Map endOffsets = consumer.endOffsets(partitions);
        log.debug("endOffsets(end): {}", (Object)System.nanoTime());
        offsets = SimpleConsumeManagerImpl.calculateInitialOffsets(consumer, partitions, offsets, endOffsets, fromBeginning, timestamp);
        ConsumerRecords partitionRecords = new ConsumerRecords(Collections.emptyMap());
        if (SimpleConsumeManagerImpl.canConsumeFromOffsets(offsets, endOffsets)) {
            consumer.assign(partitions);
            for (TopicPartition partition : partitions) {
                consumer.seek(partition, offsets.get(partition.partition()).longValue());
            }
            log.debug("poll(start): {}", (Object)System.nanoTime());
            partitionRecords = consumer.poll(POLL_TIMEOUT);
            log.debug("poll(end): {}", (Object)System.nanoTime());
        }
        ImmutableList.Builder result = new ImmutableList.Builder();
        for (TopicPartition partition : partitions) {
            List kafkaRecords = partitionRecords.records(partition);
            result.add((Object)SimpleConsumeManagerImpl.getPartitionConsumeData(kafkaRecords, partition, offsets, endOffsets));
        }
        return result.build();
    }

    private static List<PartitionConsumeData> doConsumeMultiGuaranteeProgress(KafkaConsumer<byte[], byte[]> consumer, List<TopicPartition> partitions, Map<Integer, Long> offsets, Boolean fromBeginning, Long timestamp, Integer maxPollRecords) {
        log.debug("endOffsets(start): {}", (Object)System.nanoTime());
        Map endOffsets = consumer.endOffsets(partitions);
        log.debug("endOffsets(end): {}", (Object)System.nanoTime());
        offsets = SimpleConsumeManagerImpl.calculateInitialOffsets(consumer, partitions, offsets, endOffsets, fromBeginning, timestamp);
        HashMap<TopicPartition, List> partitionRecordsMap = new HashMap<TopicPartition, List>();
        if (SimpleConsumeManagerImpl.canConsumeFromOffsets(offsets, endOffsets)) {
            int recordLimit;
            consumer.assign(partitions);
            for (TopicPartition partition : partitions) {
                consumer.seek(partition, offsets.get(partition.partition()).longValue());
            }
            int polls = 0;
            int n = recordLimit = maxPollRecords != null ? maxPollRecords - partitions.size() : Integer.MAX_VALUE;
            while (partitionRecordsMap.size() < partitions.size() && polls < 5) {
                log.debug("poll#{} (start): {}", (Object)(++polls), (Object)System.nanoTime());
                ConsumerRecords partitionRecords = consumer.poll(POLL_TIMEOUT);
                log.debug("poll#{} (end): {}", (Object)polls, (Object)System.nanoTime());
                for (TopicPartition partition : partitions) {
                    List kafkaRecords = partitionRecords.records(partition);
                    if (kafkaRecords == null || kafkaRecords.size() <= 0) continue;
                    kafkaRecords = kafkaRecords.subList(0, Math.max(1, Math.min(kafkaRecords.size(), recordLimit + 1)));
                    recordLimit -= kafkaRecords.size() - 1;
                    partitionRecordsMap.computeIfAbsent(partition, p -> new ArrayList()).addAll(kafkaRecords);
                }
                consumer.pause(partitionRecordsMap.keySet());
            }
        }
        ImmutableList.Builder result = new ImmutableList.Builder();
        for (TopicPartition partition : partitions) {
            List kafkaRecords = (List)partitionRecordsMap.get(partition);
            result.add((Object)SimpleConsumeManagerImpl.getPartitionConsumeData(kafkaRecords, partition, offsets, endOffsets));
        }
        return result.build();
    }

    private static Map<Integer, Long> calculateInitialOffsets(KafkaConsumer<byte[], byte[]> consumer, List<TopicPartition> partitions, Map<Integer, Long> offsets, Map<TopicPartition, Long> endOffsets, Boolean fromBeginning, Long timestamp) {
        Map<Integer, Long> initialOffsets;
        if (Boolean.TRUE.equals(fromBeginning)) {
            initialOffsets = consumer.beginningOffsets(partitions).entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), Map.Entry::getValue));
        } else if (Boolean.FALSE.equals(fromBeginning)) {
            initialOffsets = endOffsets.entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), Map.Entry::getValue));
        } else if (timestamp != null) {
            Map<TopicPartition, Long> timestampMap = partitions.stream().collect(Collectors.toMap(tp -> tp, tp -> timestamp));
            log.debug("offsetsForTimes(start): {}", (Object)System.nanoTime());
            Map offsetMap = consumer.offsetsForTimes(timestampMap);
            log.debug("offsetsForTimes(end): {}", (Object)System.nanoTime());
            initialOffsets = offsetMap.entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), e -> {
                if (e.getValue() != null) {
                    return ((OffsetAndTimestamp)e.getValue()).offset();
                }
                return (Long)endOffsets.get(e.getKey());
            }));
        } else {
            initialOffsets = offsets;
        }
        return initialOffsets;
    }
}

