/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.command.kafka;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import io.confluent.command.CommandKeyTypePredicate;
import io.confluent.command.CommandUtil;
import io.confluent.command.Store;
import io.confluent.command.kafka.FilteredReadOnlyKeyValueStore;
import io.confluent.command.record.Command;
import java.time.Duration;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.StoreQueryParameters;
import org.apache.kafka.streams.StreamsBuilder;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.Topology;
import org.apache.kafka.streams.kstream.Consumed;
import org.apache.kafka.streams.kstream.Materialized;
import org.apache.kafka.streams.processor.AbstractProcessor;
import org.apache.kafka.streams.processor.Processor;
import org.apache.kafka.streams.processor.ProcessorSupplier;
import org.apache.kafka.streams.state.QueryableStoreType;
import org.apache.kafka.streams.state.QueryableStoreTypes;
import org.apache.kafka.streams.state.ReadOnlyKeyValueStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommandStore
implements Store<Command.CommandKey, Command.CommandMessage, RecordMetadata> {
    public static final String DEFAULT_COMMAND_TOPIC = "_confluent-command";
    private static final Logger log = LoggerFactory.getLogger(CommandStore.class);
    private static final long DEFAULT_START_TIMEOUT = TimeUnit.SECONDS.toMillis(30L);
    private static final long DEFAULT_STOP_TIMEOUT = TimeUnit.SECONDS.toMillis(30L);
    private static final long DEFAULT_AWAIT_TIMEOUT = TimeUnit.SECONDS.toMillis(15L);
    public static final String COMMANDER = "commander";
    private final String commandTopic;
    private final KafkaStreams streamsJob;
    private final KafkaProducer<Command.CommandKey, Command.CommandMessage> producer;
    private final AtomicLong highOffset;

    protected CommandStore(String commandTopic, KafkaStreams streamsJob, KafkaProducer<Command.CommandKey, Command.CommandMessage> producer, AtomicLong highOffset) {
        this.commandTopic = commandTopic;
        this.streamsJob = streamsJob;
        this.producer = producer;
        this.highOffset = highOffset;
    }

    @Override
    public void start() throws TimeoutException, InterruptedException {
        this.start(DEFAULT_START_TIMEOUT);
    }

    public void start(long allowance) throws TimeoutException, InterruptedException {
        long timeout = System.currentTimeMillis() + allowance;
        this.streamsJob.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler(){

            @Override
            public void uncaughtException(Thread t, Throwable e) {
                log.error("streams uncaught exception thread={}", (Object)t, (Object)e);
            }
        });
        this.streamsJob.start();
        while (this.streamsJob.state() != KafkaStreams.State.RUNNING) {
            log.info("waiting for streams to be in running state. Current state is {}", (Object)this.streamsJob.state());
            if (timeout < System.currentTimeMillis()) {
                log.warn("unable to start with allowance={}", (Object)allowance);
                throw new TimeoutException();
            }
            Thread.sleep(TimeUnit.SECONDS.toMillis(1L));
        }
        log.info("Streams state is {}", (Object)this.streamsJob.state());
        ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage> store = null;
        while (store == null) {
            try {
                store = this.getStore();
            }
            catch (Exception e) {
                log.info("waiting for command store to materialize {}", log.isDebugEnabled() ? e : "");
            }
            if (timeout < System.currentTimeMillis()) {
                log.warn("unable to start with allowance={}", (Object)allowance);
                throw new TimeoutException();
            }
            Thread.sleep(TimeUnit.SECONDS.toMillis(1L));
        }
    }

    @Override
    public void close() {
        this.close(false);
    }

    public void close(boolean clean) {
        try {
            this.streamsJob.close(Duration.ofMillis(DEFAULT_STOP_TIMEOUT));
            if (clean) {
                this.streamsJob.cleanUp();
            }
        }
        catch (Exception e) {
            log.error("closing streams job", (Throwable)e);
        }
        try {
            this.producer.close(Duration.ofMillis(DEFAULT_STOP_TIMEOUT));
        }
        catch (Exception e) {
            log.error("closing producer", (Throwable)e);
        }
    }

    @Override
    public Future<RecordMetadata> updateAsync(Command.CommandKey key, Command.CommandMessage command) {
        return this.producer.send(new ProducerRecord(this.commandTopic, Integer.valueOf(0), (Object)key, (Object)command));
    }

    @Override
    public RecordMetadata update(Command.CommandKey key, Command.CommandMessage command) throws InterruptedException, TimeoutException, ExecutionException {
        return this.update(key, command, DEFAULT_AWAIT_TIMEOUT);
    }

    @Override
    public RecordMetadata update(Command.CommandKey key, Command.CommandMessage command, long timeoutMs) throws InterruptedException, TimeoutException, ExecutionException {
        if (command != null) {
            Command.CommandMessage mangledCommand = CommandUtil.createUniqueCommandMessage(command);
            return this.awaitUpdate(key, mangledCommand, timeoutMs);
        }
        return this.awaitDelete(key, timeoutMs);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RecordMetadata awaitUpdate(Command.CommandKey key, Command.CommandMessage command, long timeoutMs) throws InterruptedException, ExecutionException, TimeoutException {
        Preconditions.checkNotNull((Object)command, (Object)"command must not be null");
        long endTime = System.currentTimeMillis() + timeoutMs;
        RecordMetadata recordMetadata = this.updateAsync(key, command).get(timeoutMs, TimeUnit.MILLISECONDS);
        AtomicLong atomicLong = this.highOffset;
        synchronized (atomicLong) {
            while (this.highOffset.get() < recordMetadata.offset() && endTime > System.currentTimeMillis()) {
                this.highOffset.wait(Math.max(endTime - System.currentTimeMillis(), 0L));
            }
        }
        if (this.highOffset.get() < recordMetadata.offset()) {
            throw new TimeoutException("Timed-out waiting to see " + key + " update at offset " + recordMetadata.offset());
        }
        return recordMetadata;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RecordMetadata awaitDelete(Command.CommandKey key, long timeoutMs) throws InterruptedException, ExecutionException, TimeoutException {
        RecordMetadata recordMetadata = this.updateAsync(key, null).get(timeoutMs, TimeUnit.MILLISECONDS);
        long endTime = System.currentTimeMillis() + timeoutMs;
        AtomicLong atomicLong = this.highOffset;
        synchronized (atomicLong) {
            while (this.get(key) != null && endTime > System.currentTimeMillis()) {
                this.highOffset.wait(100L);
            }
        }
        if (this.get(key) != null) {
            throw new TimeoutException("Timed-out waiting to delete " + key);
        }
        return recordMetadata;
    }

    private Command.CommandMessage get(Command.CommandKey key) {
        return (Command.CommandMessage)this.getStore().get((Object)key);
    }

    @Override
    public ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage> getStore() {
        return (ReadOnlyKeyValueStore)this.streamsJob.store(StoreQueryParameters.fromNameAndType((String)COMMANDER, (QueryableStoreType)QueryableStoreTypes.keyValueStore()));
    }

    @Override
    public ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage> getStore(Command.CommandConfigType type) {
        return new FilteredReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>(this.getStore(), new CommandKeyTypePredicate(type));
    }

    public static void main(String[] args) throws Throwable {
        try (CommandStore commander = new Builder().build();){
            commander.start();
            Command.CommandKey key = Command.CommandKey.newBuilder().setConfigType(Command.CommandConfigType.LICENSE_INFO).setGuid("guid").build();
            commander.update(key, Command.CommandMessage.newBuilder().setLicenseInfo(Command.LicenseInfo.newBuilder().setJwt("jwt 9").build()).build());
            log.info("{}", (Object)commander.get(key));
        }
    }

    public static class Builder {
        private static final Map<String, Object> DEFAULT_PROPERTIES = ImmutableMap.of((Object)"application.id", (Object)"test-commander", (Object)"bootstrap.servers", (Object)"localhost:9092", (Object)"topology.optimization", (Object)"all");
        private String commandTopic = "_confluent-command";
        private StreamsConfig streamsConfig = null;
        private Map<String, Object> producerConfig = DEFAULT_PROPERTIES;

        public Builder topic(String topic) {
            this.commandTopic = topic;
            return this;
        }

        public Builder streamsConfig(StreamsConfig streamsConfigs) {
            this.streamsConfig = streamsConfigs;
            return this;
        }

        public Builder producerConfig(Map<String, Object> producerConfig) {
            this.producerConfig = producerConfig;
            return this;
        }

        public CommandStore build() {
            final AtomicLong highOffset = new AtomicLong();
            StreamsBuilder builder = new StreamsBuilder();
            builder.table(this.commandTopic, Consumed.with((Topology.AutoOffsetReset)Topology.AutoOffsetReset.EARLIEST).withKeySerde(CommandUtil.commandKeySerde).withValueSerde(CommandUtil.commandMessageSerde), Materialized.as((String)CommandStore.COMMANDER)).toStream().process((ProcessorSupplier)new ProcessorSupplier<Command.CommandKey, Command.CommandMessage>(){

                public Processor<Command.CommandKey, Command.CommandMessage> get() {
                    return new AbstractProcessor<Command.CommandKey, Command.CommandMessage>(){

                        /*
                         * WARNING - Removed try catching itself - possible behaviour change.
                         */
                        public void process(Command.CommandKey commandKey, Command.CommandMessage commandMessage) {
                            highOffset.set(this.context().offset());
                            AtomicLong atomicLong = highOffset;
                            synchronized (atomicLong) {
                                highOffset.notifyAll();
                            }
                        }
                    };
                }
            }, new String[0]);
            Properties props = new Properties();
            if (this.streamsConfig == null) {
                props.putAll(DEFAULT_PROPERTIES);
            } else {
                props.putAll((Map<?, ?>)this.streamsConfig.originals());
            }
            return new CommandStore(this.commandTopic, new KafkaStreams(builder.build(props), this.streamsConfig == null ? new StreamsConfig(DEFAULT_PROPERTIES) : this.streamsConfig), (KafkaProducer<Command.CommandKey, Command.CommandMessage>)new KafkaProducer(this.producerConfig, CommandUtil.commandKeySerde, CommandUtil.commandMessageSerde), highOffset);
        }
    }
}

