/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.schema.registry;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.kafka.schemaregistry.ParsedSchema;
import io.confluent.kafka.schemaregistry.client.SchemaMetadata;
import io.confluent.kafka.schemaregistry.client.SchemaRegistryClient;
import io.confluent.kafka.schemaregistry.client.rest.exceptions.RestClientException;
import io.confluent.kafka.schemaregistry.protobuf.ProtobufSchema;
import io.confluent.kafka.serializers.protobuf.AbstractKafkaProtobufSerializer;
import io.confluent.kafka.serializers.subject.DefaultReferenceSubjectNameStrategy;
import io.confluent.kafka.serializers.subject.strategy.ReferenceSubjectNameStrategy;
import io.confluent.ksql.exception.KsqlSchemaAuthorizationException;
import io.confluent.ksql.schema.registry.SchemaAndId;
import io.confluent.ksql.util.ExecutorUtil;
import io.confluent.ksql.util.KsqlConstants;
import io.confluent.ksql.util.KsqlException;
import java.io.IOException;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.kafka.common.acl.AclOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SchemaRegistryUtil {
    private static final Logger LOG = LoggerFactory.getLogger(SchemaRegistryUtil.class);
    private static final Pattern DENIED_OPERATION_STRING_PATTERN = Pattern.compile("User is denied operation (.*) on .*");
    @VisibleForTesting
    public static final int SUBJECT_NOT_FOUND_ERROR_CODE = 40401;

    private SchemaRegistryUtil() {
    }

    public static void cleanupInternalTopicSchemas(String applicationId, SchemaRegistryClient schemaRegistryClient) {
        SchemaRegistryUtil.getInternalSubjectNames(applicationId, schemaRegistryClient).forEach(subject -> SchemaRegistryUtil.tryDeleteInternalSubject(applicationId, schemaRegistryClient, subject));
    }

    public static Stream<String> getSubjectNames(SchemaRegistryClient schemaRegistryClient) {
        return SchemaRegistryUtil.getSubjectNames(schemaRegistryClient, "Could not get subject names from schema registry.");
    }

    private static Stream<String> getSubjectNames(SchemaRegistryClient schemaRegistryClient, String errorMsg) {
        try {
            return schemaRegistryClient.getAllSubjects().stream();
        }
        catch (Exception e) {
            LOG.warn(errorMsg, (Throwable)e);
            return Stream.empty();
        }
    }

    public static void deleteSubjectWithRetries(SchemaRegistryClient schemaRegistryClient, String subject) throws Exception {
        try {
            ExecutorUtil.executeWithRetries(() -> schemaRegistryClient.deleteSubject(subject), error -> SchemaRegistryUtil.isRetriableError(error));
        }
        catch (RestClientException e) {
            if (SchemaRegistryUtil.isAuthErrorCode(e)) {
                throw new KsqlSchemaAuthorizationException(AclOperation.DELETE, subject);
            }
            throw e;
        }
    }

    public static boolean subjectExists(SchemaRegistryClient srClient, String subject) {
        return SchemaRegistryUtil.getLatestSchema(srClient, subject).isPresent();
    }

    public static Optional<Integer> getLatestSchemaId(SchemaRegistryClient srClient, String topic, boolean isKey) {
        String subject = KsqlConstants.getSRSubject((String)topic, (boolean)isKey);
        return SchemaRegistryUtil.getLatestSchema(srClient, subject).map(SchemaMetadata::getId);
    }

    public static Optional<SchemaAndId> getLatestSchemaAndId(SchemaRegistryClient srClient, String topic, boolean isKey) {
        String subject = KsqlConstants.getSRSubject((String)topic, (boolean)isKey);
        return SchemaRegistryUtil.getLatestSchemaId(srClient, topic, isKey).map(id -> {
            try {
                return new SchemaAndId(srClient.getSchemaById(id.intValue()), (int)id);
            }
            catch (Exception e) {
                SchemaRegistryUtil.throwOnAuthError(e, subject);
                throw new KsqlException("Could not get schema for subject " + subject + " and id " + id, (Throwable)e);
            }
        });
    }

    private static void throwOnAuthError(Exception e, String subject) {
        AclOperation deniedOperation;
        if (SchemaRegistryUtil.isAuthErrorCode(e) && (deniedOperation = SchemaRegistryUtil.getDeniedOperation(e.getMessage())) != AclOperation.UNKNOWN) {
            throw new KsqlSchemaAuthorizationException(deniedOperation, subject);
        }
    }

    public static Optional<SchemaMetadata> getLatestSchema(SchemaRegistryClient srClient, String topic, boolean getKeySchema) {
        String subject = KsqlConstants.getSRSubject((String)topic, (boolean)getKeySchema);
        return SchemaRegistryUtil.getLatestSchema(srClient, subject);
    }

    public static Optional<SchemaMetadata> getLatestSchema(SchemaRegistryClient srClient, String subject) {
        try {
            SchemaMetadata schemaMetadata = srClient.getLatestSchemaMetadata(subject);
            return Optional.ofNullable(schemaMetadata);
        }
        catch (Exception e) {
            if (SchemaRegistryUtil.isSubjectNotFoundErrorCode(e)) {
                return Optional.empty();
            }
            SchemaRegistryUtil.throwOnAuthError(e, subject);
            throw new KsqlException("Could not get latest schema for subject " + subject, (Throwable)e);
        }
    }

    public static AclOperation getDeniedOperation(String errorMessage) {
        Matcher matcher = DENIED_OPERATION_STRING_PATTERN.matcher(errorMessage);
        if (matcher.matches()) {
            return AclOperation.fromString((String)matcher.group(1));
        }
        return AclOperation.UNKNOWN;
    }

    public static boolean isSubjectNotFoundErrorCode(Throwable error) {
        return error instanceof RestClientException && ((RestClientException)error).getErrorCode() == 40401;
    }

    public static boolean isAuthErrorCode(Throwable error) {
        return error instanceof RestClientException && (((RestClientException)error).getStatus() == 401 || ((RestClientException)error).getStatus() == 403);
    }

    private static boolean isRetriableError(Throwable error) {
        return !SchemaRegistryUtil.isSubjectNotFoundErrorCode(error) && !SchemaRegistryUtil.isAuthErrorCode(error);
    }

    private static void hardDeleteSubjectWithRetries(SchemaRegistryClient schemaRegistryClient, String subject) throws Exception {
        try {
            ExecutorUtil.executeWithRetries(() -> schemaRegistryClient.deleteSubject(subject, true), error -> SchemaRegistryUtil.isRetriableError(error));
        }
        catch (RestClientException e) {
            AclOperation deniedOperation;
            if (SchemaRegistryUtil.isAuthErrorCode(e) && (deniedOperation = SchemaRegistryUtil.getDeniedOperation(e.getMessage())) != AclOperation.UNKNOWN) {
                throw new KsqlSchemaAuthorizationException(deniedOperation, subject);
            }
            throw e;
        }
    }

    private static Stream<String> getInternalSubjectNames(String applicationId, SchemaRegistryClient schemaRegistryClient) {
        Stream<String> allSubjectNames = SchemaRegistryUtil.getSubjectNames(schemaRegistryClient, "Could not clean up the schema registry for query: " + applicationId);
        return allSubjectNames.filter(subjectName -> subjectName.startsWith(applicationId)).filter(SchemaRegistryUtil::isInternalSubject);
    }

    private static boolean isInternalSubject(String subjectName) {
        for (boolean isKey : new boolean[]{true, false}) {
            String changelog = KsqlConstants.getSRSubject((String)"-changelog", (boolean)isKey);
            String repartition = KsqlConstants.getSRSubject((String)"-repartition", (boolean)isKey);
            if (!subjectName.endsWith(changelog) && !subjectName.endsWith(repartition)) continue;
            return true;
        }
        return false;
    }

    private static void tryDeleteInternalSubject(String applicationId, SchemaRegistryClient schemaRegistryClient, String subjectName) {
        try {
            SchemaRegistryUtil.deleteSubjectWithRetries(schemaRegistryClient, subjectName);
            SchemaRegistryUtil.hardDeleteSubjectWithRetries(schemaRegistryClient, subjectName);
        }
        catch (Exception e) {
            LOG.warn("Could not clean up the schema registry for query: " + applicationId + ", subject: " + subjectName, (Throwable)e);
        }
    }

    public static int registerSchema(SchemaRegistryClient srClient, ParsedSchema parsedSchema, String topic, String subject, boolean isKey) throws KsqlSchemaAuthorizationException, KsqlException {
        try {
            if (parsedSchema instanceof ProtobufSchema) {
                ProtobufSchema resolved = AbstractKafkaProtobufSerializer.resolveDependencies((SchemaRegistryClient)srClient, (boolean)true, (boolean)false, (boolean)true, null, (ReferenceSubjectNameStrategy)new DefaultReferenceSubjectNameStrategy(), (String)topic, (boolean)isKey, (ProtobufSchema)((ProtobufSchema)parsedSchema));
                return srClient.register(subject, (ParsedSchema)resolved);
            }
            return srClient.register(subject, parsedSchema);
        }
        catch (RestClientException | IOException e) {
            AclOperation deniedOperation;
            if (SchemaRegistryUtil.isAuthErrorCode(e) && (deniedOperation = SchemaRegistryUtil.getDeniedOperation(e.getMessage())) != AclOperation.UNKNOWN) {
                throw new KsqlSchemaAuthorizationException(deniedOperation, subject);
            }
            throw new KsqlException("Could not register schema for topic: " + e.getMessage(), e);
        }
    }
}

