/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.services;

import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import io.confluent.ksql.exception.KafkaDeleteTopicsException;
import io.confluent.ksql.exception.KafkaResponseGetFailedException;
import io.confluent.ksql.exception.KsqlTopicAuthorizationException;
import io.confluent.ksql.services.KafkaClusterUtil;
import io.confluent.ksql.services.KafkaTopicClient;
import io.confluent.ksql.services.TopicValidationUtil;
import io.confluent.ksql.util.ExecutorUtil;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.KsqlServerException;
import io.confluent.ksql.util.Pair;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.AlterConfigOp;
import org.apache.kafka.clients.admin.Config;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.clients.admin.CreateTopicsOptions;
import org.apache.kafka.clients.admin.DeleteTopicsResult;
import org.apache.kafka.clients.admin.DescribeTopicsOptions;
import org.apache.kafka.clients.admin.ListOffsetsResult;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.OffsetSpec;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.acl.AclOperation;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.errors.TopicDeletionDisabledException;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class KafkaTopicClientImpl
implements KafkaTopicClient {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaTopicClient.class);
    private static final String DEFAULT_REPLICATION_PROP = "default.replication.factor";
    private static final String DELETE_TOPIC_ENABLE = "delete.topic.enable";
    private final Supplier<Admin> adminClient;

    public KafkaTopicClientImpl(Supplier<Admin> sharedAdminClient) {
        this.adminClient = Objects.requireNonNull(sharedAdminClient, "sharedAdminClient");
    }

    public void createTopic(String topic, int numPartitions, short replicationFactor, Map<String, ?> configs, CreateTopicsOptions createOptions) {
        Optional retentionMs = KafkaTopicClient.getRetentionMs(configs);
        if (this.isTopicExists(topic)) {
            this.validateTopicProperties(topic, numPartitions, replicationFactor, retentionMs);
            return;
        }
        short resolvedReplicationFactor = replicationFactor == -1 ? this.getDefaultClusterReplication() : replicationFactor;
        NewTopic newTopic = new NewTopic(topic, numPartitions, resolvedReplicationFactor);
        newTopic.configs(KafkaTopicClientImpl.toStringConfigs(configs));
        try {
            LOG.info("Creating topic '{}' {}", (Object)topic, (Object)(createOptions.shouldValidateOnly() ? "(ONLY VALIDATE)" : ""));
            ExecutorUtil.executeWithRetries(() -> (Void)this.adminClient.get().createTopics(Collections.singleton(newTopic), createOptions).all().get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new KafkaResponseGetFailedException("Failed to guarantee existence of topic " + topic, (Throwable)e);
        }
        catch (TopicExistsException e) {
            this.validateTopicProperties(topic, numPartitions, replicationFactor, retentionMs);
        }
        catch (TopicAuthorizationException e) {
            throw new KsqlTopicAuthorizationException(AclOperation.CREATE, Collections.singleton(topic));
        }
        catch (Exception e) {
            throw new KafkaResponseGetFailedException("Failed to guarantee existence of topic " + topic, (Throwable)e);
        }
    }

    private short getDefaultClusterReplication() {
        try {
            String defaultReplication = this.getConfig().get(DEFAULT_REPLICATION_PROP).value();
            return Short.parseShort(defaultReplication);
        }
        catch (KsqlServerException e) {
            throw e;
        }
        catch (Exception e) {
            throw new KsqlServerException("Could not get default replication from Kafka cluster!", (Throwable)e);
        }
    }

    public boolean isTopicExists(String topic) {
        LOG.trace("Checking for existence of topic '{}'", (Object)topic);
        try {
            ExecutorUtil.executeWithRetries(() -> (TopicDescription)((KafkaFuture)this.adminClient.get().describeTopics((Collection)ImmutableList.of((Object)topic), new DescribeTopicsOptions().includeAuthorizedOperations(true)).topicNameValues().get(topic)).get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE.and(e -> !(e instanceof UnknownTopicOrPartitionException)));
            return true;
        }
        catch (TopicAuthorizationException e2) {
            throw new KsqlTopicAuthorizationException(AclOperation.DESCRIBE, Collections.singleton(topic));
        }
        catch (Exception e3) {
            if (Throwables.getRootCause((Throwable)e3) instanceof UnknownTopicOrPartitionException) {
                return false;
            }
            throw new KafkaResponseGetFailedException("Failed to check if exists for topic: " + topic, (Throwable)e3);
        }
    }

    public Set<String> listTopicNames() {
        try {
            return ExecutorUtil.executeWithRetries(() -> (Set)this.adminClient.get().listTopics().names().get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE);
        }
        catch (Exception e) {
            throw new KafkaResponseGetFailedException("Failed to retrieve Kafka Topic names", (Throwable)e);
        }
    }

    public Map<String, TopicDescription> describeTopics(Collection<String> topicNames, Boolean skipRetriesOnFailure) {
        try {
            if (skipRetriesOnFailure.booleanValue()) {
                return (Map)this.adminClient.get().describeTopics(topicNames, new DescribeTopicsOptions().includeAuthorizedOperations(true)).allTopicNames().get();
            }
            return ExecutorUtil.executeWithRetries(() -> (Map)this.adminClient.get().describeTopics(topicNames, new DescribeTopicsOptions().includeAuthorizedOperations(true)).allTopicNames().get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE);
        }
        catch (ExecutionException e) {
            throw new KafkaResponseGetFailedException("Failed to Describe Kafka Topic(s): " + topicNames, e.getCause());
        }
        catch (TopicAuthorizationException e) {
            throw new KsqlTopicAuthorizationException(AclOperation.DESCRIBE, topicNames);
        }
        catch (Exception e) {
            throw new KafkaResponseGetFailedException("Failed to Describe Kafka Topic(s): " + topicNames, (Throwable)e);
        }
    }

    public Map<String, String> getTopicConfig(String topicName) {
        return this.topicConfig(topicName, true);
    }

    public boolean addTopicConfig(String topicName, Map<String, ?> overrides) {
        ConfigResource resource = new ConfigResource(ConfigResource.Type.TOPIC, topicName);
        Map<String, String> stringConfigs = KafkaTopicClientImpl.toStringConfigs(overrides);
        try {
            Map<String, String> existingConfig = this.topicConfig(topicName, false);
            boolean changed = stringConfigs.entrySet().stream().anyMatch(e -> !Objects.equals(existingConfig.get(e.getKey()), e.getValue()));
            if (!changed) {
                return false;
            }
            Set entries = stringConfigs.entrySet().stream().map(e -> new ConfigEntry((String)e.getKey(), (String)e.getValue())).map(ce -> new AlterConfigOp(ce, AlterConfigOp.OpType.SET)).collect(Collectors.toSet());
            Map request = Collections.singletonMap(resource, entries);
            ExecutorUtil.executeWithRetries(() -> (Void)this.adminClient.get().incrementalAlterConfigs(request).all().get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE);
            return true;
        }
        catch (UnsupportedVersionException e2) {
            return this.addTopicConfigLegacy(topicName, stringConfigs);
        }
        catch (Exception e3) {
            throw new KafkaResponseGetFailedException("Failed to set config for Kafka Topic " + topicName, (Throwable)e3);
        }
    }

    public KafkaTopicClient.TopicCleanupPolicy getTopicCleanupPolicy(String topicName) {
        String policy = this.getTopicConfig(topicName).getOrDefault("cleanup.policy", "").toLowerCase();
        if (policy.equals("compact")) {
            return KafkaTopicClient.TopicCleanupPolicy.COMPACT;
        }
        if (policy.equals("delete")) {
            return KafkaTopicClient.TopicCleanupPolicy.DELETE;
        }
        if (policy.contains("compact") && policy.contains("delete")) {
            return KafkaTopicClient.TopicCleanupPolicy.COMPACT_DELETE;
        }
        throw new KsqlException("Could not get the topic configs for : " + topicName);
    }

    public void deleteTopics(Collection<String> topicsToDelete) {
        if (topicsToDelete.isEmpty()) {
            return;
        }
        DeleteTopicsResult deleteTopicsResult = this.adminClient.get().deleteTopics(topicsToDelete);
        Map results = deleteTopicsResult.topicNameValues();
        ArrayList failList = Lists.newArrayList();
        ArrayList exceptionList = Lists.newArrayList();
        for (Map.Entry entry : results.entrySet()) {
            try {
                ((KafkaFuture)entry.getValue()).get(30L, TimeUnit.SECONDS);
            }
            catch (Exception e) {
                Throwable rootCause = ExceptionUtils.getRootCause((Throwable)e);
                if (rootCause instanceof TopicDeletionDisabledException) {
                    throw new TopicDeletionDisabledException("Topic deletion is disabled. To delete the topic, you must set 'delete.topic.enable' to true in the Kafka broker configuration.");
                }
                if (rootCause instanceof TopicAuthorizationException) {
                    throw new KsqlTopicAuthorizationException(AclOperation.DELETE, Collections.singleton(entry.getKey()));
                }
                if (rootCause instanceof UnknownTopicOrPartitionException) continue;
                LOG.error(String.format("Could not delete topic '%s'", entry.getKey()), (Throwable)e);
                failList.add(entry.getKey());
                exceptionList.add(new Pair(entry.getKey(), (Object)rootCause));
            }
        }
        if (!failList.isEmpty()) {
            throw new KafkaDeleteTopicsException("Failed to clean up topics: " + String.join((CharSequence)",", failList), (List)exceptionList);
        }
    }

    public void deleteInternalTopics(String internalTopicPrefix) {
        try {
            Set<String> topicNames = this.listTopicNames();
            ArrayList internalTopics = Lists.newArrayList();
            for (String topicName : topicNames) {
                if (!KafkaTopicClientImpl.isInternalTopic(topicName, internalTopicPrefix)) continue;
                internalTopics.add(topicName);
            }
            if (!internalTopics.isEmpty()) {
                Collections.sort(internalTopics);
                this.deleteTopics(internalTopics);
            }
        }
        catch (Exception e) {
            LOG.error("Exception while trying to clean up internal topics for application id: {}.", (Object)internalTopicPrefix, (Object)e);
        }
    }

    public Map<TopicPartition, Long> listTopicsOffsets(Collection<String> topicNames, OffsetSpec offsetSpec) {
        Map<TopicPartition, OffsetSpec> offsetsRequest = this.describeTopics(topicNames).entrySet().stream().flatMap(entry -> ((TopicDescription)entry.getValue()).partitions().stream().map(tpInfo -> new TopicPartition((String)entry.getKey(), tpInfo.partition()))).collect(Collectors.toMap(tp -> tp, tp -> offsetSpec));
        try {
            return ExecutorUtil.executeWithRetries(() -> ((Map)this.adminClient.get().listOffsets(offsetsRequest).all().get()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((ListOffsetsResult.ListOffsetsResultInfo)entry.getValue()).offset())), ExecutorUtil.RetryBehaviour.ON_RETRYABLE);
        }
        catch (TopicAuthorizationException e) {
            throw new KsqlTopicAuthorizationException(AclOperation.DESCRIBE, e.unauthorizedTopics());
        }
        catch (ExecutionException e) {
            throw new KafkaResponseGetFailedException("Failed to get topic offsets. partitions: " + offsetsRequest.keySet(), e.getCause());
        }
        catch (Exception e) {
            throw new KafkaResponseGetFailedException("Failed to get topic offsets. partitions: " + offsetsRequest.keySet(), (Throwable)e);
        }
    }

    private Config getConfig() {
        return KafkaClusterUtil.getConfig(this.adminClient.get());
    }

    private static boolean isInternalTopic(String topicName, String applicationId) {
        boolean prefixMatches = topicName.startsWith(applicationId + "-");
        boolean suffixMatches = topicName.endsWith("-changelog") || topicName.endsWith("-repartition") || topicName.matches(".+-KTABLE-FK-JOIN-SUBSCRIPTION-REGISTRATION-\\d+-topic") || topicName.matches(".+-KTABLE-FK-JOIN-SUBSCRIPTION-RESPONSE-\\d+-topic");
        return prefixMatches && suffixMatches;
    }

    private void validateTopicProperties(String topic, int requiredNumPartition, int requiredNumReplicas, Optional<Long> requiredRetentionMs) {
        TopicDescription existingTopic = this.describeTopic(topic);
        Map<String, String> existingConfig = this.getTopicConfig(topic);
        TopicValidationUtil.validateTopicProperties(requiredNumPartition, requiredNumReplicas, requiredRetentionMs, existingTopic, existingConfig);
        LOG.debug("Did not create topic {} with {} partitions, replication-factor {}, and retention {} since it exists", new Object[]{topic, requiredNumPartition, requiredNumReplicas, requiredRetentionMs});
    }

    private Map<String, String> topicConfig(String topicName, boolean includeDefaults) {
        ConfigResource resource = new ConfigResource(ConfigResource.Type.TOPIC, topicName);
        List<ConfigResource> request = Collections.singletonList(resource);
        try {
            Config config = (Config)ExecutorUtil.executeWithRetries(() -> (Map)this.adminClient.get().describeConfigs((Collection)request).all().get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE).get(resource);
            return config.entries().stream().filter(e -> e.value() != null).filter(e -> includeDefaults || !e.isDefault()).collect(Collectors.toMap(ConfigEntry::name, ConfigEntry::value));
        }
        catch (TopicAuthorizationException e2) {
            throw new KsqlTopicAuthorizationException(AclOperation.DESCRIBE_CONFIGS, e2.unauthorizedTopics());
        }
        catch (Exception e3) {
            throw new KafkaResponseGetFailedException("Failed to get config for Kafka Topic " + topicName, (Throwable)e3);
        }
    }

    private boolean addTopicConfigLegacy(String topicName, Map<String, String> overrides) {
        ConfigResource resource = new ConfigResource(ConfigResource.Type.TOPIC, topicName);
        try {
            Map<String, String> existingConfig = this.topicConfig(topicName, false);
            existingConfig.putAll(overrides);
            Set entries = existingConfig.entrySet().stream().map(e -> new ConfigEntry((String)e.getKey(), (String)e.getValue())).collect(Collectors.toSet());
            Map<ConfigResource, Config> request = Collections.singletonMap(resource, new Config(entries));
            ExecutorUtil.executeWithRetries(() -> (Void)this.adminClient.get().alterConfigs(request).all().get(), ExecutorUtil.RetryBehaviour.ON_RETRYABLE);
            return true;
        }
        catch (Exception e2) {
            throw new KafkaResponseGetFailedException("Failed to set config for Kafka Topic " + topicName, (Throwable)e2);
        }
    }

    private static Map<String, String> toStringConfigs(Map<String, ?> configs) {
        return configs.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().toString()));
    }
}

