/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.telemetry.collector;

import io.confluent.shaded.com.google.common.base.MoreObjects;
import io.confluent.shaded.com.google.common.base.Verify;
import io.confluent.telemetry.ConfluentTelemetryConfig;
import io.confluent.telemetry.Context;
import io.confluent.telemetry.MetricKey;
import io.confluent.telemetry.collector.ConfluentMetricNamingConvention;
import io.confluent.telemetry.collector.MetricsCollector;
import io.confluent.telemetry.emitter.Emitter;
import io.confluent.telemetry.metrics.SinglePointMetric;
import java.io.IOException;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import kafka.server.KafkaConfig;
import org.apache.kafka.common.config.AbstractConfig;
import org.apache.kafka.common.config.ConfigDef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VolumeMetricsCollector
implements MetricsCollector {
    public static final String VOLUME_LABEL = "volume";
    private static final Logger log = LoggerFactory.getLogger(VolumeMetricsCollector.class);
    private long lastUpdateNs;
    private Map<String, VolumeInfo> cachedMetrics = null;
    private Map<String, Map<String, String>> labelsCache = new HashMap<String, Map<String, String>>();
    private long updatePeriodMs;
    private final String[] logDirs;
    private final Context context;
    private final String diskTotalBytesName;
    private final String diskUsableBytesName;

    private VolumeMetricsCollector(Builder builder) {
        this.updatePeriodMs = builder.updatePeriodMs;
        this.logDirs = builder.logDirs;
        this.context = builder.context;
        this.diskTotalBytesName = ConfluentMetricNamingConvention.fullMetricName("io.confluent.system", VOLUME_LABEL, "disk_total_bytes");
        this.diskUsableBytesName = ConfluentMetricNamingConvention.fullMetricName("io.confluent.system", VOLUME_LABEL, "disk_usable_bytes");
    }

    private synchronized Map<String, String> labelsFor(String volumeName) {
        Map<String, String> labels = this.labelsCache.get(volumeName);
        if (labels == null) {
            labels = new HashMap<String, String>();
            if (this.context.isDebugEnabled()) {
                labels.put("library", "none");
            }
            labels.put(VOLUME_LABEL, volumeName);
            this.labelsCache.put(volumeName, labels);
        }
        return Collections.unmodifiableMap(labels);
    }

    @Override
    public void collect(Emitter emitter) {
        for (VolumeInfo volumeInfo : this.getMetrics().values()) {
            MetricKey metricKeyUsable;
            Map<String, String> labels = this.labelsFor(volumeInfo.name());
            MetricKey metricKeyTotal = new MetricKey(this.diskTotalBytesName, labels);
            if (emitter.shouldEmitMetric(metricKeyTotal)) {
                emitter.emitMetric(SinglePointMetric.gauge(metricKeyTotal, volumeInfo.totalBytes(), Instant.now()));
            }
            if (!emitter.shouldEmitMetric(metricKeyUsable = new MetricKey(this.diskUsableBytesName, labels))) continue;
            emitter.emitMetric(SinglePointMetric.gauge(metricKeyUsable, volumeInfo.usableBytes, Instant.now()));
        }
    }

    public String toString() {
        return VolumeMetricsCollector.class.getCanonicalName();
    }

    public Map<String, VolumeInfo> getMetrics() {
        long curTimeNs = System.nanoTime();
        long deltaNs = curTimeNs - this.lastUpdateNs;
        long deltaMs = TimeUnit.MILLISECONDS.convert(deltaNs, TimeUnit.NANOSECONDS);
        if (deltaMs >= this.updatePeriodMs) {
            this.cachedMetrics = null;
        }
        if (this.cachedMetrics == null) {
            this.lastUpdateNs = curTimeNs;
            this.cachedMetrics = this.refreshCachedMetrics();
        }
        return Collections.unmodifiableMap(this.cachedMetrics);
    }

    private Map<String, VolumeInfo> refreshCachedMetrics() {
        HashMap<String, FileStore> fileStoreNameToObject = new HashMap<String, FileStore>();
        HashMap fileStoreNameToLogDirs = new HashMap();
        for (String logDir : this.logDirs) {
            try {
                FileStore fileStore = this.pathToFileStore(logDir);
                fileStoreNameToObject.put(fileStore.name(), fileStore);
                if (!fileStoreNameToLogDirs.containsKey(fileStore.name())) {
                    fileStoreNameToLogDirs.put(fileStore.name(), new TreeSet());
                }
                ((Set)fileStoreNameToLogDirs.get(fileStore.name())).add(logDir);
            }
            catch (IOException e) {
                log.error("Failed to resolve path to FileStore", (Throwable)e);
            }
        }
        TreeMap<String, VolumeInfo> metrics = new TreeMap<String, VolumeInfo>();
        for (FileStore fileStore : fileStoreNameToObject.values()) {
            try {
                VolumeInfo volumeInfo = new VolumeInfo(fileStore, (Set)fileStoreNameToLogDirs.get(fileStore.name()));
                if (log.isDebugEnabled()) {
                    log.debug("Read {}", (Object)volumeInfo.toString());
                }
                metrics.put(volumeInfo.name(), volumeInfo);
            }
            catch (IOException | RuntimeException e) {
                log.error("Failed to retrieve VolumeInfo from FileStore", (Throwable)e);
            }
        }
        return metrics;
    }

    private FileStore pathToFileStore(String path) throws IOException {
        Path pathObj = Paths.get(path, new String[0]);
        return Files.getFileStore(pathObj);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(ConfluentTelemetryConfig config) {
        VolumeMetricsCollectorConfig collectorConfig = config.getVolumeMetricsCollectorConfig();
        return new Builder().setLogDirs(collectorConfig.getBrokerLogVolumes()).setUpdatePeriodMs(collectorConfig.getUpdatePeriodMs());
    }

    public static class Builder {
        private long updatePeriodMs;
        private String[] logDirs;
        private Context context;

        private Builder() {
        }

        public Builder setUpdatePeriodMs(long updatePeriodMs) {
            this.updatePeriodMs = updatePeriodMs;
            return this;
        }

        public Builder setLogDirs(String[] logDirs) {
            this.logDirs = logDirs;
            return this;
        }

        public Builder setContext(Context context) {
            this.context = context;
            return this;
        }

        public VolumeMetricsCollector build() {
            Objects.requireNonNull(this.context);
            Objects.requireNonNull(this.context.getDomain());
            Objects.requireNonNull(this.logDirs);
            Verify.verify(this.updatePeriodMs > 0L, "update interval cannot be less than 1", new Object[0]);
            return new VolumeMetricsCollector(this);
        }
    }

    static class VolumeInfo {
        private final String name;
        private final long usableBytes;
        private final long totalBytes;
        private final Set<String> logDirs;

        private VolumeInfo(FileStore fileStore, Set<String> logDirs) throws IOException {
            this.name = fileStore.name();
            this.usableBytes = fileStore.getUsableSpace();
            this.totalBytes = fileStore.getTotalSpace();
            this.logDirs = Collections.unmodifiableSet(logDirs);
        }

        public String name() {
            return this.name;
        }

        public long usableBytes() {
            return this.usableBytes;
        }

        public long totalBytes() {
            return this.totalBytes;
        }

        public Collection<String> logDirs() {
            return this.logDirs;
        }

        public String toString() {
            return MoreObjects.toStringHelper(this).add("name", this.name).add("usableBytes", this.usableBytes).add("totalBytes", this.totalBytes).add("logDirs", this.logDirs).toString();
        }
    }

    public static class VolumeMetricsCollectorConfig
    extends AbstractConfig {
        public static final String PREFIX = "confluent.telemetry.metrics.collector.volume.";
        public static final String VOLUME_METRICS_UPDATE_PERIOD_MS = "confluent.telemetry.metrics.collector.volume.update.ms";
        public static final long DEFAULT_VOLUME_METRICS_UPDATE_PERIOD = 15000L;
        public static final String VOLUME_METRICS_UPDATE_PERIOD_DOC = "The minimum interval at which to fetch new volume metrics.";
        private static final ConfigDef CONFIG = new ConfigDef().define("confluent.telemetry.metrics.collector.volume.update.ms", ConfigDef.Type.LONG, (Object)15000L, ConfigDef.Importance.LOW, "The minimum interval at which to fetch new volume metrics.");

        public VolumeMetricsCollectorConfig(Map<?, ?> originals) {
            this(originals, true);
        }

        public VolumeMetricsCollectorConfig(Map<?, ?> originals, boolean doLog) {
            super(CONFIG, originals, doLog);
        }

        String[] getBrokerLogVolumes() {
            String logDirsString = null;
            Map originals = this.originals();
            if (originals.containsKey(KafkaConfig.LogDirsProp())) {
                logDirsString = (String)originals.get(KafkaConfig.LogDirsProp());
            }
            if (logDirsString == null && originals.containsKey(KafkaConfig.LogDirProp())) {
                logDirsString = (String)originals.get(KafkaConfig.LogDirProp());
            }
            return logDirsString == null ? null : logDirsString.split("\\s*,\\s*");
        }

        long getUpdatePeriodMs() {
            return this.getLong(VOLUME_METRICS_UPDATE_PERIOD_MS);
        }
    }
}

