/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.telemetry.exporter.kafka;

import io.confluent.shaded.io.opentelemetry.proto.metrics.v1.MetricsData;
import io.confluent.telemetry.exporter.AbstractExporter;
import io.confluent.telemetry.exporter.kafka.KafkaExporterConfig;
import io.confluent.telemetry.metrics.SerializedMetric;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.time.Clock;
import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.metrics.CompoundStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Meter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaExporter
extends AbstractExporter {
    private static final Logger log = LoggerFactory.getLogger(KafkaExporter.class);
    private static final int ERROR_LOG_INTERVAL_MS = 5000;
    public static final String VERSION_HEADER_KEY = "v";
    public static final byte[] V1_HEADER_BYTES = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(1).array();
    private static final Header V1_HEADER = new Header(){

        public String key() {
            return KafkaExporter.VERSION_HEADER_KEY;
        }

        public byte[] value() {
            return V1_HEADER_BYTES;
        }
    };
    private static final Iterable<Header> V1_HEADERS = Collections.singleton(V1_HEADER);
    private boolean isTopicCreated = false;
    private final Properties adminClientProperties;
    private final String topicName;
    private final boolean createTopic;
    private final int topicReplicas;
    private final int topicPartitions;
    private final Map<String, String> topicConfig;
    private final KafkaProducer<byte[], MetricsData> producer;
    private final AtomicLong droppedEventCount = new AtomicLong();
    private final AtomicReference<Exception> droppedEventException = new AtomicReference();
    private long lastLoggedTimestamp = 0L;
    private long lastLoggedCount = 0L;
    private volatile boolean isClosed = false;
    private final String name;
    private SelfMetrics selfMetrics;

    public KafkaExporter(Builder builder) {
        this.name = Objects.requireNonNull(builder.name);
        this.adminClientProperties = Objects.requireNonNull(builder.adminClientProperties);
        this.topicName = Objects.requireNonNull(builder.topicName);
        this.topicConfig = Objects.requireNonNull(builder.topicConfig);
        this.createTopic = builder.createTopic;
        this.topicReplicas = builder.topicReplicas;
        this.topicPartitions = builder.topicPartitions;
        this.producer = new KafkaProducer(Objects.requireNonNull(builder.producerProperties));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    boolean ensureTopic() {
        try (AdminClient adminClient = this.createAdminClient();){
            try {
                adminClient.describeTopics(Collections.singleton(this.topicName)).allTopicNames().get();
                log.debug("Telemetry topic {} already exists", (Object)this.topicName);
            }
            catch (ExecutionException e) {
                if (!(e.getCause() instanceof UnknownTopicOrPartitionException)) {
                    log.warn(e.getMessage());
                    throw e;
                }
                adminClient.createTopics(Collections.singleton(new NewTopic(this.topicName, this.topicPartitions, (short)this.topicReplicas).configs(this.topicConfig))).all().get();
                log.info("Created telemetry topic {}", (Object)this.topicName);
            }
            boolean bl = true;
            return bl;
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TopicExistsException) {
                log.info("Attempted to create telemetry topic {} but the topic was already created. This may happen the first time the reporter is started and multiple brokers attempt to create the topic simultaneously.", (Object)this.topicName);
                return true;
            }
            log.error("Error checking or creating telemetry topic {}", (Object)this.topicName, (Object)e.getCause());
            return false;
        }
        catch (InterruptedException e) {
            log.warn("Telemetry topic initialization interrupted");
            return false;
        }
    }

    AdminClient createAdminClient() {
        return AdminClient.create((Properties)this.adminClientProperties);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void doEmit(SerializedMetric metric) {
        try {
            if (!this.maybeCreateTopic()) {
                return;
            }
            KafkaProducer<byte[], MetricsData> kafkaProducer = this.producer;
            synchronized (kafkaProducer) {
                long now;
                long droppedCount;
                long droppedDelta;
                if (!Thread.currentThread().isInterrupted() && !this.isClosed) {
                    log.trace("Generated metric message : {}", (Object)metric);
                    this.producer.send(new ProducerRecord(this.topicName, null, null, (Object)metric.metricsData(), V1_HEADERS), (metadata, exception) -> {
                        if (exception != null) {
                            this.selfMetrics.droppedRecordsSensor.record();
                            this.droppedEventCount.incrementAndGet();
                            this.droppedEventException.compareAndSet(null, exception);
                        } else {
                            this.selfMetrics.sentRecordsSensor.record();
                        }
                    });
                }
                if ((droppedDelta = (droppedCount = this.droppedEventCount.get()) - this.lastLoggedCount) > 0L && this.lastLoggedTimestamp + 5000L < (now = Clock.systemUTC().millis())) {
                    log.warn("Failed to produce {} metrics messages", (Object)droppedDelta, this.droppedEventException.getAndSet(null));
                    this.lastLoggedTimestamp = now;
                    this.lastLoggedCount = droppedCount;
                }
            }
        }
        catch (InterruptException interruptException) {
            // empty catch block
        }
    }

    private synchronized boolean maybeCreateTopic() {
        if (this.createTopic) {
            if (!this.isTopicCreated) {
                this.isTopicCreated = this.ensureTopic();
            }
            return this.isTopicCreated;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws Exception {
        if (this.producer != null) {
            KafkaProducer<byte[], MetricsData> kafkaProducer = this.producer;
            synchronized (kafkaProducer) {
                this.isClosed = true;
                this.producer.close(Duration.ofMillis(0L));
            }
        }
    }

    @Override
    public void setMetricsRegistry(Metrics metrics) {
        this.selfMetrics = new SelfMetrics(metrics);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(KafkaExporterConfig config) {
        return new Builder().setCreateTopic(config.isCreateTopic()).setTopicConfig(config.getTopicConfig()).setTopicName(config.getTopicName()).setTopicReplicas(config.getTopicReplicas()).setTopicPartitions(config.getTopicPartitions()).setProducerProperties(config.getProducerProperties()).setAdminClientProperties(config.getProducerProperties());
    }

    public static final class Builder {
        private String name;
        private Properties adminClientProperties;
        private String topicName;
        private boolean createTopic;
        private int topicReplicas;
        private int topicPartitions;
        private Map<String, String> topicConfig;
        private Properties producerProperties;

        private Builder() {
        }

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setAdminClientProperties(Properties adminClientProperties) {
            this.adminClientProperties = adminClientProperties;
            return this;
        }

        public Builder setTopicName(String topicName) {
            this.topicName = topicName;
            return this;
        }

        public Builder setCreateTopic(boolean createTopic) {
            this.createTopic = createTopic;
            return this;
        }

        public Builder setTopicReplicas(int topicReplicas) {
            this.topicReplicas = topicReplicas;
            return this;
        }

        public Builder setTopicPartitions(int topicPartitions) {
            this.topicPartitions = topicPartitions;
            return this;
        }

        public Builder setTopicConfig(Map<String, String> topicConfig) {
            this.topicConfig = topicConfig;
            return this;
        }

        public Builder setProducerProperties(Properties producerProperties) {
            this.producerProperties = producerProperties;
            return this;
        }

        public KafkaExporter build() {
            return new KafkaExporter(this);
        }
    }

    private class SelfMetrics {
        private static final String GROUP = "KafkaExporter";
        final Sensor droppedRecordsSensor;
        final Sensor sentRecordsSensor;

        SelfMetrics(Metrics metrics) {
            Map<String, String> tags = Collections.singletonMap("exporterName", KafkaExporter.this.name);
            String sensorSuffix = "(name=" + KafkaExporter.this.name + ")";
            this.droppedRecordsSensor = metrics.sensor("dropped-records" + sensorSuffix);
            this.droppedRecordsSensor.add((CompoundStat)new Meter(metrics.metricName("dropped-records-rate", GROUP, tags), metrics.metricName("dropped-records-total", GROUP, tags)));
            this.sentRecordsSensor = metrics.sensor("sent-records" + sensorSuffix);
            this.sentRecordsSensor.add((CompoundStat)new Meter(metrics.metricName("sent-records-rate", GROUP, tags), metrics.metricName("sent-records-total", GROUP, tags)));
        }
    }
}

