/*
 * Decompiled with CFR 0.152.
 */
package kafka.server.resource;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryNotificationInfo;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import javax.management.ListenerNotFoundException;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;
import kafka.server.BrokerReconfigurable;
import kafka.server.KafkaConfig;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.metrics.MeasurableStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.CumulativeCount;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.collection.JavaConverters;
import scala.collection.Set;

public class HeapWatcher
implements BrokerReconfigurable,
NotificationFilter,
NotificationListener {
    private static final Logger LOG = LoggerFactory.getLogger(HeapWatcher.class);
    private static final String METRICS_GROUP_NAME = HeapWatcher.class.getSimpleName();
    static final Map<String, HeapWatcher> INSTANCES = new HashMap<String, HeapWatcher>();
    private final Metrics metrics;
    private final Sensor almostOOMSensor;
    private final MemoryPoolMXBean memoryPoolMxBean;
    final MetricName almostOOMMetricName;
    private long usageThresholdBytes;
    private volatile boolean wasShutdown;
    public static Set<String> reconfigurableConfigs = JavaConverters.asScalaSet((java.util.Set)Utils.mkSet((Object[])new String[]{"confluent.heap.tenured.notify.bytes"}));

    public static synchronized Optional<HeapWatcher> maybeCreateHeapWatcher(Metrics metrics, KafkaConfig kafkaConfig) {
        MemoryPoolMXBean bean = HeapWatcher.getTenuredPoolMXBean();
        if (bean == null) {
            LOG.error("Unable to find the MemoryPoolMXBean for the tenured pool, not creating a HeapWatcher");
            return Optional.empty();
        }
        try {
            return Optional.of(HeapWatcher.instanceForBean(bean, metrics, kafkaConfig));
        }
        catch (ConfigException e) {
            LOG.error("Unable to initialize HeapWatcher due to invalid configuration, it'd be disabled", (Throwable)e);
        }
        catch (IllegalArgumentException e) {
            LOG.error("Unable to initialize HeapWatcher due to invalid argument, it'd be disabled", (Throwable)e);
        }
        return Optional.empty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static HeapWatcher instanceForBean(MemoryPoolMXBean bean, Metrics metrics, KafkaConfig kafkaConfig) {
        String beanName = bean.getName();
        Map<String, HeapWatcher> map = INSTANCES;
        synchronized (map) {
            if (INSTANCES.containsKey(beanName)) {
                return INSTANCES.get(beanName);
            }
            HeapWatcher newHeapWatcher = new HeapWatcher(bean, metrics, kafkaConfig);
            INSTANCES.put(beanName, newHeapWatcher);
            return newHeapWatcher;
        }
    }

    protected HeapWatcher(MemoryPoolMXBean memoryPoolMxBean, Metrics metrics, KafkaConfig kafkaConfig) throws IllegalArgumentException, ConfigException {
        LOG.trace("Creating HeapWatcher");
        this.wasShutdown = false;
        long threshold = HeapWatcher.usageThersholdBytesFromConfig(kafkaConfig);
        this.memoryPoolMxBean = memoryPoolMxBean;
        if (memoryPoolMxBean == null) {
            throw new IllegalArgumentException("memoryPoolMxBean argument is null");
        }
        if (!memoryPoolMxBean.isCollectionUsageThresholdSupported()) {
            throw new IllegalArgumentException(String.format("The memory pool supplied (%s) does not support a collection usage threshold", memoryPoolMxBean.getName()));
        }
        this.metrics = metrics;
        this.almostOOMSensor = this.metrics.sensor("AlmostOOM");
        this.almostOOMMetricName = metrics.metricName("almost-oom", METRICS_GROUP_NAME, "Number of times an almost-OOM has occurred");
        this.almostOOMSensor.add(this.almostOOMMetricName, (MeasurableStat)new CumulativeCount());
        this.registerWithMemoryMXBean();
        this.setCollectionUsageThreshold(threshold);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void registerWithMemoryMXBean() {
        MemoryPoolMXBean memoryPoolMXBean = this.memoryPoolMxBean;
        synchronized (memoryPoolMXBean) {
            if (this.wasShutdown) {
                LOG.error("HeapWatcher was already shutdown, won't register with memory pool again");
                return;
            }
            MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
            NotificationEmitter emitter = (NotificationEmitter)((Object)memoryMXBean);
            LOG.debug("Added HeapWatcher as a listener for the tenured heap MemoryMXBean");
            emitter.addNotificationListener(this, this, null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void unregisterWithMemoryMXBean() {
        MemoryPoolMXBean memoryPoolMXBean = this.memoryPoolMxBean;
        synchronized (memoryPoolMXBean) {
            MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
            NotificationEmitter emitter = (NotificationEmitter)((Object)memoryMXBean);
            try {
                emitter.removeNotificationListener(this, this, null);
                LOG.debug("Removed HeapWatcher as a listener for the tenured heap MemoryMXBean");
            }
            catch (ListenerNotFoundException e) {
                LOG.warn("removeNotificationListener from MemoryMXBean failed. Maybe it was never registered?", (Throwable)e);
            }
            this.wasShutdown = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setCollectionUsageThreshold(long threshold) {
        MemoryPoolMXBean memoryPoolMXBean = this.memoryPoolMxBean;
        synchronized (memoryPoolMXBean) {
            if (this.wasShutdown) {
                LOG.error("HeapWatcher was already shutdown. Won't try to set collection usage threshold");
                return;
            }
            long maxThreshold = this.memoryPoolMxBean.getUsage().getMax();
            if (maxThreshold > -1L && threshold > maxThreshold) {
                LOG.info("Asked to set collection usage threshold to %d, more than the max available to the pool %d. Disabling notifications insetad", (Object)threshold, (Object)maxThreshold);
                threshold = 0L;
            }
            LOG.info("Setting collection usage threshold to %d", (Object)threshold);
            this.memoryPoolMxBean.setCollectionUsageThreshold(threshold);
            this.usageThresholdBytes = this.memoryPoolMxBean.getCollectionUsageThreshold();
            if (this.usageThresholdBytes != threshold) {
                LOG.error("Asked the MemoryPoolMXBean to set collection threshold to %d, but it was set to %d instead", (Object)this.usageThresholdBytes, (Object)threshold);
            }
        }
    }

    private static long usageThersholdBytesFromConfig(KafkaConfig kafkaConfig) throws ConfigException {
        Long value = kafkaConfig.confluentConfig().heapWatcherTenuredNotifyBytes();
        if (value == null) {
            return 0L;
        }
        if (value < 0L) {
            throw new ConfigException(String.format("Value for %s is %d, a negative number", "confluent.heap.tenured.notify.bytes", value));
        }
        return value;
    }

    private static boolean isTenuredPoolName(String name) {
        return name.contains("Old") || name.contains("Tenured");
    }

    static MemoryPoolMXBean getTenuredPoolMXBean() {
        for (MemoryPoolMXBean poolBean : ManagementFactory.getMemoryPoolMXBeans()) {
            if (!poolBean.getType().equals((Object)MemoryType.HEAP) || !HeapWatcher.isTenuredPoolName(poolBean.getName())) continue;
            return poolBean;
        }
        return null;
    }

    private void notifyListeners() {
        this.almostOOMSensor.record();
    }

    @Override
    public boolean isNotificationEnabled(Notification notification) {
        if (this.wasShutdown) {
            LOG.trace("Notification recieved after HeapWatcher was shutdown: %s", (Object)notification);
            return false;
        }
        if (!notification.getType().equals("java.management.memory.collection.threshold.exceeded")) {
            LOG.trace("Asked about notification that isn't MEMORY_COLLECTION_THRESHOLD_EXCEEDED: %s", (Object)notification);
            return false;
        }
        MemoryNotificationInfo info = MemoryNotificationInfo.from((CompositeData)notification.getUserData());
        LOG.debug("Filtering MEMORY_COLLECTION_THRESHOLD_EXCEEDED notification with payload: %s", (Object)info);
        return HeapWatcher.isTenuredPoolName(info.getPoolName());
    }

    @Override
    public void handleNotification(Notification notification, Object handback) {
        if (this.wasShutdown) {
            LOG.trace("Notification recieved after HeapWatcher was shutdown: %s", (Object)notification);
            return;
        }
        if (LOG.isDebugEnabled()) {
            MemoryNotificationInfo info = MemoryNotificationInfo.from((CompositeData)notification.getUserData());
            LOG.debug("Handling notification with payload: %s", (Object)info);
        }
        MemoryUsage usage = this.memoryPoolMxBean.getCollectionUsage();
        long usedMemory = usage.getUsed();
        LOG.info("Got notified about the tenured pool, collection usage memory being %d (notification threshold is %d), usedMemory, usageThresholdBytes");
        if (usedMemory > this.usageThresholdBytes) {
            this.notifyListeners();
        }
    }

    public synchronized void shutdown() {
        this.unregisterWithMemoryMXBean();
        this.metrics.removeSensor(this.almostOOMSensor.name());
    }

    @Override
    public Set<String> reconfigurableConfigs() {
        return reconfigurableConfigs;
    }

    @Override
    public void validateReconfiguration(KafkaConfig newConfig) {
        HeapWatcher.usageThersholdBytesFromConfig(newConfig);
    }

    @Override
    public void reconfigure(KafkaConfig oldConfig, KafkaConfig newConfig) {
        Long newThreshold = HeapWatcher.usageThersholdBytesFromConfig(newConfig);
        if (newThreshold != this.usageThresholdBytes) {
            this.setCollectionUsageThreshold(newThreshold);
        }
    }
}

