/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.kafka.common.Cell;
import org.apache.kafka.common.PartitionPlacementStrategy;
import org.apache.kafka.common.Tenant;
import org.apache.kafka.common.errors.ResourceNotFoundException;
import org.apache.kafka.common.message.AssignTenantsToCellRequestData;
import org.apache.kafka.common.message.AssignTenantsToCellResponseData;
import org.apache.kafka.common.message.DeleteTenantsResponseData;
import org.apache.kafka.common.message.DescribeTenantsResponseData;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.metadata.RemoveTenantRecord;
import org.apache.kafka.common.metadata.TenantRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.requests.AssignTenantsToCellRequest;
import org.apache.kafka.common.requests.DeleteTenantsRequest;
import org.apache.kafka.common.requests.DescribeTenantsRequest;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.controller.CellControlManager;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.FeatureControlManager;
import org.apache.kafka.metadata.placement.CellAssignor;
import org.apache.kafka.metadata.placement.TenantDescriber;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.mutable.BoundedList;
import org.slf4j.Logger;

public class TenantControlManager
implements TenantDescriber {
    private final Logger log;
    private final FeatureControlManager featureControl;
    private final CellControlManager cellControl;
    private final PartitionPlacementStrategy defaultPartitionPlacementStrategy;
    private final short replicationFactor;

    public TenantControlManager(LogContext logContext, FeatureControlManager featureControl, CellControlManager cellControlManager, PartitionPlacementStrategy partitionPlacementStrategy, short replicationFactor) {
        this.log = logContext.logger(TenantControlManager.class);
        this.featureControl = featureControl;
        this.cellControl = cellControlManager;
        this.defaultPartitionPlacementStrategy = partitionPlacementStrategy;
        this.replicationFactor = replicationFactor;
    }

    public ControllerResult<AssignTenantsToCellResponseData> assignTenantsToCell(AssignTenantsToCellRequest request, Set<Integer> usableBrokers) {
        this.cellControl.confirmCellsSupported();
        AssignTenantsToCellResponseData data = new AssignTenantsToCellResponseData();
        ArrayList<AssignTenantsToCellResponseData.TenantAssignmentErrors> failedTenants = new ArrayList<AssignTenantsToCellResponseData.TenantAssignmentErrors>();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        for (AssignTenantsToCellRequestData.TenantToCellAssignment assignment : request.tenantToAssign()) {
            String tenantId = assignment.tenantId();
            ApiError err = this.assignTenantToCell(tenantId, assignment.cellId(), assignment.force(), usableBrokers, ((List)records)::add);
            if (ApiError.NONE.equals((Object)err)) continue;
            failedTenants.add(new AssignTenantsToCellResponseData.TenantAssignmentErrors().setTenantId(tenantId).setError(err.error().code()).setErrorMessage(err.message()));
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data.setFailedTenants(failedTenants));
    }

    public ControllerResult<DescribeTenantsResponseData> describeTenants(DescribeTenantsRequest request) {
        this.cellControl.confirmCellsSupported();
        DescribeTenantsResponseData data = new DescribeTenantsResponseData();
        List<String> tenantIds = request.tenants().isEmpty() ? this.cellControl.tenantIds() : request.tenants();
        Collections.sort(tenantIds);
        ArrayList<DescribeTenantsResponseData.TenantDescription> tenantDescriptions = new ArrayList<DescribeTenantsResponseData.TenantDescription>();
        for (String tenantId : tenantIds) {
            Optional<Cell> tenantCell = this.getTenantCell(tenantId);
            if (!tenantCell.isPresent()) {
                return ControllerResult.atomicOf(Collections.emptyList(), data.setErrorCode(Errors.TENANT_NOT_FOUND.code()).setErrorMessage(String.format("Tenant %s does not exist", tenantId)));
            }
            tenantDescriptions.add(new DescribeTenantsResponseData.TenantDescription().setTenantId(tenantId).setCellId(tenantCell.get().cellId()).setPartitionPlacementStrategy(this.defaultPartitionPlacementStrategy.code().intValue()));
        }
        return ControllerResult.atomicOf(Collections.emptyList(), data.setTenantDescriptions(tenantDescriptions));
    }

    public ControllerResult<DeleteTenantsResponseData> deleteTenants(DeleteTenantsRequest request) {
        this.cellControl.confirmCellsSupported();
        DeleteTenantsResponseData data = new DeleteTenantsResponseData();
        ArrayList tenantIds = new ArrayList(request.data().tenants());
        Collections.sort(tenantIds);
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        for (String tenantId : tenantIds) {
            Optional<Cell> tenantCellOpt = this.getTenantCell(tenantId);
            if (!tenantCellOpt.isPresent()) continue;
            this.removeTenant(tenantId, ((List)records)::add);
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data.setFailedTenants(new ArrayList()));
    }

    void replay(TenantRecord tenantRecord) {
        this.cellControl.replay(tenantRecord);
    }

    void replay(RemoveTenantRecord removeTenantRecord) {
        this.cellControl.replay(removeTenantRecord);
    }

    int createTenantToCellAssignmentIfNotExists(String tenantId, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<Cell> tenantCellOpt = this.getTenantCell(tenantId);
        if (tenantCellOpt.isPresent()) {
            return tenantCellOpt.get().cellId();
        }
        Optional<Cell> cellOpt = this.cellControl.computeUsableCell(usableBrokers, this.replicationFactor);
        if (!cellOpt.isPresent()) {
            this.log.error("Cluster is unable to create partitions due to it not having any usable cells.");
            throw new ResourceNotFoundException("Cluster is unable to create partitions");
        }
        Cell cell = cellOpt.get();
        TenantRecord record = new TenantRecord().setTenantId(tenantId).setCellId(cell.cellId());
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.TENANT_RECORD.highestSupportedVersion()));
        return cell.cellId();
    }

    ApiError createTenant(String tenantId, int cellId, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<Cell> cellOpt = this.cellControl.getCell(cellId);
        if (!cellOpt.isPresent()) {
            return new ApiError(Errors.CELL_NOT_FOUND, String.format("Cell %s does not exist", cellId));
        }
        if (this.cellControl.containsTenant(tenantId)) {
            return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s already exists", tenantId));
        }
        Cell cell = cellOpt.get();
        TenantRecord record = new TenantRecord().setTenantId(tenantId).setCellId(cell.cellId());
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.TENANT_RECORD.highestSupportedVersion()));
        return ApiError.NONE;
    }

    ApiError assignTenantToCell(String tenantId, int cellId, boolean force, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        Cell targetCell;
        Optional<Cell> sourceCellOpt = this.getTenantCell(tenantId);
        Optional<Cell> targetCellOpt = this.cellControl.getCell(cellId);
        if (!sourceCellOpt.isPresent()) {
            return new ApiError(Errors.TENANT_NOT_FOUND, String.format("Tenant %s does not exist", tenantId));
        }
        if (!targetCellOpt.isPresent()) {
            return new ApiError(Errors.CELL_NOT_FOUND, String.format("Cell %s does not exist", cellId));
        }
        Cell sourceCell = sourceCellOpt.get();
        if (sourceCell.equals((Object)(targetCell = targetCellOpt.get()))) {
            return ApiError.NONE;
        }
        if (!CellAssignor.isCellOpenForAssignment(targetCell, usableBrokers, this.replicationFactor)) {
            return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s cannot be moved to cell %s since the cell either does not have enough brokers to meet its minSize or does not have at least %s alive brokers", tenantId, targetCell.cellId(), this.replicationFactor));
        }
        if (!force) {
            if (CellControlManager.PROHIBITED_TARGET_STATES.contains(targetCell.state())) {
                return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s cannot be moved to cell %s since it is prohibited", tenantId, targetCell.cellId()));
            }
            if (CellControlManager.PROHIBITED_SOURCE_STATES.contains(sourceCell.state())) {
                return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s cannot be moved from cell %s since it is prohibited", tenantId, sourceCell.cellId()));
            }
        }
        TenantRecord record = new TenantRecord().setTenantId(tenantId).setCellId(cellId);
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.TENANT_RECORD.highestSupportedVersion()));
        this.log.info("Tenant {} is manually assigned cell {}", (Object)tenantId, (Object)cellId);
        return ApiError.NONE;
    }

    boolean removeTenant(String tenantId, Consumer<ApiMessageAndVersion> recordConsumer) {
        boolean tenantExists = this.cellControl.containsTenant(tenantId);
        if (tenantExists) {
            RemoveTenantRecord record = new RemoveTenantRecord().setTenantId(tenantId);
            recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.REMOVE_TENANT_RECORD.highestSupportedVersion()));
            this.log.info("Deleted tenant {} information", (Object)tenantId);
        } else {
            this.log.warn("Tenant {} information was already deleted", (Object)tenantId);
        }
        return tenantExists;
    }

    @Override
    public int getTenantCellId(String tenantId) {
        if (this.defaultPartitionPlacementStrategy != PartitionPlacementStrategy.TENANT_IN_CELL || !this.featureControl.metadataVersion().isCellsSupported()) {
            return -1;
        }
        return this.getTenantCell(tenantId).map(Cell::cellId).orElse(-1);
    }

    PartitionPlacementStrategy calculatePartitionPlacementStrategy(Optional<KafkaPrincipal> principalOpt) {
        if (!this.featureControl.metadataVersion().isCellsSupported()) {
            return PartitionPlacementStrategy.CLUSTER_WIDE;
        }
        return CellAssignor.calculatePartitionPlacementStrategy(principalOpt, this.defaultPartitionPlacementStrategy);
    }

    boolean isTenantCellPlacementEnabled(Optional<KafkaPrincipal> principalOpt) {
        if (!this.featureControl.metadataVersion().isCellsSupported()) {
            return false;
        }
        return CellAssignor.isTenantCellPlacementEnabled(principalOpt, this.defaultPartitionPlacementStrategy);
    }

    private Optional<Cell> getTenantCell(String tenantId) {
        Tenant tenant = this.cellControl.getTenant(tenantId);
        if (tenant == null) {
            return Optional.empty();
        }
        Optional<Cell> cellOpt = this.cellControl.getCell(tenant.cellId());
        if (!cellOpt.isPresent()) {
            this.log.error("Tenant {} is assigned to cell {}, however the cell does not exist", (Object)tenantId, (Object)tenant.cellId());
        }
        return cellOpt;
    }
}

