/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.tokenapi.jwt;

import io.confluent.common.security.util.PemUtils;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.KeyPair;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyPairHandler {
    private static final Logger log = LoggerFactory.getLogger(KeyPairHandler.class);
    private final List<KeyPair> keyPairs;
    private final String keyPairPassphrase;

    public KeyPairHandler(Path keyPairPath) {
        this.keyPairPassphrase = null;
        this.keyPairs = this.loadKeys(keyPairPath);
    }

    public KeyPairHandler(Path keyPairPath, String keyPairPassphrase) {
        this.keyPairPassphrase = keyPairPassphrase;
        this.keyPairs = this.loadKeys(keyPairPath);
    }

    private List<KeyPair> loadKeys(Path keyPairPath) {
        try {
            List<Path> keys = this.getKeyPairPaths(keyPairPath);
            List<KeyPair> keyPairs = keys.stream().map(this::tryLoadKeyPair).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
            if (keyPairs.isEmpty()) {
                throw new IllegalStateException(String.format("No keypair(s) could be loaded from config %s with path: %s. Please check logs for more info.", "token.key.path", keyPairPath));
            }
            return keyPairs;
        }
        catch (IOException e) {
            String errorMsg = String.format("Unable to load token keyPair(s) from config %s with path: %s. Please check logs for more info.", "token.key.path", keyPairPath);
            log.error(errorMsg, (Throwable)e);
            throw new IllegalStateException(errorMsg, e);
        }
    }

    private List<Path> getKeyPairPaths(Path keyPairPath) throws IOException {
        if (Files.isRegularFile(keyPairPath, new LinkOption[0])) {
            return Collections.singletonList(keyPairPath);
        }
        ArrayList<Path> keys = new ArrayList<Path>();
        Files.newDirectoryStream(keyPairPath, "*.pem").forEach(keys::add);
        return keys;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<KeyPair> tryLoadKeyPair(Path path) {
        try (InputStream is = Files.newInputStream(path, new OpenOption[0]);){
            KeyPair keyPair;
            if (this.keyPairPassphrase != null && !this.keyPairPassphrase.isEmpty()) {
                log.info("Loading keypair from path {} with a passphrase", (Object)path);
                keyPair = PemUtils.loadKeyPair((InputStream)is, (String)this.keyPairPassphrase);
            } else {
                log.info("Loading keypair from path {} without passphrase", (Object)path);
                keyPair = PemUtils.loadKeyPair((InputStream)is);
            }
            Optional<KeyPair> optional = Optional.of(keyPair);
            return optional;
        }
        catch (Exception e) {
            String errorMsg = String.format("Unable to load token keyPair from config %s with path: %s", "token.key.path", path);
            log.error(errorMsg, (Throwable)e);
            return Optional.empty();
        }
    }

    public List<KeyPair> getKeyPairs() {
        return this.keyPairs;
    }
}

