/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.replication.push;

import io.confluent.kafka.replication.push.PushSession;
import io.confluent.kafka.replication.push.Pusher;
import io.confluent.kafka.replication.push.ReplicationConfig;
import io.confluent.kafka.replication.push.buffer.BufferingAppendRecordsBuilder;
import io.confluent.kafka.replication.push.buffer.BufferingPartitionDataBuilder;
import io.confluent.kafka.replication.push.buffer.PushReplicationEvent;
import io.confluent.kafka.replication.push.buffer.RefCountingMemoryTracker;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicIdPartition;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.NetworkException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.message.AppendRecordsRequestData;
import org.apache.kafka.common.message.AppendRecordsResponseData;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.BaseRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.AppendRecordsRequest;
import org.apache.kafka.common.requests.AppendRecordsResponse;
import org.apache.kafka.common.utils.ExponentialBackoff;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.util.InterBrokerSendThread;
import org.apache.kafka.server.util.RequestAndCompletionHandler;

class PusherThread
extends InterBrokerSendThread
implements Pusher {
    private final ReplicationConfig config;
    private final Function<Integer, Optional<Node>> nodeResolver;
    private final RefCountingMemoryTracker<MemoryRecords> tracker;
    private final Time time;
    private final int lingerMs;
    private final int maxWaitMs;
    private final int retryTimeoutMs;
    private final Queue<PushReplicationEvent<?>> eventQueue;
    private final Map<PartitionReplicaKey, PushSession> activePushSessions;
    private final Map<Integer, BufferingAppendRecordsBuilder> bufferingRequestBuilders;
    private final Map<Integer, Long> latestKnownBrokerEpochs;
    final Map<Integer, Deque<RequestAndCompletionHandler>> pendingRequests;
    private final Map<Integer, NodeLoader> nodeLoaders;

    static PusherThread newPusher(int pusherId, ReplicationConfig config, KafkaClient networkClient, Function<Integer, Optional<Node>> nodeResolver, RefCountingMemoryTracker<MemoryRecords> tracker, Time time) {
        String name = String.format("Pusher-thread-%s", pusherId);
        return new PusherThread(name, config, networkClient, nodeResolver, tracker, time);
    }

    PusherThread(String name, ReplicationConfig config, KafkaClient networkClient, Function<Integer, Optional<Node>> nodeResolver, RefCountingMemoryTracker<MemoryRecords> tracker, Time time) {
        super(name, networkClient, config.requestTimeoutMs(), time);
        this.config = config;
        int maxInFlightRequests = config.maxInFlightRequests();
        if (maxInFlightRequests != 1) {
            this.log.warn("Max in-flight requests for push replication configured to be '{}', but only '1' is currently supported", (Object)maxInFlightRequests);
        }
        this.nodeResolver = nodeResolver;
        this.tracker = tracker;
        this.time = time;
        this.lingerMs = config.lingerMs();
        this.retryTimeoutMs = config.retryTimeoutMs();
        this.maxWaitMs = config.maxWaitMs();
        this.eventQueue = new ConcurrentLinkedQueue();
        this.activePushSessions = new HashMap<PartitionReplicaKey, PushSession>();
        this.bufferingRequestBuilders = new HashMap<Integer, BufferingAppendRecordsBuilder>();
        this.latestKnownBrokerEpochs = new HashMap<Integer, Long>();
        this.pendingRequests = new HashMap<Integer, Deque<RequestAndCompletionHandler>>();
        this.nodeLoaders = new HashMap<Integer, NodeLoader>();
    }

    @Override
    public CompletableFuture<Void> startPush(TopicIdPartition partition, int destinationBrokerId, PushSession pushSession) {
        PushReplicationEvent<PushSession> startPushEvent = PushReplicationEvent.forStartPush(partition, destinationBrokerId, pushSession);
        CompletableFuture<Void> startFuture = startPushEvent.payload().startFuture();
        this.enqueue(startPushEvent);
        return startFuture;
    }

    @Override
    public void onLeaderAppend(TopicIdPartition partition, int destinationBrokerId, long highWatermark, long appendOffset, AbstractRecords records) {
        PushReplicationEvent<PushReplicationEvent.RecordsPayload> leaderAppendEvent = PushReplicationEvent.forRecords(partition, destinationBrokerId, records, appendOffset, highWatermark);
        this.enqueue(leaderAppendEvent);
    }

    @Override
    public void onHighWatermarkUpdate(TopicIdPartition partition, int destinationBrokerId, long highWatermark) {
        PushReplicationEvent<PushReplicationEvent.OffsetPayload> highWatermarkUpdateEvent = PushReplicationEvent.forHighWatermarkUpdate(partition, destinationBrokerId, highWatermark);
        this.enqueue(highWatermarkUpdateEvent);
    }

    @Override
    public void onLogStartOffsetUpdate(TopicIdPartition partition, int destinationBrokerId, long logStartOffset) {
        PushReplicationEvent<PushReplicationEvent.OffsetPayload> logStartOffsetUpdateEvent = PushReplicationEvent.forLogStartOffsetUpdate(partition, destinationBrokerId, logStartOffset);
        this.enqueue(logStartOffsetUpdateEvent);
    }

    @Override
    public CompletableFuture<Void> stopPush(TopicIdPartition partition, int destinationBrokerId, boolean sendEndSessionRequest) {
        PushReplicationEvent<CompletableFuture<Void>> stopPushEvent = PushReplicationEvent.forStopPush(partition, destinationBrokerId);
        CompletableFuture<Void> stopFuture = stopPushEvent.payload();
        this.enqueue(stopPushEvent);
        return stopFuture;
    }

    private void enqueue(PushReplicationEvent<?> event) {
        this.eventQueue.offer(event);
        this.wakeup();
    }

    @Override
    public void shutdown() {
        try {
            super.shutdown();
            this.activePushSessions.clear();
            this.bufferingRequestBuilders.clear();
            this.pendingRequests.clear();
            this.nodeLoaders.clear();
            this.eventQueue.clear();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public Collection<RequestAndCompletionHandler> generateRequests() {
        ArrayList<RequestCandidate> candidates = new ArrayList<RequestCandidate>();
        while (!this.eventQueue.isEmpty()) {
            PushReplicationEvent<PushSession> currentEvent = this.eventQueue.poll();
            switch (currentEvent.type()) {
                case MEMORY_RECORDS: 
                case FILE_RECORDS: 
                case HWM_UPDATE: 
                case LSO_UPDATE: {
                    this.processPartitionUpdateEvent(currentEvent, candidates);
                    break;
                }
                case START_PUSH: {
                    PushReplicationEvent<PushSession> startEvent = currentEvent;
                    this.processStartEvent(startEvent);
                    break;
                }
                case STOP_PUSH: {
                    PushReplicationEvent<PushSession> stopEvent = currentEvent;
                    this.processStopEvent(stopEvent, candidates);
                    break;
                }
                default: {
                    this.log.error("Unknown push replication event type {}", (Object)currentEvent.type());
                    this.failSessionAfterEventProcessingFailure(currentEvent, candidates);
                }
            }
            if (candidates.isEmpty()) continue;
            break;
        }
        return this.getRequestsToSend(candidates);
    }

    protected long sendRequests(long now, long maxTimeoutMs) {
        long currentTimeoutMs = super.sendRequests(now, maxTimeoutMs);
        if (!this.bufferingRequestBuilders.isEmpty() && this.lingerMs != 0) {
            currentTimeoutMs = Math.min(currentTimeoutMs, (long)this.lingerMs);
        }
        if (!this.bufferingRequestBuilders.isEmpty() && this.maxWaitMs != 0) {
            currentTimeoutMs = Math.min(currentTimeoutMs, (long)this.maxWaitMs);
        }
        return currentTimeoutMs;
    }

    private void processPartitionUpdateEvent(PushReplicationEvent<?> event, List<RequestCandidate> candidates) {
        PartitionReplicaKey key = new PartitionReplicaKey(event);
        PushSession pushSession = this.activePushSessions.get(key);
        if (pushSession == null) {
            this.log.debug("Cannot process push replication event {} for partition replica {}; no active push session found", event, (Object)key);
            if (event.type() == PushReplicationEvent.Type.MEMORY_RECORDS) {
                PushReplicationEvent.RecordsPayload payload = (PushReplicationEvent.RecordsPayload)event.payload();
                MemoryRecords memoryRecords = (MemoryRecords)payload.records();
                this.tracker.countDown(memoryRecords);
            }
            return;
        }
        CompletableFuture<Void> startFuture = pushSession.startFuture();
        if (!startFuture.isDone() && event.type() == PushReplicationEvent.Type.FILE_RECORDS) {
            PushReplicationEvent<PushReplicationEvent.RecordsPayload> fileRecordsEvent = event;
            event = PushReplicationEvent.forTransitionRecords(fileRecordsEvent, startFuture);
        }
        this.processAndMaybeCreateRequest(event, pushSession, candidates);
    }

    private void untrackRecords(PushReplicationEvent<?> event) {
        if (event.type() == PushReplicationEvent.Type.MEMORY_RECORDS) {
            PushReplicationEvent.RecordsPayload payload = (PushReplicationEvent.RecordsPayload)event.payload();
            MemoryRecords memoryRecords = (MemoryRecords)payload.records();
            this.tracker.countDown(memoryRecords);
        }
    }

    private void processStartEvent(PushReplicationEvent<PushSession> startEvent) {
        PushSession newSession = startEvent.payload();
        PartitionReplicaKey key = new PartitionReplicaKey(startEvent);
        PushSession oldSession = this.activePushSessions.get(key);
        if (oldSession != null) {
            String errorMessage = String.format("Cannot start push session %s due to start event %s; already found an active session %s", newSession, startEvent, oldSession);
            this.log.error(errorMessage);
            IllegalStateException ise = new IllegalStateException(errorMessage);
            newSession.startFuture().completeExceptionally(ise);
            return;
        }
        if (!this.maybeAddPushSessionAndUpdateBrokerEpoch(key, newSession)) {
            return;
        }
        newSession.startFuture().complete(null);
        this.log.info("Added push session {} due to start event {}", (Object)newSession, startEvent);
    }

    private boolean maybeAddPushSessionAndUpdateBrokerEpoch(PartitionReplicaKey partitionReplicaKey, PushSession pushSession) {
        long lastKnownBrokerEpoch;
        int replicaId = partitionReplicaKey.replicaId;
        long sessionBrokerEpoch = pushSession.replicaEpoch();
        if (sessionBrokerEpoch < (lastKnownBrokerEpoch = this.latestKnownBrokerEpochs.getOrDefault(replicaId, -1L).longValue())) {
            String pushSessionStartErrorMessage = String.format("New push session %s ignored due to stale broker epoch. Latest known broker epoch is %d.", pushSession, lastKnownBrokerEpoch);
            this.log.info(pushSessionStartErrorMessage);
            pushSession.startFuture().completeExceptionally(new IllegalStateException(pushSessionStartErrorMessage));
            return false;
        }
        if (sessionBrokerEpoch > lastKnownBrokerEpoch) {
            this.latestKnownBrokerEpochs.put(replicaId, sessionBrokerEpoch);
            BufferingAppendRecordsBuilder requestBuilder = this.bufferingRequestBuilders.get(replicaId);
            if (requestBuilder != null && requestBuilder.destinationBrokerEpoch() < sessionBrokerEpoch) {
                requestBuilder.clear();
                this.bufferingRequestBuilders.remove(replicaId);
            }
        }
        this.activePushSessions.put(partitionReplicaKey, pushSession);
        return true;
    }

    private void processStopEvent(PushReplicationEvent<CompletableFuture<Void>> stopEvent, List<RequestCandidate> candidates) {
        PartitionReplicaKey key = new PartitionReplicaKey(stopEvent);
        PushSession pushSession = this.activePushSessions.remove(key);
        if (pushSession == null) {
            this.log.debug("Cannot stop push session due to stop event {}; no active session found", stopEvent);
        } else {
            this.log.info("Removed push session {} due to stop event {}", (Object)pushSession, stopEvent);
            this.processAndMaybeCreateRequest(stopEvent, pushSession, candidates);
        }
    }

    private void processAndMaybeCreateRequest(PushReplicationEvent<?> event, PushSession pushSession, List<RequestCandidate> candidates) {
        int destinationBrokerId = event.replicaId();
        long destinationBrokerEpoch = pushSession.replicaEpoch();
        if (destinationBrokerEpoch == this.latestKnownBrokerEpochs.getOrDefault(destinationBrokerId, -1L)) {
            BufferingAppendRecordsBuilder builder = this.getOrCreateBuilder(destinationBrokerId, destinationBrokerEpoch);
            if (!builder.processEvent(event, pushSession)) {
                this.bufferingRequestBuilders.remove(destinationBrokerId);
                this.createCandidateFromRemovedBuilder(builder, candidates);
                if (!this.getOrCreateBuilder(destinationBrokerId, destinationBrokerEpoch).processEvent(event, pushSession)) {
                    this.log.error("Push replication event {} could not be consumed", event);
                    this.failSessionAfterEventProcessingFailure(event, candidates);
                }
            }
        } else {
            this.log.info("Push replication event {} for push session {} ignored due to stale broker epoch. Latest known broker epoch is {}", new Object[]{event, pushSession, destinationBrokerEpoch});
            this.untrackRecords(event);
        }
    }

    private void failSessionAfterEventProcessingFailure(PushReplicationEvent<?> event, List<RequestCandidate> candidates) {
        PushReplicationEvent<CompletableFuture<Void>> stopEvent = PushReplicationEvent.forStopPush(event.topicIdPartition(), event.replicaId());
        this.processStopEvent(stopEvent, candidates);
    }

    private void failSessionAfterResponseFailure(PartitionReplicaKey key, PushSession sessionToFail) {
        PushSession activeSession = this.activePushSessions.remove(key);
        if (activeSession == null || activeSession != sessionToFail) {
            String errorMessage = String.format("Unexpected mismatch between active session %s and session to fail %s for partition replica %s", activeSession, sessionToFail, key);
            throw new IllegalStateException(errorMessage);
        }
        sessionToFail.onPushSessionEnded();
    }

    private Collection<RequestAndCompletionHandler> getRequestsToSend(List<RequestCandidate> candidates) {
        Iterator<Map.Entry<Integer, BufferingAppendRecordsBuilder>> iterator = this.bufferingRequestBuilders.entrySet().iterator();
        while (iterator.hasNext()) {
            Node destinationNode;
            BufferingAppendRecordsBuilder builder = iterator.next().getValue();
            if (!builder.isRequestReady() || (destinationNode = this.getNodeLoader(builder.destinationBrokerId()).getNode()) == null || this.hasInFlightRequests(destinationNode)) continue;
            iterator.remove();
            this.createCandidateFromRemovedBuilder(builder, candidates);
        }
        for (RequestCandidate requestCandidate : candidates) {
            this.pendingRequests.computeIfAbsent(requestCandidate.destinationBrokerId, id -> new ArrayDeque()).addLast(requestCandidate.createRequest());
        }
        if (this.pendingRequests.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<RequestAndCompletionHandler> selected = new ArrayList<RequestAndCompletionHandler>();
        for (Map.Entry<Integer, Deque<RequestAndCompletionHandler>> entry : this.pendingRequests.entrySet()) {
            int destinationBrokerId = entry.getKey();
            Deque<RequestAndCompletionHandler> pendingRequestsForNode = entry.getValue();
            if (pendingRequestsForNode.isEmpty()) continue;
            NodeLoader nodeLoader = this.getNodeLoader(destinationBrokerId);
            Node destinationNode = nodeLoader.getNode();
            if (destinationNode == null) {
                if (!nodeLoader.failed()) continue;
                this.handleNodeLoaderFailed(destinationBrokerId, pendingRequestsForNode);
                continue;
            }
            if (this.hasInFlightRequests(destinationNode)) continue;
            RequestAndCompletionHandler first = Objects.requireNonNull(pendingRequestsForNode.poll());
            if (!destinationNode.equals((Object)first.destination)) {
                first = new RequestAndCompletionHandler(first.creationTimeMs, destinationNode, first.request, first.handler);
            }
            selected.add(first);
        }
        return selected;
    }

    private void handleNodeLoaderFailed(int destinationBrokerId, Deque<RequestAndCompletionHandler> pendingRequests) {
        RequestAndCompletionHandler pendingRequest;
        while ((pendingRequest = pendingRequests.poll()) != null) {
            this.signalDisconnection(pendingRequest);
        }
        this.nodeLoaders.remove(destinationBrokerId);
    }

    private void signalDisconnection(RequestAndCompletionHandler request) {
        ClientResponse disconnectionResponse = new ClientResponse(null, request.handler, request.destination == null ? "" : request.destination.idString(), request.creationTimeMs, this.time.milliseconds(), true, null, null, null);
        request.handler.onComplete(disconnectionResponse);
    }

    private BufferingAppendRecordsBuilder getOrCreateBuilder(int destinationBrokerId, long destinationBrokerEpoch) {
        return this.bufferingRequestBuilders.computeIfAbsent(destinationBrokerId, id -> new BufferingAppendRecordsBuilder((int)id, destinationBrokerEpoch, this.config, this.tracker, this.time));
    }

    private void createCandidateFromRemovedBuilder(BufferingAppendRecordsBuilder builder, List<RequestCandidate> candidates) {
        int destinationBrokerId = builder.destinationBrokerId();
        long creationTimeMs = this.time.milliseconds();
        AppendRecordsRequestData requestData = builder.build();
        if (requestData == null) {
            return;
        }
        candidates.add(new RequestCandidate(creationTimeMs, destinationBrokerId, requestData));
    }

    void handleAppendRecordsResponse(long creationTimeMs, int destinationBrokerId, AppendRecordsRequestData requestData, ClientResponse clientResponse) {
        HashMap<PartitionReplicaKey, PushSession> matchingSessions;
        AppendRecordsRequestData endSessionRequestData;
        AuthenticationException topLevelError = null;
        AppendRecordsResponse response = null;
        if (clientResponse.authenticationException() != null) {
            topLevelError = clientResponse.authenticationException();
        } else if (clientResponse.wasDisconnected()) {
            this.getNodeLoader(destinationBrokerId).forceLoad();
            topLevelError = clientResponse.wasTimedOut() ? new TimeoutException() : new NetworkException();
        } else if (clientResponse.versionMismatch() != null) {
            topLevelError = clientResponse.versionMismatch();
        } else {
            response = (AppendRecordsResponse)clientResponse.responseBody();
            if (Errors.NONE.code() != response.data().errorCode()) {
                topLevelError = Errors.forCode((short)response.data().errorCode()).exception();
            }
        }
        if (topLevelError instanceof RetriableException && this.time.milliseconds() - creationTimeMs < (long)this.retryTimeoutMs) {
            this.retryRequest(creationTimeMs, destinationBrokerId, requestData);
            return;
        }
        if (topLevelError != null) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Received top-level response error {} for AppendRecords request {} to broker {}", new Object[]{topLevelError, requestData, destinationBrokerId});
            } else {
                this.log.info("Received top-level response error {} for AppendRecords request to broker {}", (Object)topLevelError, (Object)destinationBrokerId);
            }
        }
        if ((endSessionRequestData = this.processRequestData(requestData, destinationBrokerId, matchingSessions = new HashMap<PartitionReplicaKey, PushSession>(), (Throwable)topLevelError)) != null) {
            this.retryRequest(creationTimeMs, destinationBrokerId, endSessionRequestData);
            return;
        }
        if (topLevelError == null && !matchingSessions.isEmpty()) {
            this.processResponseData(response.data(), destinationBrokerId, matchingSessions);
        }
    }

    private void retryRequest(long creationTimeMs, int destinationBrokerId, AppendRecordsRequestData requestData) {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Retrying AppendRecords request {} with creation timestamp {} to broker {}", new Object[]{requestData, creationTimeMs, destinationBrokerId});
        }
        this.pendingRequests.computeIfAbsent(destinationBrokerId, id -> new ArrayDeque()).addFirst(new RequestAndCompletionHandler(creationTimeMs, this.getNodeLoader(destinationBrokerId).getNode(), (AbstractRequest.Builder)new AppendRecordsRequest.Builder(requestData), r -> this.handleAppendRecordsResponse(creationTimeMs, destinationBrokerId, requestData, r)));
    }

    private AppendRecordsRequestData processRequestData(AppendRecordsRequestData requestData, int destinationBrokerId, Map<PartitionReplicaKey, PushSession> matchingSessions, Throwable topLevelError) {
        long requestReplicaEpoch = requestData.replicaEpoch();
        boolean foundReplicaEpochMismatch = false;
        boolean isTopLevelRetriableError = topLevelError instanceof RetriableException;
        ArrayList<AppendRecordsRequestData.TopicData> endSessionData = null;
        for (AppendRecordsRequestData.TopicData topicData : requestData.topics()) {
            AppendRecordsRequestData.TopicData endSessionTopicData = null;
            for (AppendRecordsRequestData.PartitionData partitionData : topicData.partitions()) {
                BaseRecords records = partitionData.records();
                if (records instanceof BufferingPartitionDataBuilder.PartitionRecords) {
                    for (MemoryRecords memoryRecords : ((BufferingPartitionDataBuilder.PartitionRecords)records).memoryRecords()) {
                        this.tracker.countDown(memoryRecords);
                    }
                }
                if (foundReplicaEpochMismatch) {
                    endSessionData = null;
                    continue;
                }
                PartitionReplicaKey key = new PartitionReplicaKey(topicData.topicId(), partitionData.partitionIndex(), destinationBrokerId);
                PushSession session = this.activePushSessions.get(key);
                if (session != null) {
                    if (requestReplicaEpoch != session.replicaEpoch()) {
                        this.log.info("Replica epoch mismatch (found {} in request and {} in session for partition replica {}); skip all further request processing", new Object[]{requestReplicaEpoch, session.replicaEpoch(), key});
                        foundReplicaEpochMismatch = true;
                        continue;
                    }
                    if (partitionData.currentLeaderEpoch() != session.leaderEpoch() || partitionData.replicationSessionId() != session.replicationSessionId()) {
                        this.log.info("Active push session {} doesn't match AppendRecords request data {} for partition replica {}", new Object[]{session, partitionData, key});
                        continue;
                    }
                    if (topLevelError != null) {
                        this.failSessionAfterResponseFailure(key, session);
                        continue;
                    }
                    matchingSessions.put(key, session);
                    continue;
                }
                if (!partitionData.endReplicationSession()) {
                    this.log.info("No active push session found matching AppendRecords request data {} for partition replica {}", (Object)partitionData, (Object)key);
                    continue;
                }
                if (!isTopLevelRetriableError) continue;
                if (endSessionData == null) {
                    endSessionData = new ArrayList<AppendRecordsRequestData.TopicData>();
                }
                if (endSessionTopicData == null) {
                    endSessionTopicData = new AppendRecordsRequestData.TopicData().setTopicId(topicData.topicId());
                    endSessionData.add(endSessionTopicData);
                }
                AppendRecordsRequestData.PartitionData endSessionPartitionData = new AppendRecordsRequestData.PartitionData().setPartitionIndex(partitionData.partitionIndex()).setEndReplicationSession(true);
                endSessionTopicData.partitions().add(endSessionPartitionData);
            }
        }
        return endSessionData == null ? null : new AppendRecordsRequestData().setTopics(endSessionData);
    }

    private void processResponseData(AppendRecordsResponseData responseData, int destinationBrokerId, Map<PartitionReplicaKey, PushSession> matchingSessions) {
        for (AppendRecordsResponseData.TopicData topicData : responseData.topics()) {
            for (AppendRecordsResponseData.PartitionData partitionData : topicData.partitions()) {
                PartitionReplicaKey key = new PartitionReplicaKey(topicData.topicId(), partitionData.partitionIndex(), destinationBrokerId);
                PushSession session = matchingSessions.get(key);
                if (session == null) continue;
                if (partitionData.errorCode() != Errors.NONE.code()) {
                    this.log.info("Received AppendRecords response error {} for partition replica {}", (Object)Errors.forCode((short)partitionData.errorCode()), (Object)key);
                    this.failSessionAfterResponseFailure(key, session);
                    continue;
                }
                session.onAppendRecordsResponse(partitionData.logEndOffset(), partitionData.logStartOffset());
            }
        }
    }

    private NodeLoader getNodeLoader(int destinationBrokerId) {
        return this.nodeLoaders.computeIfAbsent(destinationBrokerId, id -> new NodeLoader(this.time, () -> this.nodeResolver.apply((Integer)id)));
    }

    private static class NodeLoader {
        private static final int MAX_ATTEMPTS = 5;
        private static final long INITIAL_BACKOFF_MS = 100L;
        private static final long MAX_BACKOFF_MS = 1000L;
        private final Time time;
        private final Supplier<Optional<Node>> nodeResolver;
        private final ExponentialBackoff backoff;
        private final int maxAttempts;
        private Node node;
        private int failedAttempts;
        private long lastAttemptMs;

        NodeLoader(Time time, Supplier<Optional<Node>> nodeResolver) {
            this(time, nodeResolver, 5);
        }

        private NodeLoader(Time time, Supplier<Optional<Node>> nodeResolver, int maxAttempts) {
            this.time = time;
            this.nodeResolver = nodeResolver;
            this.maxAttempts = maxAttempts;
            this.backoff = new ExponentialBackoff(100L, 2, 1000L, 0.0);
            this.load();
        }

        private void load() {
            Optional<Node> maybeNode = this.nodeResolver.get();
            this.lastAttemptMs = this.time.hiResClockMs();
            if (maybeNode.isPresent()) {
                this.node = maybeNode.get();
                this.failedAttempts = 0;
            } else {
                ++this.failedAttempts;
            }
        }

        Node getNode() {
            if (this.node == null && this.failedAttempts < this.maxAttempts && this.time.hiResClockMs() > this.lastAttemptMs + this.backoff.backoff((long)(this.failedAttempts - 1))) {
                this.load();
            }
            return this.node;
        }

        void forceLoad() {
            this.load();
        }

        boolean failed() {
            return this.failedAttempts >= this.maxAttempts;
        }
    }

    private static final class PartitionReplicaKey {
        private final Uuid topicId;
        private final int partitionId;
        private final int replicaId;

        public PartitionReplicaKey(TopicIdPartition topicIdPartition, int replicaId) {
            this.topicId = topicIdPartition.topicId();
            this.partitionId = topicIdPartition.partition();
            this.replicaId = replicaId;
        }

        public PartitionReplicaKey(Uuid topicId, int partitionId, int replicaId) {
            this.topicId = topicId;
            this.partitionId = partitionId;
            this.replicaId = replicaId;
        }

        public PartitionReplicaKey(PushReplicationEvent<?> event) {
            this(event.topicIdPartition(), event.replicaId());
        }

        public int hashCode() {
            return Objects.hash(this.topicId, this.partitionId, this.replicaId);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof PartitionReplicaKey)) {
                return false;
            }
            PartitionReplicaKey that = (PartitionReplicaKey)o;
            return this.topicId.equals((Object)that.topicId) && this.partitionId == that.partitionId && this.replicaId == that.replicaId;
        }

        public String toString() {
            return "PartitionReplicaKey{topicId=" + this.topicId + ", partitionId=" + this.partitionId + ", replicaId=" + this.replicaId + '}';
        }
    }

    private final class RequestCandidate {
        private final long creationTimeMs;
        private final int destinationBrokerId;
        private final AppendRecordsRequestData requestData;

        RequestCandidate(long creationTimeMs, int destinationBrokerId, AppendRecordsRequestData requestData) {
            this.creationTimeMs = creationTimeMs;
            this.destinationBrokerId = destinationBrokerId;
            this.requestData = requestData;
        }

        RequestAndCompletionHandler createRequest() {
            return new RequestAndCompletionHandler(this.creationTimeMs, PusherThread.this.getNodeLoader(this.destinationBrokerId).getNode(), (AbstractRequest.Builder)new AppendRecordsRequest.Builder(this.requestData), r -> PusherThread.this.handleAppendRecordsResponse(this.creationTimeMs, this.destinationBrokerId, this.requestData, r));
        }
    }
}

