/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.replication.push;

import io.confluent.kafka.replication.push.PushManager;
import io.confluent.kafka.replication.push.PushSession;
import io.confluent.kafka.replication.push.Pusher;
import io.confluent.kafka.replication.push.PusherThread;
import io.confluent.kafka.replication.push.ReplicationConfig;
import io.confluent.kafka.replication.push.buffer.RefCountingMemoryTracker;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.clients.NetworkClient;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicIdPartition;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PushManagerImpl
implements PushManager {
    private static final Logger log = LoggerFactory.getLogger(PushManagerImpl.class);
    private final ReplicationConfig config;
    private final Metrics metrics;
    private final List<Pusher> pushers;
    private final RefCountingMemoryTracker<MemoryRecords> tracker;
    private final AtomicReference<State> state;
    private final ConcurrentHashMap<Integer, Integer> replicaPusherMap;
    private final AtomicInteger numReplicasAssigned;

    public PushManagerImpl(ReplicationConfig config, Time time, Metrics metrics, Function<Integer, Optional<Node>> nodeResolver, Function<Integer, NetworkClient> networkClientResolver) {
        this.config = config;
        this.metrics = metrics;
        RefCountingMemoryTracker<MemoryRecords> tracker = new RefCountingMemoryTracker<MemoryRecords>(MemoryRecords::sizeInBytes, config.maxMemoryBufferBytes(), ignored -> {});
        this.pushers = Collections.unmodifiableList(PushManagerImpl.initPushers(config, time, nodeResolver, networkClientResolver, tracker));
        this.tracker = tracker;
        this.state = new AtomicReference<State>(State.NOT_STARTED);
        this.replicaPusherMap = new ConcurrentHashMap();
        this.numReplicasAssigned = new AtomicInteger(0);
    }

    @Override
    public void onLeaderAppend(TopicIdPartition topicPartition, Set<Integer> replicaIds, long highWatermark, long appendOffset, AbstractRecords records) {
        MemoryRecords memoryRecords;
        int numEnqueued = replicaIds.size();
        if (numEnqueued > 0 && records instanceof MemoryRecords && !this.tracker.initCount(memoryRecords = (MemoryRecords)records, numEnqueued)) {
            this.stopPush(topicPartition, replicaIds, true);
            return;
        }
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicPartition, replicaId);
            pusher.onLeaderAppend(topicPartition, replicaId, highWatermark, appendOffset, records);
        }
    }

    @Override
    public void onHighWatermarkUpdate(TopicIdPartition topicPartition, Set<Integer> replicaIds, long updatedHighWatermark) {
        log.trace("onHighWatermarkUpdate called with value {} for partition {} and replicas {}", new Object[]{updatedHighWatermark, topicPartition, replicaIds});
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicPartition, replicaId);
            pusher.onHighWatermarkUpdate(topicPartition, replicaId, updatedHighWatermark);
        }
    }

    @Override
    public void onLogStartOffsetUpdate(TopicIdPartition topicPartition, Set<Integer> replicaIds, long updatedLogStartOffset) {
        log.trace("onLogStartOffsetUpdate called with offset {} for partition {} and replicas {}", new Object[]{updatedLogStartOffset, topicPartition, replicaIds});
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicPartition, replicaId);
            pusher.onLogStartOffsetUpdate(topicPartition, replicaId, updatedLogStartOffset);
        }
    }

    @Override
    public CompletableFuture<Void> startPush(TopicIdPartition topicPartition, int replicaId, PushSession pushSession) {
        log.trace("startPush called with session {} for partition {} and replica {}", new Object[]{pushSession, topicPartition, replicaId});
        Pusher pusher = this.getPusher(topicPartition, replicaId);
        return pusher.startPush(topicPartition, replicaId, pushSession);
    }

    @Override
    public CompletableFuture<Void> stopPush(TopicIdPartition topicPartition, Set<Integer> replicaIds, boolean sendEndSessionRequest) {
        log.trace("stopPush called for partition {} and replicas {} with sendEndSession={}", new Object[]{topicPartition, replicaIds, sendEndSessionRequest});
        CompletableFuture[] stopPushFutures = new CompletableFuture[replicaIds.size()];
        int i = 0;
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicPartition, replicaId);
            stopPushFutures[i++] = pusher.stopPush(topicPartition, replicaId, sendEndSessionRequest);
        }
        return CompletableFuture.allOf(stopPushFutures);
    }

    @Override
    public boolean startup() {
        if (this.state.compareAndSet(State.NOT_STARTED, State.RUNNING)) {
            log.info("Starting up PushManager...");
            this.pushers.forEach(Pusher::start);
            return true;
        }
        log.info("PushManager is already started.");
        return false;
    }

    @Override
    public boolean shutdown() {
        if (this.state.compareAndSet(State.RUNNING, State.SHUTDOWN)) {
            log.info("Shutting down PushManager...");
            this.tracker.close();
            this.shutdownPushers();
            return true;
        }
        log.info("PushManager is not running.");
        return false;
    }

    @Override
    public boolean isActive() {
        return this.state.get() == State.RUNNING;
    }

    public Pusher getPusher(TopicIdPartition topicIdPartition, int destinationBrokerId) {
        if (!this.isActive()) {
            String errorMessage = "PushManager has already been shut down!";
            log.error(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
        Integer pusherId = this.replicaPusherMap.get(destinationBrokerId);
        if (pusherId == null) {
            pusherId = this.replicaPusherMap.computeIfAbsent(destinationBrokerId, k -> Math.abs(this.numReplicasAssigned.getAndIncrement() % this.pushers.size()));
        }
        return this.pushers.get(pusherId);
    }

    private static List<Pusher> initPushers(ReplicationConfig config, Time time, Function<Integer, Optional<Node>> nodeResolver, Function<Integer, NetworkClient> networkClientResolver, RefCountingMemoryTracker<MemoryRecords> tracker) {
        ArrayList<Pusher> newPushers = new ArrayList<Pusher>();
        for (int pusherId = 0; pusherId < config.maxPushers(); ++pusherId) {
            PusherThread newPusher = PusherThread.newPusher(pusherId, config, (KafkaClient)networkClientResolver.apply(pusherId), nodeResolver, tracker, time);
            newPushers.add(newPusher);
        }
        return newPushers;
    }

    private void shutdownPushers() {
        this.pushers.forEach(Pusher::shutdown);
    }

    private static enum State {
        NOT_STARTED,
        RUNNING,
        SHUTDOWN;

    }
}

