/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.databalancing;

import io.confluent.common.utils.SystemTime;
import io.confluent.common.utils.Time;
import io.confluent.kafka.databalancing.DefaultRebalancer;
import io.confluent.kafka.databalancing.ProposedRebalance;
import io.confluent.kafka.databalancing.ReassignmentFailedException;
import io.confluent.kafka.databalancing.RebalancerAdmin;
import io.confluent.kafka.databalancing.RebalancerConfig;
import io.confluent.kafka.databalancing.metric.MetricsCollector;
import io.confluent.kafka.databalancing.throttle.Throttle;
import io.confluent.kafka.databalancing.topology.Broker;
import io.confluent.kafka.databalancing.topology.ClusterAssignment;
import io.confluent.kafka.databalancing.topology.ClusterReassignment;
import io.confluent.kafka.databalancing.topology.PartitionAssignment;
import io.confluent.kafka.databalancing.topology.PartitionReassignment;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.NewPartitionReassignment;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IncrementalRebalancer
extends DefaultRebalancer {
    private static final Logger log = LoggerFactory.getLogger(IncrementalRebalancer.class);
    private final int maxConcurrentMovesPerLeader;
    private final Time time;

    public IncrementalRebalancer(RebalancerConfig config, RebalancerAdmin rebalancerAdmin) {
        super(config, rebalancerAdmin);
        this.time = new SystemTime();
        this.maxConcurrentMovesPerLeader = config.getInt("confluent.rebalancer.max.concurrent.moves.per.leader");
    }

    public IncrementalRebalancer(RebalancerConfig config, RebalancerAdmin rebalancerAdmin, MetricsCollector metricsCollector, Throttle throttler, Time time) {
        super(config, rebalancerAdmin, metricsCollector, throttler);
        this.time = time;
        this.maxConcurrentMovesPerLeader = config.getInt("confluent.rebalancer.max.concurrent.moves.per.leader");
    }

    @Override
    public void startRebalance(ProposedRebalance proposedRebalance, long replicationQuota) {
        IncrementalRebalanceExecutor executor = new IncrementalRebalanceExecutor(proposedRebalance, this.rebalancerAdmin, this.throttle, this.time, this.maxConcurrentMovesPerLeader);
        this.throttle.limit(replicationQuota, proposedRebalance.reassignment().brokers());
        executor.run();
        this.throttle.disengage();
    }

    private static class PendingReassignment
    implements Comparable<PendingReassignment> {
        final TopicPartition partition;
        final PartitionAssignment sourceAssignment;
        final PartitionAssignment targetAssignment;
        final long sizeInBytes;

        private PendingReassignment(TopicPartition partition, PartitionAssignment sourceAssignment, PartitionAssignment targetAssignment, long sizeInBytes) {
            this.partition = partition;
            this.sourceAssignment = sourceAssignment;
            this.targetAssignment = targetAssignment;
            this.sizeInBytes = sizeInBytes;
        }

        @Override
        public int compareTo(PendingReassignment o) {
            int res = Long.compare(this.sizeInBytes, o.sizeInBytes);
            if (res != 0) {
                return res;
            }
            res = this.partition.topic().compareTo(o.partition.topic());
            if (res != 0) {
                return res;
            }
            return Integer.compare(this.partition.partition(), o.partition.partition());
        }
    }

    private static class LeaderReassignments {
        private Map<TopicPartition, PendingReassignment> inProgress = new HashMap<TopicPartition, PendingReassignment>();
        private PriorityQueue<PendingReassignment> awaiting = new PriorityQueue();

        private LeaderReassignments() {
        }

        PendingReassignment poll() {
            PendingReassignment reassignment = this.awaiting.poll();
            if (reassignment != null) {
                this.inProgress.put(reassignment.partition, reassignment);
            }
            return reassignment;
        }

        void offer(PendingReassignment reassignment) {
            this.awaiting.offer(reassignment);
        }

        Map<TopicPartition, PendingReassignment> checkAndComplete(Set<TopicPartition> currentInProgress) {
            List allFinished = this.inProgress.keySet().stream().filter(tp -> !currentInProgress.contains(tp)).collect(Collectors.toList());
            if (allFinished.isEmpty()) {
                return Collections.emptyMap();
            }
            HashMap<TopicPartition, PendingReassignment> finishedReassignments = new HashMap<TopicPartition, PendingReassignment>(allFinished.size());
            for (TopicPartition finished : allFinished) {
                finishedReassignments.put(finished, this.inProgress.remove(finished));
            }
            return finishedReassignments;
        }

        int numInProgress() {
            return this.inProgress.size();
        }

        int numRemaining() {
            return this.inProgress.size() + this.awaiting.size();
        }
    }

    private static class IncrementalRebalanceExecutor
    implements Runnable {
        private static final int MIN_BACKOFF_MS = 100;
        private static final int TIMEOUT_MS = 30000;
        private static final int MAX_BACKOFF_MS = 5000;
        private final Map<Broker, LeaderReassignments> reassignments = new HashMap<Broker, LeaderReassignments>();
        private final RebalancerAdmin rebalancerAdmin;
        private final Throttle throttle;
        private final int maxConcurrentMovesByLeader;
        private final Time time;
        private int backoffMs = 100;
        private Map<TopicPartition, PendingReassignment> retryReassignments;
        private long retryDeadlineMs;

        IncrementalRebalanceExecutor(ProposedRebalance rebalance, RebalancerAdmin rebalancerAdmin, Throttle throttle, Time time, int maxConcurrentMovesByLeader) {
            this.rebalancerAdmin = rebalancerAdmin;
            this.throttle = throttle;
            this.time = time;
            this.maxConcurrentMovesByLeader = maxConcurrentMovesByLeader;
            this.buildReassignments(rebalance);
        }

        @Override
        public void run() {
            while (!this.reassignments.isEmpty()) {
                this.maybeSubmitReassignment();
                this.backoff();
                if (this.retryReassignments != null) continue;
                this.checkCompletion();
            }
        }

        private Map<TopicPartition, PendingReassignment> buildNextReassignment() {
            long currentTimeMs = this.time.milliseconds();
            if (this.retryReassignments != null) {
                if (currentTimeMs > this.retryDeadlineMs) {
                    throw new ReassignmentFailedException("Failed to submit reassignment " + this.retryReassignments + " after " + 30000 + "ms");
                }
                this.printNewReassignment("Retry", this.retryReassignments);
                return this.retryReassignments;
            }
            HashMap<TopicPartition, PendingReassignment> newReassignments = new HashMap<TopicPartition, PendingReassignment>();
            for (LeaderReassignments leaderReassignments : this.reassignments.values()) {
                this.buildNewAssignment(leaderReassignments, newReassignments);
            }
            if (!newReassignments.isEmpty()) {
                this.printNewReassignment("Begin", newReassignments);
                this.retryDeadlineMs = currentTimeMs + 30000L;
                this.retryReassignments = newReassignments;
            }
            return newReassignments;
        }

        private void printNewReassignment(String verb, Map<TopicPartition, PendingReassignment> reassignments) {
            for (Map.Entry<TopicPartition, PendingReassignment> newReassignmentEntry : reassignments.entrySet()) {
                PendingReassignment newReassignment = newReassignmentEntry.getValue();
                System.out.println(verb + " reassignment of " + newReassignmentEntry.getKey() + " from " + newReassignment.sourceAssignment + " to " + newReassignment.targetAssignment);
            }
        }

        private void maybeSubmitReassignment() {
            Map<TopicPartition, PendingReassignment> newReassignments = this.buildNextReassignment();
            if (!newReassignments.isEmpty()) {
                try {
                    this.throttle.throttleReplicas(this.reassignmentFromPendingMap(newReassignments));
                    this.submitReassignment(newReassignments);
                    this.retryReassignments = null;
                    this.resetBackoff();
                }
                catch (RuntimeException e) {
                    log.warn("Failed to submit reassignment of partitions {}", newReassignments.keySet(), (Object)e);
                }
            }
        }

        private ClusterReassignment reassignmentFromPendingMap(Map<TopicPartition, PendingReassignment> pendingMap) {
            return new ClusterReassignment(pendingMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> {
                PendingReassignment pending = (PendingReassignment)entry.getValue();
                return new PartitionReassignment(pending.sourceAssignment.replicaIds(), pending.targetAssignment.replicaIds());
            })));
        }

        private void submitReassignment(Map<TopicPartition, PendingReassignment> reassignment) {
            Map<TopicPartition, NewPartitionReassignment> newAssignments = reassignment.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> {
                PartitionAssignment assignment = ((PendingReassignment)entry.getValue()).targetAssignment;
                return NewPartitionReassignment.ofReplicasAndObservers(assignment.replicaIds(), assignment.observerIds());
            }));
            this.rebalancerAdmin.createPartitionReassignment(newAssignments);
        }

        private void checkCompletion() {
            Set<TopicPartition> inProgress = this.rebalancerAdmin.currentReassignment().topicPartitions();
            ArrayList<Broker> finished = new ArrayList<Broker>();
            HashMap<TopicPartition, PendingReassignment> completed = new HashMap<TopicPartition, PendingReassignment>();
            for (Map.Entry<Broker, LeaderReassignments> entry : this.reassignments.entrySet()) {
                Broker broker = entry.getKey();
                LeaderReassignments leaderReassignments = entry.getValue();
                completed.putAll(leaderReassignments.checkAndComplete(inProgress));
                if (leaderReassignments.numRemaining() != 0) continue;
                finished.add(broker);
            }
            for (Map.Entry<Broker, LeaderReassignments> entry : completed.entrySet()) {
                PendingReassignment newReassignment = (PendingReassignment)((Object)entry.getValue());
                System.out.println("Completed reassignment of " + entry.getKey() + " from " + newReassignment.sourceAssignment + " to " + newReassignment.targetAssignment);
            }
            this.throttle.unthrottleReplicas(this.reassignmentFromPendingMap(completed));
            this.reassignments.keySet().removeAll(finished);
        }

        private void buildNewAssignment(LeaderReassignments leaderReassignments, Map<TopicPartition, PendingReassignment> newReassignments) {
            PendingReassignment pending;
            while (leaderReassignments.numInProgress() < this.maxConcurrentMovesByLeader && (pending = leaderReassignments.poll()) != null) {
                newReassignments.put(pending.partition, pending);
            }
        }

        private void resetBackoff() {
            this.backoffMs = 100;
        }

        private void backoff() {
            this.time.sleep((long)this.backoffMs);
            this.backoffMs = Math.min(this.backoffMs * 2, 5000);
        }

        private void buildReassignments(ProposedRebalance rebalance) {
            ClusterAssignment currentAssignment = rebalance.currentAssignment();
            for (Map.Entry<TopicPartition, PartitionAssignment> entry : rebalance.proposedAssignment().asMap().entrySet()) {
                TopicPartition topicPartition = entry.getKey();
                PartitionAssignment sourceAssignment = currentAssignment.assignment(topicPartition);
                Broker currentLeader = sourceAssignment.preferredLeader().get();
                PartitionAssignment targetAssignment = entry.getValue();
                if (targetAssignment.equals(sourceAssignment)) continue;
                LeaderReassignments leaderReassignments = this.reassignments.computeIfAbsent(currentLeader, b -> new LeaderReassignments());
                leaderReassignments.offer(new PendingReassignment(topicPartition, sourceAssignment, targetAssignment, rebalance.partitionSize(topicPartition)));
            }
        }
    }
}

