/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.databalancing;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.kafka.databalancing.MutableRebalanceContext;
import io.confluent.kafka.databalancing.RebalanceContext;
import io.confluent.kafka.databalancing.RebalancePolicy;
import io.confluent.kafka.databalancing.Utils;
import io.confluent.kafka.databalancing.constraint.Constraints;
import io.confluent.kafka.databalancing.constraint.RebalanceConstraints;
import io.confluent.kafka.databalancing.exception.ValidationException;
import io.confluent.kafka.databalancing.topology.Broker;
import io.confluent.kafka.databalancing.topology.ClusterAssignment;
import io.confluent.kafka.databalancing.topology.PartitionAssignment;
import io.confluent.kafka.databalancing.topology.Replica;
import io.confluent.kafka.databalancing.topology.TopologyUtils;
import io.confluent.kafka.databalancing.view.BrokerCountFairView;
import io.confluent.kafka.databalancing.view.BrokerFairView;
import io.confluent.kafka.databalancing.view.BrokerSizeFairView;
import io.confluent.kafka.databalancing.view.ClusterView;
import io.confluent.kafka.databalancing.view.RackCountFairView;
import io.confluent.kafka.databalancing.view.RackSizeFairView;
import io.confluent.kafka.databalancing.view.TopicViewSupplier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.metadata.TopicPlacement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MovesOptimisedRebalancePolicy
implements RebalancePolicy {
    private static final Logger logger = LoggerFactory.getLogger(MovesOptimisedRebalancePolicy.class);
    private int replicasMoved = 0;
    private int leadersMoved = 0;
    private int firstObserversMoved = 0;

    @Override
    public ClusterAssignment enforcePlacementConstraints(MutableRebalanceContext context) {
        int step = 0;
        logger.info("**** {}. Ensure partitions match topic placement constraint ****", (Object)step++);
        Set<TopicPartition> partitionsWithBrokersRemoved = this.removeBrokers(context);
        logger.info("**** {}. Ensure no under-replicated partitions ****", (Object)step++);
        this.fullyReplicate(context, partitionsWithBrokersRemoved);
        this.log(context);
        return context.buildAssignment();
    }

    @Override
    public ClusterAssignment rebalancePartitions(MutableRebalanceContext context) {
        String rackLogMessageFragment;
        List<String> racks;
        this.ensureOfflineBrokerAreRemoved(context);
        List<String> processedRacks = racks = Utils.sorted(context.allRacks());
        if (racks.isEmpty()) {
            processedRacks = Arrays.asList(new String[]{null});
            logger.info("Executing cluster-wide as no racks configured");
        }
        int step = 0;
        logger.info("**** {}. Ensure partitions match rack or topic placement constraints ****", (Object)step++);
        Set<TopicPartition> partitionsWithBrokersRemoved = this.removeBrokers(context);
        logger.info("**** {}. Ensure no under-replicated partitions ****", (Object)step++);
        this.fullyReplicate(context, partitionsWithBrokersRemoved);
        List<String> topics = Utils.sorted(context.topics());
        if (!racks.isEmpty()) {
            for (String topic : topics) {
                logger.info("**** {}. Ensure balanced replica count across racks for topic: {} ****", (Object)step++, (Object)topic);
                this.replicaTopicFairness(context, new RackCountFairView(context, topic));
                logger.info("**** {}. Ensure balanced replica size across racks for topic: {} ****", (Object)step++, (Object)topic);
                this.replicaTopicFairness(context, new RackSizeFairView(context, topic));
            }
        }
        for (String rack : processedRacks) {
            rackLogMessageFragment = this.rackLogMessageFragment(rack);
            for (String topic : Utils.sorted(context.topics())) {
                logger.info("**** {}. Ensure balanced replica count for topic: {} {} ****", new Object[]{step++, topic, rackLogMessageFragment});
                this.replicaTopicFairness(context, new BrokerCountFairView(context, rack, topic));
                logger.info("**** {}. Ensure balanced replica size for topic: {} {} ****", new Object[]{step++, topic, rackLogMessageFragment});
                this.replicaTopicFairness(context, new BrokerSizeFairView(context, rack, topic));
            }
        }
        for (String topic : topics) {
            logger.info("**** {}. Ensure balanced leaders for topic: {} {} ****", new Object[]{step++, topic, this.maybeClusterWide(racks)});
            this.leaderTopicFairness(context, new BrokerCountFairView(context, null, topic));
        }
        for (String rack : processedRacks) {
            rackLogMessageFragment = this.rackLogMessageFragment(rack);
            for (String topic : topics) {
                logger.info("**** {}. Ensure balanced first observer count count for topic: {} {} ****", new Object[]{step++, topic, rackLogMessageFragment});
                this.firstObserverTopicFairness(context, new BrokerCountFairView(context, rack, topic));
            }
        }
        if (!racks.isEmpty()) {
            logger.info("**** {}. Ensure balanced replica count across racks ****", (Object)step++);
            this.replicaFairness(context, new RackCountFairView(context, null), TopicViewSupplier.rackCount(context));
            logger.info("**** {}. Ensure balanced replica size across racks ****", (Object)step++);
            this.replicaSizeFairness(context, new RackSizeFairView(context, null), TopicViewSupplier.rackSize(context));
        }
        for (String rack : processedRacks) {
            rackLogMessageFragment = this.rackLogMessageFragment(rack);
            logger.info("**** {}. Ensure balanced replica count {} ****", (Object)step++, (Object)rackLogMessageFragment);
            this.replicaFairness(context, new BrokerCountFairView(context, rack, null), TopicViewSupplier.brokerCount(context, rack));
            logger.info("**** {}. Ensure balanced replica size {} ****", (Object)step++, (Object)rackLogMessageFragment);
            this.replicaSizeFairness(context, new BrokerSizeFairView(context, rack, null), TopicViewSupplier.brokerSize(context, rack));
        }
        for (String rack : processedRacks) {
            rackLogMessageFragment = this.rackLogMessageFragment(rack);
            logger.info("**** {}. Ensure balanced first observer count {} ****", (Object)step++, (Object)rackLogMessageFragment);
            this.firstObserverFairness(context, new BrokerCountFairView(context, rack, null), TopicViewSupplier.brokerCount(context, rack));
        }
        logger.info("**** {}. Ensure balanced leaders for brokers {} ****", (Object)step++, (Object)this.maybeClusterWide(racks));
        BrokerCountFairView leaderBrokerView = new BrokerCountFairView(context, null, null);
        this.leaderFairness(context, leaderBrokerView, TopicViewSupplier.brokerCount(context, null));
        this.log(context);
        return context.buildAssignment();
    }

    private String maybeClusterWide(List<String> racks) {
        return racks.isEmpty() ? "" : "(cluster-wide)";
    }

    private String rackLogMessageFragment(String rack) {
        if (rack == null) {
            return "";
        }
        return "across brokers on rack " + rack;
    }

    private void replicaTopicFairness(final MutableRebalanceContext context, ClusterView topicView) {
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (final Replica abovePar : topicView.replicasOnAboveParBrokers()) {
                if (!topicView.replicasOnAboveParBrokers().contains(abovePar)) continue;
                final ClusterView tView = topicView;
                Predicate<Broker> fairnessPredicate = new Predicate<Broker>(){

                    @Override
                    public boolean test(Broker belowPar) {
                        boolean replicaFairness = tView.compareReplicaFairness(abovePar.topicPartition(), abovePar.broker(), belowPar).isGreater();
                        Optional<TopicPlacement> topicPlacement = context.topicPlacement(abovePar.topicPartition().topic());
                        boolean matchesObserverPlacement = Utils.brokerMatchesObserverPlacement(context, belowPar, topicPlacement);
                        if (matchesObserverPlacement) {
                            boolean firstObserverFairness = tView.compareFirstObserverFairness(abovePar.topicPartition(), abovePar.broker(), belowPar).isGreaterOrEqual();
                            return replicaFairness && firstObserverFairness;
                        }
                        return replicaFairness;
                    }
                };
                List<Broker> belowParBrokers = topicView.brokersWithBelowParReplicaFairness();
                if (!this.moveReplicaToBelowParBroker(context, topicView.constraints(), abovePar, belowParBrokers, fairnessPredicate)) continue;
                moveFound = true;
                topicView = topicView.refresh(context);
            }
        }
    }

    private void firstObserverFairness(MutableRebalanceContext context, BrokerFairView clusterView, TopicViewSupplier<?> topicViewSupplier) {
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (Replica abovePar : clusterView.firstObserversOnAboveParBrokers()) {
                if (!clusterView.firstObserversOnAboveParBrokers().contains(abovePar)) continue;
                BrokerFairView cView = clusterView;
                BrokerFairView tView = (BrokerFairView)topicViewSupplier.get(abovePar.topicPartition().topic());
                BooleanPredicate<Broker> fairnessPredicate = (belowPar, swap) -> {
                    boolean clusterFirstObserverFairness = cView.compareFirstObserverFairness(abovePar.topicPartition(), abovePar.broker(), (Broker)belowPar).isGreater();
                    boolean topicFirstObserverFairness = tView.compareFirstObserverFairness(abovePar.topicPartition(), abovePar.broker(), (Broker)belowPar).isGreaterOrEqual();
                    if (swap) {
                        return clusterFirstObserverFairness && topicFirstObserverFairness;
                    }
                    boolean clusterReplicaFairness = cView.compareReplicaFairness(abovePar.topicPartition(), abovePar.broker(), (Broker)belowPar).isGreaterOrEqual();
                    boolean topicReplicaFairness = tView.compareReplicaFairness(abovePar.topicPartition(), abovePar.broker(), (Broker)belowPar).isGreaterOrEqual();
                    return clusterFirstObserverFairness && topicFirstObserverFairness && clusterReplicaFairness && topicReplicaFairness;
                };
                List<Broker> belowParBrokers = clusterView.brokersWithBelowParFirstObserverFairness();
                if (!this.moveFirstObserverToBelowParBroker(context, clusterView.constraints(), abovePar, belowParBrokers, fairnessPredicate)) continue;
                moveFound = true;
                clusterView = (BrokerFairView)clusterView.refresh(context);
                topicViewSupplier.clear();
            }
        }
    }

    private void firstObserverTopicFairness(MutableRebalanceContext context, BrokerFairView topicView) {
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (Replica abovePar : topicView.firstObserversOnAboveParBrokers()) {
                if (!topicView.firstObserversOnAboveParBrokers().contains(abovePar)) continue;
                BrokerFairView tView = topicView;
                BooleanPredicate<Broker> fairnessPredicate = (belowPar, swap) -> {
                    boolean topicFirstObserverFairness = tView.compareFirstObserverFairness(abovePar.topicPartition(), abovePar.broker(), (Broker)belowPar).isGreater();
                    if (swap) {
                        return topicFirstObserverFairness;
                    }
                    boolean replicaTopicFairness = tView.compareReplicaFairness(abovePar.topicPartition(), abovePar.broker(), (Broker)belowPar).isGreaterOrEqual();
                    return topicFirstObserverFairness && replicaTopicFairness;
                };
                List<Broker> belowParBrokers = topicView.brokersWithBelowParFirstObserverFairness();
                if (!this.moveFirstObserverToBelowParBroker(context, topicView.constraints(), abovePar, belowParBrokers, fairnessPredicate)) continue;
                moveFound = true;
                topicView = (BrokerFairView)topicView.refresh(context);
            }
        }
    }

    private boolean moveFirstObserverToBelowParBroker(MutableRebalanceContext context, RebalanceConstraints constraints, Replica abovePar, Collection<Broker> belowParBrokers, BooleanPredicate<Broker> fairnessPredicate) {
        logger.debug("Moving {}, # belowPar candidates: {} ", (Object)abovePar, (Object)belowParBrokers.size());
        for (Broker belowPar : belowParBrokers) {
            boolean obeysPartition = constraints.obeysPartitionConstraint(abovePar.topicPartition(), belowPar);
            boolean obeysRack = constraints.obeysRackConstraint(abovePar.topicPartition(), abovePar.topicPlacement(), abovePar.broker(), belowPar);
            boolean obeysDiskSpace = constraints.obeysDiskSpaceConstraint(abovePar.topicPartition(), belowPar);
            boolean fairnessNoSwap = fairnessPredicate.test(belowPar, false);
            boolean fairnessSwap = fairnessPredicate.test(belowPar, true);
            PartitionAssignment partitionAssignment = context.assignment(abovePar.topicPartition());
            boolean belowParBrokerIsObserver = partitionAssignment.observers().contains(belowPar);
            if (obeysRack && obeysPartition && obeysDiskSpace && fairnessNoSwap) {
                this.moveFirstObserver(context, abovePar.topicPartition(), belowPar, false);
                return true;
            }
            if (obeysRack && belowParBrokerIsObserver && !obeysPartition && obeysDiskSpace && fairnessSwap) {
                this.moveFirstObserver(context, abovePar.topicPartition(), belowPar, true);
                return true;
            }
            logger.debug("Move to {} failed due to rack/below par broker is observer/partition/disk space/fairness constraints: {}, {}, {}, {} {}", new Object[]{belowPar, obeysRack, belowParBrokerIsObserver, obeysPartition, obeysDiskSpace, fairnessNoSwap || fairnessSwap});
        }
        logger.debug("First observer {} could not be moved despite attempting {} different brokers", (Object)abovePar, (Object)belowParBrokers.size());
        return false;
    }

    private void moveFirstObserver(MutableRebalanceContext context, TopicPartition partition, Broker newFirstObserver, boolean swap) {
        Optional<Broker> currentFirstObserver = context.firstObserver(partition);
        currentFirstObserver.ifPresent(firstObserver -> {
            if (firstObserver.equals(newFirstObserver)) {
                logger.debug("Movement was not made as both source and destination broker are the same: {}", (Object)firstObserver.id());
            } else {
                if (swap) {
                    context.swapObservers(partition, newFirstObserver);
                } else {
                    context.makeFirstObserver(partition, newFirstObserver);
                }
                ++this.firstObserversMoved;
                logger.debug("Moved partition {} ({} MB) from broker {} ({} MB) to broker {} ({} MB)", new Object[]{partition, TopologyUtils.formattedPartitionSize(context, partition), firstObserver.id(), TopologyUtils.formattedBrokerSize(context, firstObserver), newFirstObserver.id(), TopologyUtils.formattedBrokerSize(context, newFirstObserver)});
            }
        });
    }

    void replicaSizeFairness(MutableRebalanceContext context, ClusterView clusterView, final TopicViewSupplier<?> topicViewSupplier) {
        ClusterView view = this.replicaFairness(context, clusterView, topicViewSupplier);
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (final Replica abovePar : view.replicasOnAboveParBrokers()) {
                if (!view.replicasOnAboveParBrokers().contains(abovePar)) continue;
                final ClusterView cView = view;
                Predicate<Replica> fairnessPredicate = new Predicate<Replica>(){
                    private final ClusterView aboveParTopicView;
                    {
                        this.aboveParTopicView = topicViewSupplier.get(abovePar.topicPartition().topic());
                    }

                    @Override
                    public boolean test(Replica belowParReplica) {
                        Broker belowParBroker = belowParReplica.broker();
                        Object belowParTopicView = topicViewSupplier.get(belowParReplica.topicPartition().topic());
                        boolean clusterFairness = cView.compareReplicaFairness(abovePar.broker(), abovePar.topicPartition(), belowParBroker, belowParReplica.topicPartition()).isGreater();
                        boolean topicFairnessOut = this.aboveParTopicView.compareReplicaFairness(abovePar.broker(), abovePar.topicPartition(), belowParBroker, belowParReplica.topicPartition()).isGreaterOrEqual();
                        boolean topicFairnessBack = belowParTopicView.compareReplicaFairness(abovePar.broker(), abovePar.topicPartition(), belowParBroker, belowParReplica.topicPartition()).isGreaterOrEqual();
                        return clusterFairness && topicFairnessOut && topicFairnessBack;
                    }
                };
                ArrayList<Replica> belowParReplicas = new ArrayList<Replica>();
                for (Broker belowParBroker : view.brokersWithBelowParReplicaFairness()) {
                    for (TopicPartition belowParPartition : TopologyUtils.sortPartitionsBySize(context, context.replicas(belowParBroker))) {
                        Replica replica = new Replica(belowParPartition, context.topicPlacement(belowParPartition.topic()), belowParBroker);
                        belowParReplicas.add(replica);
                    }
                }
                if (!this.switchReplicaWithBelowParBrokerReplica(context, view.constraints(), abovePar, belowParReplicas, fairnessPredicate)) continue;
                moveFound = true;
                view = view.refresh(context);
                topicViewSupplier.clear();
            }
        }
    }

    private boolean switchReplicaWithBelowParBrokerReplica(MutableRebalanceContext context, RebalanceConstraints constraints, Replica abovePar, Collection<Replica> belowParReplicas, Predicate<Replica> fairnessPredicate) {
        logger.debug("Moving {}, # belowPar replica candidates: {} ", (Object)abovePar, (Object)belowParReplicas.size());
        for (Replica belowParReplica : belowParReplicas) {
            Broker belowParBroker = belowParReplica.broker();
            TopicPartition belowParPartition = belowParReplica.topicPartition();
            boolean obeysDiskSpaceOut = constraints.obeysDiskSpaceConstraint(abovePar.topicPartition(), abovePar.broker(), belowParReplica.topicPartition());
            boolean obeysDiskSpaceBack = constraints.obeysDiskSpaceConstraint(belowParPartition, belowParBroker, abovePar.topicPartition());
            boolean obeysPartitionOut = constraints.obeysPartitionConstraint(abovePar.topicPartition(), belowParBroker);
            boolean obeysPartitionBack = constraints.obeysPartitionConstraint(belowParPartition, abovePar.broker());
            boolean obeysRackOut = constraints.obeysRackConstraint(abovePar.topicPartition(), abovePar.topicPlacement(), abovePar.broker(), belowParBroker);
            boolean obeysRackBack = constraints.obeysRackConstraint(belowParPartition, belowParReplica.topicPlacement(), belowParBroker, abovePar.broker());
            boolean fairness = fairnessPredicate.test(belowParReplica);
            if (!obeysDiskSpaceOut || !obeysDiskSpaceBack || !obeysPartitionOut || !obeysPartitionBack || !fairness || !obeysRackOut || !obeysRackBack) continue;
            this.movePartition(context, abovePar.topicPartition(), abovePar.broker(), belowParBroker);
            this.movePartition(context, belowParReplica.topicPartition(), belowParBroker, abovePar.broker());
            return true;
        }
        logger.debug("Replica {} could not be moved despite attempting {} different brokers", (Object)abovePar, (Object)belowParReplicas.size());
        return false;
    }

    private ValidationException buildMoveException(Broker broker, TopicPartition tp, MoveResult moveResult) {
        StringBuilder constraintErrorMessage = new StringBuilder();
        Iterator<MoveAttempt> moveIterator = moveResult.getFailedMoves().iterator();
        while (moveIterator.hasNext()) {
            constraintErrorMessage.append(moveIterator.next().errorMessage());
            if (!moveIterator.hasNext()) continue;
            constraintErrorMessage.append(", ");
        }
        return new ValidationException(String.format("ERROR: Could not move partition %s from removed broker %s, due to the following constraint errors: %s.", tp, broker.id(), constraintErrorMessage));
    }

    private MoveAttempt failedConstraints(Constraints constraints, Replica replica, Broker destination) {
        ArrayList<ConstraintType> failedConstraints = new ArrayList<ConstraintType>();
        if (!constraints.obeysPartitionConstraint(replica.topicPartition(), destination)) {
            failedConstraints.add(ConstraintType.PARTITION);
        }
        if (!constraints.obeysRackConstraint(replica.topicPartition(), replica.topicPlacement(), replica.broker(), destination)) {
            failedConstraints.add(ConstraintType.RACK);
        }
        if (!constraints.obeysDiskSpaceConstraint(replica.topicPartition(), destination)) {
            failedConstraints.add(ConstraintType.DISK_SPACE);
        }
        return new MoveAttempt(replica, destination, failedConstraints);
    }

    private boolean hasTopicPlacementViolation(TopicPartition partition, TopicPlacement placement, MutableRebalanceContext context) {
        PartitionAssignment assignment = context.assignment(partition);
        List<Broker> syncReplicas = assignment.syncReplicas();
        boolean allOnlineSyncReplicasMatch = syncReplicas.stream().allMatch(broker -> context.isOffline((Broker)broker) || placement.matchesReplicas(context.brokerProperties((Broker)broker)));
        if (!allOnlineSyncReplicasMatch || syncReplicas.size() != Utils.getExpectedReplicaCount(placement) || this.isConstraintCountExceeded(context, placement.replicas(), syncReplicas)) {
            return true;
        }
        List<Broker> observers = assignment.observers;
        boolean allOnlineObserversMatch = observers.stream().allMatch(broker -> context.isOffline((Broker)broker) || placement.matchesObservers(context.brokerProperties((Broker)broker)));
        return !allOnlineObserversMatch || observers.size() != Utils.getExpectedObserverCount(placement) || this.isConstraintCountExceeded(context, placement.observers(), observers);
    }

    private boolean isConstraintCountExceeded(RebalanceContext context, List<TopicPlacement.ConstraintCount> constraintCounts, List<Broker> brokers) {
        for (TopicPlacement.ConstraintCount constraintCount : constraintCounts) {
            long numMatching = brokers.stream().filter(broker -> constraintCount.matches(context.brokerProperties((Broker)broker))).count();
            if (numMatching <= (long)constraintCount.count()) continue;
            return true;
        }
        return false;
    }

    @VisibleForTesting
    Set<TopicPartition> removeBrokers(MutableRebalanceContext context) {
        HashSet<TopicPartition> partitionsWithBrokersRemoved = new HashSet<TopicPartition>();
        Set<Broker> brokersToBeRemoved = context.brokersToBeRemoved();
        for (TopicPartition tp : context.allPartitions()) {
            Optional<TopicPlacement> topicPlacementOpt;
            int currentReplicaSize = context.brokers(tp).size();
            if (!brokersToBeRemoved.isEmpty()) {
                context.brokers(tp).stream().filter(brokersToBeRemoved::contains).forEach(broker -> context.removeReplica(tp, (Broker)broker));
            }
            if ((topicPlacementOpt = context.topicPlacement(tp.topic())).isPresent()) {
                if (this.hasTopicPlacementViolation(tp, topicPlacementOpt.get(), context) || context.brokers(tp).size() != currentReplicaSize) {
                    this.removeBrokersNotMatchingConstraints(context, tp, topicPlacementOpt.get());
                }
            } else {
                this.removeExtraBrokersOnRack(context, tp);
            }
            if (currentReplicaSize == context.brokers(tp).size()) continue;
            partitionsWithBrokersRemoved.add(tp);
        }
        return partitionsWithBrokersRemoved;
    }

    protected void removeBrokersNotMatchingConstraints(MutableRebalanceContext context, TopicPartition tp, TopicPlacement topicPlacement) {
        List<Broker> currentBrokers = context.brokers(tp);
        List<Broker> remainingBrokers = new ArrayList<Broker>(currentBrokers);
        for (Broker broker2 : currentBrokers) {
            context.removeReplica(tp, broker2);
        }
        List<Broker> newSyncReplicas = this.getBrokersMatchingConstraints(context, remainingBrokers, topicPlacement.replicas());
        for (Broker broker3 : newSyncReplicas) {
            context.addSyncReplica(tp, broker3);
        }
        remainingBrokers = remainingBrokers.stream().filter(replica -> !newSyncReplicas.contains(replica)).collect(Collectors.toList());
        List<Broker> list = this.getBrokersMatchingConstraints(context, remainingBrokers, topicPlacement.observers());
        for (Broker observer : list) {
            context.addObserver(tp, observer);
        }
        List list2 = remainingBrokers.stream().filter(context::isOffline).collect(Collectors.toList());
        if (newSyncReplicas.size() < Utils.getExpectedReplicaCount(topicPlacement) && !list2.isEmpty()) {
            throw new ValidationException("Cannot reassign partition " + tp + " with offline brokers " + list2 + " since there are insufficient replicas to satisfy synchronous replication  constraints " + topicPlacement.replicas() + " in current replica set " + currentBrokers);
        }
        PartitionAssignment assignment = context.assignment(tp);
        int expectedObserverCount = Utils.getExpectedObserverCount(topicPlacement);
        int currentObserverCount = assignment.observers.size();
        list2.stream().limit(expectedObserverCount - currentObserverCount).forEach(broker -> context.addObserver(tp, (Broker)broker));
    }

    private void removeExtraBrokersOnRack(MutableRebalanceContext context, TopicPartition tp) {
        Set<String> allRacks = context.allRetainedRacks();
        if (allRacks.isEmpty()) {
            return;
        }
        List<Broker> currentBrokers = context.brokers(tp);
        Set<String> currentBrokerRacks = context.racks(currentBrokers);
        int extraRacks = allRacks.size() - currentBrokerRacks.size();
        if (extraRacks > 0 && currentBrokerRacks.size() < currentBrokers.size()) {
            HashSet<String> racks = new HashSet<String>();
            for (Broker broker : currentBrokers) {
                String rack = context.brokerRack(broker);
                if (racks.contains(rack) && extraRacks > 0) {
                    --extraRacks;
                    context.removeReplica(tp, broker);
                    continue;
                }
                racks.add(rack);
            }
        }
    }

    @VisibleForTesting
    List<Broker> getBrokersMatchingConstraints(MutableRebalanceContext context, List<Broker> brokers, List<TopicPlacement.ConstraintCount> constraints) {
        ArrayList<Broker> brokersMatchingConstraint = new ArrayList<Broker>(brokers.size());
        brokers = new ArrayList<Broker>(brokers);
        while (!brokers.isEmpty()) {
            Broker broker = brokers.get(0);
            Map<String, String> brokerRackMap = context.brokerProperties(broker);
            Optional<TopicPlacement.ConstraintCount> brokersConstraint = constraints.stream().filter(constraint -> constraint.matches(brokerRackMap)).findFirst();
            if (brokersConstraint.isPresent()) {
                TopicPlacement.ConstraintCount constraint2 = brokersConstraint.get();
                Map<Boolean, List<Broker>> partitions = brokers.stream().collect(Collectors.partitioningBy(matchedBroker -> constraint2.matches(context.brokerProperties((Broker)matchedBroker))));
                brokersMatchingConstraint.addAll(partitions.get(Boolean.TRUE).stream().limit(constraint2.count()).collect(Collectors.toList()));
                brokers = partitions.get(Boolean.FALSE);
                continue;
            }
            brokers.remove(0);
        }
        return brokersMatchingConstraint;
    }

    void fullyReplicate(MutableRebalanceContext context, Set<TopicPartition> partitionsWithBrokersRemoved) {
        Constraints constraints = new Constraints(context, null);
        block0: for (TopicPartition tp : context.allPartitions()) {
            int replicationFactor = context.replicationFactor(tp.topic());
            Set<String> racks = TopologyUtils.racksFor(context, tp);
            Optional<TopicPlacement> topicPlacement = context.topicPlacement(tp.topic());
            while (replicationFactor > context.brokers(tp).size()) {
                if (this.addReplica(context, constraints, racks, tp, topicPlacement)) continue;
                if (partitionsWithBrokersRemoved.contains(tp)) {
                    throw new ValidationException("Unable to satisfy rack constraints for partition: " + tp);
                }
                logger.warn("Could not create replica due to either rack, partition or disk space constraints. Thus partition {} will remain under-replicated.", (Object)tp);
                continue block0;
            }
        }
    }

    @VisibleForTesting
    boolean addReplica(MutableRebalanceContext context, Constraints constraints, Set<String> racks, TopicPartition tp, Optional<TopicPlacement> topicPlacement) {
        List<Broker> leastLoadedBrokers = TopologyUtils.leastLoadedBrokersPreferringOtherRacks(context, racks);
        ReplicaCandidates candidates = this.filterByTopicPlacement(context, leastLoadedBrokers, tp, topicPlacement);
        for (Broker candidateBroker : candidates) {
            if (!constraints.obeysPartitionConstraint(tp, candidateBroker) || !constraints.obeysRackConstraint(tp, topicPlacement, null, candidateBroker) || !constraints.obeysDiskSpaceConstraint(tp, candidateBroker)) continue;
            if (candidates.shouldBeObserver) {
                context.addObserver(tp, candidateBroker);
            } else {
                context.addSyncReplica(tp, candidateBroker);
            }
            return true;
        }
        return false;
    }

    ReplicaCandidates filterByTopicPlacement(MutableRebalanceContext context, List<Broker> candidateBrokers, TopicPartition tp, Optional<TopicPlacement> topicPlacementOpt) {
        List<Broker> candidates;
        List<Broker> candidates2;
        List<Broker> currentBrokers = context.brokers(tp);
        if (!topicPlacementOpt.isPresent()) {
            return new ReplicaCandidates(candidateBrokers, false);
        }
        TopicPlacement topicPlacement = topicPlacementOpt.get();
        Map.Entry<List<Broker>, List<Broker>> partitionedBrokers = Utils.partitionBrokers(context, topicPlacementOpt, context.syncReplicas(tp), context.observers(tp));
        List<Broker> currentReplicas = partitionedBrokers.getKey();
        List<Broker> currentObservers = partitionedBrokers.getValue();
        List<Broker> offlineBrokers = currentBrokers.stream().filter(context::isOffline).collect(Collectors.toList());
        int replicaCount = Utils.getExpectedReplicaCount(topicPlacement);
        if (replicaCount > currentReplicas.size() && offlineBrokers.isEmpty() && !(candidates2 = this.getCandidateReplicas(context, topicPlacement.replicas(), currentReplicas, Collections.emptyList(), candidateBrokers)).isEmpty()) {
            return new ReplicaCandidates(candidates2, false);
        }
        int observerCount = Utils.getExpectedObserverCount(topicPlacement);
        if (observerCount > currentObservers.size() + offlineBrokers.size() && !(candidates = this.getCandidateReplicas(context, topicPlacement.observers(), currentObservers, offlineBrokers, candidateBrokers)).isEmpty()) {
            return new ReplicaCandidates(candidates, true);
        }
        throw new ValidationException("Unable to satisfy topic placement constraint. Current brokers: [" + Utils.mkString(currentBrokers, ",") + "] Offline brokers: [" + Utils.mkString(offlineBrokers, ",") + "] Candidate brokers: [" + Utils.mkString(candidateBrokers, ",") + "] Topic placement constraints: " + topicPlacementOpt);
    }

    private List<Broker> getCandidateReplicas(MutableRebalanceContext context, List<TopicPlacement.ConstraintCount> constraints, List<Broker> currentOnlineReplicas, List<Broker> currentOfflineReplicas, List<Broker> candidateReplicas) {
        for (TopicPlacement.ConstraintCount constraint : constraints) {
            long matchingBrokersCount = currentOnlineReplicas.stream().filter(broker -> constraint.matches(context.brokerProperties((Broker)broker))).count();
            if (matchingBrokersCount + (long)currentOfflineReplicas.size() >= (long)constraint.count()) continue;
            return candidateReplicas.stream().filter(broker -> constraint.matches(context.brokerProperties((Broker)broker)) && !currentOnlineReplicas.contains(broker)).collect(Collectors.toList());
        }
        return Collections.emptyList();
    }

    private ClusterView replicaFairness(final MutableRebalanceContext context, ClusterView clusterView, final TopicViewSupplier<?> topicViewSupplier) {
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (final Replica abovePar : clusterView.replicasOnAboveParBrokers()) {
                if (!clusterView.replicasOnAboveParBrokers().contains(abovePar)) continue;
                final ClusterView view = clusterView;
                Predicate<Broker> fairnessPredicate = new Predicate<Broker>(){
                    private final ClusterView topicView;
                    {
                        this.topicView = topicViewSupplier.get(abovePar.topicPartition().topic());
                    }

                    @Override
                    public boolean test(Broker belowPar) {
                        boolean clusterFairness = view.compareReplicaFairness(abovePar.topicPartition(), abovePar.broker(), belowPar).isGreater();
                        boolean topicFairness = this.topicView.compareReplicaFairness(abovePar.topicPartition(), abovePar.broker(), belowPar).isGreaterOrEqual();
                        Optional<TopicPlacement> topicPlacement = context.topicPlacement(abovePar.topicPartition().topic());
                        boolean matchesObserverPlacement = Utils.brokerMatchesObserverPlacement(context, belowPar, topicPlacement);
                        if (matchesObserverPlacement) {
                            boolean firstObserverFairnessTopicView = this.topicView.compareFirstObserverFairness(abovePar.topicPartition(), abovePar.broker(), belowPar).isGreaterOrEqual();
                            boolean firstObserverFairnessClusterView = view.compareFirstObserverFairness(abovePar.topicPartition(), abovePar.broker(), belowPar).isGreaterOrEqual();
                            return clusterFairness && topicFairness && firstObserverFairnessTopicView && firstObserverFairnessClusterView;
                        }
                        return clusterFairness && topicFairness;
                    }
                };
                List<Broker> belowParBrokers = clusterView.brokersWithBelowParReplicaFairness();
                if (!this.moveReplicaToBelowParBroker(context, clusterView.constraints(), abovePar, belowParBrokers, fairnessPredicate)) continue;
                moveFound = true;
                clusterView = clusterView.refresh(context);
                topicViewSupplier.clear();
            }
        }
        return clusterView;
    }

    private boolean moveReplicaToBelowParBroker(MutableRebalanceContext context, RebalanceConstraints constraints, Replica abovePar, Collection<Broker> belowParBrokers, Predicate<Broker> fairnessPredicate) {
        logger.debug("Moving {}, # belowPar candidates: {} ", (Object)abovePar, (Object)belowParBrokers.size());
        for (Broker belowPar : belowParBrokers) {
            boolean obeysPartition = constraints.obeysPartitionConstraint(abovePar.topicPartition(), belowPar);
            boolean obeysRack = constraints.obeysRackConstraint(abovePar.topicPartition(), abovePar.topicPlacement(), abovePar.broker(), belowPar);
            boolean obeysDiskSpace = constraints.obeysDiskSpaceConstraint(abovePar.topicPartition(), belowPar);
            boolean fairness = fairnessPredicate.test(belowPar);
            if (obeysRack && obeysPartition && obeysDiskSpace && fairness) {
                this.movePartition(context, abovePar.topicPartition(), abovePar.broker(), belowPar);
                return true;
            }
            logger.debug("Move to {} failed due to rack/partition/disk space/fairness constraints: {}, {}, {}, {}", new Object[]{belowPar, obeysRack, obeysPartition, obeysDiskSpace, fairness});
        }
        logger.debug("Replica {} could not be moved despite attempting {} different brokers", (Object)abovePar, (Object)belowParBrokers.size());
        return false;
    }

    private void movePartition(MutableRebalanceContext context, TopicPartition partition, Broker from, Broker to) {
        if (to.equals(from)) {
            logger.debug("Movement was not made as both source and destination broker are the same: {}", (Object)from.id());
        } else {
            context.movePartition(partition, from, to);
            ++this.replicasMoved;
            if (logger.isDebugEnabled()) {
                logger.debug("Moved partition {} ({} MB) from broker {} ({} MB) to broker {} ({} MB)", new Object[]{partition, TopologyUtils.formattedPartitionSize(context, partition), from.id(), TopologyUtils.formattedBrokerSize(context, from), to.id(), TopologyUtils.formattedBrokerSize(context, to)});
            }
        }
    }

    @VisibleForTesting
    void leaderTopicFairness(final MutableRebalanceContext context, ClusterView topicView) {
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (TopicPartition aboveParLeader : topicView.leadersOnAboveParBrokers()) {
                ClusterView tView;
                BiPredicate<Broker> fairnessPredicate;
                if (!topicView.leadersOnAboveParBrokers().contains(aboveParLeader) || !this.changeLeadership(context, aboveParLeader, fairnessPredicate = new BiPredicate<Broker>(tView = topicView, aboveParLeader){
                    final /* synthetic */ ClusterView val$tView;
                    final /* synthetic */ TopicPartition val$aboveParLeader;
                    {
                        this.val$tView = clusterView;
                        this.val$aboveParLeader = topicPartition;
                    }

                    @Override
                    public boolean test(Broker leader, Broker follower) {
                        return this.val$tView.compareLeaderFairness(this.val$aboveParLeader, leader, follower).isGreater();
                    }
                })) continue;
                moveFound = true;
                topicView = topicView.refresh(context);
            }
        }
        moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (final TopicPartition aboveParLeaderPartition : topicView.leadersOnAboveParBrokers()) {
                ClusterView tView;
                Predicate<Replica> fairnessPredicate;
                List<Broker> belowParBrokers;
                if (!topicView.leadersOnAboveParBrokers().contains(aboveParLeaderPartition) || !this.switchLeaderToOtherBroker(context, aboveParLeaderPartition, belowParBrokers = topicView.brokersWithBelowParLeaderFairness(), fairnessPredicate = new Predicate<Replica>(tView = topicView){
                    private final Broker aboveParLeaderBroker;
                    final /* synthetic */ ClusterView val$tView;
                    {
                        this.val$tView = clusterView;
                        this.aboveParLeaderBroker = context.brokers(aboveParLeaderPartition).get(0);
                    }

                    @Override
                    public boolean test(Replica belowParReplica) {
                        Broker belowParBroker = belowParReplica.broker();
                        TopicPartition belowParFollower = belowParReplica.topicPartition();
                        boolean obeysDiskSpaceOut = this.val$tView.constraints().obeysDiskSpaceConstraint(aboveParLeaderPartition, this.aboveParLeaderBroker, belowParFollower);
                        boolean obeysDiskSpaceBack = this.val$tView.constraints().obeysDiskSpaceConstraint(belowParFollower, belowParBroker, aboveParLeaderPartition);
                        boolean obeysPartitionOut = this.val$tView.constraints().obeysPartitionConstraint(aboveParLeaderPartition, belowParBroker);
                        boolean obeysPartitionBack = this.val$tView.constraints().obeysPartitionConstraint(belowParFollower, this.aboveParLeaderBroker);
                        boolean fairness = this.val$tView.compareLeaderFairness(aboveParLeaderPartition, this.aboveParLeaderBroker, belowParBroker).isGreater();
                        Optional<TopicPlacement> topicPlacement = context.topicPlacement(aboveParLeaderPartition.topic());
                        boolean obeysRackOut = this.val$tView.constraints().obeysRackConstraint(aboveParLeaderPartition, topicPlacement, this.aboveParLeaderBroker, belowParBroker);
                        boolean obeysRackIn = this.val$tView.constraints().obeysRackConstraint(belowParReplica.topicPartition(), belowParReplica.topicPlacement(), belowParBroker, this.aboveParLeaderBroker);
                        return obeysDiskSpaceOut && obeysDiskSpaceBack && obeysPartitionOut && obeysPartitionBack && fairness && obeysRackIn && obeysRackOut;
                    }
                })) continue;
                moveFound = true;
                topicView = topicView.refresh(context);
            }
        }
    }

    @VisibleForTesting
    void leaderFairness(final MutableRebalanceContext context, ClusterView clusterView, TopicViewSupplier<?> topicViewSupplier) {
        boolean moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (TopicPartition aboveParLeader : clusterView.leadersOnAboveParBrokers()) {
                Object tView;
                ClusterView cView;
                BiPredicate<Broker> fairnessPredicate;
                if (!clusterView.leadersOnAboveParBrokers().contains(aboveParLeader) || !this.changeLeadership(context, aboveParLeader, fairnessPredicate = new BiPredicate<Broker>(cView = clusterView, aboveParLeader, tView = topicViewSupplier.get(aboveParLeader.topic())){
                    final /* synthetic */ ClusterView val$cView;
                    final /* synthetic */ TopicPartition val$aboveParLeader;
                    final /* synthetic */ ClusterView val$tView;
                    {
                        this.val$cView = clusterView;
                        this.val$aboveParLeader = topicPartition;
                        this.val$tView = clusterView2;
                    }

                    @Override
                    public boolean test(Broker leader, Broker follower) {
                        return this.val$cView.compareLeaderFairness(this.val$aboveParLeader, leader, follower).isGreater() && this.val$tView.compareLeaderFairness(this.val$aboveParLeader, leader, follower).isGreaterOrEqual();
                    }
                })) continue;
                moveFound = true;
                clusterView = clusterView.refresh(context);
                topicViewSupplier.clear();
            }
        }
        moveFound = true;
        while (moveFound) {
            moveFound = false;
            for (final TopicPartition aboveParLeaderPartition : clusterView.leadersOnAboveParBrokers()) {
                Object topicView;
                ClusterView cView;
                Predicate<Replica> fairnessPredicate;
                List<Broker> belowParBrokers;
                if (!clusterView.leadersOnAboveParBrokers().contains(aboveParLeaderPartition) || !this.switchLeaderToOtherBroker(context, aboveParLeaderPartition, belowParBrokers = clusterView.brokersWithBelowParLeaderFairness(), fairnessPredicate = new Predicate<Replica>(cView = clusterView, topicView = topicViewSupplier.get(aboveParLeaderPartition.topic())){
                    private final Broker aboveParLeaderBroker;
                    final /* synthetic */ ClusterView val$cView;
                    final /* synthetic */ ClusterView val$topicView;
                    {
                        this.val$cView = clusterView;
                        this.val$topicView = clusterView2;
                        this.aboveParLeaderBroker = context.brokers(aboveParLeaderPartition).get(0);
                    }

                    @Override
                    public boolean test(Replica belowParReplica) {
                        Broker belowParBroker = belowParReplica.broker();
                        TopicPartition belowParFollower = belowParReplica.topicPartition();
                        boolean obeysDiskSpaceOut = this.val$cView.constraints().obeysDiskSpaceConstraint(aboveParLeaderPartition, this.aboveParLeaderBroker, belowParFollower);
                        boolean obeysDiskSpaceBack = this.val$cView.constraints().obeysDiskSpaceConstraint(belowParFollower, belowParBroker, aboveParLeaderPartition);
                        boolean obeysPartitionOut = this.val$cView.constraints().obeysPartitionConstraint(aboveParLeaderPartition, belowParBroker);
                        boolean obeysPartitionBack = this.val$cView.constraints().obeysPartitionConstraint(belowParFollower, this.aboveParLeaderBroker);
                        boolean clusterFairness = this.val$cView.compareLeaderFairness(aboveParLeaderPartition, this.aboveParLeaderBroker, belowParBroker).isGreater();
                        boolean topicFairness = this.val$topicView.compareLeaderFairness(aboveParLeaderPartition, this.aboveParLeaderBroker, belowParBroker).isGreaterOrEqual();
                        Optional<TopicPlacement> topicPlacement = context.topicPlacement(aboveParLeaderPartition.topic());
                        boolean obeysRackOut = this.val$topicView.constraints().obeysRackConstraint(aboveParLeaderPartition, topicPlacement, this.aboveParLeaderBroker, belowParBroker);
                        boolean obeysRackIn = this.val$topicView.constraints().obeysRackConstraint(belowParReplica.topicPartition(), belowParReplica.topicPlacement(), belowParBroker, this.aboveParLeaderBroker);
                        return obeysDiskSpaceOut && obeysDiskSpaceBack && obeysPartitionOut && obeysPartitionBack && clusterFairness && topicFairness && obeysRackIn && obeysRackOut;
                    }
                })) continue;
                moveFound = true;
                clusterView = clusterView.refresh(context);
                topicViewSupplier.clear();
            }
        }
    }

    boolean changeLeadership(MutableRebalanceContext context, TopicPartition aboveParLeader, BiPredicate<Broker> fairnessPredicate) {
        List<Broker> replicas = context.syncReplicas(aboveParLeader);
        Broker leader = replicas.get(0);
        for (Broker follower : replicas.subList(1, replicas.size())) {
            if (!fairnessPredicate.test(leader, follower)) continue;
            this.makeLeader(context, aboveParLeader, follower);
            return true;
        }
        return false;
    }

    private boolean switchLeaderToOtherBroker(final MutableRebalanceContext context, TopicPartition aboveParLeader, Collection<Broker> belowParBrokers, Predicate<Replica> fairnessPredicate) {
        Broker aboveParLeaderBroker = context.brokers(aboveParLeader).get(0);
        ArrayList<Replica> belowParFollowers = new ArrayList<Replica>();
        for (Broker belowParBroker : belowParBrokers) {
            for (TopicPartition belowParFollower : context.syncFollowers(belowParBroker)) {
                Optional<TopicPlacement> topicPlacement = context.topicPlacement(belowParFollower.topic());
                Replica replica = new Replica(belowParFollower, topicPlacement, belowParBroker);
                if (!Utils.brokersMatchSameConstraint(context, aboveParLeaderBroker, belowParBroker, topicPlacement)) continue;
                belowParFollowers.add(replica);
            }
        }
        final long leaderSize = context.partitionSize(aboveParLeader);
        Collections.sort(belowParFollowers, new Comparator<Replica>(){

            @Override
            public int compare(Replica r1, Replica r2) {
                long s1 = context.partitionSize(r1.topicPartition());
                long s2 = context.partitionSize(r2.topicPartition());
                int result = Long.compare(Math.abs(leaderSize - s1), Math.abs(leaderSize - s2));
                if (result != 0) {
                    return result;
                }
                if (s1 != s2) {
                    return Long.compare(s1, s2);
                }
                if (r1.broker().id() != r2.broker().id()) {
                    return Integer.compare(r1.broker().id(), r2.broker().id());
                }
                return TopologyUtils.topicPartitionComparator.compare(r1.topicPartition(), r2.topicPartition());
            }
        });
        for (Replica replica : belowParFollowers) {
            if (!fairnessPredicate.test(replica)) continue;
            Broker belowParBroker = replica.broker();
            this.movePartition(context, aboveParLeader, aboveParLeaderBroker, belowParBroker);
            this.movePartition(context, replica.topicPartition(), belowParBroker, aboveParLeaderBroker);
            return true;
        }
        return false;
    }

    void makeLeader(MutableRebalanceContext context, TopicPartition topicPartition, Broker newLeader) {
        Broker currentLeader = context.leader(topicPartition);
        if (!newLeader.equals(currentLeader)) {
            context.makeLeader(topicPartition, newLeader);
            logger.debug("Leadership moved brokers: [{} -> {}] for partition {}:{}", new Object[]{currentLeader, newLeader, topicPartition, context.brokers(topicPartition)});
            ++this.leadersMoved;
        } else {
            System.out.println("Leadership change was not made as " + newLeader + " was already the leader for partition " + topicPartition + " - see: " + context.brokers(topicPartition));
        }
    }

    private void ensureOfflineBrokerAreRemoved(RebalanceContext context) {
        if (!context.brokersToBeRemoved().containsAll(context.offlineBrokers())) {
            throw new ValidationException("There are offline brokers " + context.offlineBrokers() + " which are not among the brokers to be removed " + context.brokersToBeRemoved());
        }
    }

    public int replicasMoved() {
        return this.replicasMoved;
    }

    public int leadersMoved() {
        return this.leadersMoved;
    }

    public int firstObserversMoved() {
        return this.firstObserversMoved;
    }

    private void log(RebalanceContext context) {
        logger.debug("Racks to replica Counts {}", TopologyUtils.rackReplicaCounts(context, null, null));
        logger.debug("Racks to leader Counts {}", TopologyUtils.rackLeaderCounts(context, null, null));
        logger.debug("Broker to replica Counts {}", TopologyUtils.brokerReplicaCounts(context, context.allBrokers(), null));
        logger.debug("Broker to leader Counts {}", TopologyUtils.brokerLeaderCounts(context, context.allBrokers(), null));
        logger.debug("Broker to first observer Counts {}", TopologyUtils.brokerFirstObserverCounts(context, context.allBrokers(), null));
        logger.debug("Number of replicas moves {}", (Object)this.replicasMoved);
        logger.debug("Number of first observer moves {}", (Object)this.firstObserversMoved);
        logger.debug("Number of leader moves {}", (Object)this.leadersMoved);
    }

    public static class ReplicaCandidates
    implements Iterable<Broker> {
        final List<Broker> brokers;
        final boolean shouldBeObserver;

        private ReplicaCandidates(List<Broker> brokers, boolean shouldBeObserver) {
            this.brokers = brokers;
            this.shouldBeObserver = shouldBeObserver;
        }

        @Override
        public Iterator<Broker> iterator() {
            return this.brokers.iterator();
        }
    }

    static interface BiPredicate<T> {
        public boolean test(T var1, T var2);
    }

    static interface BooleanPredicate<T> {
        public boolean test(T var1, boolean var2);
    }

    static interface Predicate<T> {
        public boolean test(T var1);
    }

    private static class MoveResult {
        public MoveAttempt successfulMove;
        public List<MoveAttempt> failedMoves;

        public boolean success() {
            return this.successfulMove != null;
        }

        public List<MoveAttempt> getFailedMoves() {
            return this.failedMoves;
        }

        private MoveResult(List<MoveAttempt> failedMoves) {
            this.failedMoves = failedMoves;
        }

        private MoveResult(MoveAttempt successfulMove, List<MoveAttempt> failedMoves) {
            this.successfulMove = successfulMove;
            this.failedMoves = failedMoves;
        }
    }

    private static class MoveAttempt {
        final Replica replica;
        final Broker destination;
        final List<ConstraintType> failedConstraints;

        private MoveAttempt(Replica replica, Broker destination, List<ConstraintType> failedConstraints) {
            this.replica = replica;
            this.destination = destination;
            this.failedConstraints = failedConstraints;
        }

        boolean allowMove() {
            return this.failedConstraints.isEmpty();
        }

        boolean allowMoveWithRelaxedRackConstraint() {
            return this.failedConstraints.size() == 1 && this.failedConstraints.get(0) == ConstraintType.RACK;
        }

        String errorMessage() {
            StringBuilder constraintsMsg = new StringBuilder();
            Iterator<ConstraintType> iterator = this.failedConstraints.iterator();
            while (iterator.hasNext()) {
                constraintsMsg.append((Object)iterator.next());
                if (!iterator.hasNext()) continue;
                constraintsMsg.append(", ");
            }
            return String.format("[%s -> %s] failed constraints: [%s]", this.replica, this.destination.id(), constraintsMsg);
        }
    }

    private static enum ConstraintType {
        RACK,
        DISK_SPACE,
        PARTITION;

    }
}

