/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.SdkClientException;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.PropertiesFileCredentialsProvider;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsV2Request;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.S3VersionSummary;
import com.amazonaws.services.s3.model.SSEAwsKeyManagementParams;
import com.amazonaws.services.s3.model.VersionListing;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClient;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder;
import io.confluent.kafka.storage.checksum.E2EChecksumProtectedFileType;
import io.confluent.kafka.storage.checksum.E2EChecksumStore;
import io.confluent.kafka.storage.utils.E2EChecksumUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import kafka.server.KafkaConfig;
import kafka.tier.exceptions.E2EChecksumInvalidException;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.AutoAbortingGenericInputStream;
import kafka.tier.store.BucketHealthResult;
import kafka.tier.store.S3AutoAbortingInputStream;
import kafka.tier.store.S3TierObjectStoreConfig;
import kafka.tier.store.S3VersionInformation;
import kafka.tier.store.ThrottledFileInputStream;
import kafka.tier.store.TierObjectAttribute;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.TierObjectStoreUtils;
import kafka.tier.store.VersionInformation;
import kafka.utils.CoreUtils;
import kafka.utils.Throttler;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.kafka.common.utils.ByteBufferInputStream;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3TierObjectStore
implements TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(S3TierObjectStore.class);
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final AmazonS3 client;
    private final String bucket;
    private final String prefix;
    private final String sseAlgorithm;
    private final String sseCustomerEncryptionKey;
    private final int autoAbortThresholdBytes;
    private final Optional<E2EChecksumStore> checksumStoreOpt;
    private static final int DEFAULT_S3_DELETE_BATCH_SIZE = 500;
    private static final String CRC32C = "CRC32C";
    private static final String CRC32C_HEADER = "x-amz-checksum-crc32c";
    private static final String ERROR_CODE_BAD_DIGEST = "BadDigest";

    public S3TierObjectStore(S3TierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this(S3TierObjectStore.client(config), config, checksumStoreOpt);
    }

    S3TierObjectStore(AmazonS3 client, S3TierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.client = client;
        this.bucket = config.s3Bucket;
        this.prefix = config.s3Prefix;
        this.sseAlgorithm = config.s3SseAlgorithm;
        this.sseCustomerEncryptionKey = config.s3SseCustomerEncryptionKey;
        this.autoAbortThresholdBytes = config.s3AutoAbortThresholdBytes;
        this.expectBucket(this.bucket, config.s3Region, config.s3EndpointOverride);
        this.checksumStoreOpt = checksumStoreOpt;
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.S3;
    }

    private Set<String> listObjectsWithoutVersions(String keyPrefix) {
        HashSet<String> objects = new HashSet<String>();
        try {
            ListObjectsV2Result result;
            ListObjectsV2Request req = new ListObjectsV2Request().withBucketName(this.bucket).withPrefix(keyPrefix).withMaxKeys(Integer.valueOf(1000));
            do {
                result = this.client.listObjectsV2(req);
                for (S3ObjectSummary objectSummary : result.getObjectSummaries()) {
                    objects.add(objectSummary.getKey());
                }
                String token = result.getNextContinuationToken();
                req.setContinuationToken(token);
            } while (result.isTruncated());
            log.debug("TierObjectStore listObjects versions: false keyPrefix:  " + keyPrefix + " # of objects returned: " + objects.size());
        }
        catch (SdkClientException e) {
            throw new TierObjectStoreRetriableException("Failed to list the objects (without versions) with prefix: " + keyPrefix + " " + (Object)((Object)e));
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception while listing objects (without versions) with prefix: " + keyPrefix + " " + e);
        }
        return objects;
    }

    @Override
    public Map<String, List<VersionInformation>> listObject(String keyPrefix, boolean getVersionInfo) {
        HashMap<String, List<VersionInformation>> results = new HashMap<String, List<VersionInformation>>();
        if (!getVersionInfo) {
            for (String object : this.listObjectsWithoutVersions(keyPrefix)) {
                results.put(object, new ArrayList());
            }
            return results;
        }
        try {
            ListVersionsRequest request = new ListVersionsRequest().withBucketName(this.bucket).withPrefix(keyPrefix).withMaxResults(Integer.valueOf(1000));
            VersionListing versionListings = this.client.listVersions(request);
            while (true) {
                for (S3VersionSummary versionSummary : versionListings.getVersionSummaries()) {
                    results.putIfAbsent(versionSummary.getKey(), new ArrayList());
                    S3VersionInformation versionInfo = new S3VersionInformation(versionSummary.getVersionId(), versionSummary.isDeleteMarker(), versionSummary.isLatest());
                    ((List)results.get(versionSummary.getKey())).add(versionInfo);
                }
                if (!versionListings.isTruncated()) break;
                versionListings = this.client.listNextBatchOfVersions(versionListings);
            }
            if (log.isDebugEnabled()) {
                StringBuilder allBlobs = new StringBuilder();
                results.forEach((key, versions) -> allBlobs.append("[").append((String)key).append("->").append(Arrays.toString(versions.toArray())).append("] "));
                log.debug("TierObjectStore listObjects versions: true keyPrefix: " + keyPrefix + " " + allBlobs);
            }
        }
        catch (SdkClientException e) {
            throw new TierObjectStoreRetriableException("Failed to list the objects with prefix: " + keyPrefix + " " + (Object)((Object)e));
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception while listing objects with prefix: " + keyPrefix + " " + e);
        }
        return results;
    }

    @Override
    public TierObjectStoreResponse getObject(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType, Integer byteOffsetStart, Integer byteOffsetEnd, VersionInformation versionInformation) {
        S3Object object;
        GetObjectRequest request;
        String key = this.keyPath(objectMetadata, fileType);
        String fullPath = "s3://" + this.bucket + "/" + key;
        GetObjectRequest getObjectRequest = request = versionInformation != null ? new GetObjectRequest(this.bucket, key, versionInformation.getVersionId()) : new GetObjectRequest(this.bucket, key);
        if (byteOffsetStart != null && byteOffsetEnd != null) {
            request.setRange((long)byteOffsetStart.intValue(), (long)byteOffsetEnd.intValue());
        } else if (byteOffsetStart != null && byteOffsetStart != 0) {
            request.setRange((long)byteOffsetStart.intValue());
        } else if (byteOffsetEnd != null) {
            throw new IllegalStateException("Cannot specify a byteOffsetEnd without specifying a byteOffsetStart");
        }
        log.debug("Fetching object from {}, with range {} - {}", new Object[]{fullPath, byteOffsetStart, byteOffsetEnd});
        try {
            object = this.client.getObject(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to fetch object from %s, metadata: %s type: %s range %s-%s", new Object[]{fullPath, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when fetching object from %s, metadata: %s type: %s range %s-%s", new Object[]{fullPath, objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        S3ObjectInputStream inputStream = object.getObjectContent();
        return new S3TierObjectStoreResponse(inputStream, this.autoAbortThresholdBytes, object.getObjectMetadata().getContentLength());
    }

    @Override
    public ByteBuffer getSnapshot(TierObjectStore.ObjectStoreMetadata metadata, TierObjectStore.FileType fileType, int estimatedBufferSize) {
        ByteBuffer buffer;
        if (fileType != TierObjectStore.FileType.TIER_PARTITION_STATE_METADATA_SNAPSHOT && fileType != TierObjectStore.FileType.TIER_STATE_SNAPSHOT && fileType != TierObjectStore.FileType.PRODUCER_STATE) {
            throw new IllegalArgumentException("getSnapshot does not support the given fileType: " + (Object)((Object)fileType));
        }
        try (TierObjectStoreResponse response = this.getObject(metadata, fileType);){
            buffer = ByteBuffer.wrap(Utils.readFullyToArray((InputStream)response.getInputStream(), (int)estimatedBufferSize));
        }
        catch (Exception e) {
            for (Throwable cause = e.getCause(); cause != null && cause != cause.getCause(); cause = cause.getCause()) {
                if (!(cause instanceof AmazonS3Exception) || !((AmazonS3Exception)cause).getErrorCode().equals("NoSuchKey")) continue;
                throw new TierObjectStoreFatalException("Snapshot object not found in object store.", cause);
            }
            throw new TierObjectStoreRetriableException("Encountered an exception when fetching snapshot from object store.", e);
        }
        return buffer;
    }

    @Override
    public TierObjectStore.OpaqueData prepPutSegment() throws TierObjectStoreRetriableException, IOException {
        return TierObjectStore.OpaqueData.ZEROED;
    }

    @Override
    public void putObject(TierObjectStore.ObjectStoreMetadata objectMetadata, File file, TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        String key = this.keyPath(objectMetadata, fileType);
        String fullPath = "s3://" + this.bucket + "/" + key;
        try {
            this.putFile(key, metadata, file, fileType.toE2EChecksumProtectedFileType());
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object to %s with metadata %s, file %s, type %s", new Object[]{fullPath, objectMetadata, file, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object to %s with metadata %s, file %s, type %s", new Object[]{fullPath, objectMetadata, file, fileType}), e);
        }
    }

    @Override
    public void putBuffer(TierObjectStore.ObjectStoreMetadata objectMetadata, ByteBuffer buffer, TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        String key = this.keyPath(objectMetadata, fileType);
        String fullPath = "s3://" + this.bucket + "/" + key;
        try {
            this.putBuf(key, metadata, buffer);
        }
        catch (AmazonClientException ex) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object to %s with metadata %s, buffer %s, type %s", new Object[]{fullPath, objectMetadata, buffer, fileType}), ex);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object to %s with metadata %s, buffer %s, type %s", new Object[]{fullPath, objectMetadata, buffer, fileType}), e);
        }
    }

    @Override
    public void restoreObjectByCopy(TierObjectStore.ObjectMetadata objectMetadata, String key, VersionInformation lastLiveVersion) {
        String lastLiveVersionId = lastLiveVersion.getVersionId();
        String fullPath = "s3://" + this.bucket + "/" + key;
        try {
            CopyObjectRequest request = new CopyObjectRequest(this.bucket, key, lastLiveVersionId, this.bucket, key).withNewObjectMetadata(this.objectMetadata(null));
            this.setKmsParams(request);
            log.debug("restore object for {}-{}", (Object)fullPath, (Object)lastLiveVersionId);
            this.client.copyObject(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to restore object %s (version: %s)", fullPath, lastLiveVersionId), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when restoring object %s (version: %s)", fullPath, lastLiveVersionId), e);
        }
    }

    private void setKmsParams(CopyObjectRequest request) {
        if (this.usesKms()) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
    }

    @Override
    public void putSegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<File> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState, Optional<Throttler> throttlerOpt) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            if (throttlerOpt.isPresent()) {
                this.putFileWithThrottling(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData, E2EChecksumProtectedFileType.SEGMENT, throttlerOpt.get());
                this.putFileWithThrottling(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData, E2EChecksumProtectedFileType.OFFSET_INDEX, throttlerOpt.get());
                this.putFileWithThrottling(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData, E2EChecksumProtectedFileType.TIMESTAMP_INDEX, throttlerOpt.get());
                if (producerStateSnapshotData.isPresent()) {
                    this.putFileWithThrottling(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, producerStateSnapshotData.get(), E2EChecksumProtectedFileType.PRODUCER_STATE, throttlerOpt.get());
                }
            } else {
                this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData, E2EChecksumProtectedFileType.SEGMENT);
                this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData, E2EChecksumProtectedFileType.OFFSET_INDEX);
                this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData, E2EChecksumProtectedFileType.TIMESTAMP_INDEX);
                producerStateSnapshotData.ifPresent(file -> this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, (File)file, E2EChecksumProtectedFileType.PRODUCER_STATE));
            }
            transactionIndexData.ifPresent(abortedTxnsBuf -> this.putBufMaybeEncrypt(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, (ByteBuffer)abortedTxnsBuf));
            epochState.ifPresent(buf -> this.putBufMaybeEncrypt(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, (ByteBuffer)buf));
        }
        catch (AmazonClientException e) {
            AmazonS3Exception ex;
            if (e instanceof AmazonS3Exception && (ex = (AmazonS3Exception)e).getErrorCode().equals(ERROR_CODE_BAD_DIGEST)) {
                throw new E2EChecksumInvalidException(objectMetadata, (Throwable)ex);
            }
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
        finally {
            this.checksumStoreOpt.ifPresent(checksumStore -> TierObjectStoreUtils.postPutSegmentCleanup(checksumStore, segmentData, offsetIndexData, timestampIndexData));
        }
    }

    @Override
    public void putInMemorySegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<ByteBuffer> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData, E2EChecksumProtectedFileType.SEGMENT);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData, E2EChecksumProtectedFileType.OFFSET_INDEX);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData, E2EChecksumProtectedFileType.TIMESTAMP_INDEX);
            producerStateSnapshotData.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, (ByteBuffer)buf));
            transactionIndexData.ifPresent(abortedTxnsBuf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, (ByteBuffer)abortedTxnsBuf));
            epochState.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, (ByteBuffer)buf));
        }
        catch (AmazonClientException e) {
            AmazonS3Exception ex;
            if (e instanceof AmazonS3Exception && (ex = (AmazonS3Exception)e).getErrorCode().equals(ERROR_CODE_BAD_DIGEST)) {
                throw new E2EChecksumInvalidException(objectMetadata, (Throwable)ex);
            }
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
        finally {
            this.checksumStoreOpt.ifPresent(checksumStore -> TierObjectStoreUtils.postPutSegmentCleanup(checksumStore, segmentData, offsetIndexData, timestampIndexData));
        }
    }

    @Override
    public void deleteSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        List<DeleteObjectsRequest.KeyVersion> keys = this.keysForSegment(objectMetadata);
        DeleteObjectsRequest request = new DeleteObjectsRequest(this.bucket).withKeys(keys);
        try {
            this.client.deleteObjects(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to delete segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting segment: " + objectMetadata, e);
        }
    }

    @Override
    public TierObjectAttribute objectExists(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType) throws TierObjectStoreRetriableException {
        TierObjectAttribute result = new TierObjectAttribute(false);
        try {
            String key = this.keyPath(objectMetadata, fileType);
            ObjectMetadata metadata = this.client.getObjectMetadata(this.bucket, key);
            log.debug("objectExists at s3://{}/{} with metadata {}", new Object[]{this.bucket, key, metadata});
            result.exist = true;
            result.size = metadata.getContentLength();
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                result.exist = false;
            }
            throw new TierObjectStoreRetriableException("Failed to check object existence: " + objectMetadata + " type: " + (Object)((Object)fileType), e);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to check object existence: " + objectMetadata + " type: " + (Object)((Object)fileType), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when checking object existence: " + objectMetadata + " type: " + (Object)((Object)fileType), e);
        }
        return result;
    }

    @Override
    public void deleteVersions(List<TierObjectStore.KeyAndVersion> keys) {
        ArrayList<DeleteObjectsRequest.KeyVersion> s3Keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        for (TierObjectStore.KeyAndVersion key : keys) {
            DeleteObjectsRequest.KeyVersion keyVersion = key.versionId() == null ? new DeleteObjectsRequest.KeyVersion(key.key()) : new DeleteObjectsRequest.KeyVersion(key.key(), key.versionId());
            log.debug("Deleting object {} {}", (Object)keyVersion.getKey(), (Object)keyVersion.getVersion());
            s3Keys.add(keyVersion);
            if (s3Keys.size() < 500) continue;
            this.makeDeleteObjectsCall(s3Keys);
            s3Keys.clear();
        }
        if (!s3Keys.isEmpty()) {
            this.makeDeleteObjectsCall(s3Keys);
        }
    }

    private void makeDeleteObjectsCall(List<DeleteObjectsRequest.KeyVersion> s3Keys) {
        DeleteObjectsRequest request = new DeleteObjectsRequest(this.bucket).withKeys(s3Keys);
        try {
            log.debug("Sending a batch delete request");
            this.client.deleteObjects(request);
        }
        catch (MultiObjectDeleteException e) {
            log.error("S3 reported errors while deleting the following versioned objects:");
            e.getErrors().forEach(err -> log.error("Blob Key: " + err.getKey() + " Blob VersionId: " + err.getVersionId() + " Error Code: " + err.getCode() + " Error Message: " + err.getMessage()));
            throw new TierObjectStoreRetriableException("Failed to delete " + e.getErrors().size() + " versioned objects", e);
        }
        catch (SdkClientException e) {
            log.error(Arrays.toString(e.getStackTrace()));
            throw new TierObjectStoreRetriableException("Failed to delete versioned objects", e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting versioned objects", e);
        }
    }

    @Override
    public void close() {
        this.client.shutdown();
    }

    private String keyPath(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType) {
        return objectMetadata.toPath(this.prefix, fileType);
    }

    private ObjectMetadata objectMetadata(Map<String, String> userMetadata) {
        ObjectMetadata metadata = new ObjectMetadata();
        if (this.sseAlgorithm != null) {
            metadata.setSSEAlgorithm(this.sseAlgorithm);
        }
        if (userMetadata != null) {
            metadata.setUserMetadata(userMetadata);
        }
        return metadata;
    }

    private void setKmsParams(PutObjectRequest request) {
        if (this.usesKms()) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
    }

    private void putFile(String key, Map<String, String> metadata, File file, E2EChecksumProtectedFileType fileType) {
        ObjectMetadata s3ObjMetadata = this.objectMetadata(metadata);
        Optional<String> crc = this.getCrcAndSetHeader(fileType, file, metadata, s3ObjMetadata);
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, file).withMetadata(s3ObjMetadata);
        this.setKmsParams(request);
        if (crc.isPresent()) {
            log.debug("Uploading object to s3://{}/{} with crc {}", new Object[]{this.bucket, key, crc.get()});
        } else {
            log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        }
        this.client.putObject(request);
    }

    private void putFileWithThrottling(String key, Map<String, String> metadata, File file, E2EChecksumProtectedFileType fileType, Throttler throttler) throws IOException {
        ObjectMetadata s3Metadata = this.objectMetadata(metadata);
        Optional<String> crc = this.getCrcAndSetHeader(fileType, file, metadata, s3Metadata);
        try (ThrottledFileInputStream fileInputStream = new ThrottledFileInputStream(file, throttler);){
            s3Metadata.setContentLength(file.length());
            PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)fileInputStream, s3Metadata);
            this.setKmsParams(request);
            if (crc.isPresent()) {
                log.debug("Uploading object to s3 with throttling://{}/{} with crc {}", new Object[]{this.bucket, key, crc.get()});
            } else {
                log.debug("Uploading object to s3 with throttling://{}/{}", (Object)this.bucket, (Object)key);
            }
            this.client.putObject(request);
        }
    }

    private Optional<String> getCrcAndSetHeader(E2EChecksumProtectedFileType fileType, File file, Map<String, String> metadata, ObjectMetadata s3ObjMetadata) {
        if (!this.checksumStoreOpt.isPresent()) {
            return Optional.empty();
        }
        E2EChecksumStore checksumStore = this.checksumStoreOpt.get();
        Optional crc = Optional.empty();
        if (checksumStore.checksumProtectionEnabled(fileType)) {
            crc = E2EChecksumUtils.getBase64CrcFromStore((E2EChecksumStore)checksumStore, (File)file, metadata);
            crc.ifPresent(s -> s3ObjMetadata.setHeader(CRC32C_HEADER, s));
        }
        return crc;
    }

    public void putBuf(String key, Map<String, String> userMetadata, ByteBuffer buf) {
        ObjectMetadata s3Metadata = this.objectMetadata(userMetadata);
        s3Metadata.setContentLength((long)(buf.limit() - buf.position()));
        s3Metadata.setContentMD5(CoreUtils.toBase64(CoreUtils.md5hash(buf)));
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)new ByteBufferInputStream(buf.duplicate()), s3Metadata);
        this.setKmsParams(request);
        log.debug("Uploading buffer to {}", (Object)key);
        this.client.putObject(request);
    }

    private boolean usesKms() {
        return this.sseCustomerEncryptionKey != null && this.sseAlgorithm.equals("aws:kms");
    }

    private void putBufMaybeEncrypt(String key, Map<String, String> metadata, ByteBuffer buf) {
        ObjectMetadata s3metadata = this.objectMetadata(metadata);
        s3metadata.setContentLength((long)(buf.limit() - buf.position()));
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)new ByteBufferInputStream(buf.duplicate()), s3metadata);
        this.setKmsParams(request);
        log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        this.client.putObject(request);
    }

    private List<DeleteObjectsRequest.KeyVersion> keysForSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        block5: for (TierObjectStore.FileType fileType : this.getFileTypesPerSegment()) {
            switch (fileType) {
                case TRANSACTION_INDEX: {
                    if (!objectMetadata.hasAbortedTxns()) continue block5;
                    keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, fileType)));
                    continue block5;
                }
                case EPOCH_STATE: {
                    if (!objectMetadata.hasEpochState()) continue block5;
                    keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, fileType)));
                    continue block5;
                }
                case PRODUCER_STATE: {
                    if (!objectMetadata.hasProducerState()) continue block5;
                    keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, fileType)));
                    continue block5;
                }
            }
            keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, fileType)));
        }
        return keys;
    }

    @Override
    public BucketHealthResult checkBucketHealth() {
        try {
            ByteBuffer payload = TierObjectStoreUtils.timeHealthPayload();
            TierObjectStore.HealthMetadata metadata = new TierObjectStore.HealthMetadata(this.clusterIdOpt, this.brokerIdOpt);
            String key = metadata.toPath(this.prefix, TierObjectStore.FileType.HEALTH_CHECK);
            this.putBufMaybeEncrypt(key, metadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt), payload);
            try (InputStream inputStream = this.getObject(metadata, TierObjectStore.FileType.HEALTH_CHECK).getInputStream();){
                int read;
                while ((read = inputStream.read()) > 0) {
                    log.trace("Bucket probe read {} bytes", (Object)read);
                }
            }
            this.client.deleteObject(this.bucket, key);
            return BucketHealthResult.HEALTHY;
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 400 && e.getErrorCode().startsWith("KMS.")) {
                log.error("Bucket health checker resulted in a BYOK related error with error code: {}, status code: {}", new Object[]{e.getErrorCode(), e.getStatusCode(), e});
                return BucketHealthResult.BYOK;
            }
            if (e.getStatusCode() == 403 && Objects.equals(e.getErrorCode(), "AccessDenied")) {
                log.error("Bucket health checker resulted in a permission error for customer key: {}", (Object)(this.usesKms() ? "not enabled" : this.sseCustomerEncryptionKey), (Object)e);
                if (this.usesKms()) {
                    return BucketHealthResult.BYOK;
                }
                return BucketHealthResult.PERMISSION;
            }
            log.error("Bucket health checker returned an unclassified error for status code: {} error code: {}", new Object[]{e.getStatusCode(), e.getErrorCode(), e});
            return BucketHealthResult.UNCLASSIFIED;
        }
        catch (Exception e) {
            log.error("Bucket health checker returned unclassified error", (Throwable)e);
            return BucketHealthResult.UNCLASSIFIED;
        }
    }

    public static String validateAndGetS3RegionName(String s3Region) {
        try {
            return Regions.fromName((String)s3Region).getName();
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Configured " + KafkaConfig.TierS3RegionProp() + " '" + s3Region + "' is not known");
        }
    }

    public static AmazonS3 client(S3TierObjectStoreConfig config) throws TierObjectStoreFatalException {
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        clientConfiguration.setUserAgentPrefix(config.s3UserAgentPrefix);
        SSLConnectionSocketFactory sslConnectionSocketFactory = S3TierObjectStore.getSSLConnectionSocketFactory(config);
        if (sslConnectionSocketFactory != null) {
            clientConfiguration.getApacheHttpClientConfig().setSslSocketFactory((ConnectionSocketFactory)sslConnectionSocketFactory);
        }
        AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard();
        builder.setClientConfiguration(clientConfiguration);
        if (config.s3ForcePathStyleAccess.booleanValue()) {
            builder.setPathStyleAccessEnabled(Boolean.valueOf(true));
        }
        if (config.s3SignerOverride.isPresent() && !config.s3SignerOverride.get().isEmpty()) {
            clientConfiguration.setSignerOverride(config.s3SignerOverride.get());
        }
        if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
            String s3RegionName = S3TierObjectStore.validateAndGetS3RegionName(config.s3Region);
            builder.setEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(config.s3EndpointOverride.get(), s3RegionName));
        } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
            builder.setRegion(config.s3Region);
        }
        log.debug("AWS_METADATA_SERVICE_TIMEOUT is {} seconds", (Object)System.getenv("AWS_METADATA_SERVICE_TIMEOUT"));
        AWSCredentialsProvider provider = config.s3CredFilePath.map(PropertiesFileCredentialsProvider::new).orElse((AWSCredentialsProvider)new DefaultAWSCredentialsProviderChain());
        if (config.assumeRoleArn.isPresent()) {
            AWSSecurityTokenServiceClientBuilder stsClientBuilder = AWSSecurityTokenServiceClient.builder();
            stsClientBuilder.setCredentials(provider);
            if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
                String s3RegionName = S3TierObjectStore.validateAndGetS3RegionName(config.s3Region);
                stsClientBuilder.setRegion(s3RegionName);
            } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
                stsClientBuilder.setRegion(config.s3Region);
            }
            AWSSecurityTokenService stsClient = (AWSSecurityTokenService)stsClientBuilder.build();
            provider = new STSAssumeRoleSessionCredentialsProvider.Builder(config.assumeRoleArn.get(), "tiered-storage").withStsClient(stsClient).build();
        }
        builder.setCredentials(provider);
        return (AmazonS3)builder.build();
    }

    private void expectBucket(String bucket, String expectedRegion, Optional<String> endpointOverride) throws TierObjectStoreFatalException {
        try {
            String actualRegion = this.client.getBucketLocation(bucket);
            if (actualRegion.equals("US") && expectedRegion.equals("us-east-1")) {
                return;
            }
            if (!expectedRegion.equals(actualRegion)) {
                log.warn("Bucket region {} does not match expected region {}", (Object)actualRegion, (Object)expectedRegion);
            }
        }
        catch (AmazonClientException ex) {
            if (endpointOverride.isPresent() && !endpointOverride.get().isEmpty()) {
                log.warn("On-prem store does not implement S3 API's GetBucketLocation. Skipping check which ensures that actual bucket region matches expected region.");
            }
            throw new TierObjectStoreFatalException("Failed to validate that bucket location for " + bucket + " matches location " + expectedRegion + "; unable to call GetBucketLocation", ex);
        }
    }

    private static SSLConnectionSocketFactory getSSLConnectionSocketFactory(S3TierObjectStoreConfig config) throws TierObjectStoreFatalException {
        boolean hasCustomKeyStore;
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;
        boolean hasCustomTrustStore = config.s3SslTrustStoreLocation.isPresent() && !config.s3SslTrustStoreLocation.get().isEmpty();
        boolean bl = hasCustomKeyStore = config.s3SslKeyStoreLocation.isPresent() && !config.s3SslKeyStoreLocation.get().isEmpty();
        if (hasCustomTrustStore || hasCustomKeyStore) {
            try {
                SSLContextBuilder sslContextBuilder = SSLContexts.custom();
                if (hasCustomTrustStore) {
                    KeyStore trustStore = KeyStore.getInstance(config.s3SslTrustStoreType.get());
                    FileInputStream trustKeyStoreFile = new FileInputStream(config.s3SslTrustStoreLocation.get());
                    trustStore.load(trustKeyStoreFile, config.s3SslTrustStorePassword.get().value().toCharArray());
                    sslContextBuilder.loadTrustMaterial(trustStore, null);
                }
                if (hasCustomKeyStore) {
                    KeyStore keyStore = KeyStore.getInstance(config.s3SslKeyStoreType.get());
                    FileInputStream identityKeyStoreFile = new FileInputStream(config.s3SslKeyStoreLocation.get());
                    keyStore.load(identityKeyStoreFile, config.s3SslKeyStorePassword.get().value().toCharArray());
                    sslContextBuilder.loadKeyMaterial(keyStore, config.s3SslKeyPassword.get().value().toCharArray(), (aliases, socket) -> "confluent.kafka");
                }
                sslContextBuilder.setProtocol(config.s3SslProtocol);
                SSLContext sslContext = sslContextBuilder.build();
                sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContext, config.s3SslEnabledProtocols.toArray(new String[0]), null, (HostnameVerifier)new DefaultHostnameVerifier());
            }
            catch (Exception e) {
                throw new TierObjectStoreFatalException("Failed to load keystore or trust store for tiered object store", e);
            }
        }
        return sslConnectionSocketFactory;
    }

    private static class S3TierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final AutoAbortingGenericInputStream inputStream;

        S3TierObjectStoreResponse(S3ObjectInputStream inputStream, long autoAbortSize, long streamSize) {
            this.inputStream = new S3AutoAbortingInputStream(inputStream, autoAbortSize, streamSize);
        }

        @Override
        public void close() {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }
}

