/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.function.udf.geo;

import com.google.common.collect.Lists;
import com.google.common.collect.Streams;
import io.confluent.ksql.function.KsqlFunctionException;
import io.confluent.ksql.function.udf.Udf;
import io.confluent.ksql.function.udf.UdfDescription;
import io.confluent.ksql.function.udf.UdfParameter;
import io.confluent.ksql.util.GrammaticalJoiner;
import java.util.List;
import java.util.stream.Stream;

@UdfDescription(name="geo_distance", author="Confluent", description="Compute the distance between two points on the surface of the earth, according to the Haversine formula for \"great circle distance\".")
public class GeoDistance {
    private static final double EARTH_RADIUS_KM = 6371.0;
    private static final double EARTH_RADIUS_MILES = 3959.0;
    private static final List<String> VALID_RADIUS_NAMES_MILES = Lists.newArrayList((Object[])new String[]{"mi", "mile", "miles"});
    private static final List<String> VALID_RADIUS_NAMES_KMS = Lists.newArrayList((Object[])new String[]{"km", "kilometer", "kilometers", "kilometre", "kilometres"});
    private static final String VALID_VALUES = "one of " + GrammaticalJoiner.or().join(Streams.concat((Stream[])new Stream[]{VALID_RADIUS_NAMES_MILES.stream(), VALID_RADIUS_NAMES_KMS.stream()}));

    @Udf(description="The 2 input points should be specified as (lat, lon) pairs, measured in decimal degrees. An optional fifth parameter allows to specify either \"MI\" (miles) or \"KM\" (kilometers) as the desired unit for the output measurement. Default is KM.")
    public Double geoDistance(@UdfParameter(description="The latitude of the first point in decimal degrees.") double lat1, @UdfParameter(description="The longitude of the first point in decimal degrees.") double lon1, @UdfParameter(description="The latitude of the second point in decimal degrees.") double lat2, @UdfParameter(description="The longitude of the second point in decimal degrees.") double lon2, @UdfParameter(description="The units for the return value. Either MILES or KM.") String units) {
        GeoDistance.validateLatLonValues(lat1, lon1, lat2, lon2);
        double chosenRadius = GeoDistance.selectEarthRadiusToUse(units);
        double deltaLat = Math.toRadians(lat2 - lat1);
        double deltaLon = Math.toRadians(lon2 - lon1);
        double lat1Radians = Math.toRadians(lat1);
        double lat2Radians = Math.toRadians(lat2);
        double a = GeoDistance.haversin(deltaLat) + GeoDistance.haversin(deltaLon) * Math.cos(lat1Radians) * Math.cos(lat2Radians);
        double distanceInRadians = 2.0 * Math.asin(Math.sqrt(a));
        return distanceInRadians * chosenRadius;
    }

    @Udf(description="The 2 input points should be specified as (lat, lon) pairs, measured in decimal degrees. The distance returned is in kilometers.")
    public Double geoDistance(@UdfParameter(description="The latitude of the first point in decimal degrees.") double lat1, @UdfParameter(description="The longitude of the first point in decimal degrees.") double lon1, @UdfParameter(description="The latitude of the second point in decimal degrees.") double lat2, @UdfParameter(description="The longitude of the second point in decimal degrees.") double lon2) {
        return this.geoDistance(lat1, lon1, lat2, lon2, VALID_RADIUS_NAMES_KMS.get(0));
    }

    private static void validateLatLonValues(double lat1, double lon1, double lat2, double lon2) {
        if (lat1 < -90.0 || lat2 < -90.0 || lat1 > 90.0 || lat2 > 90.0) {
            throw new KsqlFunctionException("valid latitude values for GeoDistance function are in the range of -90 to 90 decimal degrees");
        }
        if (lon1 < -180.0 || lon2 < -180.0 || lon1 > 180.0 || lon2 > 180.0) {
            throw new KsqlFunctionException("valid longitude values for GeoDistance function are in the range of -180 to +180 decimal degrees");
        }
    }

    private static double selectEarthRadiusToUse(String units) {
        double chosenRadius = 6371.0;
        if (units != null && units.trim().length() > 0) {
            String outputUnit = units.trim().toLowerCase();
            if (VALID_RADIUS_NAMES_MILES.contains(outputUnit)) {
                chosenRadius = 3959.0;
            } else if (VALID_RADIUS_NAMES_KMS.contains(outputUnit)) {
                chosenRadius = 6371.0;
            } else {
                throw new KsqlFunctionException("GeoDistance function units parameter must be " + VALID_VALUES + ". Values are case-insensitive.");
            }
        }
        return chosenRadius;
    }

    private static double haversin(double val) {
        return Math.pow(Math.sin(val / 2.0), 2.0);
    }
}

