/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.telemetry.collector;

import io.confluent.shaded.com.google.common.annotations.VisibleForTesting;
import io.confluent.telemetry.MetricKey;
import io.confluent.telemetry.collector.LastValueTracker;
import io.confluent.telemetry.collector.MetricNamingStrategy;
import io.confluent.telemetry.collector.MetricsCollector;
import io.confluent.telemetry.emitter.Emitter;
import io.confluent.telemetry.metrics.SinglePointMetric;
import java.lang.reflect.Field;
import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.metrics.KafkaMetric;
import org.apache.kafka.common.metrics.Measurable;
import org.apache.kafka.common.metrics.stats.CumulativeSum;
import org.apache.kafka.common.metrics.stats.WindowedCount;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaMetricsCollector
implements MetricsCollector {
    private static final Logger log = LoggerFactory.getLogger(KafkaMetricsCollector.class);
    private final StateLedger ledger;
    private final Clock clock;
    private final MetricNamingStrategy<MetricName> metricNamingStrategy;
    private static final Field METRIC_VALUE_PROVIDER_FIELD;

    public KafkaMetricsCollector(MetricNamingStrategy<MetricName> metricNamingStrategy) {
        this(metricNamingStrategy, Clock.systemUTC());
    }

    @VisibleForTesting
    KafkaMetricsCollector(MetricNamingStrategy<MetricName> metricNamingStrategy, Clock clock) {
        this.metricNamingStrategy = metricNamingStrategy;
        this.clock = clock;
        this.ledger = new StateLedger();
    }

    public void init(List<KafkaMetric> metrics) {
        this.ledger.init(metrics);
    }

    public void metricChange(KafkaMetric metric) {
        this.ledger.metricChange(metric);
    }

    public void metricRemoval(KafkaMetric metric) {
        this.ledger.metricRemoval(metric);
    }

    @VisibleForTesting
    Set<MetricKey> getTrackedMetrics() {
        return this.ledger.metricMap.keySet();
    }

    @Override
    public void collect(Emitter emitter) {
        for (Map.Entry<MetricKey, KafkaMetric> entry : this.ledger.getMetrics()) {
            MetricKey metricKey = entry.getKey();
            KafkaMetric metric = entry.getValue();
            try {
                this.collectMetric(emitter, metricKey, metric);
            }
            catch (Exception e) {
                log.error("Unexpected error processing Kafka metric {}", (Object)metricKey, (Object)e);
            }
        }
    }

    private void collectMetric(Emitter emitter, MetricKey metricKey, KafkaMetric metric) {
        Object metricValue;
        try {
            metricValue = metric.metricValue();
        }
        catch (Exception e) {
            log.warn("Failed to retrieve metric value {}", (Object)metricKey.getName(), (Object)e);
            return;
        }
        if (KafkaMetricsCollector.isMeasurable(metric)) {
            Measurable measurable = metric.measurable();
            Double value = (Double)metricValue;
            if (measurable instanceof WindowedCount || measurable instanceof CumulativeSum) {
                this.collectSum(metricKey, value, emitter);
                this.collectDelta(metricKey, value, emitter);
            } else {
                this.collectGauge(metricKey, value, emitter);
            }
        } else if (metricValue instanceof Number) {
            Number value = (Number)metricValue;
            this.collectGauge(metricKey, value, emitter);
        } else {
            log.debug("Skipping non-measurable gauge metric {}", (Object)metricKey.getName());
        }
    }

    private void collectDelta(MetricKey originalKey, Double value, Emitter emitter) {
        MetricKey metricKey = this.metricNamingStrategy.derivedMetricKey(originalKey, "delta");
        if (!emitter.shouldEmitMetric(metricKey)) {
            return;
        }
        Instant timestamp = this.clock.instant();
        LastValueTracker.InstantAndValue<Double> instantAndValue = this.ledger.delta(originalKey, timestamp, value);
        emitter.emitMetric(SinglePointMetric.deltaSum(metricKey, instantAndValue.getValue(), true, timestamp, instantAndValue.getIntervalStart()));
    }

    private void collectSum(MetricKey metricKey, double value, Emitter emitter) {
        if (!emitter.shouldEmitMetric(metricKey)) {
            return;
        }
        emitter.emitMetric(SinglePointMetric.sum(metricKey, value, true, this.clock.instant(), this.ledger.instantAdded(metricKey)));
    }

    private void collectSum(MetricKey metricKey, long value, Emitter emitter) {
        if (!emitter.shouldEmitMetric(metricKey)) {
            return;
        }
        emitter.emitMetric(SinglePointMetric.sum(metricKey, value, true, this.clock.instant()));
    }

    private void collectGauge(MetricKey metricKey, Number value, Emitter emitter) {
        if (!emitter.shouldEmitMetric(metricKey)) {
            return;
        }
        emitter.emitMetric(SinglePointMetric.gauge(metricKey, value, this.clock.instant()));
    }

    public String toString() {
        return this.getClass().getCanonicalName();
    }

    private static boolean isMeasurable(KafkaMetric metric) {
        try {
            Object provider = METRIC_VALUE_PROVIDER_FIELD.get(metric);
            return provider instanceof Measurable;
        }
        catch (Exception e) {
            throw new KafkaException((Throwable)e);
        }
    }

    static {
        try {
            METRIC_VALUE_PROVIDER_FIELD = KafkaMetric.class.getDeclaredField("metricValueProvider");
            METRIC_VALUE_PROVIDER_FIELD.setAccessible(true);
        }
        catch (Exception e) {
            throw new KafkaException((Throwable)e);
        }
    }

    private class StateLedger {
        private final Map<MetricKey, KafkaMetric> metricMap = new ConcurrentHashMap<MetricKey, KafkaMetric>();
        private final LastValueTracker<Double> doubleDeltas = new LastValueTracker();
        private final ConcurrentMap<MetricKey, Instant> metricAdded = new ConcurrentHashMap<MetricKey, Instant>();

        private StateLedger() {
        }

        private Instant instantAdded(MetricKey metricKey) {
            return this.metricAdded.computeIfAbsent(metricKey, x -> KafkaMetricsCollector.this.clock.instant());
        }

        public void init(List<KafkaMetric> metrics) {
            log.debug("initializing Kafka metrics collector");
            for (KafkaMetric m4 : metrics) {
                this.metricMap.put(KafkaMetricsCollector.this.metricNamingStrategy.metricKey(m4.metricName()), m4);
            }
        }

        public void metricChange(KafkaMetric metric) {
            MetricKey metricKey = KafkaMetricsCollector.this.metricNamingStrategy.metricKey(metric.metricName());
            this.metricMap.put(metricKey, metric);
            if (this.doubleDeltas.contains(metricKey)) {
                log.warn("Registering a new metric {} which already has a last value tracked. Removing metric from delta register.", (Object)metric.metricName(), (Object)new Exception());
                this.doubleDeltas.remove(metricKey);
            }
            this.metricAdded.put(metricKey, KafkaMetricsCollector.this.clock.instant());
        }

        public void metricRemoval(KafkaMetric metric) {
            log.debug("removing kafka metric : {}", (Object)metric.metricName());
            MetricKey metricKey = KafkaMetricsCollector.this.metricNamingStrategy.metricKey(metric.metricName());
            this.metricMap.remove(metricKey);
            this.doubleDeltas.remove(metricKey);
            this.metricAdded.remove(metricKey);
        }

        public Iterable<? extends Map.Entry<MetricKey, KafkaMetric>> getMetrics() {
            return this.metricMap.entrySet();
        }

        public LastValueTracker.InstantAndValue<Double> delta(MetricKey metricKey, Instant now, Double value) {
            Optional<LastValueTracker.InstantAndValue<Double>> lastValue = this.doubleDeltas.getAndSet(metricKey, now, value);
            return lastValue.map(last -> new LastValueTracker.InstantAndValue<Double>(last.getIntervalStart(), value - (Double)last.getValue())).orElse(new LastValueTracker.InstantAndValue<Double>(this.instantAdded(metricKey), value));
        }
    }
}

