/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.topic;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import kafka.cluster.BrokerEndPoint;
import kafka.server.BrokerBlockingSender;
import kafka.server.KafkaConfig;
import kafka.server.RemoteLeaderEndPoint;
import kafka.server.RemoteLeaderRequestBuilder;
import kafka.server.ReplicaManager;
import kafka.tier.TopicIdPartition;
import kafka.tier.state.OffsetAndEpoch;
import kafka.tier.state.TierPartitionState;
import kafka.tier.tools.RecoveryUtils;
import kafka.tier.topic.TierTopic;
import kafka.tier.topic.TierTopicManagerConfig;
import org.apache.kafka.clients.admin.ConfluentAdmin;
import org.apache.kafka.clients.admin.PartitionResult;
import org.apache.kafka.clients.admin.ReplicaStatusOptions;
import org.apache.kafka.clients.admin.ReplicaStatusResult;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.message.OffsetForLeaderEpochRequestData;
import org.apache.kafka.common.message.OffsetForLeaderEpochResponseData;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.common.MetadataVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.Function0;

public class TierTopicDataLossValidator {
    private static final Logger log = LoggerFactory.getLogger(TierTopicDataLossValidator.class);
    private static final int DATA_LOSS_DETECTION_TIMEOUT_MS = 900000;
    private static final int DATA_LOSS_DETECTION_THREAD_POOL_SIZE_MAX = 10;
    private final TierTopicManagerConfig config;
    private final TierTopic tierTopic;
    private final ReplicaManager replicaManager;
    private final Time time;
    private final Metrics metrics;
    private final Map<Integer, RemoteLeaderEndPoint> brokerToLeaderEndPoint = new ConcurrentHashMap<Integer, RemoteLeaderEndPoint>();

    public TierTopicDataLossValidator(TierTopicManagerConfig config, TierTopic tierTopic, ReplicaManager replicaManager, Time time, Metrics metrics) {
        this.config = config;
        this.tierTopic = tierTopic;
        this.replicaManager = replicaManager;
        this.time = time;
        this.metrics = metrics;
    }

    boolean detectDataLossInTierTopicHead(Producer<byte[], byte[]> producer) throws InterruptedException {
        if (!this.config.enableTierTopicDataLossDetection.booleanValue()) {
            log.info("Skip data loss detection in tier topic as {} is disabled.", (Object)KafkaConfig.TierTopicDataLossDetectionEnableProp());
            return false;
        }
        if (this.replicaManager.logManager().hadCleanShutdown()) {
            log.info("Skip data loss detection in tier topic as broker had clean shutdown.");
            return false;
        }
        log.info("Broker did not have clean shutdown. Validating data loss in tier topic head.");
        long startTimeMs = this.time.milliseconds();
        Map<TopicPartition, OffsetAndEpoch> tierTopicPartitionsToValidate = this.getTierTopicMaxOffsetAndEpochByFtps(this.tierTopic);
        if (tierTopicPartitionsToValidate.isEmpty()) {
            log.info("Skip data loss detection in tier topic as there are no tier topic partitions to validate.");
            return false;
        }
        ConfluentAdmin adminClient = this.createAdminClient();
        if (adminClient == null) {
            log.warn("Skip data loss detection in tier topic as admin client is not created.");
            return false;
        }
        ExecutorService executorService = null;
        try {
            HashMap<TopicPartition, Long> dataLossPartitionEndOffset = new HashMap<TopicPartition, Long>();
            Map<TopicPartition, PartitionResult> leaderInfo = this.getTierTopicPartitionLeaderInfo(tierTopicPartitionsToValidate.keySet(), adminClient);
            while (!tierTopicPartitionsToValidate.isEmpty()) {
                log.info("Try to detect data loss on {} tier topic partitions", (Object)tierTopicPartitionsToValidate.size());
                Map<Integer, Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition>> requestByLeader = this.buildRequests(tierTopicPartitionsToValidate, leaderInfo);
                if (executorService == null && !requestByLeader.isEmpty()) {
                    executorService = Executors.newFixedThreadPool(Math.min(requestByLeader.size(), 10));
                }
                ArrayList<Future<Map<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset>>> offsetForLeaderEpochTasks = new ArrayList<Future<Map<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset>>>();
                for (Map.Entry<Integer, Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition>> entry : requestByLeader.entrySet()) {
                    offsetForLeaderEpochTasks.add(executorService.submit(new OffsetForLeaderEpochTask(entry.getKey(), entry.getValue())));
                }
                HashMap<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset> epochEndOffsetsResults = new HashMap<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset>();
                offsetForLeaderEpochTasks.forEach(future -> {
                    try {
                        epochEndOffsetsResults.putAll((Map)future.get());
                    }
                    catch (Exception e) {
                        log.error("Failed to finish the OffsetForLeaderEpoch task", (Throwable)e);
                    }
                });
                Map<TopicPartition, PartitionResult> latestLeaderInfo = this.getTierTopicPartitionLeaderInfo(tierTopicPartitionsToValidate.keySet(), adminClient);
                this.validateDataLoss(tierTopicPartitionsToValidate, dataLossPartitionEndOffset, epochEndOffsetsResults, leaderInfo, latestLeaderInfo);
                leaderInfo = latestLeaderInfo;
                if (this.time.milliseconds() - startTimeMs > 900000L) {
                    log.warn("Tier topic data loss detection timed out after {} ms. {} partitions remaining to be validated: {}", new Object[]{900000, tierTopicPartitionsToValidate.size(), tierTopicPartitionsToValidate.keySet()});
                    break;
                }
                Thread.sleep(TimeUnit.SECONDS.toMillis(15L));
            }
            this.fenceUserTopicPartitions(dataLossPartitionEndOffset, producer);
            long durationMs = this.time.milliseconds() - startTimeMs;
            if (tierTopicPartitionsToValidate.isEmpty()) {
                log.info("Successfully finished detecting data loss in tier topic in {} ms - {} partitions are found to have data loss: {}", new Object[]{durationMs, dataLossPartitionEndOffset.size(), dataLossPartitionEndOffset.keySet()});
            } else {
                log.error("Failed to fully detect data loss in all partitions in tier topic head in {} ms, yet {} partitions are found to have data loss: {}", new Object[]{durationMs, dataLossPartitionEndOffset.size(), dataLossPartitionEndOffset.keySet()});
            }
            boolean bl = !dataLossPartitionEndOffset.isEmpty();
            return bl;
        }
        catch (Exception e) {
            log.error("Failed to detect data loss in tier topic head", (Throwable)e);
            throw e;
        }
        finally {
            for (RemoteLeaderEndPoint leaderEndPoint : this.brokerToLeaderEndPoint.values()) {
                leaderEndPoint.close();
            }
            if (executorService != null) {
                executorService.shutdown();
            }
            adminClient.close();
        }
    }

    void validateDataLoss(Map<TopicPartition, OffsetAndEpoch> partitionToValidate, Map<TopicPartition, Long> dataLossPartitionEndOffset, Map<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset> epochEndOffsetsResMap, Map<TopicPartition, PartitionResult> leaderInfoBefore, Map<TopicPartition, PartitionResult> leaderInfoAfter) {
        epochEndOffsetsResMap.forEach((tp, epochEndOffset) -> {
            OffsetAndEpoch offsetAndEpochByFtps = partitionToValidate.getOrDefault(tp, OffsetAndEpoch.EMPTY);
            PartitionResult leaderBefore = leaderInfoBefore.getOrDefault(tp, new PartitionResult(Collections.emptyList()));
            PartitionResult leaderAfter = leaderInfoAfter.getOrDefault(tp, new PartitionResult(Collections.emptyList()));
            if (offsetAndEpochByFtps != OffsetAndEpoch.EMPTY && leaderBefore.leaderId() == leaderAfter.leaderId() && leaderBefore.leaderEpoch().equals(leaderAfter.leaderEpoch())) {
                if (epochEndOffset.errorCode() != Errors.NONE.code()) {
                    log.error("OffsetForLeaderEpoch request to node {} for partition {} returned error code {}", new Object[]{leaderAfter.leaderId(), tp, epochEndOffset.errorCode()});
                } else {
                    boolean hasDataLoss;
                    partitionToValidate.remove(tp);
                    boolean bl = hasDataLoss = epochEndOffset.leaderEpoch() != offsetAndEpochByFtps.epoch().orElse(-1).intValue() || epochEndOffset.endOffset() <= offsetAndEpochByFtps.offset();
                    if (hasDataLoss) {
                        dataLossPartitionEndOffset.put((TopicPartition)tp, epochEndOffset.endOffset());
                        log.error("Data loss detected in partition {}: Leader epoch in the response {}, leader epoch in the request {}; end offset in response {}, highest committed offset in request {}", new Object[]{tp, epochEndOffset.leaderEpoch(), offsetAndEpochByFtps.epoch(), epochEndOffset.endOffset(), offsetAndEpochByFtps.offset()});
                    } else {
                        log.debug("No data loss detected in partition {}: Leader epoch in the response {}, leader epoch in the request {}; end offset in response {}, highest committed offset in request {}", new Object[]{tp, epochEndOffset.leaderEpoch(), offsetAndEpochByFtps.epoch(), epochEndOffset.endOffset(), offsetAndEpochByFtps.offset()});
                    }
                }
            }
        });
    }

    Map<TopicPartition, OffsetAndEpoch> getTierTopicMaxOffsetAndEpochByFtps(TierTopic tierTopic) {
        HashMap<TopicPartition, OffsetAndEpoch> partitionToOffsetEpoch = new HashMap<TopicPartition, OffsetAndEpoch>();
        this.replicaManager.logManager().allLogs().foreach(log -> {
            TierPartitionState tierPartitionState;
            if (!log.isDeleted() && !log.isStray() && log.topicIdPartition().isDefined() && !(tierPartitionState = log.tierPartitionState()).status().hasError() && tierPartitionState.lastLocalMaterializedSrcOffsetAndEpoch() != OffsetAndEpoch.EMPTY && tierPartitionState.lastLocalMaterializedSrcOffsetAndEpoch().epoch().isPresent()) {
                TopicPartition tierTopicPartition = tierTopic.toTierTopicPartition((TopicIdPartition)log.topicIdPartition().get());
                OffsetAndEpoch prevMaxOffsetAndEpoch = partitionToOffsetEpoch.getOrDefault(tierTopicPartition, OffsetAndEpoch.EMPTY);
                if (tierPartitionState.lastLocalMaterializedSrcOffsetAndEpoch().offset() > prevMaxOffsetAndEpoch.offset()) {
                    partitionToOffsetEpoch.put(tierTopicPartition, tierPartitionState.lastLocalMaterializedSrcOffsetAndEpoch());
                }
            }
            return null;
        });
        return partitionToOffsetEpoch;
    }

    ConfluentAdmin createAdminClient() {
        Map<String, Object> interBrokerClientConfigs = this.config.interBrokerClientConfigs.get();
        if (interBrokerClientConfigs.isEmpty() || !interBrokerClientConfigs.containsKey("bootstrap.servers")) {
            log.warn("Failed to create admin client - cannot resolve bootstrap server.");
            return null;
        }
        Properties adminClientProps = new Properties();
        adminClientProps.putAll(interBrokerClientConfigs);
        return ConfluentAdmin.create((Properties)adminClientProps);
    }

    Map<TopicPartition, PartitionResult> getTierTopicPartitionLeaderInfo(Set<TopicPartition> tierTopicPartitions, ConfluentAdmin adminClient) {
        HashMap<TopicPartition, PartitionResult> partitionResults = new HashMap<TopicPartition, PartitionResult>();
        ReplicaStatusResult replicaStatusResult = adminClient.replicaStatus(tierTopicPartitions, new ReplicaStatusOptions());
        replicaStatusResult.partitionResults().forEach((topicPartition, result) -> {
            PartitionResult partitionResult;
            try {
                partitionResult = (PartitionResult)result.get();
            }
            catch (Exception e) {
                log.warn("Failed to get PartitionResult for tier topic partition {}", topicPartition, (Object)e);
                partitionResult = new PartitionResult(Collections.emptyList());
            }
            partitionResults.put((TopicPartition)topicPartition, partitionResult);
        });
        return partitionResults;
    }

    private Map<Integer, Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition>> buildRequests(Map<TopicPartition, OffsetAndEpoch> partitionToValidate, Map<TopicPartition, PartitionResult> tierTopicPartitionLeaderInfo) {
        HashMap<Integer, Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition>> requests = new HashMap<Integer, Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition>>();
        tierTopicPartitionLeaderInfo.forEach((tierTopicPartition, partitionResult) -> {
            OffsetAndEpoch offsetAndEpoch = partitionToValidate.getOrDefault(tierTopicPartition, OffsetAndEpoch.EMPTY);
            if (partitionResult.leaderId() != -1 && offsetAndEpoch.epoch().isPresent()) {
                requests.computeIfAbsent(partitionResult.leaderId(), v -> new HashMap()).put(tierTopicPartition, new OffsetForLeaderEpochRequestData.OffsetForLeaderPartition().setPartition(tierTopicPartition.partition()).setLeaderEpoch(offsetAndEpoch.epoch().get().intValue()));
            }
        });
        return requests;
    }

    private void fenceUserTopicPartitions(Map<TopicPartition, Long> dataLossPartitionEndOffset, Producer<byte[], byte[]> producer) {
        if (dataLossPartitionEndOffset.isEmpty()) {
            TierTopicDataLossValidator.log.debug("No tier topic partition has data loss, no need to fence user topic partitions.");
            return;
        }
        if (!this.config.enableTierTopicFencingDuringDataLoss.booleanValue()) {
            TierTopicDataLossValidator.log.info("Skip fencing user topic partitions as {} is disabled", (Object)KafkaConfig.TierTopicFencingDuringDataLossEnableProp());
            return;
        }
        HashMap partitionToFence = new HashMap();
        this.replicaManager.logManager().allLogs().foreach(log -> {
            TopicPartition tierTopicPartition;
            long tierTopicPartitionEndOffset;
            long materializedOffset;
            if (!log.isStray() && !log.isDeleted() && log.topicIdPartition().isDefined() && !log.tierPartitionState().status().hasError() && (materializedOffset = log.tierPartitionState().lastLocalMaterializedSrcOffsetAndEpoch().offset()) >= (tierTopicPartitionEndOffset = dataLossPartitionEndOffset.getOrDefault(tierTopicPartition = this.tierTopic.toTierTopicPartition((TopicIdPartition)log.topicIdPartition().get()), Long.MAX_VALUE).longValue())) {
                partitionToFence.put(log.topicIdPartition().get(), materializedOffset);
            }
            return null;
        });
        TierTopicDataLossValidator.log.info("Fencing {} user topic partitions for {} tier topic partitions", (Object)partitionToFence.size(), (Object)dataLossPartitionEndOffset.size());
        for (Map.Entry entry : partitionToFence.entrySet()) {
            TopicIdPartition partition = (TopicIdPartition)entry.getKey();
            long materializedOffset = (Long)entry.getValue();
            try {
                RecoveryUtils.injectTierTopicEventsUntilOffset(producer, partition, this.tierTopic.topicName(), this.tierTopic.numPartitions().getAsInt(), materializedOffset + 1L, false);
            }
            catch (Exception e) {
                TierTopicDataLossValidator.log.error("Failed to send tier partition fence event for partition {}", (Object)partition, (Object)e);
            }
        }
    }

    RemoteLeaderEndPoint getLeaderEndPoint(int nodeId, int leaderId, Map<Integer, RemoteLeaderEndPoint> brokerToLeaderEndPoint) {
        RemoteLeaderEndPoint leaderEndPoint = brokerToLeaderEndPoint.get(leaderId);
        if (leaderEndPoint != null) {
            return leaderEndPoint;
        }
        Node leaderNode = (Node)this.replicaManager.metadataCache().getAliveBrokerNode(leaderId, this.replicaManager.config().interBrokerListenerName()).getOrElse(() -> Node.noNode());
        if (leaderNode == Node.noNode()) {
            log.error("No broker node found for id {} when building RemoteLeaderEndPoint.", (Object)leaderId);
            return null;
        }
        BrokerEndPoint brokerEndPoint = new BrokerEndPoint(leaderNode.id(), leaderNode.host(), leaderNode.port());
        BrokerBlockingSender blockingSender = new BrokerBlockingSender(brokerEndPoint, this.replicaManager.config(), this.metrics, this.time, -1, String.format("tier-topic-data-loss-detection-%d-to-%d-", nodeId, leaderId), new LogContext(), "tier-topic-data-loss-validator");
        RemoteLeaderRequestBuilder requestBuilder = new RemoteLeaderRequestBuilder(this.replicaManager.config(), (Function0<MetadataVersion>)((Function0)() -> this.replicaManager.metadataCache().metadataVersion()), this.replicaManager.brokerEpochSupplier());
        leaderEndPoint = new RemoteLeaderEndPoint(String.format("[RemoteLeaderEndPoint NodeId=%d]", leaderNode.id()), blockingSender, null, requestBuilder, null, this.replicaManager.config(), this.replicaManager, null, (Function0<MetadataVersion>)((Function0)() -> this.replicaManager.metadataCache().metadataVersion()));
        log.debug("Created a remote LeaderEndPoint for broker {}", (Object)leaderId);
        brokerToLeaderEndPoint.put(leaderId, leaderEndPoint);
        return leaderEndPoint;
    }

    class OffsetForLeaderEpochTask
    implements Callable<Map<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset>> {
        final int leaderId;
        final Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition> request;

        OffsetForLeaderEpochTask(int leaderId, Map<TopicPartition, OffsetForLeaderEpochRequestData.OffsetForLeaderPartition> request) {
            this.leaderId = leaderId;
            this.request = request;
        }

        @Override
        public Map<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset> call() throws Exception {
            RemoteLeaderEndPoint leaderEndPoint = TierTopicDataLossValidator.this.getLeaderEndPoint(((TierTopicDataLossValidator)TierTopicDataLossValidator.this).config.brokerId, this.leaderId, TierTopicDataLossValidator.this.brokerToLeaderEndPoint);
            if (leaderEndPoint == null) {
                log.error("Failed to get LeaderEndPoint for node {}", (Object)this.leaderId);
                return Collections.emptyMap();
            }
            log.debug("Sending OffsetForLeaderEpoch request to node {} for {} partitions: {}", new Object[]{this.leaderId, this.request.size(), this.request});
            Map<TopicPartition, OffsetForLeaderEpochResponseData.EpochEndOffset> result = leaderEndPoint.fetchEpochEndOffsetsAsJava(this.request);
            log.debug("Completed OffsetForLeaderEpoch request to node {} for {} partitions: {}", new Object[]{this.leaderId, result.size(), result});
            return result;
        }
    }
}

