/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.multitenant;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.confluent.kafka.multitenant.SslCertificateSpecification;
import io.confluent.kafka.multitenant.utils.Utils;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.clients.admin.AlterConfigOp;
import org.apache.kafka.clients.admin.AlterConfigsOptions;
import org.apache.kafka.clients.admin.AlterConfigsResult;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.clients.admin.ConfluentAdmin;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.config.internals.ConfluentConfigs;
import org.apache.kafka.common.network.ListenerName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslCertificateManager {
    private static final Logger LOG = LoggerFactory.getLogger(SslCertificateManager.class);
    private static final Long CLOSE_TIMEOUT_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final AlterConfigsOptions ALTER_OPTIONS = new AlterConfigsOptions().timeoutMs(Integer.valueOf(30000));
    private final String brokerId;
    private final String sslCertsDir;
    private final String sslSpecFilename;
    private final SslCertsDirListener dirListener;
    private final Thread dirListenerThread;
    private ConfluentAdmin adminClient;
    private AtomicBoolean adminClientCreated = new AtomicBoolean(false);
    private byte[] currentFullchainBytes = null;
    private byte[] currentPrivkeyBytes = null;
    private final List<String> sslListenerNames;

    SslCertificateManager(Map<String, ?> configs) {
        this(configs.get("broker.id"), configs.get("multitenant.metadata.ssl.certs.path"), null, ConfluentConfigs.sslListenerNames(configs));
    }

    SslCertificateManager(Object brokerId, Object sslCertsPath, ConfluentAdmin adminClient, List<String> sslListenerNames) {
        this.brokerId = this.getBrokerId(brokerId);
        this.sslCertsDir = this.getSslCertsDirConfig(sslCertsPath);
        this.sslSpecFilename = this.getSslSpecFilename(sslCertsPath);
        this.adminClient = adminClient;
        this.sslListenerNames = sslListenerNames;
        if (this.sslListenerNames.isEmpty()) {
            LOG.warn("No TLS-enabled listeners were configured");
        }
        if (this.adminClient != null) {
            this.adminClientCreated.compareAndSet(false, true);
        }
        if (this.sslCertsDir != null && !this.sslListenerNames.isEmpty()) {
            this.dirListener = new SslCertsDirListener(this.sslCertsDir);
            this.dirListenerThread = new Thread((Runnable)this.dirListener, "confluent-ssl-certs-change-listener");
        } else {
            this.dirListener = null;
            this.dirListenerThread = null;
        }
    }

    public void createAdminClient(Map<String, Object> interBrokerClientConfigs) {
        if (this.brokerId != null && this.sslCertsDir != null) {
            this.adminClient = Utils.createConfluentAdmin(interBrokerClientConfigs);
        }
        if (this.adminClient != null) {
            this.adminClientCreated.compareAndSet(false, true);
        }
    }

    public void close() {
        if (this.dirListener != null) {
            this.dirListener.close();
        }
        if (this.adminClient != null) {
            this.adminClient.close(Duration.ofMillis(CLOSE_TIMEOUT_MS));
        }
    }

    private String getBrokerId(Object brokerIdValue) {
        if (brokerIdValue == null) {
            return null;
        }
        return brokerIdValue.toString();
    }

    private String getSslCertsDirConfig(Object sslCertPath) {
        String sslCertDir = null;
        if (sslCertPath != null) {
            try {
                sslCertDir = sslCertPath.toString().substring(0, sslCertPath.toString().lastIndexOf("/"));
            }
            catch (IndexOutOfBoundsException e) {
                LOG.warn("Ssl cert spec config not in the correct format {}", sslCertPath, (Object)e);
            }
        }
        return sslCertDir;
    }

    private String getSslSpecFilename(Object sslCertPath) {
        String sslSpecFilename = null;
        if (sslCertPath != null) {
            try {
                sslSpecFilename = sslCertPath.toString().substring(sslCertPath.toString().lastIndexOf("/") + 1);
            }
            catch (IndexOutOfBoundsException e) {
                LOG.warn("Ssl cert spec config not in the correct format {}", sslCertPath, (Object)e);
            }
        }
        return sslSpecFilename;
    }

    void startWatching() throws IOException {
        if (this.dirListener != null) {
            this.dirListener.register();
            this.dirListenerThread.start();
        }
    }

    void shutdown() {
        try {
            if (this.dirListener != null) {
                this.dirListenerThread.interrupt();
                this.dirListenerThread.join(CLOSE_TIMEOUT_MS);
            }
        }
        catch (InterruptedException e) {
            LOG.error("Shutting down ssl certs reader thread was interrupted", (Throwable)e);
        }
    }

    ConfluentAdmin getAdminClient() {
        return this.adminClient;
    }

    void loadSslCertFiles() {
        if (!this.adminClientCreated.get()) {
            LOG.info("Admin client instance not created, would not proceed to certificate update");
            return;
        }
        Path sslCertsSpecFile = this.getSslCertsFilePath(this.sslSpecFilename);
        if (!Files.exists(sslCertsSpecFile, new LinkOption[0])) {
            LOG.warn("No spec file found at path: {}", (Object)sslCertsSpecFile);
            return;
        }
        this.getSslCertificateSpecifications();
    }

    private void getSslCertificateSpecifications() {
        try {
            Path specPath = this.getSslCertsFilePath(this.sslSpecFilename);
            ObjectMapper objectMapper = new ObjectMapper();
            SslCertificateSpecification sslSpec = (SslCertificateSpecification)objectMapper.readValue(specPath.toFile(), SslCertificateSpecification.class);
            this.getSslConfigs(sslSpec);
        }
        catch (Exception e) {
            LOG.error("Error occurred while retrieving certificate specifications", (Throwable)e);
        }
    }

    private void getSslConfigs(SslCertificateSpecification sslSpec) throws Exception {
        Path sslPemFullchainFilePath = this.getSslCertsFilePath(sslSpec.sslPemFullchainFilename());
        if (!Files.exists(sslPemFullchainFilePath, new LinkOption[0])) {
            LOG.warn("No fullchain file found at path: {} ", (Object)sslPemFullchainFilePath);
            return;
        }
        this.checkCertificateX509(sslPemFullchainFilePath);
        Path sslPemPrivkeyFilePath = this.getSslCertsFilePath(sslSpec.sslPemPrivkeyFilename());
        if (!Files.exists(sslPemPrivkeyFilePath, new LinkOption[0])) {
            LOG.warn("No privkey file found at path: {} ", (Object)sslPemPrivkeyFilePath);
            return;
        }
        Path pkcsCertFilePath = this.getSslCertsFilePath(sslSpec.pkcsCertFilename());
        if (!Files.exists(pkcsCertFilePath, new LinkOption[0])) {
            LOG.warn("No pkcs file found at path: {} ", (Object)pkcsCertFilePath);
            return;
        }
        this.checkCertificatePKCS(sslSpec, pkcsCertFilePath);
        byte[] syncFullchainBytes = this.pemToBytes(sslPemFullchainFilePath);
        byte[] syncPrivkeyBytes = this.pemToBytes(sslPemPrivkeyFilePath);
        if (!this.comparePEMCertificates(syncFullchainBytes, syncPrivkeyBytes)) {
            ArrayList sslCertConfig = new ArrayList();
            this.sslListenerNames.forEach(listenerName -> {
                String configPrefix = new ListenerName(listenerName).configPrefix();
                ConfigEntry keystoreLocation = new ConfigEntry(configPrefix + "ssl.keystore.location", pkcsCertFilePath.toString());
                ConfigEntry keystoreType = new ConfigEntry(configPrefix + "ssl.keystore.type", sslSpec.sslKeystoreType());
                sslCertConfig.add(new AlterConfigOp(keystoreLocation, AlterConfigOp.OpType.SET));
                sslCertConfig.add(new AlterConfigOp(keystoreType, AlterConfigOp.OpType.SET));
            });
            ConfigResource configResource = new ConfigResource(ConfigResource.Type.BROKER, this.brokerId);
            HashMap<ConfigResource, Collection<AlterConfigOp>> alterSslCertConfig = new HashMap<ConfigResource, Collection<AlterConfigOp>>();
            alterSslCertConfig.put(configResource, sslCertConfig);
            this.invokeAdminClient(alterSslCertConfig, syncFullchainBytes, syncPrivkeyBytes);
        }
    }

    private void checkCertificateX509(Path path) {
        try {
            CertificateFactory fac = CertificateFactory.getInstance("X509");
            Collection<? extends Certificate> certs = fac.generateCertificates(Files.newInputStream(path, new OpenOption[0]));
            for (Certificate certificate : certs) {
                Path certFileName = path.getFileName();
                this.checkCert(certificate, certFileName.toString());
            }
        }
        catch (Exception e) {
            LOG.error("Error getting certificates as X509.", (Throwable)e);
        }
    }

    private void checkCertificatePKCS(SslCertificateSpecification sslSpec, Path path) {
        try {
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            keyStore.load(Files.newInputStream(path, new OpenOption[0]), sslSpec.getSslKeystorePassword().toCharArray());
            Enumeration<String> aliases = keyStore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                this.checkCert(keyStore.getCertificate(alias), path.getFileName().toString());
            }
        }
        catch (Exception e) {
            LOG.error("Unable to check PKCS key store", (Throwable)e);
        }
    }

    private void checkCert(Certificate cert, String certFileName) {
        LOG.info("Checking cert " + certFileName);
        try {
            if (cert instanceof X509Certificate) {
                X509Certificate c = (X509Certificate)cert;
                c.checkValidity();
            } else {
                LOG.warn(String.format("Certificate %s not instance of X509Certificate", certFileName));
            }
        }
        catch (CertificateExpiredException | CertificateNotYetValidException e) {
            LOG.error(String.format("Certificate %s is invalid based on from/to dates.", certFileName), (Throwable)e);
        }
        catch (Exception e) {
            LOG.error(String.format("Error checking certificate %s validity", certFileName), (Throwable)e);
        }
    }

    private synchronized void invokeAdminClient(Map<ConfigResource, Collection<AlterConfigOp>> alterSslCertConfig, byte[] syncFullchainBytes, byte[] syncPrivkeyBytes) throws Exception {
        AlterConfigsResult alterConfigsResult = this.adminClient.incrementalAlterConfigs(alterSslCertConfig, ALTER_OPTIONS);
        alterConfigsResult.all().get();
        this.currentFullchainBytes = (byte[])syncFullchainBytes.clone();
        this.currentPrivkeyBytes = (byte[])syncPrivkeyBytes.clone();
        LOG.info("Updated SSL certificate files for broker {} with {}", (Object)this.brokerId, (Object)alterConfigsResult);
    }

    private boolean comparePEMCertificates(byte[] syncFullchainBytes, byte[] syncPrivkeyBytes) {
        return Arrays.equals(this.currentFullchainBytes, syncFullchainBytes) && Arrays.equals(this.currentPrivkeyBytes, syncPrivkeyBytes);
    }

    private byte[] pemToBytes(Path path) throws Exception {
        String filepath = path.toString();
        File file = new File(filepath);
        FileInputStream fis = new FileInputStream(filepath);
        byte[] fileData = new byte[(int)file.length()];
        fis.read(fileData);
        return fileData;
    }

    private Path getSslCertsFilePath(String filename) {
        return Paths.get(this.sslCertsDir, filename);
    }

    class SslCertsDirListener
    implements Runnable {
        private WatchService watchService = null;
        private Path sslCertsPath = null;
        private static final String DATA_DIR_NAME = "..data";

        SslCertsDirListener(String sslCertsDir) {
            this.sslCertsPath = Paths.get(sslCertsDir, new String[0]);
        }

        void register() throws IOException {
            if (this.watchService == null) {
                this.watchService = FileSystems.getDefault().newWatchService();
                if (!Files.exists(this.sslCertsPath, new LinkOption[0])) {
                    LOG.info("{} doesn't exists yet, creating", (Object)this.sslCertsPath);
                    Files.createDirectories(this.sslCertsPath, new FileAttribute[0]);
                }
                this.sslCertsPath.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.OVERFLOW);
                LOG.info("Registered to watch {} for ssl certs updates", (Object)this.sslCertsPath);
            } else {
                LOG.warn("Trying to reregister an existing SslCertsDirListener");
            }
        }

        public void close() {
            if (this.watchService != null) {
                try {
                    this.watchService.close();
                    this.watchService = null;
                    LOG.info("Closed watcher for ssl certs at {}", (Object)this.sslCertsPath);
                }
                catch (IOException ioe) {
                    LOG.error("Failed to shutdown watcher for ssl certs at {}", (Object)this.sslCertsPath, (Object)ioe);
                }
            } else {
                LOG.warn("Trying to close a null watchService");
            }
        }

        @Override
        public void run() {
            try {
                this.doRun();
            }
            catch (InterruptedException ie) {
                LOG.warn("Watching {} for ssl certs was interrupted.", (Object)this.sslCertsPath);
            }
            catch (Exception e) {
                LOG.warn("Stopping watching for ssl certs. ", (Throwable)e);
            }
            finally {
                this.close();
            }
        }

        private void doRun() throws InterruptedException {
            WatchKey key;
            do {
                key = this.watchService.take();
                for (WatchEvent<?> event : key.pollEvents()) {
                    LOG.debug("Got event: {} {}", event.kind(), event.context());
                    Path filename = this.sslCertsPath.resolve((Path)event.context());
                    if (!filename.getFileName().toString().equals(DATA_DIR_NAME)) continue;
                    SslCertificateManager.this.loadSslCertFiles();
                }
            } while (key.reset());
            LOG.warn("Ssl certs dir no longer watched");
        }
    }
}

