/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.controlcenter.httpclient;

import com.damnhandy.uri.template.UriUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.CharStreams;
import io.confluent.controlcenter.ControlCenterConfig;
import io.confluent.controlcenter.httpclient.HttpCredential;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import javax.ws.rs.InternalServerErrorException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.api.ContentProvider;
import org.eclipse.jetty.client.api.Request;
import org.eclipse.jetty.client.api.Response;
import org.eclipse.jetty.client.util.InputStreamResponseListener;
import org.eclipse.jetty.client.util.StringContentProvider;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpHeaderValue;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Client {
    private static final Logger log = LoggerFactory.getLogger(Client.class);
    private static final long DEFAULT_TIMEOUT_SEC = 15L;
    protected static final String MAX_REQUESTS_EXCEEDED_MESSAGE = "Client aborted http request because too many requests (Queue Size: %s) are already queued. Please check if request destination (%s) is running and reachable. If so, please try restarting C3 or wait until the request queue clears up. You can also increase request queue size with \"confluent.controlcenter.mds.client.max.requests.queued.per.destination\". Root Cause: %s";
    private final SslContextFactory sslContextFactory;
    private final HttpClient client;
    private final ObjectMapper objectMapper;
    private final long timeoutSec;

    public Client(SslContextFactory sslContextFactory, ObjectMapper objectMapper, long timeoutSec, ControlCenterConfig controlCenterConfig) {
        Preconditions.checkNotNull((Object)objectMapper);
        this.sslContextFactory = sslContextFactory;
        this.objectMapper = objectMapper;
        this.timeoutSec = timeoutSec;
        this.client = this.sslContextFactory == null ? new HttpClient() : new HttpClient(this.sslContextFactory);
        try {
            this.configureClient(controlCenterConfig);
            this.client.start();
        }
        catch (Exception e) {
            log.error("exception starting HttpClient", (Throwable)e);
        }
    }

    private void configureClient(ControlCenterConfig controlCenterConfig) {
        Integer customMaxRequestsQueuedPerDestination = controlCenterConfig.getInt("confluent.controlcenter.mds.client.max.requests.queued.per.destination");
        Integer customIdleTimeout = controlCenterConfig.getInt("confluent.controlcenter.mds.client.idle.timeout");
        if (customMaxRequestsQueuedPerDestination != null) {
            this.client.setMaxRequestsQueuedPerDestination(customMaxRequestsQueuedPerDestination.intValue());
        }
        if (customIdleTimeout != null) {
            this.client.setIdleTimeout((long)customIdleTimeout.intValue());
        }
    }

    public <O> O makeRequestNoBody(String url, HttpMethod httpMethod, HttpCredential credential, TypeReference<O> returnTypeRef) {
        return Client.makeRequestNoBody(this.client, this.objectMapper, this.timeoutSec, url, httpMethod, credential, returnTypeRef);
    }

    public <O> O makeRequestNoBody(String url, HttpMethod httpMethod, HttpCredential credential, TypeReference<O> returnTypeRef, Set<Integer> allowedStatusCodes) {
        return Client.makeRequestNoBody(this.client, this.objectMapper, this.timeoutSec, url, httpMethod, credential, returnTypeRef, allowedStatusCodes);
    }

    @VisibleForTesting
    static <O> O makeRequestNoBody(HttpClient httpClient, ObjectMapper objectMapper, long timeoutSec, String url, HttpMethod httpMethod, HttpCredential credential, TypeReference<O> returnTypeRef) {
        return Client.makeRequestNoBody(httpClient, objectMapper, timeoutSec, url, httpMethod, credential, returnTypeRef, (Set<Integer>)ImmutableSet.of((Object)200));
    }

    @VisibleForTesting
    static <O> O makeRequestNoBody(HttpClient httpClient, ObjectMapper objectMapper, long timeoutSec, String url, HttpMethod httpMethod, HttpCredential credential, TypeReference<O> returnTypeRef, Set<Integer> allowedStatusCodes) {
        try {
            InputStreamResponseListener listener = new InputStreamResponseListener();
            Client.prepareRequest(httpClient, url, httpMethod, credential).send((Response.CompleteListener)listener);
            return Client.parseResponse(objectMapper, timeoutSec, url, returnTypeRef, allowedStatusCodes, listener);
        }
        catch (IOException | InterruptedException | ExecutionException | TimeoutException e) {
            int maxRequests = httpClient.getMaxRequestsQueuedPerDestination();
            if (Client.maxRequestsExceeded(e, maxRequests)) {
                throw new RuntimeException(String.format(MAX_REQUESTS_EXCEEDED_MESSAGE, maxRequests, url, Client.rootCauseMessageFrom(e)), e.getCause());
            }
            throw new RuntimeException(e);
        }
    }

    public <I, O> O makeRequestWithContent(String url, HttpMethod httpMethod, HttpCredential credential, MimeTypes.Type contentType, I content, TypeReference<O> returnTypeRef) {
        return Client.makeRequestWithContent(this.client, this.objectMapper, this.timeoutSec, url, httpMethod, credential, contentType, content, returnTypeRef, (Set<Integer>)ImmutableSet.of((Object)200));
    }

    public <I, O> O makeRequestWithContent(String url, HttpMethod httpMethod, HttpCredential credential, MimeTypes.Type contentType, I content, TypeReference<O> returnTypeRef, Set<Integer> allowedStatusCodes) {
        return Client.makeRequestWithContent(this.client, this.objectMapper, this.timeoutSec, url, httpMethod, credential, contentType, content, returnTypeRef, allowedStatusCodes);
    }

    @VisibleForTesting
    static <I, O> O makeRequestWithContent(HttpClient httpClient, ObjectMapper objectMapper, long timeoutSec, String url, HttpMethod httpMethod, HttpCredential credential, MimeTypes.Type contentType, I content, TypeReference<O> returnTypeRef, Set<Integer> allowedStatusCodes) {
        try {
            InputStreamResponseListener listener = new InputStreamResponseListener();
            Client.prepareRequest(httpClient, url, httpMethod, credential).content((ContentProvider)new StringContentProvider(contentType.toString(), Client.serializeObject(objectMapper, content), StandardCharsets.UTF_8)).send((Response.CompleteListener)listener);
            return Client.parseResponse(objectMapper, timeoutSec, url, returnTypeRef, allowedStatusCodes, listener);
        }
        catch (IOException | InterruptedException | ExecutionException | TimeoutException e) {
            int maxRequests = httpClient.getMaxRequestsQueuedPerDestination();
            if (Client.maxRequestsExceeded(e, maxRequests)) {
                throw new RuntimeException(String.format(MAX_REQUESTS_EXCEEDED_MESSAGE, maxRequests, url, Client.rootCauseMessageFrom(e)), e.getCause());
            }
            throw new RuntimeException(e);
        }
    }

    private static String rootCauseMessageFrom(Exception exception) {
        if (exception == null) {
            return "";
        }
        return exception.getCause() == null ? exception.getMessage() : exception.getCause().getMessage();
    }

    private static boolean maxRequestsExceeded(Exception e, int maxRequests) {
        return e.getCause() instanceof RejectedExecutionException && e.getCause().getMessage().contains(String.format(" %d ", maxRequests));
    }

    @VisibleForTesting
    static Request prepareRequest(HttpClient httpClient, String url, HttpMethod httpMethod, HttpCredential credential) {
        Request request = httpClient.newRequest(url).method(httpMethod);
        if (!httpMethod.isIdempotent()) {
            request = request.header(HttpHeader.CONNECTION, HttpHeaderValue.CLOSE.asString());
        }
        if (credential != null) {
            request = request.header(HttpHeader.AUTHORIZATION, credential.authorizationHeaderValue());
        }
        return request;
    }

    @VisibleForTesting
    static <O> O parseResponse(ObjectMapper objectMapper, long timeoutSec, String url, TypeReference<O> returnTypeRef, Set<Integer> allowedStatusCodes, InputStreamResponseListener listener) throws InterruptedException, TimeoutException, ExecutionException, IOException {
        Response response = listener.get(timeoutSec, TimeUnit.SECONDS);
        if (allowedStatusCodes.contains(response.getStatus())) {
            try (InputStream responseContent = listener.getInputStream();){
                O o = Client.deserializeObject(objectMapper, responseContent, returnTypeRef);
                return o;
            }
        }
        log.error("expected response code from {} to be one of {}, but was {}", new Object[]{url, allowedStatusCodes, response.getStatus()});
        throw new WebApplicationException("unexpected response code from " + url, Response.Status.BAD_GATEWAY);
    }

    @VisibleForTesting
    static <T> String serializeObject(ObjectMapper objectMapper, T object) throws IOException {
        if (object instanceof String) {
            return (String)object;
        }
        return objectMapper.writeValueAsString(object);
    }

    @VisibleForTesting
    static <T> T deserializeObject(ObjectMapper objectMapper, InputStream inputStream, TypeReference<T> typeRef) throws IOException {
        if (typeRef.getType().equals(String.class)) {
            return (T)CharStreams.toString((Readable)new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        }
        if (typeRef.getType().equals(Void.class)) {
            return null;
        }
        return (T)objectMapper.readValue(inputStream, typeRef);
    }

    public static String urlEncode(String s) {
        try {
            return UriUtil.encode((String)s);
        }
        catch (UnsupportedEncodingException e) {
            throw new InternalServerErrorException();
        }
    }

    public static <T> T makeRequestWithRetries(Function<String, T> makeRequest, List<String> urls, int retriesEach, String errorMessage) {
        ArrayList<String> shuffledUrls = new ArrayList<String>(urls);
        Collections.shuffle(shuffledUrls);
        Throwable lastException = null;
        for (int r = 0; r < retriesEach; ++r) {
            for (String url : shuffledUrls) {
                try {
                    return makeRequest.apply(url);
                }
                catch (Exception e) {
                    lastException = e;
                    log.warn("failed to make request to {}", (Object)url);
                }
            }
        }
        if (lastException != null) {
            if (log.isDebugEnabled()) {
                log.warn("failed to make request", lastException);
            } else {
                log.warn("failed to make request: {}", (Object)lastException.getMessage());
            }
        }
        throw new WebApplicationException(errorMessage, Response.Status.BAD_GATEWAY);
    }

    @VisibleForTesting
    HttpClient getClient() {
        return this.client;
    }
}

