/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.controlcenter.license;

import com.google.inject.Inject;
import io.confluent.controlcenter.ControlCenterConfig;
import io.confluent.controlcenter.license.LicenseHolder;
import io.confluent.controlcenter.license.LicenseReader;
import io.confluent.license.InvalidLicenseException;
import io.confluent.license.License;
import io.confluent.license.LicenseManager;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.apache.commons.codec.binary.Base64;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LicenseService {
    private static final Logger log = LoggerFactory.getLogger(LicenseService.class);
    private final LicenseHolder licenseHolder;
    private final LicenseManager licenseManager;
    private final ControlCenterConfig controlCenterConfig;

    @Inject
    public LicenseService(LicenseHolder licenseHolder, ControlCenterConfig controlCenterConfig) {
        this.licenseHolder = Objects.requireNonNull(licenseHolder, "licenseHolder cannot be null");
        this.controlCenterConfig = Objects.requireNonNull(controlCenterConfig, "controlCenterConfig cannot be null");
        this.licenseManager = licenseHolder.getLicenseManager();
    }

    public long getExpiration() {
        return Optional.ofNullable(this.licenseHolder.getCurrentLicense()).map(License::expirationDate).map(Date::getTime).orElseGet(() -> {
            log.warn("No current license or expiration date available");
            return Long.MIN_VALUE;
        });
    }

    public String getAudience() {
        return Optional.ofNullable(this.licenseHolder.getCurrentLicense()).map(License::audience).filter(audience -> !audience.isEmpty()).map(audience -> (String)audience.get(0)).orElse(null);
    }

    public String getLicenseType() throws MalformedClaimException, InvalidJwtException {
        License currentLicense = this.licenseHolder.getCurrentLicense();
        if (currentLicense == null) {
            log.warn("No current license available for license type check");
            return null;
        }
        return currentLicense.type().toString();
    }

    public String getPayload() {
        License currentLicense = this.licenseHolder.getCurrentLicense();
        if (currentLicense == null) {
            log.warn("No current license available for payload extraction");
            return null;
        }
        String[] licenseArr = currentLicense.serializedForm().split("\\.");
        if (licenseArr.length < 2) {
            log.warn("Invalid license format - insufficient parts");
            return null;
        }
        Base64 base64Url = new Base64(true);
        try {
            return new String(base64Url.decode(licenseArr[1]), StandardCharsets.UTF_8);
        }
        catch (Throwable t) {
            log.warn("Could not convert decoded license body to String", t);
            return null;
        }
    }

    public String getString() {
        if (!this.controlCenterConfig.getBoolean("confluent.controlcenter.license.manager.enable").booleanValue()) {
            return null;
        }
        License currentLicense = this.licenseHolder.getCurrentLicense();
        if (currentLicense == null) {
            log.warn("No current license available for string extraction");
            return null;
        }
        return currentLicense.serializedForm();
    }

    public List<License> registerAndValidateLicense(String licenseString) throws InvalidLicenseException {
        if (licenseString == null || licenseString.trim().isEmpty()) {
            log.warn("Cannot set null or empty license string");
            throw new InvalidLicenseException("License string cannot be null or empty");
        }
        return LicenseReader.readAndRegisterFromString(licenseString, this.licenseManager);
    }

    public List<License> getAllLicenses() {
        return this.licenseHolder.getLicenses();
    }

    public List<License> getLatestLicenses() {
        return LicenseReader.readAndRegisterFromString("", this.licenseManager);
    }

    public License getCurrentLicense() {
        return this.licenseHolder.getCurrentLicense();
    }
}

