/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.controlcenter.tools;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.hash.Hashing;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.inject.Injector;
import com.google.inject.Key;
import com.google.inject.TypeLiteral;
import com.google.protobuf.ProtocolStringList;
import io.confluent.command.cluster_metadata.CommandClusterMetadata;
import io.confluent.command.kafka.CommandStore;
import io.confluent.command.record.Command;
import io.confluent.controlcenter.BootstrapClientConfigSupplier;
import io.confluent.controlcenter.ControlCenterConfig;
import io.confluent.controlcenter.StaticClusterConfigSupplier;
import io.confluent.controlcenter.command.CommandModule;
import io.confluent.controlcenter.rest.ControlCenterRestModule;
import io.confluent.controlcenter.util.ConfigUtils;
import io.confluent.controlcenter.util.InjectorFactory;
import io.confluent.rbacapi.entities.HostInfo;
import io.confluent.security.authorizer.Scope;
import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.kafka.common.security.auth.SecurityProtocol;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.state.KeyValueIterator;
import org.apache.kafka.streams.state.ReadOnlyKeyValueStore;
import org.apache.log4j.Appender;
import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.PatternLayout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterDataExporter {
    private static final Logger log = LoggerFactory.getLogger(ClusterDataExporter.class);
    private static final String CONFIG_OPT = "config";
    private static final String FILE_PATH_OPT = "outfile";
    private static final String KAFKA_CLUSTER_DEFAULT_DISPLAY_ID = "default";
    private static final String KAFKA_CLUSTER_DEFAULT_DISPLAY_NAME = "Unknown";

    public static void main(String[] args) throws Exception {
        BasicConfigurator.configure((Appender)new ConsoleAppender((Layout)new PatternLayout(), "System.err"));
        CommandLine cmdLine = ClusterDataExporter.getCmdLine(args);
        ControlCenterConfig config = new ControlCenterConfig(cmdLine.getOptionValue(CONFIG_OPT));
        String path = cmdLine.getOptionValue(FILE_PATH_OPT).trim();
        try (CommandTopicClusters commandTopicClusters = new CommandTopicClusters(config);){
            ClusterDataExporter.writeJsonFile(commandTopicClusters.getAllKafkaClusters(), path);
        }
        System.out.println("Done\n----------------------------------------------------------------------------------\nNote: for all the clusters in the exported JSON, please check and update 'protocol' property value.");
    }

    private static void writeJsonFile(List<ClusterInfo> clusterInfo, String path) {
        try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(path, true), StandardCharsets.UTF_8));){
            ObjectMapper mapper = new ObjectMapper();
            String jsonString = mapper.writeValueAsString(clusterInfo);
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            JsonParser jp = new JsonParser();
            JsonElement je = jp.parse(jsonString);
            String prettyJsonString = gson.toJson(je);
            out.write(prettyJsonString);
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        }
    }

    private static CommandLine getCmdLine(String[] args) throws ParseException {
        Options options = new Options().addOption(Option.builder((String)FILE_PATH_OPT).hasArg().required(true).longOpt(FILE_PATH_OPT).desc("Output Json file to export the cluster information").build());
        Options completeOptions = new Options();
        options.getOptions().forEach(arg_0 -> ((Options)completeOptions).addOption(arg_0));
        completeOptions.addOption(Option.builder((String)CONFIG_OPT).hasArg().required(true).longOpt(CONFIG_OPT).desc("C3 Configuration file's path and name.").build());
        DefaultParser parser = new DefaultParser();
        try {
            return parser.parse(completeOptions, args);
        }
        catch (ParseException e) {
            HelpFormatter formatter = new HelpFormatter();
            System.out.println(e.getMessage());
            formatter.printHelp("control-center-export --cluster props_file", "", options, "", true);
            System.exit(1);
            return null;
        }
    }

    static class CommandTopicClusters
    implements AutoCloseable {
        ControlCenterConfig config;
        CommandStore commandStore;
        ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage> clusterMetadataStore;
        Map<String, String> securityProtocolMap;

        CommandTopicClusters(ControlCenterConfig config) {
            Injector injector = InjectorFactory.createInjectorForScripts(config);
            this.config = config;
            this.commandStore = this.getCommandStore(injector, config);
            this.clusterMetadataStore = this.getClusterMetadataStore(injector);
            this.securityProtocolMap = this.getSecurityProtocolMap(injector);
        }

        public List<ClusterInfo> getAllKafkaClusters() {
            ArrayList<ClusterInfo> clusters = new ArrayList<ClusterInfo>();
            try (KeyValueIterator iter = this.clusterMetadataStore.all();){
                while (iter.hasNext()) {
                    KeyValue entry = (KeyValue)iter.next();
                    CommandClusterMetadata.ClusterMetadata clusterMetadata = ((Command.CommandMessage)entry.value).getClusterMetadata();
                    if (!clusterMetadata.hasKafka() || !this.validCluster(clusterMetadata, ((Command.CommandKey)entry.key).getGuid())) continue;
                    ProtocolStringList actualBootstrapServers = clusterMetadata.getKafka().getBootstrapServersList();
                    List<HostInfo> hosts = this.getHostList((List<String>)actualBootstrapServers);
                    SecurityProtocol securityProtocol = this.getSecurityProtocol((List<String>)actualBootstrapServers);
                    ClusterInfo cluster = new ClusterInfo(clusterMetadata.getDisplayName(), Scope.kafkaClusterScope((String)((Command.CommandKey)entry.key).getGuid()), hosts, securityProtocol);
                    clusters.add(cluster);
                }
            }
            return clusters;
        }

        @Override
        public void close() {
            this.commandStore.close(true);
        }

        private ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage> getClusterMetadataStore(Injector injector) {
            TypeLiteral<ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>> literal = new TypeLiteral<ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>>(){};
            return (ReadOnlyKeyValueStore)injector.getInstance(Key.get((TypeLiteral)literal, CommandModule.ExporterClusterMetadataStore.class));
        }

        private CommandStore getCommandStore(Injector injector, ControlCenterConfig config) {
            CommandStore commandStore = (CommandStore)injector.getInstance(Key.get(CommandStore.class, CommandModule.ExporterCommandStore.class));
            try {
                commandStore.start(config.getLong("confluent.controlcenter.command.streams.start.timeout").longValue());
            }
            catch (InterruptedException | TimeoutException e) {
                System.out.println(e.getMessage());
                System.exit(-1);
            }
            return commandStore;
        }

        @VisibleForTesting
        Map<String, String> getSecurityProtocolMap(Injector injector) {
            BootstrapClientConfigSupplier bootstrapClusterConfigSupplier = (BootstrapClientConfigSupplier)injector.getInstance(BootstrapClientConfigSupplier.class);
            StaticClusterConfigSupplier additionalClustersConfigSupplier = (StaticClusterConfigSupplier)injector.getInstance(Key.get(StaticClusterConfigSupplier.class, ControlCenterRestModule.KafkaClusterConfigSupplier.class));
            ImmutableMap.Builder securityProtocolMap = ImmutableMap.builder();
            Consumer<Map> configConsumer = config -> {
                List<String> expectedBootstrapServers = ConfigUtils.getList(config, "bootstrap.servers");
                String securityProtocol = (String)config.get("security.protocol");
                if (CollectionUtils.isNotEmpty(expectedBootstrapServers) && !Strings.isNullOrEmpty((String)securityProtocol)) {
                    String expectedHashId = this.hash(expectedBootstrapServers);
                    securityProtocolMap.put((Object)expectedHashId, (Object)securityProtocol);
                }
            };
            configConsumer.accept((Map)bootstrapClusterConfigSupplier.get());
            additionalClustersConfigSupplier.getConfigurations().values().forEach(configConsumer);
            return securityProtocolMap.build();
        }

        private List<HostInfo> getHostList(List<String> bootstrapServers) {
            return bootstrapServers.stream().map(h -> {
                String[] host = h.split(":");
                return new HostInfo(host[0], Integer.parseInt(host[1]));
            }).collect(Collectors.toList());
        }

        @VisibleForTesting
        SecurityProtocol getSecurityProtocol(List<String> actualBootstrapServers) {
            String actualHashId = this.hash(new ArrayList<String>(actualBootstrapServers));
            String securityProtocol = this.securityProtocolMap.getOrDefault(actualHashId, null);
            return Strings.isNullOrEmpty((String)securityProtocol) ? SecurityProtocol.SASL_PLAINTEXT : SecurityProtocol.forName((String)securityProtocol);
        }

        @VisibleForTesting
        void setSecurityProtocolMap(Map<String, String> securityProtocolMap) {
            this.securityProtocolMap = securityProtocolMap;
        }

        private boolean validCluster(CommandClusterMetadata.ClusterMetadata clusterMetadata, String clusterID) {
            return clusterID != null && !clusterID.isEmpty() && !clusterID.equals(ClusterDataExporter.KAFKA_CLUSTER_DEFAULT_DISPLAY_ID) && clusterMetadata.getKafka().getBootstrapServersList().size() > 0 && !clusterMetadata.getDisplayName().equals(ClusterDataExporter.KAFKA_CLUSTER_DEFAULT_DISPLAY_NAME);
        }

        private String hash(List<String> bootstrapServers) {
            Collections.sort(bootstrapServers);
            return Hashing.sha256().hashString((CharSequence)ConfigUtils.toString(bootstrapServers), StandardCharsets.UTF_8).toString();
        }
    }

    private static class ClusterInfo {
        public final String clusterName;
        public final Scope scope;
        public final List<HostInfo> hosts;
        public final SecurityProtocol securityProtocol;

        @JsonCreator
        ClusterInfo(@JsonProperty(value="clusterName") String clusterName, @JsonProperty(value="scope") Scope scope, @JsonProperty(value="hosts") List<HostInfo> hosts, @JsonProperty(value="protocol") SecurityProtocol securityProtocol) {
            this.clusterName = clusterName;
            this.scope = scope;
            ArrayList<HostInfo> temp = new ArrayList<HostInfo>(hosts);
            Collections.sort(temp);
            this.hosts = temp;
            this.securityProtocol = securityProtocol;
        }

        @JsonProperty(value="protocol")
        private SecurityProtocol getSecurityProtocol() {
            return this.securityProtocol;
        }
    }
}

