/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.multitenant;

import io.confluent.kafka.multitenant.SslCertificateManager;
import io.confluent.kafka.multitenant.Utils;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.AlterConfigsOptions;
import org.apache.kafka.clients.admin.ConfluentAdmin;
import org.apache.kafka.clients.admin.MockAdminClient;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.ArgumentMatchers;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.exceptions.base.MockitoAssertionError;
import org.mockito.verification.VerificationMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslCertificateManagerTest {
    private static final long TEST_MAX_WAIT_MS = TimeUnit.SECONDS.toMillis(60L);
    private static final String SSL_CERTS_DIR = "mnt/sslcerts/";
    private static final String DATA_DIR = "..data";
    private static final String BROKER_ID = "0";
    private static final URL TEST_SSL_CERTS_MAY = SslCertificateManagerTest.class.getResource("/cert_exp_may");
    private static final URL TEST_SSL_CERTS_AUG = SslCertificateManagerTest.class.getResource("/cert_exp_aug");
    private static final URL TEST_ROOT = SslCertificateManagerTest.class.getResource("/");
    private ConfluentAdmin mockAdminClient;
    private SslCertificateManager sslCache;
    private String sslCertsPath;
    private Path tempDir;
    private static MockedStatic<LoggerFactory> loggerFactory;
    private static Logger mockLog;

    @BeforeAll
    public static void beforeAll() {
        loggerFactory = Mockito.mockStatic(LoggerFactory.class);
        mockLog = (Logger)Mockito.mock(Logger.class);
        loggerFactory.when(() -> LoggerFactory.getLogger(SslCertificateManager.class)).thenReturn((Object)mockLog);
    }

    @AfterAll
    public static void afterAll() {
        loggerFactory.close();
    }

    @BeforeEach
    public void setUp() throws Exception {
        this.tempDir = TestUtils.tempDirectory().toPath();
        System.out.println("root resource: " + TEST_ROOT.getPath());
        Node node = new Node(0, "localhost", 9092);
        this.sslCertsPath = this.tempDir.toRealPath(new LinkOption[0]) + "/" + SSL_CERTS_DIR + "spec.json";
        this.mockAdminClient = (ConfluentAdmin)Mockito.spy((Object)new MockAdminClient(Collections.singletonList(node), node));
        this.sslCache = new SslCertificateManager((Object)BROKER_ID, (Object)this.sslCertsPath, this.mockAdminClient, Arrays.asList("EXTERNAL"));
    }

    @AfterEach
    public void teardown() {
        this.sslCache.shutdown();
        this.sslCache.close();
    }

    @Test
    public void testAdminClientInvokedAfterCertificateSync() throws Exception {
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
    }

    @Test
    public void testCertsCouldNotBeParsed() {
        try (MockedStatic certFactory = Mockito.mockStatic(CertificateFactory.class);
             MockedStatic ksFactory = Mockito.mockStatic(KeyStore.class);){
            CertificateException ce = new CertificateException("something bad");
            certFactory.when(() -> CertificateFactory.getInstance("X509")).thenThrow(new Throwable[]{ce});
            KeyStoreException kse = new KeyStoreException("something worse");
            ksFactory.when(() -> KeyStore.getInstance("PKCS12")).thenThrow(new Throwable[]{kse});
            Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
            Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
            this.sslCache.loadSslCertFiles();
            ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
            certFactory.verify(() -> CertificateFactory.getInstance((String)ArgumentMatchers.any()), Mockito.times((int)1));
            ksFactory.verify(() -> KeyStore.getInstance((String)ArgumentMatchers.any()), Mockito.times((int)1));
            ((Logger)Mockito.verify((Object)mockLog, (VerificationMode)Mockito.times((int)1))).error(ArgumentMatchers.contains((String)"X509"), (Throwable)ArgumentMatchers.eq((Object)ce));
            ((Logger)Mockito.verify((Object)mockLog, (VerificationMode)Mockito.times((int)1))).error(ArgumentMatchers.contains((String)"PKCS"), (Throwable)ArgumentMatchers.eq((Object)kse));
        }
        catch (Exception e) {
            Assertions.fail((Throwable)e);
        }
    }

    @Test
    public void testDynamicConfigsAfterCertificateSyncOneListener() throws Exception {
        this.testDynamicConfigsAfterCertificateSync(Arrays.asList("EXTERNAL"));
    }

    @Test
    public void testDynamicConfigsAfterCertificateSyncTwoListeners() throws Exception {
        this.testDynamicConfigsAfterCertificateSync(Arrays.asList("EXTERNAL", "EXTERNAL_BACKCHANNEL"));
    }

    private void testDynamicConfigsAfterCertificateSync(List<String> sslListenerNames) throws Exception {
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        this.sslCache = new SslCertificateManager((Object)BROKER_ID, (Object)this.sslCertsPath, this.mockAdminClient, sslListenerNames);
        this.sslCache.loadSslCertFiles();
        ArgumentCaptor configsCaptor = ArgumentCaptor.forClass(Map.class);
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)configsCaptor.capture(), (AlterConfigsOptions)ArgumentMatchers.any());
        Map configs = (Map)configsCaptor.getValue();
        HashSet<String> configNames = new HashSet<String>();
        if (sslListenerNames.isEmpty()) {
            configNames.add("listener.name.external.ssl.keystore.location");
            configNames.add("listener.name.external.ssl.keystore.type");
        } else {
            sslListenerNames.forEach(ln -> {
                String configPrefix = new ListenerName(ln).configPrefix();
                configNames.add(configPrefix + "ssl.keystore.location");
                configNames.add(configPrefix + "ssl.keystore.type");
            });
        }
        Set alterConfigNames = configs.values().stream().flatMap(alterConfigOps -> alterConfigOps.stream()).map(alterConfigOp -> alterConfigOp.configEntry().name()).collect(Collectors.toSet());
        Assertions.assertEquals(configNames, alterConfigNames);
    }

    @Test
    public void testAdminClientNotInvokedWithoutReadPermissionForCerts() throws Exception {
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        String fullchainFilepath = this.tempDir.toRealPath(new LinkOption[0]) + "/" + SSL_CERTS_DIR + "/" + DATA_DIR + "/fullchain.pem";
        String privkeyFilepath = this.tempDir.toRealPath(new LinkOption[0]) + "/" + SSL_CERTS_DIR + "/" + DATA_DIR + "/privkey.pem";
        Files.setPosixFilePermissions(Paths.get(fullchainFilepath, new String[0]), PosixFilePermissions.fromString("-wx-wx-wx"));
        Files.setPosixFilePermissions(Paths.get(privkeyFilepath, new String[0]), PosixFilePermissions.fromString("-wx-wx-wx"));
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)0))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
    }

    @Test
    public void testAdminClientNotInvokedWithoutSpecFile() throws Exception {
        String specfile = "spec.json";
        Utils.moveFile(specfile, TEST_SSL_CERTS_AUG, TEST_ROOT);
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)0))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.moveFile(specfile, TEST_ROOT, TEST_SSL_CERTS_AUG);
    }

    @Test
    public void testAdminClientNotInvokedWithoutPKCSCertificate() throws Exception {
        String pkcsfile = "pkcs.p12";
        Utils.moveFile(pkcsfile, TEST_SSL_CERTS_AUG, TEST_ROOT);
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)0))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.moveFile(pkcsfile, TEST_ROOT, TEST_SSL_CERTS_AUG);
    }

    @Test
    public void testAdminClientNotInvokedWithoutPrivkeyPemFile() throws Exception {
        String privkeyfile = "privkey.pem";
        Utils.moveFile(privkeyfile, TEST_SSL_CERTS_AUG, TEST_ROOT);
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_ROOT, SSL_CERTS_DIR);
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)0))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.moveFile(privkeyfile, TEST_ROOT, TEST_SSL_CERTS_AUG);
    }

    @Test
    public void testAdminClientNotInvokedWithoutFullchainPemFile() throws Exception {
        String fullchainfile = "fullchain.pem";
        Utils.moveFile(fullchainfile, TEST_SSL_CERTS_AUG, TEST_ROOT);
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)0))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.moveFile(fullchainfile, TEST_ROOT, TEST_SSL_CERTS_AUG);
    }

    private boolean verifyIncerementalAlterConfigsCalls(int wantedNumberOfInvocations) {
        try {
            ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)wantedNumberOfInvocations))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
            return true;
        }
        catch (MockitoAssertionError e) {
            return false;
        }
    }

    @Test
    public void testAdminClientInvocationOnIdenticalSslCertsSync() throws Exception {
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        this.sslCache.startWatching();
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
    }

    @Test
    public void testAdminClientInvocationOnDifferentSslCertsSync() throws Exception {
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        this.sslCache.startWatching();
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        TestUtils.waitForCondition(() -> this.verifyIncerementalAlterConfigsCalls(2), (long)TEST_MAX_WAIT_MS, (String)"Should call after sync");
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        TestUtils.waitForCondition(() -> this.verifyIncerementalAlterConfigsCalls(3), (long)TEST_MAX_WAIT_MS, (String)"Should call after sync");
    }

    @Test
    public void testWatchServiceDoesNotTerminateOnDirectoryDeletion() throws Exception {
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        this.sslCache.startWatching();
        this.sslCache.loadSslCertFiles();
        ((ConfluentAdmin)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.deleteFiles(this.tempDir, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempDir, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        TestUtils.waitForCondition(() -> this.verifyIncerementalAlterConfigsCalls(2), (long)TEST_MAX_WAIT_MS, (String)"Should call after sync");
    }
}

