/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafkarest.controllers;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import io.confluent.kafkarest.controllers.CloudUIInternalModule;
import io.confluent.kafkarest.controllers.KafkaConsumerProvider;
import io.confluent.kafkarest.controllers.SimpleConsumeManager;
import io.confluent.kafkarest.entities.v3.PartitionConsumeData;
import io.confluent.kafkarest.entities.v3.PartitionConsumeRecord;
import io.confluent.kafkarest.entities.v3.PartitionOffsetData;
import io.confluent.kafkarest.entities.v3.PartitionsOffsetsData;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.ws.rs.BadRequestException;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SimpleConsumeManagerImpl
implements SimpleConsumeManager {
    private static final Duration POLL_TIMEOUT = Duration.ofSeconds(1L);
    private static final int MAX_POLLS = 5;
    private static final int MAX_RESPONSE_BYTES = 0x1400000;
    private static final Logger log = LoggerFactory.getLogger(SimpleConsumeManagerImpl.class);
    private final KafkaConsumerProvider consumerProvider;
    private final Executor simpleConsumeExecutor;

    @Inject
    public SimpleConsumeManagerImpl(KafkaConsumerProvider consumerProvider, @CloudUIInternalModule.SimpleConsumeExecutor ExecutorService simpleConsumeExecutor) {
        this.consumerProvider = Objects.requireNonNull(consumerProvider);
        this.simpleConsumeExecutor = Objects.requireNonNull(simpleConsumeExecutor);
    }

    @Override
    public CompletableFuture<PartitionConsumeData> consumeFromPartition(@Nonnull TopicPartition partition, Boolean fromBeginning, Long timestamp, Long offset, Integer maxPollRecords, Integer fetchMaxBytes, Integer messageMaxBytes) {
        log.debug("consumeFromPartition(start): {}", (Object)System.nanoTime());
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(maxPollRecords, fetchMaxBytes);
        return CompletableFuture.supplyAsync(() -> {
            if (Boolean.TRUE.equals(fromBeginning)) {
                Map beginningOffsets = consumer.beginningOffsets(Collections.singleton(partition));
                return this.doConsume(consumer, partition, (Long)beginningOffsets.get(partition), messageMaxBytes);
            }
            if (Boolean.FALSE.equals(fromBeginning)) {
                Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
                return this.doConsume(consumer, partition, (Long)endOffsets.get(partition), messageMaxBytes);
            }
            if (timestamp != null) {
                Map<TopicPartition, Long> timestampMap = Collections.singletonMap(partition, timestamp);
                log.debug("offsetsForTimes(start): {}", (Object)System.nanoTime());
                Map offsetMap = consumer.offsetsForTimes(timestampMap);
                log.debug("offsetsForTimes(end): {}", (Object)System.nanoTime());
                OffsetAndTimestamp offsetData = (OffsetAndTimestamp)offsetMap.get(partition);
                if (offsetData == null) {
                    Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
                    return this.doConsume(consumer, partition, (Long)endOffsets.get(partition), messageMaxBytes);
                }
                return this.doConsume(consumer, partition, offsetData.offset(), messageMaxBytes);
            }
            Map beginningOffsets = consumer.beginningOffsets(Collections.singleton(partition));
            return this.doConsume(consumer, partition, Long.max((Long)beginningOffsets.get(partition), offset), messageMaxBytes);
        }, this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("consumeFromPartition(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }

    private KafkaConsumer<byte[], byte[]> getConsumerWithConfigs(Integer maxPollRecords, Integer fetchMaxBytes) {
        Properties consumerConfigOverride = new Properties();
        if (maxPollRecords != null) {
            consumerConfigOverride.put("max.poll.records", maxPollRecords);
        }
        if (fetchMaxBytes != null) {
            consumerConfigOverride.put("fetch.max.bytes", fetchMaxBytes);
        }
        try {
            return this.consumerProvider.getConsumer(consumerConfigOverride);
        }
        catch (ConfigException ce) {
            throw new BadRequestException(ce.getMessage());
        }
    }

    private <T> T handleConsumeResult(T result, Throwable error, KafkaConsumer<byte[], byte[]> consumer) {
        this.consumerProvider.releaseConsumer(consumer);
        if (error != null) {
            throw SimpleConsumeManagerImpl.processError(error);
        }
        return result;
    }

    @VisibleForTesting
    PartitionConsumeData doConsume(KafkaConsumer<byte[], byte[]> consumer, TopicPartition partition, long offset, Integer messageMaxBytes) {
        List singlePartitionRecords = Collections.emptyList();
        Set<TopicPartition> singlePartitionSet = Collections.singleton(partition);
        Map<Integer, Long> offsets = Collections.singletonMap(partition.partition(), offset);
        log.debug("endOffsets(start): {}", (Object)System.nanoTime());
        Map endOffsets = consumer.endOffsets(singlePartitionSet);
        log.debug("endOffsets(end): {}", (Object)System.nanoTime());
        if (SimpleConsumeManagerImpl.canConsumeFromOffsets(offsets, endOffsets)) {
            consumer.assign(singlePartitionSet);
            consumer.seek(partition, offset);
            log.debug("poll(start): {}", (Object)System.nanoTime());
            singlePartitionRecords = consumer.poll(POLL_TIMEOUT).records(partition);
            log.debug("poll(end): {}", (Object)System.nanoTime());
        }
        return SimpleConsumeManagerImpl.getPartitionConsumeData(singlePartitionRecords, partition, offsets, endOffsets, messageMaxBytes);
    }

    @VisibleForTesting
    static boolean canConsumeFromOffsets(Map<Integer, Long> offsets, Map<TopicPartition, Long> endOffsets) {
        if (offsets == null || offsets.isEmpty()) {
            return false;
        }
        for (Map.Entry<TopicPartition, Long> endOffset : endOffsets.entrySet()) {
            long offset;
            long partitionEndOffset = endOffset.getValue();
            if (partitionEndOffset < (offset = offsets.get(endOffset.getKey().partition()).longValue())) continue;
            return true;
        }
        return false;
    }

    private static PartitionConsumeData getPartitionConsumeData(List<ConsumerRecord<byte[], byte[]>> singlePartitionRecords, @Nonnull TopicPartition partition, Map<Integer, Long> offsets, @Nonnull Map<TopicPartition, Long> endOffsets, Integer messageMaxBytes) {
        int numRecords;
        int partitionId = partition.partition();
        long nextOffset = endOffsets.get(partition);
        if (offsets != null && offsets.containsKey(partitionId)) {
            nextOffset = Math.min(nextOffset, offsets.get(partitionId));
        }
        int n = numRecords = singlePartitionRecords != null ? singlePartitionRecords.size() : 0;
        if (numRecords > 0) {
            nextOffset = singlePartitionRecords.get(numRecords - 1).offset() + 1L;
        }
        return PartitionConsumeData.builder().setPartitionId(partitionId).setNextOffset(nextOffset).setRecords(PartitionConsumeRecord.listFromConsumerRecordList(singlePartitionRecords, messageMaxBytes)).build();
    }

    private static RuntimeException processError(Throwable t) {
        if (t instanceof CompletionException) {
            t = t.getCause();
        }
        if (t instanceof IllegalArgumentException || t instanceof IllegalStateException) {
            return new BadRequestException(t.getMessage());
        }
        if (t instanceof TimeoutException) {
            return new io.confluent.kafkarest.exceptions.TimeoutException(t);
        }
        if (t instanceof RuntimeException) {
            return (RuntimeException)t;
        }
        return new RuntimeException(t);
    }

    @Override
    public CompletableFuture<List<PartitionConsumeData>> consumeFromMultiplePartitions(@Nonnull String topicName, int numPartitions, Map<Integer, Long> offsets, Boolean fromBeginning, Long timestamp, Integer maxPollRecords, Integer fetchMaxBytes, Integer messageMaxBytes, boolean guaranteeProgressForEachPartition) {
        log.debug("consumeFromMultiPartitions(start): {}", (Object)System.nanoTime());
        Integer recordsLimit = maxPollRecords != null && maxPollRecords > numPartitions ? maxPollRecords : 2000;
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(recordsLimit, fetchMaxBytes);
        List<TopicPartition> partitions = SimpleConsumeManagerImpl.getPartitions(topicName, numPartitions, offsets);
        Supplier<List> consumeLambda = guaranteeProgressForEachPartition ? () -> this.doConsumeMultiGuaranteeProgress(consumer, partitions, offsets, fromBeginning, timestamp, maxPollRecords, messageMaxBytes) : () -> this.doConsumeMulti(consumer, partitions, offsets, fromBeginning, timestamp, messageMaxBytes);
        return CompletableFuture.supplyAsync(consumeLambda, this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("consumeFromMultiPartitions(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }

    private static List<TopicPartition> getPartitions(String topicName, int numPartitions, Map<Integer, Long> offsets) {
        if (offsets == null || offsets.isEmpty()) {
            return IntStream.range(0, numPartitions).mapToObj(partitionId -> new TopicPartition(topicName, partitionId)).collect(Collectors.toList());
        }
        return offsets.keySet().stream().map(partitionId -> new TopicPartition(topicName, partitionId.intValue())).collect(Collectors.toList());
    }

    @VisibleForTesting
    List<PartitionConsumeData> doConsumeMulti(KafkaConsumer<byte[], byte[]> consumer, List<TopicPartition> partitions, Map<Integer, Long> offsets, Boolean fromBeginning, Long timestamp, Integer messageMaxBytes) {
        log.debug("endOffsets(start): {}", (Object)System.nanoTime());
        Map endOffsets = consumer.endOffsets(partitions);
        log.debug("endOffsets(end): {}", (Object)System.nanoTime());
        offsets = SimpleConsumeManagerImpl.calculateInitialOffsets(consumer, partitions, offsets, endOffsets, fromBeginning, timestamp);
        ConsumerRecords partitionRecords = new ConsumerRecords(Collections.emptyMap());
        if (SimpleConsumeManagerImpl.canConsumeFromOffsets(offsets, endOffsets)) {
            consumer.assign(partitions);
            for (TopicPartition partition : partitions) {
                consumer.seek(partition, offsets.get(partition.partition()).longValue());
            }
            log.debug("poll(start): {}", (Object)System.nanoTime());
            partitionRecords = consumer.poll(POLL_TIMEOUT);
            log.debug("poll(end): {}", (Object)System.nanoTime());
        }
        ImmutableList.Builder result = new ImmutableList.Builder();
        for (TopicPartition partition : partitions) {
            List kafkaRecords = partitionRecords.records(partition);
            result.add((Object)SimpleConsumeManagerImpl.getPartitionConsumeData(kafkaRecords, partition, offsets, endOffsets, messageMaxBytes));
        }
        return result.build();
    }

    @VisibleForTesting
    public static int getOverallSerializedMessageSize(List<ConsumerRecord<byte[], byte[]>> kafkaRecords) {
        int result = 0;
        for (ConsumerRecord<byte[], byte[]> record : kafkaRecords) {
            result += record.serializedKeySize() + record.serializedValueSize();
        }
        return result;
    }

    @VisibleForTesting
    List<PartitionConsumeData> doConsumeMultiGuaranteeProgress(KafkaConsumer<byte[], byte[]> consumer, List<TopicPartition> partitions, Map<Integer, Long> offsets, Boolean fromBeginning, Long timestamp, Integer recordLimit, Integer messageMaxBytes) {
        log.debug("endOffsets(start): {}", (Object)System.nanoTime());
        Map endOffsets = consumer.endOffsets(partitions);
        log.debug("endOffsets(end): {}", (Object)System.nanoTime());
        offsets = SimpleConsumeManagerImpl.calculateInitialOffsets(consumer, partitions, offsets, endOffsets, fromBeginning, timestamp);
        HashMap<TopicPartition, List> partitionRecordsMap = new HashMap<TopicPartition, List>();
        if (SimpleConsumeManagerImpl.canConsumeFromOffsets(offsets, endOffsets)) {
            consumer.assign(partitions);
            for (TopicPartition partition : partitions) {
                consumer.seek(partition, offsets.get(partition.partition()).longValue());
            }
            int polls = 0;
            int responseSize = 0;
            while (partitionRecordsMap.size() < partitions.size() && polls < 5 && responseSize <= 0x1400000) {
                log.debug("poll#{} (start): {}", (Object)(++polls), (Object)System.nanoTime());
                ConsumerRecords partitionRecords = consumer.poll(POLL_TIMEOUT);
                log.debug("poll#{} (end): {}", (Object)polls, (Object)System.nanoTime());
                for (TopicPartition partition : partitions) {
                    List<ConsumerRecord<byte[], byte[]>> kafkaRecords = partitionRecords.records(partition);
                    if (kafkaRecords == null || kafkaRecords.size() <= 0) continue;
                    kafkaRecords = kafkaRecords.subList(0, Math.max(1, Math.min(kafkaRecords.size(), recordLimit)));
                    responseSize += SimpleConsumeManagerImpl.getOverallSerializedMessageSize(kafkaRecords);
                    recordLimit = recordLimit - kafkaRecords.size();
                    partitionRecordsMap.computeIfAbsent(partition, p -> new ArrayList()).addAll(kafkaRecords);
                }
                consumer.pause(partitionRecordsMap.keySet());
            }
        }
        ImmutableList.Builder result = new ImmutableList.Builder();
        for (TopicPartition partition : partitions) {
            List kafkaRecords = (List)partitionRecordsMap.get(partition);
            result.add((Object)SimpleConsumeManagerImpl.getPartitionConsumeData(kafkaRecords, partition, offsets, endOffsets, messageMaxBytes));
        }
        return result.build();
    }

    private static Map<Integer, Long> calculateInitialOffsets(KafkaConsumer<byte[], byte[]> consumer, List<TopicPartition> partitions, @Nullable Map<Integer, Long> offsets, Map<TopicPartition, Long> endOffsets, @Nullable Boolean fromBeginning, @Nullable Long timestamp) {
        Map<Integer, Long> initialOffsets;
        if (Boolean.TRUE.equals(fromBeginning)) {
            initialOffsets = consumer.beginningOffsets(partitions).entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), Map.Entry::getValue));
        } else if (Boolean.FALSE.equals(fromBeginning)) {
            initialOffsets = endOffsets.entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), Map.Entry::getValue));
        } else if (timestamp != null) {
            Map<TopicPartition, Long> timestampMap = partitions.stream().collect(Collectors.toMap(tp -> tp, tp -> timestamp));
            log.debug("offsetsForTimes(start): {}", (Object)System.nanoTime());
            Map offsetMap = consumer.offsetsForTimes(timestampMap);
            log.debug("offsetsForTimes(end): {}", (Object)System.nanoTime());
            initialOffsets = offsetMap.entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), e -> {
                if (e.getValue() != null) {
                    return ((OffsetAndTimestamp)e.getValue()).offset();
                }
                return (Long)endOffsets.get(e.getKey());
            }));
        } else {
            initialOffsets = offsets != null ? consumer.beginningOffsets(partitions).entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), e -> Long.max((Long)e.getValue(), (Long)offsets.get(((TopicPartition)e.getKey()).partition())))) : endOffsets.entrySet().stream().collect(Collectors.toMap(e -> ((TopicPartition)e.getKey()).partition(), Map.Entry::getValue));
        }
        return initialOffsets;
    }

    @Override
    public CompletableFuture<Long> getOffsetForPartition(@Nonnull TopicPartition partition, Boolean fromBeginning, Long timestamp) {
        log.debug("getOffsetForPartition(start): {}", (Object)System.nanoTime());
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(null, null);
        return CompletableFuture.supplyAsync(() -> {
            long nextOffset;
            if (Boolean.TRUE.equals(fromBeginning)) {
                Map beginningOffsets = consumer.beginningOffsets(Collections.singleton(partition));
                nextOffset = (Long)beginningOffsets.get(partition);
            } else if (Boolean.FALSE.equals(fromBeginning)) {
                Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
                nextOffset = (Long)endOffsets.get(partition);
            } else {
                Map<TopicPartition, Long> timestampMap = Collections.singletonMap(partition, Objects.requireNonNull(timestamp));
                log.debug("offsetsForTimes(start): {}", (Object)System.nanoTime());
                Map offsetMap = consumer.offsetsForTimes(timestampMap);
                log.debug("offsetsForTimes(end): {}", (Object)System.nanoTime());
                OffsetAndTimestamp offsetData = (OffsetAndTimestamp)offsetMap.get(partition);
                if (offsetData == null) {
                    Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
                    nextOffset = (Long)endOffsets.get(partition);
                } else {
                    nextOffset = offsetData.offset();
                }
            }
            return nextOffset;
        }, this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("getOffsetForPartition(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }

    @Override
    public CompletableFuture<PartitionsOffsetsData> getOffsetsForPartitions(@Nonnull String topicName, int numPartitions, Boolean fromBeginning, Long timestamp) {
        log.debug("getOffsetsForPartitions(start): {}", (Object)System.nanoTime());
        KafkaConsumer<byte[], byte[]> consumer = this.getConsumerWithConfigs(null, null);
        List<TopicPartition> partitions = SimpleConsumeManagerImpl.getPartitions(topicName, numPartitions, null);
        return CompletableFuture.supplyAsync(() -> {
            log.debug("beginningOffsets(start): {}", (Object)System.nanoTime());
            Map beginningOffsets = consumer.beginningOffsets((Collection)partitions);
            log.debug("beginningOffsets(end): {}", (Object)System.nanoTime());
            log.debug("endOffsets(start): {}", (Object)System.nanoTime());
            Map endOffsets = consumer.endOffsets((Collection)partitions);
            log.debug("endOffsets(end): {}", (Object)System.nanoTime());
            long totalRecords = partitions.stream().mapToLong(tp -> (Long)endOffsets.get(tp) - (Long)beginningOffsets.get(tp)).sum();
            Map<Integer, Long> offsetsMap = SimpleConsumeManagerImpl.calculateInitialOffsets(consumer, partitions, null, endOffsets, fromBeginning, timestamp);
            PartitionsOffsetsData result = new PartitionsOffsetsData(totalRecords);
            offsetsMap.forEach((partition, offset) -> result.addOffset(PartitionOffsetData.builder().setPartitionId((Integer)partition).setNextOffset((Long)offset).build()));
            return result;
        }, this.simpleConsumeExecutor).handle((result, error) -> {
            log.debug("getOffsetsForPartitions(end): {}", (Object)System.nanoTime());
            return this.handleConsumeResult((Object)result, (Throwable)error, consumer);
        });
    }
}

