/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.engine;

import com.google.common.util.concurrent.AbstractScheduledService;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.confluent.ksql.query.QueryRegistry;
import io.confluent.ksql.services.KafkaTopicClient;
import io.confluent.ksql.services.ServiceContext;
import io.confluent.ksql.util.KsqlConfig;
import io.confluent.ksql.util.QueryApplicationId;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.kafka.streams.StreamsConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransientQueryCleanupService
extends AbstractScheduledService {
    private static final Logger LOG = LoggerFactory.getLogger(TransientQueryCleanupService.class);
    private final Pattern internalTopicPrefixPattern;
    private final Pattern transientAppIdPattern;
    private final Set<String> queriesGuaranteedToBeRunningAtSomePoint;
    private final String stateDir;
    private final KafkaTopicClient topicClient;
    private final int initialDelay;
    private final int intervalPeriod;
    private Optional<Set<String>> localCommandsQueryAppIds;
    private QueryRegistry queryRegistry;
    private int numLeakedTopics;
    private int numLeakedStateDirs;
    private int numLeakedTopicsFailedToCleanUp;
    private int numLeakedStateDirsFailedToCleanUp;

    public TransientQueryCleanupService(ServiceContext serviceContext, KsqlConfig ksqlConfig) {
        String internalTopicPrefix = QueryApplicationId.buildInternalTopicPrefix((KsqlConfig)ksqlConfig, (boolean)false);
        this.internalTopicPrefixPattern = Pattern.compile(internalTopicPrefix);
        this.transientAppIdPattern = Pattern.compile(internalTopicPrefix + ".*_[0-9]\\d*_[0-9]\\d*");
        this.initialDelay = ksqlConfig.getInt("ksql.transient.query.cleanup.service.initial.delay.seconds");
        this.intervalPeriod = ksqlConfig.getInt("ksql.transient.query.cleanup.service.period.seconds");
        this.stateDir = ksqlConfig.getKsqlStreamConfigProps().getOrDefault("state.dir", StreamsConfig.configDef().defaultValues().get("state.dir")).toString();
        this.topicClient = serviceContext.getTopicClient();
        this.localCommandsQueryAppIds = Optional.empty();
        this.queriesGuaranteedToBeRunningAtSomePoint = new HashSet<String>();
        this.numLeakedTopics = 0;
        this.numLeakedStateDirs = 0;
    }

    protected void runOneIteration() {
        this.findAndDeleteLeakedTopics();
        this.findAndDeleteLeakedStateDirs();
    }

    public AbstractScheduledService.Scheduler scheduler() {
        return AbstractScheduledService.Scheduler.newFixedRateSchedule((long)this.initialDelay, (long)this.intervalPeriod, (TimeUnit)TimeUnit.SECONDS);
    }

    public void setQueryRegistry(QueryRegistry queryRegistry) {
        this.queryRegistry = queryRegistry;
    }

    @SuppressFBWarnings(value={"EI_EXPOSE_REP2"})
    public void setLocalCommandsQueryAppIds(Set<String> ids) {
        this.localCommandsQueryAppIds = Optional.of(ids);
    }

    private void findAndDeleteLeakedTopics() {
        try {
            List<String> leakedTopics = this.findLeakedTopics();
            this.numLeakedTopics = leakedTopics.size();
            LOG.info("Cleaning up {} leaked topics: {}", (Object)this.numLeakedTopics, leakedTopics);
            this.getTopicClient().deleteTopics(leakedTopics);
            this.numLeakedTopicsFailedToCleanUp = this.findLeakedTopics().size();
        }
        catch (Throwable t) {
            LOG.error("Failed to clean up topics with exception: " + t.getMessage(), t);
        }
    }

    private void findAndDeleteLeakedStateDirs() {
        try {
            List<String> leakedStateDirs = this.findLeakedStateDirs();
            this.numLeakedStateDirs = leakedStateDirs.size();
            LOG.info("Cleaning up {} leaked state directories: {}", (Object)this.numLeakedStateDirs, leakedStateDirs.stream().map(file -> this.stateDir + "/" + file).collect(Collectors.toList()));
            leakedStateDirs.forEach(this::deleteLeakedStateDir);
            this.numLeakedStateDirsFailedToCleanUp = this.findLeakedStateDirs().size();
        }
        catch (Throwable t) {
            LOG.error("Failed to clean up state directories with exception: " + t.getMessage(), t);
        }
    }

    private void deleteLeakedStateDir(String filename) {
        String path = this.stateDir + "/" + filename;
        Path pathName = Paths.get(path, new String[0]);
        try {
            Files.deleteIfExists(pathName);
        }
        catch (IOException e) {
            LOG.info("Transient Query Cleanup Service failed to delete leaked state directory: " + path, (Throwable)e);
        }
    }

    List<String> findLeakedTopics() {
        return this.getTopicClient().listTopicNames().stream().filter(this::isLeaked).collect(Collectors.toList());
    }

    List<String> findLeakedStateDirs() {
        return this.listAllStateFiles().stream().filter(this::isLeaked).collect(Collectors.toList());
    }

    List<String> listAllStateFiles() {
        File folder = new File(this.stateDir);
        File[] listOfFiles = folder.listFiles();
        if (listOfFiles == null) {
            return Collections.emptyList();
        }
        return Arrays.stream(listOfFiles).map(File::getName).collect(Collectors.toList());
    }

    boolean isLeaked(String resource) {
        if (this.foundInLocalCommands(resource)) {
            return true;
        }
        if (!this.internalTopicPrefixPattern.matcher(resource).find()) {
            return false;
        }
        if (!this.isCorrespondingQueryTerminated(resource)) {
            return false;
        }
        Matcher appIdMatcher = this.transientAppIdPattern.matcher(resource);
        if (appIdMatcher.find()) {
            return this.wasQueryGuaranteedToBeRunningAtSomePoint(appIdMatcher.group());
        }
        return false;
    }

    boolean isCorrespondingQueryTerminated(String appId) {
        return this.queryRegistry.getAllLiveQueries().stream().map(qm -> qm.getQueryId().toString()).noneMatch(appId::contains);
    }

    public void registerRunningQuery(String appId) {
        this.queriesGuaranteedToBeRunningAtSomePoint.add(appId);
    }

    boolean wasQueryGuaranteedToBeRunningAtSomePoint(String appId) {
        return this.queriesGuaranteedToBeRunningAtSomePoint.contains(appId);
    }

    boolean foundInLocalCommands(String resourceName) {
        return this.localCommandsQueryAppIds.map(strings -> strings.stream().anyMatch(resourceName::contains)).orElse(false);
    }

    KafkaTopicClient getTopicClient() {
        return this.topicClient;
    }

    String getStateDir() {
        return this.stateDir;
    }

    public int getNumLeakedTopics() {
        return this.numLeakedTopics;
    }

    public int getNumLeakedStateDirs() {
        return this.numLeakedStateDirs;
    }

    public int getNumLeakedTopicsFailedToCleanUp() {
        return this.numLeakedTopicsFailedToCleanUp;
    }

    public int getNumLeakedStateDirsFailedToCleanUp() {
        return this.numLeakedStateDirsFailedToCleanUp;
    }
}

