/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.RequestCompletionHandler;
import org.apache.kafka.clients.consumer.CommitFailedException;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.consumer.OffsetResetStrategy;
import org.apache.kafka.clients.consumer.RetriableCommitFailedException;
import org.apache.kafka.clients.consumer.internals.CommitRequestManager;
import org.apache.kafka.clients.consumer.internals.CoordinatorRequestManager;
import org.apache.kafka.clients.consumer.internals.NetworkClientDelegate;
import org.apache.kafka.clients.consumer.internals.OffsetCommitCallbackInvoker;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.message.OffsetCommitRequestData;
import org.apache.kafka.common.message.OffsetCommitResponseData;
import org.apache.kafka.common.message.OffsetFetchRequestData;
import org.apache.kafka.common.metrics.KafkaMetric;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.AbstractResponse;
import org.apache.kafka.common.requests.OffsetCommitRequest;
import org.apache.kafka.common.requests.OffsetCommitResponse;
import org.apache.kafka.common.requests.OffsetFetchRequest;
import org.apache.kafka.common.requests.OffsetFetchResponse;
import org.apache.kafka.common.requests.RequestHeader;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class CommitRequestManagerTest {
    private long retryBackoffMs = 100L;
    private long retryBackoffMaxMs = 1000L;
    private String consumerMetricGroupPrefix = "consumer";
    private static final String CONSUMER_COORDINATOR_METRICS = "consumer-coordinator-metrics";
    private Node mockedNode = new Node(1, "host1", 9092);
    private SubscriptionState subscriptionState;
    private LogContext logContext;
    private MockTime time;
    private CoordinatorRequestManager coordinatorRequestManager;
    private OffsetCommitCallbackInvoker offsetCommitCallbackInvoker;
    private Metrics metrics = new Metrics();
    private Properties props;
    private final int defaultApiTimeoutMs = 60000;

    @BeforeEach
    public void setup() {
        this.logContext = new LogContext();
        this.time = new MockTime(0L);
        this.subscriptionState = new SubscriptionState(new LogContext(), OffsetResetStrategy.EARLIEST);
        this.coordinatorRequestManager = (CoordinatorRequestManager)Mockito.mock(CoordinatorRequestManager.class);
        this.offsetCommitCallbackInvoker = (OffsetCommitCallbackInvoker)Mockito.mock(OffsetCommitCallbackInvoker.class);
        this.props = new Properties();
        this.props.put("auto.commit.interval.ms", (Object)100);
        this.props.put("key.deserializer", StringDeserializer.class);
        this.props.put("value.deserializer", StringDeserializer.class);
    }

    @Test
    public void testPollSkipIfCoordinatorUnknown() {
        CommitRequestManager commitRequestManger = this.create(false, 0L);
        this.assertPoll(false, 0, commitRequestManger);
        HashMap<TopicPartition, OffsetAndMetadata> offsets = new HashMap<TopicPartition, OffsetAndMetadata>();
        offsets.put(new TopicPartition("t1", 0), new OffsetAndMetadata(0L));
        commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), false);
        this.assertPoll(false, 0, commitRequestManger);
    }

    @Test
    public void testPollEnsureManualCommitSent() {
        CommitRequestManager commitRequestManger = this.create(false, 0L);
        this.assertPoll(0, commitRequestManger);
        HashMap<TopicPartition, OffsetAndMetadata> offsets = new HashMap<TopicPartition, OffsetAndMetadata>();
        offsets.put(new TopicPartition("t1", 0), new OffsetAndMetadata(0L));
        commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), false);
        this.assertPoll(1, commitRequestManger);
    }

    @Test
    public void testPollEnsureAutocommitSent() {
        TopicPartition tp = new TopicPartition("t1", 1);
        this.subscriptionState.assignFromUser(Collections.singleton(tp));
        this.subscriptionState.seek(tp, 100L);
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        this.assertPoll(0, commitRequestManger);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        List<NetworkClientDelegate.FutureCompletionHandler> pollResults = this.assertPoll(1, commitRequestManger);
        pollResults.forEach(v -> v.onComplete(this.mockOffsetCommitResponse("t1", 1, (short)1, Errors.NONE)));
        Assertions.assertEquals((double)0.03, (double)((Double)this.getMetric("commit-rate").metricValue()), (double)0.01);
        Assertions.assertEquals((Object)1.0, (Object)this.getMetric("commit-total").metricValue());
    }

    @Test
    public void testPollEnsureCorrectInflightRequestBufferSize() {
        CommitRequestManager commitManager = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        HashMap<TopicPartition, OffsetAndMetadata> offsets1 = new HashMap<TopicPartition, OffsetAndMetadata>();
        offsets1.put(new TopicPartition("test", 0), new OffsetAndMetadata(10L));
        offsets1.put(new TopicPartition("test", 1), new OffsetAndMetadata(20L));
        HashMap<TopicPartition, OffsetAndMetadata> offsets2 = new HashMap<TopicPartition, OffsetAndMetadata>();
        offsets2.put(new TopicPartition("test", 3), new OffsetAndMetadata(20L));
        offsets2.put(new TopicPartition("test", 4), new OffsetAndMetadata(20L));
        ArrayList<CompletableFuture> commitFutures = new ArrayList<CompletableFuture>();
        ArrayList<CompletableFuture> fetchFutures = new ArrayList<CompletableFuture>();
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        commitFutures.add(commitManager.addOffsetCommitRequest(offsets1, Optional.of(expirationTimeMs), false));
        fetchFutures.add(commitManager.addOffsetFetchRequest(Collections.singleton(new TopicPartition("test", 0)), expirationTimeMs));
        commitFutures.add(commitManager.addOffsetCommitRequest(offsets2, Optional.of(expirationTimeMs), false));
        fetchFutures.add(commitManager.addOffsetFetchRequest(Collections.singleton(new TopicPartition("test", 1)), expirationTimeMs));
        NetworkClientDelegate.PollResult result = commitManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)4, (int)result.unsentRequests.size());
        Assertions.assertTrue((boolean)result.unsentRequests.stream().anyMatch(r -> r.requestBuilder() instanceof OffsetCommitRequest.Builder));
        Assertions.assertTrue((boolean)result.unsentRequests.stream().anyMatch(r -> r.requestBuilder() instanceof OffsetFetchRequest.Builder));
        Assertions.assertFalse((boolean)commitManager.pendingRequests.hasUnsentRequests());
        Assertions.assertEquals((int)2, (int)commitManager.pendingRequests.inflightOffsetFetches.size());
        commitFutures.forEach(f -> f.complete(null));
        fetchFutures.forEach(f -> f.complete(null));
        Assertions.assertEquals((int)0, (int)commitManager.pendingRequests.inflightOffsetFetches.size());
    }

    @Test
    public void testPollEnsureEmptyPendingRequestAfterPoll() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), false);
        Assertions.assertEquals((int)1, (int)commitRequestManger.unsentOffsetCommitRequests().size());
        Assertions.assertEquals((int)1, (int)commitRequestManger.poll((long)this.time.milliseconds()).unsentRequests.size());
        Assertions.assertTrue((boolean)commitRequestManger.unsentOffsetCommitRequests().isEmpty());
        CommitRequestManagerTest.assertEmptyPendingRequests(commitRequestManger);
    }

    @Test
    public void testAsyncAutocommitNotRetriedAfterException() {
        long commitInterval = this.retryBackoffMs * 2L;
        CommitRequestManager commitRequestManger = this.create(true, commitInterval);
        TopicPartition tp = new TopicPartition("topic", 1);
        this.subscriptionState.assignFromUser(Collections.singleton(tp));
        this.subscriptionState.seek(tp, 100L);
        this.time.sleep(commitInterval);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        List<NetworkClientDelegate.FutureCompletionHandler> futures = this.assertPoll(1, commitRequestManger);
        futures.get(0).onComplete(this.mockOffsetCommitResponse("topic", 1, (short)1, Errors.COORDINATOR_LOAD_IN_PROGRESS));
        this.time.sleep(this.retryBackoffMs);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        this.assertPoll(0, commitRequestManger);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        this.time.sleep(commitInterval);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        futures = this.assertPoll(1, commitRequestManger);
        CommitRequestManagerTest.assertEmptyPendingRequests(commitRequestManger);
        futures.get(0).onComplete(this.mockOffsetCommitResponse("topic", 1, (short)1, Errors.NONE));
    }

    @ParameterizedTest
    @MethodSource(value={"offsetCommitExceptionSupplier"})
    public void testAutoCommitSyncRetriedAfterExpectedRetriableException(Errors error) {
        long commitInterval = this.retryBackoffMs * 2L;
        CommitRequestManager commitRequestManger = this.create(true, commitInterval);
        TopicPartition tp = new TopicPartition("topic", 1);
        this.subscriptionState.assignFromUser(Collections.singleton(tp));
        this.subscriptionState.seek(tp, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        this.time.sleep(commitInterval);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        CompletableFuture commitResult = commitRequestManger.maybeAutoCommitAllConsumedNow(Optional.of(expirationTimeMs), false);
        this.sendAndVerifyOffsetCommitRequestFailedAndMaybeRetried(commitRequestManger, error, commitResult);
        this.assertExceptionHandling(commitRequestManger, error, true);
    }

    @ParameterizedTest
    @MethodSource(value={"commitSyncExpectedExceptions"})
    public void testCommitSyncFailsWithExpectedException(Errors commitError, Class<? extends Exception> expectedException) {
        CommitRequestManager commitRequestManger = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        Long expirationTimeMs = this.time.milliseconds() + this.retryBackoffMs;
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.of(expirationTimeMs), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, commitError);
        TestUtils.assertFutureThrows(commitResult, expectedException);
    }

    private static Stream<Arguments> commitSyncExpectedExceptions() {
        return Stream.of(Arguments.of((Object[])new Object[]{Errors.FENCED_INSTANCE_ID, CommitFailedException.class}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_MEMBER_ID, CommitFailedException.class}), Arguments.of((Object[])new Object[]{Errors.OFFSET_METADATA_TOO_LARGE, Errors.OFFSET_METADATA_TOO_LARGE.exception().getClass()}), Arguments.of((Object[])new Object[]{Errors.INVALID_COMMIT_OFFSET_SIZE, Errors.INVALID_COMMIT_OFFSET_SIZE.exception().getClass()}), Arguments.of((Object[])new Object[]{Errors.GROUP_AUTHORIZATION_FAILED, Errors.GROUP_AUTHORIZATION_FAILED.exception().getClass()}), Arguments.of((Object[])new Object[]{Errors.CORRUPT_MESSAGE, KafkaException.class}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_SERVER_ERROR, KafkaException.class}));
    }

    @Test
    public void testOffsetCommitFailsWithCommitFailedExceptionIfUnknownMemberId() {
        long commitInterval = this.retryBackoffMs * 2L;
        CommitRequestManager commitRequestManger = this.create(true, commitInterval);
        TopicPartition tp = new TopicPartition("topic", 1);
        this.subscriptionState.assignFromUser(Collections.singleton(tp));
        this.subscriptionState.seek(tp, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        this.time.sleep(commitInterval);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        CompletableFuture commitResult = commitRequestManger.maybeAutoCommitAllConsumedNow(Optional.of(expirationTimeMs), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, Errors.UNKNOWN_MEMBER_ID);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        TestUtils.assertFutureThrows(commitResult, CommitFailedException.class);
    }

    @Test
    public void testOffsetCommitFailsWithCommitFailedExceptionIfStaleMemberEpochNotRetried() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.of(this.time.milliseconds() + 60000L), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, Errors.STALE_MEMBER_EPOCH);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        TestUtils.assertFutureThrows(commitResult, CommitFailedException.class);
    }

    @Test
    public void testCommitAsyncFailsWithRetriableOnStaleMemberEpoch() {
        CommitRequestManager commitRequestManger = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), true);
        this.completeOffsetCommitRequestWithError(commitRequestManger, Errors.STALE_MEMBER_EPOCH);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        TestUtils.assertFutureThrows(commitResult, RetriableCommitFailedException.class);
    }

    @Test
    public void testCommitAsyncFailsWithRetriableOnCoordinatorDisconnected() {
        CommitRequestManager commitRequestManager = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        CompletableFuture commitResult = commitRequestManager.addOffsetCommitRequest(offsets, Optional.empty(), true);
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        NetworkClientDelegate.UnsentRequest req = (NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0);
        ClientResponse response = this.mockOffsetCommitResponseDisconnected("topic", 1, (short)1, req);
        response.onComplete();
        Assertions.assertTrue((boolean)commitResult.isDone());
        TestUtils.assertFutureThrows(commitResult, RetriableCommitFailedException.class);
        this.assertCoordinatorDisconnect();
    }

    @Test
    public void testAutocommitEnsureOnlyOneInflightRequest() {
        TopicPartition t1p = new TopicPartition("topic1", 0);
        this.subscriptionState.assignFromUser(Collections.singleton(t1p));
        this.subscriptionState.seek(t1p, 100L);
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        List<NetworkClientDelegate.FutureCompletionHandler> futures = this.assertPoll(1, commitRequestManger);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        this.assertPoll(0, commitRequestManger);
        CommitRequestManagerTest.assertEmptyPendingRequests(commitRequestManger);
        futures.get(0).onComplete(this.buildOffsetCommitClientResponse(new OffsetCommitResponse(0, new HashMap())));
        this.assertPoll(1, commitRequestManger);
    }

    @Test
    public void testAutocommitInterceptorsInvoked() {
        TopicPartition t1p = new TopicPartition("topic1", 0);
        this.subscriptionState.assignFromUser(Collections.singleton(t1p));
        this.subscriptionState.seek(t1p, 100L);
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        List<NetworkClientDelegate.FutureCompletionHandler> futures = this.assertPoll(1, commitRequestManger);
        futures.get(0).onComplete(this.buildOffsetCommitClientResponse(new OffsetCommitResponse(0, new HashMap())));
        ((OffsetCommitCallbackInvoker)Mockito.verify((Object)this.offsetCommitCallbackInvoker)).enqueueInterceptorInvocation((Map)ArgumentMatchers.eq(Collections.singletonMap(t1p, new OffsetAndMetadata(100L))));
    }

    @Test
    public void testAutocommitInterceptorsNotInvokedOnError() {
        TopicPartition t1p = new TopicPartition("topic1", 0);
        this.subscriptionState.assignFromUser(Collections.singleton(t1p));
        this.subscriptionState.seek(t1p, 100L);
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        List<NetworkClientDelegate.FutureCompletionHandler> futures = this.assertPoll(1, commitRequestManger);
        futures.get(0).onComplete(this.buildOffsetCommitClientResponse(new OffsetCommitResponse(0, Collections.singletonMap(t1p, Errors.NETWORK_EXCEPTION))));
        ((OffsetCommitCallbackInvoker)Mockito.verify((Object)this.offsetCommitCallbackInvoker, (VerificationMode)Mockito.never())).enqueueInterceptorInvocation((Map)ArgumentMatchers.any());
    }

    @Test
    public void testAutoCommitEmptyOffsetsDoesNotGenerateRequest() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        CompletableFuture result = commitRequestManger.maybeAutoCommitAllConsumedAsync();
        Assertions.assertTrue((boolean)commitRequestManger.pendingRequests.unsentOffsetCommits.isEmpty());
        Assertions.assertTrue((boolean)result.isDone());
        Assertions.assertFalse((boolean)result.isCompletedExceptionally());
    }

    @Test
    public void testAutoCommitEmptyDoesNotLeaveInflightRequestFlagOn() {
        TopicPartition t1p = new TopicPartition("topic1", 0);
        this.subscriptionState.assignFromUser(Collections.singleton(t1p));
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        CompletableFuture result = commitRequestManger.maybeAutoCommitAllConsumedAsync();
        Assertions.assertTrue((boolean)result.isDone());
        Assertions.assertFalse((boolean)result.isCompletedExceptionally());
        this.subscriptionState.seek(t1p, 100L);
        this.time.sleep(100L);
        commitRequestManger.updateAutoCommitTimer(this.time.milliseconds());
        result = commitRequestManger.maybeAutoCommitAllConsumedAsync();
        Assertions.assertFalse((boolean)commitRequestManger.pendingRequests.unsentOffsetCommits.isEmpty());
        Assertions.assertFalse((boolean)result.isDone());
    }

    @Test
    public void testOffsetFetchRequestEnsureDuplicatedRequestSucceed() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>();
        partitions.add(new TopicPartition("t1", 0));
        List<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>> futures = this.sendAndVerifyDuplicatedOffsetFetchRequests(commitRequestManger, partitions, 2, Errors.NONE);
        futures.forEach(f -> {
            Assertions.assertTrue((boolean)f.isDone());
            Assertions.assertFalse((boolean)f.isCompletedExceptionally());
        });
        commitRequestManger.poll(0L);
        CommitRequestManagerTest.assertEmptyPendingRequests(commitRequestManger);
    }

    @ParameterizedTest
    @MethodSource(value={"offsetFetchExceptionSupplier"})
    public void testOffsetFetchRequestErroredRequests(Errors error, boolean isRetriable) {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>();
        partitions.add(new TopicPartition("t1", 0));
        List<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>> futures = this.sendAndVerifyDuplicatedOffsetFetchRequests(commitRequestManger, partitions, 5, error);
        if (isRetriable) {
            this.testRetriable(commitRequestManger, futures);
        } else {
            this.testNonRetriable(futures);
            CommitRequestManagerTest.assertEmptyPendingRequests(commitRequestManger);
        }
    }

    @ParameterizedTest
    @MethodSource(value={"offsetFetchRetriableCoordinatorErrors"})
    public void testOffsetFetchMarksCoordinatorUnknownOnRetriableCoordinatorErrors(Errors error, boolean shouldRediscoverCoordinator) {
        CommitRequestManager commitRequestManager = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>();
        partitions.add(new TopicPartition("t1", 0));
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        CompletableFuture result = commitRequestManager.addOffsetFetchRequest(partitions, expirationTimeMs);
        this.completeOffsetFetchRequestWithError(commitRequestManager, partitions, error);
        Assertions.assertFalse((boolean)result.isDone());
        if (shouldRediscoverCoordinator) {
            this.assertCoordinatorDisconnect();
        }
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        this.time.sleep(this.retryBackoffMs);
        res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
    }

    @Test
    public void testOffsetFetchMarksCoordinatorUnknownOnCoordinatorDisconnectedAndRetries() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>();
        partitions.add(new TopicPartition("t1", 0));
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        CompletableFuture result = commitRequestManger.addOffsetFetchRequest(partitions, expirationTimeMs);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        NetworkClientDelegate.UnsentRequest request = (NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0);
        ClientResponse response = this.buildOffsetFetchClientResponseDisconnected(request);
        response.onComplete();
        Assertions.assertFalse((boolean)result.isDone());
        this.assertCoordinatorDisconnect();
        this.time.sleep(100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        Assertions.assertEquals((int)1, (int)commitRequestManger.pendingRequests.inflightOffsetFetches.size());
    }

    @ParameterizedTest
    @MethodSource(value={"offsetCommitExceptionSupplier"})
    public void testOffsetCommitRequestErroredRequestsNotRetriedForAsyncCommit(Errors error) {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, error);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        Assertions.assertTrue((boolean)commitResult.isCompletedExceptionally());
        if (error.exception() instanceof RetriableException) {
            TestUtils.assertFutureThrows(commitResult, RetriableCommitFailedException.class);
        }
        this.assertExceptionHandling(commitRequestManger, error, false);
    }

    @Test
    public void testOffsetCommitSyncTimeoutNotReturnedOnPollAndFails() {
        CommitRequestManager commitRequestManger = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        Long expirationTimeMs = this.time.milliseconds() + this.retryBackoffMs * 2L;
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.of(expirationTimeMs), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, Errors.REQUEST_TIMED_OUT);
        this.time.sleep(this.retryBackoffMs);
        this.completeOffsetCommitRequestWithError(commitRequestManger, Errors.REQUEST_TIMED_OUT);
        Assertions.assertFalse((boolean)commitResult.isDone());
        this.time.sleep(this.retryBackoffMs);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        Assertions.assertTrue((boolean)commitResult.isCompletedExceptionally());
    }

    @Test
    public void testOffsetCommitSyncFailedWithRetriableThrowsTimeoutWhenRetryTimeExpires() {
        CommitRequestManager commitRequestManger = this.create(false, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        long expirationTimeMs = this.time.milliseconds() + this.retryBackoffMs * 2L;
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.of(expirationTimeMs), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, Errors.COORDINATOR_NOT_AVAILABLE);
        this.time.sleep(expirationTimeMs);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        TestUtils.assertFutureThrows(commitResult, TimeoutException.class);
    }

    @Test
    public void testOffsetCommitAsyncFailedWithRetriableThrowsRetriableCommitException() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        Errors retriableError = Errors.COORDINATOR_NOT_AVAILABLE;
        CompletableFuture commitResult = commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), false);
        this.completeOffsetCommitRequestWithError(commitRequestManger, retriableError);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertTrue((boolean)commitResult.isDone());
        Assertions.assertTrue((boolean)commitResult.isCompletedExceptionally());
        this.assertExceptionHandling(commitRequestManger, retriableError, false);
        TestUtils.assertFutureThrows(commitResult, RetriableCommitFailedException.class);
    }

    @Test
    public void testEnsureBackoffRetryOnOffsetCommitRequestTimeout() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        commitRequestManger.addOffsetCommitRequest(offsets, Optional.of(expirationTimeMs), false);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        ((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).handler().onFailure(this.time.milliseconds(), (RuntimeException)((Object)new TimeoutException()));
        Assertions.assertTrue((boolean)commitRequestManger.pendingRequests.hasUnsentRequests());
        Assertions.assertEquals((int)1, (int)commitRequestManger.unsentOffsetCommitRequests().size());
        this.assertRetryBackOff(commitRequestManger, this.retryBackoffMs);
    }

    private void assertCoordinatorDisconnect() {
        ((CoordinatorRequestManager)Mockito.verify((Object)this.coordinatorRequestManager)).markCoordinatorUnknown((String)ArgumentMatchers.any(), ArgumentMatchers.anyLong());
    }

    private void assertExceptionHandling(CommitRequestManager commitRequestManger, Errors errors, boolean requestShouldBeRetried) {
        long remainBackoffMs = requestShouldBeRetried ? this.retryBackoffMs : Long.MAX_VALUE;
        switch (errors) {
            case NOT_COORDINATOR: 
            case COORDINATOR_NOT_AVAILABLE: 
            case REQUEST_TIMED_OUT: {
                ((CoordinatorRequestManager)Mockito.verify((Object)this.coordinatorRequestManager)).markCoordinatorUnknown((String)ArgumentMatchers.any(), ArgumentMatchers.anyLong());
                this.assertPollDoesNotReturn(commitRequestManger, remainBackoffMs);
                break;
            }
            case UNKNOWN_TOPIC_OR_PARTITION: 
            case COORDINATOR_LOAD_IN_PROGRESS: {
                if (!requestShouldBeRetried) break;
                this.assertRetryBackOff(commitRequestManger, remainBackoffMs);
                break;
            }
            case GROUP_AUTHORIZATION_FAILED: {
                break;
            }
            case TOPIC_AUTHORIZATION_FAILED: 
            case OFFSET_METADATA_TOO_LARGE: 
            case INVALID_COMMIT_OFFSET_SIZE: {
                this.assertPollDoesNotReturn(commitRequestManger, Long.MAX_VALUE);
                break;
            }
            case FENCED_INSTANCE_ID: {
                this.assertPollDoesNotReturn(commitRequestManger, Long.MAX_VALUE);
                break;
            }
            default: {
                if (errors.exception() instanceof RetriableException && requestShouldBeRetried) {
                    this.assertRetryBackOff(commitRequestManger, remainBackoffMs);
                    break;
                }
                this.assertPollDoesNotReturn(commitRequestManger, Long.MAX_VALUE);
            }
        }
    }

    private void assertPollDoesNotReturn(CommitRequestManager commitRequestManager, long assertNextPollMs) {
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        Assertions.assertEquals((long)assertNextPollMs, (long)res.timeUntilNextPollMs);
    }

    private void assertRetryBackOff(CommitRequestManager commitRequestManager, long retryBackoffMs) {
        this.assertPollDoesNotReturn(commitRequestManager, retryBackoffMs);
        this.time.sleep(retryBackoffMs - 1L);
        this.assertPollDoesNotReturn(commitRequestManager, 1L);
        this.time.sleep(1L);
        this.assertPoll(1, commitRequestManager);
    }

    @Test
    public void testSyncOffsetFetchFailsWithStaleEpochAndRetriesWithNewEpoch() {
        CommitRequestManager commitRequestManager = this.create(false, 100L);
        Set<TopicPartition> partitions = Collections.singleton(new TopicPartition("t1", 0));
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        commitRequestManager.addOffsetFetchRequest(partitions, expirationTimeMs);
        int newEpoch = 8;
        String memberId = "member1";
        commitRequestManager.onMemberEpochUpdated(Optional.of(newEpoch), Optional.of(memberId));
        this.completeOffsetFetchRequestWithError(commitRequestManager, partitions, Errors.STALE_MEMBER_EPOCH);
        Assertions.assertEquals((int)0, (int)commitRequestManager.pendingRequests.inflightOffsetFetches.size());
        Assertions.assertEquals((int)1, (int)commitRequestManager.pendingRequests.unsentOffsetFetches.size());
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        this.time.sleep(this.retryBackoffMs);
        res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        OffsetFetchRequestData reqData = (OffsetFetchRequestData)((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).requestBuilder().build().data();
        Assertions.assertEquals((int)1, (int)reqData.groups().size());
        Assertions.assertEquals((int)newEpoch, (int)((OffsetFetchRequestData.OffsetFetchRequestGroup)reqData.groups().get(0)).memberEpoch());
        Assertions.assertEquals((Object)memberId, (Object)((OffsetFetchRequestData.OffsetFetchRequestGroup)reqData.groups().get(0)).memberId());
    }

    @Test
    public void testSyncOffsetFetchFailsWithStaleEpochAndNotRetriedIfMemberNotInGroupAnymore() {
        CommitRequestManager commitRequestManager = this.create(false, 100L);
        Set<TopicPartition> partitions = Collections.singleton(new TopicPartition("t1", 0));
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        CompletableFuture requestResult = commitRequestManager.addOffsetFetchRequest(partitions, expirationTimeMs);
        commitRequestManager.onMemberEpochUpdated(Optional.empty(), Optional.empty());
        this.completeOffsetFetchRequestWithError(commitRequestManager, partitions, Errors.STALE_MEMBER_EPOCH);
        Assertions.assertTrue((boolean)requestResult.isDone());
        Assertions.assertTrue((boolean)requestResult.isCompletedExceptionally());
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
    }

    @Test
    public void testOffsetCommitFailsWithStaleEpochAndRetriesWithNewEpoch() {
        CommitRequestManager commitRequestManager = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        commitRequestManager.addOffsetCommitRequest(offsets, Optional.of(expirationTimeMs), true);
        int newEpoch = 8;
        String memberId = "member1";
        commitRequestManager.onMemberEpochUpdated(Optional.of(newEpoch), Optional.of(memberId));
        this.completeOffsetCommitRequestWithError(commitRequestManager, Errors.STALE_MEMBER_EPOCH);
        Assertions.assertEquals((int)0, (int)commitRequestManager.pendingRequests.inflightOffsetFetches.size());
        Assertions.assertEquals((int)1, (int)commitRequestManager.pendingRequests.unsentOffsetCommits.size());
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        this.time.sleep(this.retryBackoffMs);
        res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        OffsetCommitRequestData reqData = (OffsetCommitRequestData)((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).requestBuilder().build().data();
        Assertions.assertEquals((int)newEpoch, (int)reqData.generationIdOrMemberEpoch());
        Assertions.assertEquals((Object)memberId, (Object)reqData.memberId());
    }

    @Test
    public void testEnsureCommitSensorRecordsMetric() {
        CommitRequestManager commitRequestManager = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        this.commitOffsetWithAssertedLatency(commitRequestManager, 100L);
        this.commitOffsetWithAssertedLatency(commitRequestManager, 101L);
        Assertions.assertEquals((Object)100.5, (Object)this.getMetric("commit-latency-avg").metricValue());
        Assertions.assertEquals((Object)101.0, (Object)this.getMetric("commit-latency-max").metricValue());
        Assertions.assertEquals((double)0.066, (double)((Double)this.getMetric("commit-rate").metricValue()), (double)0.001);
        Assertions.assertEquals((Object)2.0, (Object)this.getMetric("commit-total").metricValue());
    }

    private void commitOffsetWithAssertedLatency(CommitRequestManager commitRequestManager, long latencyMs) {
        String topic = "topic";
        boolean partition = true;
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        long commitCreationTimeMs = this.time.milliseconds();
        commitRequestManager.addOffsetCommitRequest(offsets, Optional.empty(), true);
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        this.time.sleep(latencyMs);
        long commitReceivedTimeMs = this.time.milliseconds();
        ((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).future().complete(this.mockOffsetCommitResponse("topic", 1, (short)1, commitCreationTimeMs, commitReceivedTimeMs, Errors.NONE));
    }

    private void completeOffsetFetchRequestWithError(CommitRequestManager commitRequestManager, Set<TopicPartition> partitions, Errors error) {
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        ((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).future().complete(this.buildOffsetFetchClientResponse((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0), partitions, error));
    }

    private void completeOffsetCommitRequestWithError(CommitRequestManager commitRequestManager, Errors error) {
        NetworkClientDelegate.PollResult res = commitRequestManager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        ((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).future().complete(this.mockOffsetCommitResponse("topic", 1, (short)1, error));
    }

    private void testRetriable(CommitRequestManager commitRequestManger, List<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>> futures) {
        futures.forEach(f -> Assertions.assertFalse((boolean)f.isDone()));
        this.time.sleep(100L);
        commitRequestManger.poll(this.time.milliseconds());
        futures.forEach(f -> Assertions.assertFalse((boolean)f.isDone()));
    }

    private void testNonRetriable(List<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>> futures) {
        futures.forEach(f -> Assertions.assertTrue((boolean)f.isCompletedExceptionally()));
    }

    private static Stream<Arguments> offsetCommitExceptionSupplier() {
        return Stream.of(Arguments.of((Object[])new Object[]{Errors.NOT_COORDINATOR}), Arguments.of((Object[])new Object[]{Errors.COORDINATOR_LOAD_IN_PROGRESS}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_SERVER_ERROR}), Arguments.of((Object[])new Object[]{Errors.GROUP_AUTHORIZATION_FAILED}), Arguments.of((Object[])new Object[]{Errors.OFFSET_METADATA_TOO_LARGE}), Arguments.of((Object[])new Object[]{Errors.INVALID_COMMIT_OFFSET_SIZE}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_TOPIC_OR_PARTITION}), Arguments.of((Object[])new Object[]{Errors.COORDINATOR_NOT_AVAILABLE}), Arguments.of((Object[])new Object[]{Errors.REQUEST_TIMED_OUT}), Arguments.of((Object[])new Object[]{Errors.FENCED_INSTANCE_ID}), Arguments.of((Object[])new Object[]{Errors.TOPIC_AUTHORIZATION_FAILED}), Arguments.of((Object[])new Object[]{Errors.STALE_MEMBER_EPOCH}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_MEMBER_ID}));
    }

    private static Stream<Arguments> offsetFetchExceptionSupplier() {
        return Stream.of(Arguments.of((Object[])new Object[]{Errors.NOT_COORDINATOR, true}), Arguments.of((Object[])new Object[]{Errors.COORDINATOR_LOAD_IN_PROGRESS, true}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_SERVER_ERROR, false}), Arguments.of((Object[])new Object[]{Errors.GROUP_AUTHORIZATION_FAILED, false}), Arguments.of((Object[])new Object[]{Errors.OFFSET_METADATA_TOO_LARGE, false}), Arguments.of((Object[])new Object[]{Errors.INVALID_COMMIT_OFFSET_SIZE, false}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_TOPIC_OR_PARTITION, false}), Arguments.of((Object[])new Object[]{Errors.COORDINATOR_NOT_AVAILABLE, true}), Arguments.of((Object[])new Object[]{Errors.REQUEST_TIMED_OUT, false}), Arguments.of((Object[])new Object[]{Errors.FENCED_INSTANCE_ID, false}), Arguments.of((Object[])new Object[]{Errors.TOPIC_AUTHORIZATION_FAILED, false}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_MEMBER_ID, false}), Arguments.of((Object[])new Object[]{Errors.STALE_MEMBER_EPOCH, false}));
    }

    private static Stream<Arguments> offsetFetchRetriableCoordinatorErrors() {
        return Stream.of(Arguments.of((Object[])new Object[]{Errors.NOT_COORDINATOR, true}), Arguments.of((Object[])new Object[]{Errors.COORDINATOR_NOT_AVAILABLE, true}), Arguments.of((Object[])new Object[]{Errors.COORDINATOR_LOAD_IN_PROGRESS, false}));
    }

    @ParameterizedTest
    @MethodSource(value={"partitionDataErrorSupplier"})
    public void testOffsetFetchRequestPartitionDataError(Errors error, boolean isRetriable) {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>();
        TopicPartition tp1 = new TopicPartition("t1", 2);
        TopicPartition tp2 = new TopicPartition("t2", 3);
        partitions.add(tp1);
        partitions.add(tp2);
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        CompletableFuture future = commitRequestManger.addOffsetFetchRequest(partitions, expirationTimeMs);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        HashMap<TopicPartition, OffsetFetchResponse.PartitionData> topicPartitionData = new HashMap<TopicPartition, OffsetFetchResponse.PartitionData>();
        topicPartitionData.put(tp1, new OffsetFetchResponse.PartitionData(100L, Optional.of(1), "metadata", error));
        topicPartitionData.put(tp2, new OffsetFetchResponse.PartitionData(100L, Optional.of(1), "metadata", Errors.NONE));
        ((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).handler().onComplete(this.buildOffsetFetchClientResponse((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0), topicPartitionData, Errors.NONE, false));
        if (isRetriable) {
            this.testRetriable(commitRequestManger, Collections.singletonList(future));
        } else {
            this.testNonRetriable(Collections.singletonList(future));
        }
    }

    @Test
    public void testSignalClose() {
        CommitRequestManager commitRequestManger = this.create(true, 100L);
        Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        Map<TopicPartition, OffsetAndMetadata> offsets = Collections.singletonMap(new TopicPartition("topic", 1), new OffsetAndMetadata(0L));
        commitRequestManger.addOffsetCommitRequest(offsets, Optional.empty(), false);
        commitRequestManger.signalClose();
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        OffsetCommitRequestData data = (OffsetCommitRequestData)((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).requestBuilder().build().data();
        Assertions.assertEquals((Object)"topic", (Object)((OffsetCommitRequestData.OffsetCommitRequestTopic)data.topics().get(0)).name());
    }

    private static void assertEmptyPendingRequests(CommitRequestManager commitRequestManger) {
        Assertions.assertTrue((boolean)commitRequestManger.pendingRequests.inflightOffsetFetches.isEmpty());
        Assertions.assertTrue((boolean)commitRequestManger.pendingRequests.unsentOffsetFetches.isEmpty());
        Assertions.assertTrue((boolean)commitRequestManger.pendingRequests.unsentOffsetCommits.isEmpty());
    }

    private static Stream<Arguments> partitionDataErrorSupplier() {
        return Stream.of(Arguments.of((Object[])new Object[]{Errors.UNSTABLE_OFFSET_COMMIT, true}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_TOPIC_OR_PARTITION, false}), Arguments.of((Object[])new Object[]{Errors.TOPIC_AUTHORIZATION_FAILED, false}), Arguments.of((Object[])new Object[]{Errors.UNKNOWN_SERVER_ERROR, false}));
    }

    private List<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>> sendAndVerifyDuplicatedOffsetFetchRequests(CommitRequestManager commitRequestManger, Set<TopicPartition> partitions, int numRequest, Errors error) {
        ArrayList<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>> futures = new ArrayList<CompletableFuture<Map<TopicPartition, OffsetAndMetadata>>>();
        long expirationTimeMs = this.time.milliseconds() + 60000L;
        for (int i = 0; i < numRequest; ++i) {
            futures.add(commitRequestManger.addOffsetFetchRequest(partitions, expirationTimeMs));
        }
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)1, (int)res.unsentRequests.size());
        ((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0)).handler().onComplete(this.buildOffsetFetchClientResponse((NetworkClientDelegate.UnsentRequest)res.unsentRequests.get(0), partitions, error));
        res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        return futures;
    }

    private void sendAndVerifyOffsetCommitRequestFailedAndMaybeRetried(CommitRequestManager commitRequestManger, Errors error, CompletableFuture<Void> commitResult) {
        this.completeOffsetCommitRequestWithError(commitRequestManger, error);
        NetworkClientDelegate.PollResult res = commitRequestManger.poll(this.time.milliseconds());
        Assertions.assertEquals((int)0, (int)res.unsentRequests.size());
        if (error.exception() instanceof RetriableException) {
            Assertions.assertFalse((boolean)commitResult.isDone());
        } else {
            Assertions.assertTrue((boolean)commitResult.isDone());
            Assertions.assertTrue((boolean)commitResult.isCompletedExceptionally());
        }
    }

    private List<NetworkClientDelegate.FutureCompletionHandler> assertPoll(int numRes, CommitRequestManager manager) {
        return this.assertPoll(true, numRes, manager);
    }

    private List<NetworkClientDelegate.FutureCompletionHandler> assertPoll(boolean coordinatorDiscovered, int numRes, CommitRequestManager manager) {
        if (coordinatorDiscovered) {
            Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.of(this.mockedNode));
        } else {
            Mockito.when((Object)this.coordinatorRequestManager.coordinator()).thenReturn(Optional.empty());
        }
        NetworkClientDelegate.PollResult res = manager.poll(this.time.milliseconds());
        Assertions.assertEquals((int)numRes, (int)res.unsentRequests.size());
        return res.unsentRequests.stream().map(NetworkClientDelegate.UnsentRequest::handler).collect(Collectors.toList());
    }

    private CommitRequestManager create(boolean autoCommitEnabled, long autoCommitInterval) {
        this.props.setProperty("auto.commit.interval.ms", String.valueOf(autoCommitInterval));
        this.props.setProperty("enable.auto.commit", String.valueOf(autoCommitEnabled));
        if (autoCommitEnabled) {
            this.props.setProperty("group.id", TestUtils.randomString(10));
        }
        return (CommitRequestManager)Mockito.spy((Object)new CommitRequestManager((Time)this.time, this.logContext, this.subscriptionState, new ConsumerConfig(this.props), this.coordinatorRequestManager, this.offsetCommitCallbackInvoker, "group-id", Optional.of("group-instance-id"), this.retryBackoffMs, this.retryBackoffMaxMs, OptionalDouble.of(0.0), this.metrics));
    }

    private ClientResponse buildOffsetFetchClientResponse(NetworkClientDelegate.UnsentRequest request, Set<TopicPartition> topicPartitions, Errors error) {
        HashMap<TopicPartition, OffsetFetchResponse.PartitionData> topicPartitionData = new HashMap<TopicPartition, OffsetFetchResponse.PartitionData>();
        topicPartitions.forEach(tp -> topicPartitionData.put((TopicPartition)tp, new OffsetFetchResponse.PartitionData(100L, Optional.of(1), "metadata", Errors.NONE)));
        return this.buildOffsetFetchClientResponse(request, topicPartitionData, error, false);
    }

    private ClientResponse buildOffsetFetchClientResponseDisconnected(NetworkClientDelegate.UnsentRequest request) {
        return this.buildOffsetFetchClientResponse(request, Collections.emptyMap(), Errors.NONE, true);
    }

    private ClientResponse buildOffsetCommitClientResponse(OffsetCommitResponse commitResponse) {
        short apiVersion = 1;
        return new ClientResponse(new RequestHeader(ApiKeys.OFFSET_COMMIT, apiVersion, "", 1), null, "-1", this.time.milliseconds(), this.time.milliseconds(), false, null, null, (AbstractResponse)commitResponse);
    }

    public ClientResponse mockOffsetCommitResponse(String topic, int partition, short apiKeyVersion, Errors error) {
        return this.mockOffsetCommitResponse(topic, partition, apiKeyVersion, this.time.milliseconds(), this.time.milliseconds(), error);
    }

    public ClientResponse mockOffsetCommitResponse(String topic, int partition, short apiKeyVersion, long createdTimeMs, long receivedTimeMs, Errors error) {
        OffsetCommitResponseData responseData = new OffsetCommitResponseData().setTopics(Arrays.asList(new OffsetCommitResponseData.OffsetCommitResponseTopic().setName(topic).setPartitions(Collections.singletonList(new OffsetCommitResponseData.OffsetCommitResponsePartition().setErrorCode(error.code()).setPartitionIndex(partition)))));
        OffsetCommitResponse response = (OffsetCommitResponse)Mockito.mock(OffsetCommitResponse.class);
        Mockito.when((Object)response.data()).thenReturn((Object)responseData);
        return new ClientResponse(new RequestHeader(ApiKeys.OFFSET_COMMIT, apiKeyVersion, "", 1), null, "-1", createdTimeMs, receivedTimeMs, false, null, null, (AbstractResponse)new OffsetCommitResponse(responseData));
    }

    public ClientResponse mockOffsetCommitResponseDisconnected(String topic, int partition, short apiKeyVersion, NetworkClientDelegate.UnsentRequest unsentRequest) {
        OffsetCommitResponseData responseData = new OffsetCommitResponseData().setTopics(Arrays.asList(new OffsetCommitResponseData.OffsetCommitResponseTopic().setName(topic).setPartitions(Collections.singletonList(new OffsetCommitResponseData.OffsetCommitResponsePartition().setErrorCode(Errors.NONE.code()).setPartitionIndex(partition)))));
        OffsetCommitResponse response = (OffsetCommitResponse)Mockito.mock(OffsetCommitResponse.class);
        Mockito.when((Object)response.data()).thenReturn((Object)responseData);
        return new ClientResponse(new RequestHeader(ApiKeys.OFFSET_COMMIT, apiKeyVersion, "", 1), (RequestCompletionHandler)unsentRequest.handler(), "-1", this.time.milliseconds(), this.time.milliseconds(), true, null, null, (AbstractResponse)new OffsetCommitResponse(responseData));
    }

    private ClientResponse buildOffsetFetchClientResponse(NetworkClientDelegate.UnsentRequest request, Map<TopicPartition, OffsetFetchResponse.PartitionData> topicPartitionData, Errors error, boolean disconnected) {
        AbstractRequest abstractRequest = request.requestBuilder().build();
        Assertions.assertTrue((boolean)(abstractRequest instanceof OffsetFetchRequest));
        OffsetFetchRequest offsetFetchRequest = (OffsetFetchRequest)abstractRequest;
        OffsetFetchResponse response = new OffsetFetchResponse(error, topicPartitionData);
        return new ClientResponse(new RequestHeader(ApiKeys.OFFSET_FETCH, offsetFetchRequest.version(), "", 1), (RequestCompletionHandler)request.handler(), "-1", this.time.milliseconds(), this.time.milliseconds(), disconnected, null, null, (AbstractResponse)response);
    }

    private KafkaMetric getMetric(String name) {
        return (KafkaMetric)this.metrics.metrics().get(this.metrics.metricName(name, CONSUMER_COORDINATOR_METRICS));
    }
}

