/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.network.AbstractProxyProtocolEngineTest;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.network.ProxyProtocolCommand;
import org.apache.kafka.common.network.ProxyProtocolEngine;
import org.apache.kafka.common.network.ProxyProtocolV2Engine;
import org.apache.kafka.common.network.ProxyTlv;
import org.apache.kafka.common.network.ProxyTlvType;
import org.apache.kafka.common.utils.LogContext;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class ProxyProtocolV2EngineTest
extends AbstractProxyProtocolEngineTest {
    private String bytesToHexStr(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes) {
            String s = Integer.toHexString(b & 0xFF);
            if (s.length() == 1) {
                sb.append("0");
            }
            sb.append(s);
        }
        return sb.toString();
    }

    private byte[] hexStrToBytes(String hex) {
        byte[] bytes = new byte[hex.length() / 2];
        for (int i = 0; i < hex.length(); i += 2) {
            bytes[i / 2] = (byte)Integer.parseInt(hex.substring(i, i + 2), 16);
        }
        return bytes;
    }

    @Test
    public void testLkcId() throws IOException {
        ProxyProtocolEngine engine = this.newClientPPE("127.0.0.1", 8888, DEFAULT_COMMAND, "lkc-4proxy");
        this.assertValidEngineState(engine, true, false, engine.clientAddress(), engine.clientPort(), engine.command(), Collections.singletonList(engine.tlv(ProxyTlvType.LKC_ID)));
        String destinationAddress = engine.clientAddress() instanceof Inet4Address ? "127.0.0.1" : "2001:db8:85a3:8d3:1319:8a2e:370:7349";
        byte[] header = engine.emitHeaders(InetAddress.getByName(destinationAddress), 9092);
        ByteBuffer buf = ByteBuffer.wrap(header);
        engine = this.newServerPPE();
        this.assertValidEngineState(engine, false, false, null, -1, null);
        engine.processHeaders(buf);
        this.assertValidEngineState(engine, true, true, InetAddress.getByName("127.0.0.1"), 8888, DEFAULT_COMMAND, Collections.singletonList(DEFAULT_TLV));
    }

    @Test
    public void testCCloudTrafficType() throws IOException {
        ProxyProtocolEngine engine = this.newClientPPE("127.0.0.1", 8888, DEFAULT_COMMAND, "lkc-4proxy", DEFAULT_CCLOUD_TRAFFIC_TYPE);
        this.assertValidEngineState(engine, true, false, engine.clientAddress(), engine.clientPort(), engine.command(), Arrays.asList(engine.tlv(ProxyTlvType.LKC_ID), engine.tlv(ProxyTlvType.CCLOUD_TRAFFIC_TYPE)));
        String destinationAddress = engine.clientAddress() instanceof Inet4Address ? "127.0.0.1" : "2001:db8:85a3:8d3:1319:8a2e:370:7349";
        byte[] header = engine.emitHeaders(InetAddress.getByName(destinationAddress), 9092);
        ByteBuffer buf = ByteBuffer.wrap(header);
        engine = this.newServerPPE();
        this.assertValidEngineState(engine, false, false, null, -1, null);
        engine.processHeaders(buf);
        this.assertValidEngineState(engine, true, true, InetAddress.getByName("127.0.0.1"), 8888, DEFAULT_COMMAND, Arrays.asList(DEFAULT_TLV, DEFAULT_CCLOUD_TRAFFIC_TYPE_TLV));
    }

    @Test
    public void testProxyProtocolV2EngineBasic() throws IOException {
        ProxyProtocolEngine engine = this.newServerPPE();
        this.assertValidEngineState(engine, false, false, null, -1, null);
        ByteBuffer buf = this.putStream("0D0A0D0A000D0A515549540A2111000CD83AD0EE0000000022B822B80D0A03");
        engine.processHeaders(buf);
        byte[] sourceAddress = this.hexStrToBytes("D83AD0EE");
        InetAddress address = InetAddress.getByAddress(sourceAddress);
        this.assertValidEngineState(engine, true, true, address, 8888, ProxyProtocolCommand.PROXY);
        Assertions.assertEquals((int)3, (int)buf.remaining());
    }

    @Test
    public void testProxyProtocolV2EngineIpv6() throws IOException {
        ProxyProtocolEngine engine = this.newServerPPE();
        this.assertValidEngineState(engine, false, false, null, -1, null);
        ByteBuffer buf = this.putStream("0D0A0D0A000D0A515549540A2121002420010DB8AC10FE0100000000000000000000000000000000000000000000000022B822B80D0A");
        engine.processHeaders(buf);
        byte[] sourceAddress = this.hexStrToBytes("20010DB8AC10FE010000000000000000");
        InetAddress address = InetAddress.getByAddress(sourceAddress);
        this.assertValidEngineState(engine, true, true, address, 8888, ProxyProtocolCommand.PROXY);
        Assertions.assertEquals((int)2, (int)buf.remaining());
    }

    @Test
    public void testProxyProtocolV2EngineWithMultipleSegments() throws IOException {
        ProxyProtocolEngine engine = this.newServerPPE();
        ByteBuffer buf = ByteBuffer.allocate(31);
        this.putStreamAndFlip(buf, "0D0A0D0A000D");
        engine.processHeaders(buf);
        buf.compact();
        Assertions.assertFalse((boolean)engine.hasClientInformation());
        Assertions.assertFalse((boolean)engine.ready());
        this.putStreamAndFlip(buf, "0A515549540A21");
        engine.processHeaders(buf);
        buf.compact();
        Assertions.assertFalse((boolean)engine.hasClientInformation());
        Assertions.assertFalse((boolean)engine.ready());
        this.putStreamAndFlip(buf, "11000CD83AD0EE0000000022B822B80D0A03");
        engine.processHeaders(buf);
        buf.compact();
        Assertions.assertTrue((boolean)engine.hasClientInformation());
        Assertions.assertTrue((boolean)engine.ready());
        byte[] sourceAddress = this.hexStrToBytes("D83AD0EE");
        InetAddress address = InetAddress.getByAddress(sourceAddress);
        Assertions.assertEquals((Object)address, (Object)engine.clientAddress());
        Assertions.assertEquals((int)8888, (int)engine.clientPort());
    }

    @Test
    public void testNonProxyHeaderFallbackEnabled() throws IOException {
        ProxyProtocolEngine engine = this.newServerPPE();
        engine.configure(Collections.singletonMap("confluent.proxy.protocol.fallback.enabled", true));
        ByteBuffer buf = this.putStream("AD0EE0000000022B822B0A0D");
        engine.processHeaders(buf);
        buf.compact();
        Assertions.assertFalse((boolean)engine.hasClientInformation());
        Assertions.assertTrue((boolean)engine.ready());
        buf.flip();
        Assertions.assertEquals((int)12, (int)buf.remaining());
    }

    @Test
    public void testProxyProtocolV2LocalConnection() throws IOException {
        ProxyProtocolEngine engine = this.newServerPPE();
        this.assertValidEngineState(engine, false, false, null, -1, null);
        String payload = "CD83AD0EE0000000022B822B";
        ByteBuffer buf = this.putStream("0D0A0D0A000D0A515549540A20000000" + payload);
        engine.processHeaders(buf);
        this.assertValidEngineState(engine, false, true, null, -1, ProxyProtocolCommand.LOCAL);
        Assertions.assertEquals((int)this.hexStrToBytes(payload).length, (int)buf.remaining());
    }

    @Test
    public void testMalformedLkcIdForClient() {
        String invalidLkcId = "this-is-not-a-valid-lkc-id";
        Assertions.assertThrows(ConfigException.class, () -> this.newClientPPE("216.58.208.238", 8888, DEFAULT_COMMAND, invalidLkcId));
    }

    @Test
    public void testLkcIdForClientIsOptional() {
        Assertions.assertNotNull((Object)this.newClientPPE("216.58.208.238", 8888, DEFAULT_COMMAND, "lkc-4proxy").tlv(ProxyTlvType.LKC_ID));
        Assertions.assertNull((Object)this.newClientPPE("216.58.208.238", 8888, DEFAULT_COMMAND, null).tlv(ProxyTlvType.LKC_ID));
    }

    @Test
    public void testAddressAndPortOptionalInLocalMode() {
        Assertions.assertNotNull((Object)this.newClientPPE(null, null, ProxyProtocolCommand.LOCAL, "lkc-4proxy"));
        Assertions.assertThrows(ConfigException.class, () -> this.newClientPPE(null, null, ProxyProtocolCommand.PROXY, "lkc-4proxy"));
    }

    @Test
    public void testMalformedLkcIdForServer() throws IOException {
        String invalidLkcId = "this-is-not-a-valid-lkc-id";
        ProxyTlv tlv = ProxyTlv.createForSubtype((ProxyTlvType)ProxyTlvType.LKC_ID, (String)invalidLkcId);
        String tlvHexString = this.bytesToHexStr(ProxyProtocolV2Engine.tlvBytes((ProxyTlv)tlv));
        String addressLengthHexString = "00" + Integer.toHexString(12 + tlvHexString.length() / 2);
        ProxyProtocolEngine engine = this.newServerPPE();
        ByteBuffer buf = this.putStream("0D0A0D0A000D0A515549540A2111" + addressLengthHexString + "D83AD0EE7f00000122B82384" + tlvHexString);
        engine.processHeaders(buf);
    }

    @Test
    public void testMultipleTlvs() throws IOException {
        String hostName = "636C75737465722E636F6E666C75656E742E696F";
        ProxyTlvType authorityTlvType = new ProxyTlvType.Builder(2).build();
        ProxyTlv authorityTlv = new ProxyTlv(authorityTlvType, this.hexStrToBytes(hostName));
        ArrayList<ProxyTlv> tlvs = new ArrayList<ProxyTlv>();
        tlvs.add(DEFAULT_TLV);
        tlvs.add(DEFAULT_CCLOUD_TRAFFIC_TYPE_TLV);
        tlvs.add(authorityTlv);
        String tlvHexString = this.bytesToHexStr(ProxyProtocolV2Engine.tlvBytes(tlvs));
        String addressLengthHexString = "00" + Integer.toHexString(12 + tlvHexString.length() / 2);
        ProxyProtocolEngine engine = this.newServerPPE();
        ByteBuffer buf = ByteBuffer.wrap(this.generateHeader("0D0A0D0A000D0A515549540A2111" + addressLengthHexString + "D83AD0EE7f00000122B82384" + tlvHexString));
        this.assertValidEngineState(engine, false, false, null, -1, null);
        engine.processHeaders(buf);
        this.assertValidEngineState(engine, true, true, InetAddress.getByName("216.58.208.238"), 8888, DEFAULT_COMMAND, Arrays.asList(DEFAULT_TLV, DEFAULT_CCLOUD_TRAFFIC_TYPE_TLV));
        ProxyTlv actualTlv = engine.tlv(authorityTlvType);
        Assertions.assertNotNull((Object)actualTlv);
        Assertions.assertEquals((Object)authorityTlv, (Object)actualTlv);
        String actualSniValue = new String(actualTlv.rawValue(), StandardCharsets.UTF_8);
        Assertions.assertEquals((Object)"cluster.confluent.io", (Object)actualSniValue);
    }

    @Override
    protected ProxyProtocolEngine newServerPPE() {
        return new ProxyProtocolV2Engine(Mode.SERVER, new LogContext());
    }

    @Override
    protected ProxyProtocolEngine newClientPPE() {
        return new ProxyProtocolV2Engine(Mode.CLIENT, new LogContext());
    }

    @Override
    protected byte[] generateHeader(String header) {
        return this.hexStrToBytes(header);
    }

    @Override
    protected List<String> getInvalidProxyHeaders() {
        return Arrays.asList("0D0A0D0A000D0A515549540A2112000CD83AD0EE0000000022B822B8", "0D0A0D0A000D0A515549540A2122002420010DB8AC10FE0100000000000000000000000000000000000000000000000022B822B80D0A", "0D0D0D0A000D0A515549540A2111000CD83AD0EE0000000022B822B80D0A03", "0D0A0D0A000D0A515549540A1111000CD83AD0EE0000000022B822B80D0A03", "0D0A0D0A000D0A515549540A2211000CD83AD0EE0000000022B822B80D0A03");
    }
}

