/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.execution.streams.materialization.ks;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import com.google.errorprone.annotations.Immutable;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.confluent.ksql.GenericKey;
import io.confluent.ksql.execution.streams.RoutingFilter;
import io.confluent.ksql.execution.streams.RoutingOptions;
import io.confluent.ksql.execution.streams.materialization.Locator;
import io.confluent.ksql.execution.streams.materialization.MaterializationException;
import io.confluent.ksql.util.KsqlHostInfo;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.KeyQueryMetadata;
import org.apache.kafka.streams.StreamsMetadata;
import org.apache.kafka.streams.Topology;
import org.apache.kafka.streams.TopologyDescription;
import org.apache.kafka.streams.processor.internals.StreamsMetadataState;
import org.apache.kafka.streams.processor.internals.namedtopology.KafkaStreamsNamedTopologyWrapper;
import org.apache.kafka.streams.state.HostInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KsLocator
implements Locator {
    private static final Logger LOG = LoggerFactory.getLogger(KsLocator.class);
    private final String storeName;
    private final KafkaStreams kafkaStreams;
    private final Topology topology;
    private final Serializer<GenericKey> keySerializer;
    private final URL localHost;
    private final boolean sharedRuntimesEnabled;
    private final String queryId;

    KsLocator(String stateStoreName, KafkaStreams kafkaStreams, Topology topology, Serializer<GenericKey> keySerializer, URL localHost, boolean sharedRuntimesEnabled, String queryId) {
        this.kafkaStreams = Objects.requireNonNull(kafkaStreams, "kafkaStreams");
        this.topology = Objects.requireNonNull(topology, "topology");
        this.keySerializer = Objects.requireNonNull(keySerializer, "keySerializer");
        this.storeName = Objects.requireNonNull(stateStoreName, "stateStoreName");
        this.localHost = Objects.requireNonNull(localHost, "localHost");
        this.sharedRuntimesEnabled = sharedRuntimesEnabled;
        this.queryId = Objects.requireNonNull(queryId, "queryId");
    }

    @Override
    public List<Locator.KsqlPartitionLocation> locate(List<Locator.KsqlKey> keys, RoutingOptions routingOptions, RoutingFilter.RoutingFilterFactory routingFilterFactory, boolean isRangeScan) {
        if (isRangeScan && keys.isEmpty()) {
            throw new IllegalStateException("Query is range scan but found no range keys.");
        }
        ImmutableList.Builder partitionLocations = ImmutableList.builder();
        Set<Integer> filterPartitions = routingOptions.getPartitions();
        Optional<Set<Locator.KsqlKey>> keySet = keys.isEmpty() ? Optional.empty() : Optional.of(Sets.newHashSet(keys));
        List<PartitionMetadata> metadata = keys.size() == 1 && keys.get(0).getKey().size() == 1 && !isRangeScan ? this.getMetadataForKeys(keys, filterPartitions) : this.getMetadataForAllPartitions(filterPartitions, keySet);
        if (metadata.isEmpty()) {
            MaterializationException materializationException = new MaterializationException("Cannot determine which host contains the required partitions to serve the pull query. \nThe underlying persistent query may be restarting (e.g. as a result of ALTER SYSTEM) view the status of your by issuing <DESCRIBE foo>.");
            LOG.debug(materializationException.getMessage());
            throw materializationException;
        }
        for (PartitionMetadata partitionMetadata : metadata) {
            LOG.debug("Handling pull query for partition {} of state store {}.", (Object)partitionMetadata.getPartition(), (Object)this.storeName);
            HostInfo activeHost = partitionMetadata.getActiveHost();
            Set<HostInfo> standByHosts = partitionMetadata.getStandbyHosts();
            int partition = partitionMetadata.getPartition();
            Optional<Set<Locator.KsqlKey>> partitionKeys = partitionMetadata.getKeys();
            LOG.debug("Active host {}, standby {}, partition {}.", new Object[]{activeHost, standByHosts, partition});
            List<Locator.KsqlNode> filteredHosts = this.getFilteredHosts(routingOptions, routingFilterFactory, activeHost, standByHosts, partition);
            partitionLocations.add((Object)new PartitionLocation(partitionKeys, partition, filteredHosts));
        }
        return partitionLocations.build();
    }

    private List<PartitionMetadata> getMetadataForKeys(List<Locator.KsqlKey> keys, Set<Integer> filterPartitions) {
        LinkedHashMap<Integer, KeyQueryMetadata> metadataByPartition = new LinkedHashMap<Integer, KeyQueryMetadata>();
        HashMap<Integer, Set> keysByPartition = new HashMap<Integer, Set>();
        for (Locator.KsqlKey key : keys) {
            KeyQueryMetadata metadata2 = this.getKeyQueryMetadata(key);
            if (metadata2.equals((Object)KeyQueryMetadata.NOT_AVAILABLE)) {
                LOG.debug("KeyQueryMetadata not available for state store '{}' and key {}", (Object)this.storeName, (Object)key);
                throw new MaterializationException(String.format("Materialized data for key %s is not available yet. Please try again later.", key));
            }
            LOG.debug("Handling pull query for key {} in partition {} of state store {}.", new Object[]{key, metadata2.partition(), this.storeName});
            if (filterPartitions.size() > 0 && !filterPartitions.contains(metadata2.partition())) {
                LOG.debug("Ignoring key {} in partition {} because parition is not included in lookup.", (Object)key, (Object)metadata2.partition());
                continue;
            }
            keysByPartition.computeIfAbsent(metadata2.partition(), k -> new LinkedHashSet());
            ((Set)keysByPartition.get(metadata2.partition())).add(key);
            metadataByPartition.putIfAbsent(metadata2.partition(), metadata2);
        }
        return metadataByPartition.values().stream().map(metadata -> {
            HostInfo activeHost = metadata.activeHost();
            Set standByHosts = metadata.standbyHosts();
            return new PartitionMetadata(activeHost, standByHosts, metadata.partition(), Optional.of(keysByPartition.get(metadata.partition())));
        }).collect(Collectors.toList());
    }

    private List<PartitionMetadata> getMetadataForAllPartitions(Set<Integer> filterPartitions, Optional<Set<Locator.KsqlKey>> keys) {
        Set<String> sourceTopicSuffixes = this.findSubtopologySourceTopicSuffixes();
        HashMap activeHostByPartition = new HashMap();
        HashMap standbyHostsByPartition = new HashMap();
        Collection<StreamsMetadata> streamsMetadataCollection = this.getStreamsMetadata();
        for (StreamsMetadata streamsMetadata : streamsMetadataCollection) {
            streamsMetadata.topicPartitions().forEach(tp -> {
                if (sourceTopicSuffixes.stream().anyMatch(suffix -> tp.topic().endsWith((String)suffix))) {
                    activeHostByPartition.compute(tp.partition(), (partition, hostInfo) -> {
                        if (hostInfo != null && !streamsMetadata.hostInfo().equals(hostInfo)) {
                            throw new IllegalStateException("Should only be one active host per partition");
                        }
                        return streamsMetadata.hostInfo();
                    });
                }
            });
            streamsMetadata.standbyTopicPartitions().forEach(tp -> {
                if (sourceTopicSuffixes.stream().anyMatch(suffix -> tp.topic().endsWith((String)suffix))) {
                    standbyHostsByPartition.computeIfAbsent(tp.partition(), p -> new HashSet());
                    ((Set)standbyHostsByPartition.get(tp.partition())).add(streamsMetadata.hostInfo());
                }
            });
        }
        Set partitions = Streams.concat((Stream[])new Stream[]{activeHostByPartition.keySet().stream(), standbyHostsByPartition.keySet().stream()}).collect(Collectors.toSet());
        ArrayList<PartitionMetadata> metadataList = new ArrayList<PartitionMetadata>();
        for (Integer partition : partitions) {
            if (filterPartitions.size() > 0 && !filterPartitions.contains(partition)) {
                LOG.debug("Ignoring partition {} because partition is not included in lookup.", (Object)partition);
                continue;
            }
            HostInfo activeHost = activeHostByPartition.getOrDefault(partition, StreamsMetadataState.UNKNOWN_HOST);
            Set<HostInfo> standbyHosts = standbyHostsByPartition.getOrDefault(partition, Collections.emptySet());
            metadataList.add(new PartitionMetadata(activeHost, standbyHosts, partition, keys));
        }
        return metadataList;
    }

    @VisibleForTesting
    protected KeyQueryMetadata getKeyQueryMetadata(Locator.KsqlKey key) {
        if (this.sharedRuntimesEnabled && this.kafkaStreams instanceof KafkaStreamsNamedTopologyWrapper) {
            return ((KafkaStreamsNamedTopologyWrapper)this.kafkaStreams).queryMetadataForKey(this.storeName, (Object)key.getKey(), this.keySerializer, this.queryId);
        }
        try {
            return this.kafkaStreams.queryMetadataForKey(this.storeName, (Object)key.getKey(), this.keySerializer);
        }
        catch (IllegalStateException e) {
            return KeyQueryMetadata.NOT_AVAILABLE;
        }
    }

    @VisibleForTesting
    protected Collection<StreamsMetadata> getStreamsMetadata() {
        if (this.sharedRuntimesEnabled && this.kafkaStreams instanceof KafkaStreamsNamedTopologyWrapper) {
            return ((KafkaStreamsNamedTopologyWrapper)this.kafkaStreams).streamsMetadataForStore(this.storeName, this.queryId);
        }
        return this.kafkaStreams.streamsMetadataForStore(this.storeName);
    }

    private Set<String> findSubtopologySourceTopicSuffixes() {
        for (TopologyDescription.Subtopology subtopology : this.topology.describe().subtopologies()) {
            boolean containsStateStore = false;
            for (TopologyDescription.Node node : subtopology.nodes()) {
                TopologyDescription.Processor processor;
                if (!(node instanceof TopologyDescription.Processor) || !(processor = (TopologyDescription.Processor)node).stores().contains(this.storeName)) continue;
                containsStateStore = true;
            }
            if (!containsStateStore) continue;
            for (TopologyDescription.Node node : subtopology.nodes()) {
                if (!(node instanceof TopologyDescription.Source)) continue;
                TopologyDescription.Source source = (TopologyDescription.Source)node;
                Preconditions.checkNotNull((Object)source.topicSet(), (Object)"Expecting topic set, not regex");
                return source.topicSet();
            }
            throw new IllegalStateException("Failed to find source with topics");
        }
        throw new IllegalStateException("Failed to find state store " + this.storeName);
    }

    private List<Locator.KsqlNode> getFilteredHosts(RoutingOptions routingOptions, RoutingFilter.RoutingFilterFactory routingFilterFactory, HostInfo activeHost, Set<HostInfo> standByHosts, int partition) {
        Object allHosts;
        if (routingOptions.getIsSkipForwardRequest()) {
            LOG.debug("Before filtering: Local host {} ", (Object)this.localHost);
            allHosts = ImmutableList.of((Object)new KsqlHostInfo(this.localHost.getHost(), this.localHost.getPort()));
        } else {
            LOG.debug("Before filtering: Active host {} , standby hosts {}", (Object)activeHost, standByHosts);
            allHosts = Stream.concat(Stream.of(activeHost), standByHosts.stream()).map(this::asKsqlHost).collect(Collectors.toList());
        }
        RoutingFilter routingFilter = routingFilterFactory.createRoutingFilter(routingOptions, (List<KsqlHostInfo>)allHosts, activeHost, this.queryId, this.storeName, partition);
        ImmutableList filteredHosts = (ImmutableList)allHosts.stream().map(routingFilter::filter).map(this::asNode).collect(ImmutableList.toImmutableList());
        LOG.debug("Filtered and ordered hosts: {}", (Object)filteredHosts);
        return filteredHosts;
    }

    @VisibleForTesting
    KsqlHostInfo asKsqlHost(HostInfo hostInfo) {
        return new KsqlHostInfo(hostInfo.host(), hostInfo.port());
    }

    @VisibleForTesting
    Locator.KsqlNode asNode(RoutingFilter.Host host) {
        return new Node(this.isLocalHost(host.getSourceInfo()), this.buildLocation(host.getSourceInfo()), host);
    }

    private boolean isLocalHost(KsqlHostInfo hostInfo) {
        if (hostInfo.port() != this.localHost.getPort()) {
            return false;
        }
        return hostInfo.host().equalsIgnoreCase(this.localHost.getHost()) || hostInfo.host().equalsIgnoreCase("localhost");
    }

    private URI buildLocation(KsqlHostInfo remoteInfo) {
        try {
            return new URL(this.localHost.getProtocol(), remoteInfo.host(), remoteInfo.port(), "/").toURI();
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to convert remote host info to URL. remoteInfo: " + remoteInfo);
        }
    }

    private static class PartitionMetadata {
        private final HostInfo activeHost;
        private final Set<HostInfo> standbyHosts;
        private final int partition;
        private final Optional<Set<Locator.KsqlKey>> keys;

        PartitionMetadata(HostInfo activeHost, Set<HostInfo> standbyHosts, int partition, Optional<Set<Locator.KsqlKey>> keys) {
            this.activeHost = activeHost;
            this.standbyHosts = standbyHosts;
            this.partition = partition;
            this.keys = keys;
        }

        public HostInfo getActiveHost() {
            return this.activeHost;
        }

        public Set<HostInfo> getStandbyHosts() {
            return this.standbyHosts;
        }

        public int getPartition() {
            return this.partition;
        }

        public Optional<Set<Locator.KsqlKey>> getKeys() {
            return this.keys;
        }
    }

    @VisibleForTesting
    public static final class PartitionLocation
    implements Locator.KsqlPartitionLocation {
        private final Optional<Set<Locator.KsqlKey>> keys;
        private final int partition;
        private final ImmutableList<Locator.KsqlNode> nodes;

        public PartitionLocation(Optional<Set<Locator.KsqlKey>> keys, int partition, List<Locator.KsqlNode> nodes) {
            this.keys = keys;
            this.partition = partition;
            this.nodes = ImmutableList.copyOf(nodes);
        }

        @Override
        public Optional<Set<Locator.KsqlKey>> getKeys() {
            return this.keys;
        }

        @Override
        public Locator.KsqlPartitionLocation removeFilteredHosts() {
            return new PartitionLocation(this.keys, this.partition, this.nodes.stream().filter(node -> node.getHost().isSelected()).collect(Collectors.toList()));
        }

        @Override
        public Locator.KsqlPartitionLocation removeHeadHost() {
            if (this.nodes.isEmpty()) {
                return new PartitionLocation(this.keys, this.partition, this.nodes.stream().collect(Collectors.toList()));
            }
            Locator.KsqlNode headNode = (Locator.KsqlNode)this.nodes.get(0);
            return new PartitionLocation(this.keys, this.partition, this.nodes.stream().filter(node -> !node.equals(headNode)).collect(Collectors.toList()));
        }

        @Override
        @SuppressFBWarnings(value={"EI_EXPOSE_REP"}, justification="nodes is ImmutableList")
        public List<Locator.KsqlNode> getNodes() {
            return this.nodes;
        }

        @Override
        public int getPartition() {
            return this.partition;
        }

        public String toString() {
            return " PartitionLocations {keys: " + this.keys + " , partition: " + this.partition + " , nodes: " + this.nodes + " } ";
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            PartitionLocation that = (PartitionLocation)o;
            return this.partition == that.partition && Objects.equals(this.keys, that.keys) && this.nodes.size() == that.nodes.size() && this.nodes.containsAll(that.nodes);
        }

        public int hashCode() {
            return Objects.hash(this.keys, this.partition, this.nodes);
        }
    }

    @Immutable
    @VisibleForTesting
    public static final class Node
    implements Locator.KsqlNode {
        private final boolean local;
        private final URI location;
        private final RoutingFilter.Host host;

        private Node(boolean local, URI location, RoutingFilter.Host hostInfo) {
            this.local = local;
            this.location = Objects.requireNonNull(location, "location");
            this.host = Objects.requireNonNull(hostInfo, "hostInfo");
        }

        @Override
        public boolean isLocal() {
            return this.local;
        }

        @Override
        public URI location() {
            try {
                return new URI(this.location.toString());
            }
            catch (URISyntaxException fatalError) {
                throw new IllegalStateException("Could not deep copy URI: " + this.location);
            }
        }

        @Override
        public RoutingFilter.Host getHost() {
            return this.host;
        }

        public String toString() {
            return "Node{local = " + this.local + ", location = " + this.location + ", Host = " + this.host + "}";
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Node that = (Node)o;
            return this.local == that.local && this.location.equals(that.location);
        }

        public int hashCode() {
            return Objects.hash(this.local, this.location);
        }
    }
}

