/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.license;

import io.confluent.license.CPForCCLicense;
import io.confluent.license.CSFLELicense;
import io.confluent.license.ExpiredLicenseException;
import io.confluent.license.ExpiredLicensesException;
import io.confluent.license.InvalidLicenseException;
import io.confluent.license.License;
import io.confluent.license.LicenseAutoGenerationPolicy;
import io.confluent.license.LicenseChanged;
import io.confluent.license.LicenseManagerPurpose;
import io.confluent.license.LicenseManagers;
import io.confluent.license.LicenseSelectionStrategy;
import io.confluent.license.LicenseStore;
import io.confluent.license.store.ConfluentLicenseStore;
import io.confluent.license.util.StringUtils;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.common.errors.ClusterAuthorizationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.utils.Time;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.InvalidJwtSignatureException;
import org.jose4j.lang.JoseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LicenseManager {
    private static final Logger log = LoggerFactory.getLogger(LicenseManager.class);
    protected static final long WARN_THRESHOLD_DAYS = 10L;
    private final String invalidLicenseErrorMessage = "Supplied license is invalid.";
    private final ConfluentLicenseStore licenseStore;
    private final Time time;
    private final ClusterClient primaryClusterClient;
    private final Map<String, ClusterClient> clusterClients = new ConcurrentHashMap<String, ClusterClient>();
    private final CopyOnWriteArrayList<Consumer<LicenseChanged>> listeners = new CopyOnWriteArrayList();
    private final License configuredLicense;
    private Map<License.DeploymentScope, InstalledLicense> inMemoryLicenseState = new HashMap<License.DeploymentScope, InstalledLicense>();
    private final LicenseAutoGenerationPolicy licenseAutoGenerationPolicy;
    private boolean isConfiguredLicenseInvalid = false;
    private boolean allowInvalidLicenseToRun = false;
    private long timeUntilLicenseExpirationMs = -1L;
    static final long DEFAULT_INITIAL_DELAY = 1L;
    static final long DEFAULT_PERIOD = 1L;
    private final LicenseManagerPurpose licenseManagerPurpose;
    static final TimeUnit DEFAULT_TIMEUNIT = TimeUnit.DAYS;
    private final AtomicReference<Function<List<License>, License>> selectionFunction = new AtomicReference();

    public LicenseManager(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig) {
        this(topic, producerConfig, consumerConfig, topicConfig, LicenseManagerPurpose.PRIMARY);
    }

    public LicenseManager(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig, LicenseManagerPurpose licenseManagerPurpose) {
        this(new BasicClusterClient(topicConfig), new LicenseStore(topic, producerConfig, consumerConfig, topicConfig), Time.SYSTEM, licenseManagerPurpose);
    }

    public LicenseManager(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig, Duration topicCreateTimeout, Duration retryBackoffMinMs, Duration retryBackoffMaxMs) {
        this(new BasicClusterClient(topicConfig), new LicenseStore(topic, producerConfig, consumerConfig, topicConfig, topicCreateTimeout, retryBackoffMinMs, retryBackoffMaxMs, Time.SYSTEM), Time.SYSTEM);
    }

    public LicenseManager(Map<String, Object> topicConfig, LicenseStore licenseStore, String configuredLicense, boolean startStore) {
        this(new BasicClusterClient(topicConfig), licenseStore, Time.SYSTEM, configuredLicense, startStore, false);
    }

    public LicenseManager(Map<String, Object> topicConfig, LicenseStore licenseStore, String configuredLicense, boolean startStore, boolean allowInvalidLicenseToRun) {
        this(new BasicClusterClient(topicConfig), licenseStore, Time.SYSTEM, configuredLicense, startStore, allowInvalidLicenseToRun);
    }

    protected LicenseManager(ClusterClient primaryClusterClient, LicenseStore licenseStore, Time time) {
        this(primaryClusterClient, licenseStore, time, LicenseManagerPurpose.PRIMARY);
    }

    protected LicenseManager(ClusterClient primaryClusterClient, LicenseStore licenseStore, Time time, LicenseManagerPurpose licenseManagerPurpose) {
        this(primaryClusterClient, licenseStore, time, null, true, false, licenseManagerPurpose);
    }

    protected LicenseManager(ClusterClient primaryClusterClient, LicenseStore licenseStore, Time time, String configuredLicenseStr, boolean startStore, boolean allowInvalidLicenseToRun) {
        this(primaryClusterClient, licenseStore, time, configuredLicenseStr, startStore, allowInvalidLicenseToRun, LicenseManagerPurpose.PRIMARY);
    }

    protected LicenseManager(ClusterClient primaryClusterClient, LicenseStore licenseStore, Time time, String configuredLicenseStr, boolean startStore, boolean allowInvalidLicenseToRun, LicenseManagerPurpose licenseManagerPurpose) {
        this.licenseManagerPurpose = licenseManagerPurpose;
        this.licenseStore = licenseStore;
        this.time = time;
        this.configuredLicense = this.readAndValidateConfiguredLicense(configuredLicenseStr);
        this.primaryClusterClient = primaryClusterClient;
        this.allowInvalidLicenseToRun = allowInvalidLicenseToRun;
        this.licenseAutoGenerationPolicy = LicenseAutoGenerationPolicy.DEFAULT;
        if (startStore) {
            licenseStore.start();
        }
    }

    public LicenseManager(ConfluentLicenseStore licenseStore, String configuredLicenseStr) {
        this(licenseStore, configuredLicenseStr, LicenseManagerPurpose.PRIMARY);
    }

    public LicenseManager(ConfluentLicenseStore licenseStore, String configuredLicenseStr, LicenseAutoGenerationPolicy licenseAutoGenerationPolicy) {
        this(licenseStore, configuredLicenseStr, LicenseManagerPurpose.PRIMARY, Time.SYSTEM, licenseAutoGenerationPolicy);
    }

    public LicenseManager(ConfluentLicenseStore licenseStore, String configuredLicenseStr, LicenseManagerPurpose licenseManagerPurpose) {
        this(licenseStore, configuredLicenseStr, licenseManagerPurpose, Time.SYSTEM);
    }

    public LicenseManager(ConfluentLicenseStore licenseStore, String configuredLicenseStr, LicenseManagerPurpose licenseManagerPurpose, Time time) {
        this(licenseStore, configuredLicenseStr, licenseManagerPurpose, time, LicenseAutoGenerationPolicy.DEFAULT);
    }

    public LicenseManager(ConfluentLicenseStore licenseStore, String configuredLicenseStr, LicenseManagerPurpose licenseManagerPurpose, Time time, LicenseAutoGenerationPolicy licenseAutoGenerationPolicy) {
        this.licenseManagerPurpose = licenseManagerPurpose;
        this.licenseStore = licenseStore;
        this.time = time;
        this.primaryClusterClient = null;
        this.configuredLicense = this.readAndValidateConfiguredLicense(configuredLicenseStr);
        this.licenseAutoGenerationPolicy = licenseAutoGenerationPolicy;
    }

    public void addCluster(String key, Map<String, Object> adminConfig) {
        if (adminConfig != null && !adminConfig.isEmpty()) {
            this.addCluster(key, new BasicClusterClient(adminConfig));
        }
    }

    public void addCluster(String key, ClusterClient client) {
        Objects.nonNull(key);
        Objects.nonNull(client);
        this.clusterClients.put(key, client);
    }

    public boolean removeCluster(String key) {
        return key != null && this.clusterClients.remove(key) != null;
    }

    public boolean addListener(Consumer<LicenseChanged> listener) {
        return listener != null && this.listeners.addIfAbsent(listener);
    }

    public boolean removeListener(Consumer<LicenseChanged> listener) {
        return listener != null && this.listeners.remove(listener);
    }

    public void start() {
        this.start(1L, 1L, DEFAULT_TIMEUNIT);
    }

    public void setSchedulingStrategy(Function<List<License>, License> selectionFunction) {
        this.selectionFunction.set(selectionFunction);
    }

    public void start(long initialDelay, long period, TimeUnit timeUnit) {
        LicenseSelectionStrategy licenseSelectionStrategy = new LicenseSelectionStrategy(this.selectionFunction.get());
        if (initialDelay < 0L) {
            throw new IllegalArgumentException("The initial delay should be non-negative.");
        }
        if (period <= 0L) {
            throw new IllegalArgumentException("The period should be positive.");
        }
        LicenseManagers.INSTANCE.start(this, initialDelay, period, timeUnit, licenseSelectionStrategy);
    }

    public void stop() {
        if (!LicenseManagers.INSTANCE.stop(this)) {
            this.doStop();
        }
    }

    protected void doStart() {
        this.licenseStore.start();
    }

    protected void doStop() {
        this.licenseStore.stop();
    }

    public License configuredLicense() {
        return this.configuredLicense;
    }

    public License registerOrValidateAddOnLicense(String license) throws InvalidLicenseException {
        License latestLicense;
        PublicKey publicKey = LicenseManager.loadPublicKey();
        License newLicense = this.readNewOrConfiguredLicense(license, publicKey);
        long now = this.time.milliseconds();
        String reason = "";
        String storedLicenseStr = this.licenseStore.licenseScan();
        License storedLicense = null;
        if (StringUtils.isNotBlank(storedLicenseStr)) {
            try {
                storedLicense = this.readLicense(publicKey, storedLicenseStr, true);
                License.expiration(storedLicense.jwtClaims());
            }
            catch (Throwable t) {
                if (newLicense == null) {
                    throw new InvalidLicenseException("Stored license is invalid", t);
                }
                reason = "New license replaces invalid stored license (" + t.getMessage() + ").";
                storedLicense = null;
            }
        }
        if (newLicense != null) {
            try {
                License.expiration(newLicense.jwtClaims());
            }
            catch (Throwable t) {
                if (storedLicense != null) {
                    newLicense = storedLicense;
                    reason = "Using stored license because new license has an invalid expiration.";
                    log.warn(reason, t);
                }
                throw new InvalidLicenseException("No stored license, and error extracting expiration date from valid license", t);
            }
        }
        if (storedLicense == null) {
            if (newLicense == null && this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
                if (this.isConfiguredLicenseInvalid && this.allowInvalidLicenseToRun) {
                    this.notifyLicense(null, null, null, "Supplied license is invalid.");
                    throw new InvalidLicenseException("Supplied license is invalid.");
                }
                newLicense = this.generateLicense(now);
            }
            this.maybeStoreLicense(newLicense);
            latestLicense = newLicense;
        } else {
            latestLicense = this.storeIfNewLicenseIsBetter(newLicense, storedLicense);
        }
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
            assert (latestLicense != null);
        } else if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.CSFLE) && latestLicense == null) {
            throw new InvalidLicenseException("No valid license found");
        }
        License finalLicense = this.getLicense(latestLicense);
        this.checkLicense(finalLicense, storedLicense, now, reason);
        return finalLicense;
    }

    public License registerOrValidateLicense(String license) throws InvalidLicenseException {
        Map<License.DeploymentScope, License> activeLicenses = this.doRegisterOrValidateMultiLicense(license, TrialLicenceGenerationMode.IF_NO_PRIMARY_LICENSE);
        return activeLicenses.get((Object)License.DeploymentScope.CP) != null ? activeLicenses.get((Object)License.DeploymentScope.CP) : activeLicenses.get((Object)License.DeploymentScope.CP_FOR_CC);
    }

    public List<License> registerOrValidateMultiLicense(String license, boolean skipTrialLicenseGeneration) throws InvalidLicenseException {
        Map<License.DeploymentScope, License> activeLicenses = this.doRegisterOrValidateMultiLicense(license, skipTrialLicenseGeneration ? TrialLicenceGenerationMode.NEVER : TrialLicenceGenerationMode.IF_NO_CP_LICENSE);
        ArrayList<License> result = new ArrayList<License>(activeLicenses.values());
        return Collections.unmodifiableList(result);
    }

    private Map<License.DeploymentScope, License> doRegisterOrValidateMultiLicense(String license, TrialLicenceGenerationMode trialLicenseGenerationMode) {
        License generatedLicense;
        long now = this.time.milliseconds();
        HashMap<License.DeploymentScope, License> activeLicenses = new HashMap<License.DeploymentScope, License>();
        PublicKey publicKey = LicenseManager.loadPublicKey();
        License newLicense = this.readNewOrConfiguredLicense(license, publicKey);
        Map<License.DeploymentScope, InstalledLicense> initialLicenseState = this.getInstalledLicenses(publicKey);
        InstalledLicense finalStoredLicense = this.storeNewLicenseIfBetter(newLicense, initialLicenseState);
        HashMap<License.DeploymentScope, InstalledLicense> finalLicenseState = new HashMap<License.DeploymentScope, InstalledLicense>(initialLicenseState);
        if (finalStoredLicense != null) {
            finalLicenseState.put(finalStoredLicense.getLicense().deploymentScope(), finalStoredLicense);
        }
        activeLicenses.putAll(this.selectValidStoredLicenses(finalLicenseState));
        boolean isNewLicenseInvalidOrExpired = false;
        try {
            isNewLicenseInvalidOrExpired = newLicense != null && !newLicense.isValid();
        }
        catch (Throwable t) {
            isNewLicenseInvalidOrExpired = true;
        }
        if (!isNewLicenseInvalidOrExpired && (generatedLicense = this.generateTrialLicenseIfNeeded(trialLicenseGenerationMode, finalLicenseState, activeLicenses)) != null) {
            finalLicenseState.put(License.DeploymentScope.CP, new InstalledLicense(generatedLicense.serializedForm(), generatedLicense, LicenseStatus.ACTIVE));
            activeLicenses.put(License.DeploymentScope.CP, generatedLicense);
        }
        this.notifyLicenseChanges(this.inMemoryLicenseState, finalLicenseState, now);
        this.validateFinalLicenseState(newLicense, isNewLicenseInvalidOrExpired, finalLicenseState);
        this.inMemoryLicenseState = finalLicenseState;
        return activeLicenses;
    }

    private License generateTrialLicenseIfNeeded(TrialLicenceGenerationMode trialLicenseGenerationMode, Map<License.DeploymentScope, InstalledLicense> finalLicenseState, Map<License.DeploymentScope, License> activeLicenses) {
        boolean hasCPLicenseStored;
        boolean bl = hasCPLicenseStored = finalLicenseState.get((Object)License.DeploymentScope.CP) != null;
        if (trialLicenseGenerationMode.equals((Object)TrialLicenceGenerationMode.NEVER) || hasCPLicenseStored) {
            return null;
        }
        if (trialLicenseGenerationMode.equals((Object)TrialLicenceGenerationMode.IF_NO_PRIMARY_LICENSE) && activeLicenses.isEmpty() || trialLicenseGenerationMode.equals((Object)TrialLicenceGenerationMode.IF_NO_CP_LICENSE)) {
            log.debug("Generating license (trial/free-tier) subject to eligibility.");
            License generatedLicense = this.generateLicense(this.time.milliseconds());
            if (generatedLicense.isTrial()) {
                this.licenseStore.registerLicense(License.DeploymentScope.CP, generatedLicense.serializedForm());
            }
            return this.getPrimaryLicense(generatedLicense);
        }
        return null;
    }

    InstalledLicense storeNewLicenseIfBetter(License newLicense, Map<License.DeploymentScope, InstalledLicense> initialLicenseState) {
        if (newLicense != null && this.isNewJwtHasValidExpiry(newLicense, initialLicenseState)) {
            if (newLicense.deploymentScope() == License.DeploymentScope.CP) {
                return this.storeBestLicense(newLicense, initialLicenseState, License.DeploymentScope.CP);
            }
            if (newLicense.deploymentScope() == License.DeploymentScope.CP_FOR_CC) {
                return this.storeBestLicense(newLicense, initialLicenseState, License.DeploymentScope.CP_FOR_CC);
            }
        }
        return null;
    }

    InstalledLicense storeBestLicense(License newLicense, Map<License.DeploymentScope, InstalledLicense> initialLicenseState, License.DeploymentScope deploymentScope) {
        InstalledLicense cpLicense = initialLicenseState.get((Object)deploymentScope);
        License storedLicense = null;
        if (cpLicense != null && cpLicense.isValid().booleanValue()) {
            storedLicense = this.storeIfNewLicenseIsBetter(newLicense, cpLicense.getLicense());
        } else if (newLicense != null && newLicense.isValid()) {
            storedLicense = newLicense;
            this.licenseStore.registerLicense(deploymentScope, newLicense.serializedForm());
        }
        return storedLicense != null ? new InstalledLicense(storedLicense.serializedForm(), storedLicense, LicenseStatus.ACTIVE) : null;
    }

    private Map<License.DeploymentScope, InstalledLicense> getInstalledLicenses(PublicKey publicKey) {
        Map<License.DeploymentScope, String> installedLicenseStrs = this.licenseStore.getLatestLicenses();
        Map<License.DeploymentScope, InstalledLicense> installedLicenses = this.parseAndValidateStoredLicenses(publicKey, installedLicenseStrs);
        return installedLicenses;
    }

    private Map<License.DeploymentScope, InstalledLicense> parseAndValidateStoredLicenses(PublicKey publicKey, Map<License.DeploymentScope, String> licenseStrings) {
        if (licenseStrings == null || licenseStrings.isEmpty()) {
            return Collections.emptyMap();
        }
        return licenseStrings.entrySet().stream().filter(entry -> StringUtils.isNotBlank((String)entry.getValue())).collect(Collectors.toMap(Map.Entry::getKey, entry -> this.readStoredLicenseStrsWithState(publicKey, (License.DeploymentScope)((Object)((Object)entry.getKey())), (String)entry.getValue())));
    }

    boolean hasAnyActiveLicenseInstalled(Map<License.DeploymentScope, InstalledLicense> licenseState) {
        boolean hasAnyActiveLicense = false;
        for (InstalledLicense installed : licenseState.values()) {
            if (installed.getStatus() != LicenseStatus.ACTIVE) continue;
            hasAnyActiveLicense = true;
        }
        return hasAnyActiveLicense;
    }

    private boolean isNewJwtHasValidExpiry(License newLicense, Map<License.DeploymentScope, InstalledLicense> licenseState) {
        if (newLicense == null) {
            return false;
        }
        boolean anyValidLicensePresent = this.hasAnyActiveLicenseInstalled(licenseState);
        boolean isNewLicenseJwtHasValidExpiry = true;
        try {
            License.expiration(newLicense.jwtClaims());
        }
        catch (Throwable t) {
            isNewLicenseJwtHasValidExpiry = false;
            if (anyValidLicensePresent) {
                log.warn("Using stored license because new license has an invalid expiration.", t);
            }
            throw new InvalidLicenseException("No valid stored license found, and error extracting expiration date from valid license. Please install a valid license.", t);
        }
        return isNewLicenseJwtHasValidExpiry;
    }

    private Map<License.DeploymentScope, License> selectValidStoredLicenses(Map<License.DeploymentScope, InstalledLicense> licenseState) throws InvalidLicenseException {
        HashMap<License.DeploymentScope, License> installedActiveLicenses = new HashMap<License.DeploymentScope, License>();
        for (License.DeploymentScope scope : License.DeploymentScope.values()) {
            InstalledLicense installed = licenseState.get((Object)scope);
            if (installed == null || !installed.isValid().booleanValue()) continue;
            installedActiveLicenses.put(scope, installed.getLicense());
        }
        return installedActiveLicenses;
    }

    void notifyLicenseChanges(Map<License.DeploymentScope, InstalledLicense> initialLicensesState, Map<License.DeploymentScope, InstalledLicense> finalLicensesStates, long now) {
        ArrayList<License> activeLicenses = new ArrayList<License>();
        activeLicenses.addAll(this.selectValidStoredLicenses(finalLicensesStates).values());
        for (License.DeploymentScope scope : License.DeploymentScope.values()) {
            License newStoredLicense;
            InstalledLicense initialLicenseState = initialLicensesState.get((Object)scope);
            InstalledLicense finalLicenseState = finalLicensesStates.get((Object)scope);
            License oldStoredLicense = initialLicenseState != null ? initialLicenseState.getLicense() : null;
            License license = newStoredLicense = finalLicenseState != null ? finalLicenseState.getLicense() : null;
            if (newStoredLicense == null) continue;
            String reason = "";
            if (initialLicenseState != null && initialLicenseState.isInvalid().booleanValue() && finalLicenseState != null && finalLicenseState.isValid().booleanValue()) {
                String errorMessage = initialLicenseState.getErrorMessage();
                reason = "New license replaces invalid stored license" + (errorMessage != null ? " (" + errorMessage + ")" : "");
            }
            if (initialLicenseState != null && initialLicenseState.isExpired().booleanValue() && finalLicenseState != null && finalLicenseState.isValid().booleanValue()) {
                reason = "New license replaces expired stored license";
            }
            try {
                this.checkPrimaryLicense(newStoredLicense, oldStoredLicense, activeLicenses, now, reason);
            }
            catch (ExpiredLicenseException e) {
                String licenseType = scope == License.DeploymentScope.CP_FOR_CC ? "Enterprise license for 'CP for CC' Subscription" : "TRIAL or Enterprise license for CP Subscription";
                log.warn("Expired {} license: {}", (Object)licenseType, (Object)e.getMessage());
            }
        }
    }

    List<License> getExpiredLicenses(Map<License.DeploymentScope, InstalledLicense> licenseState) {
        ArrayList<License> expiredLicenses = new ArrayList<License>();
        for (InstalledLicense installed : licenseState.values()) {
            if (installed.getStatus() != LicenseStatus.EXPIRED) continue;
            expiredLicenses.add(installed.getLicense());
        }
        return expiredLicenses;
    }

    List<License> getInvalidLicenses(Map<License.DeploymentScope, InstalledLicense> licenseState) {
        ArrayList<License> expiredLicenses = new ArrayList<License>();
        for (InstalledLicense installed : licenseState.values()) {
            if (installed.getStatus() != LicenseStatus.INVALID) continue;
            expiredLicenses.add(installed.getLicense());
        }
        return expiredLicenses;
    }

    private void validateFinalLicenseState(License newLicense, boolean isNewLicenseInvalidOrExpired, Map<License.DeploymentScope, InstalledLicense> finalLicenseState) throws InvalidLicenseException {
        if (this.hasAnyActiveLicenseInstalled(finalLicenseState)) {
            return;
        }
        if (newLicense != null && isNewLicenseInvalidOrExpired) {
            throw new InvalidLicenseException("Supplied license is invalid or expired. No valid stored license found. Please install a valid license.");
        }
        List<License> expiredLicenses = this.getExpiredLicenses(finalLicenseState);
        List<License> invalidLicenses = this.getInvalidLicenses(finalLicenseState);
        if (!expiredLicenses.isEmpty() && finalLicenseState.size() == expiredLicenses.size()) {
            throw new ExpiredLicensesException(expiredLicenses, "Stored License has expired. Please install a valid license.");
        }
        if (!invalidLicenses.isEmpty() && finalLicenseState.size() == invalidLicenses.size()) {
            throw new InvalidLicenseException("Stored license is invalid. Please install a valid license.");
        }
        throw new InvalidLicenseException("No valid license found. Please install a valid license.");
    }

    private License readNewOrConfiguredLicense(String license, PublicKey publicKey) throws InvalidLicenseException {
        if (StringUtils.isNotBlank(license)) {
            try {
                License newLicense = this.readLicense(publicKey, license, false);
                log.debug("Found valid new license: {}", (Object)newLicense);
                return newLicense;
            }
            catch (InvalidLicenseException e) {
                this.notifyLicense(null, null, null, "Supplied license is invalid. " + e.getMessage());
                throw new InvalidLicenseException("Supplied license is invalid. " + e.getMessage(), (Throwable)((Object)e));
            }
        }
        return this.configuredLicense;
    }

    private void maybeStoreLicense(License license) throws InvalidLicenseException {
        if (license == null) {
            return;
        }
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY) && !license.isFreeTier()) {
            log.debug("Storing license: {}", (Object)license);
            this.licenseStore.registerLicense(license.serializedForm());
        } else if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.CSFLE) && license.type().equals((Object)License.Type.CSFLE)) {
            log.debug("Storing CSFLE license: {}", (Object)license);
            this.licenseStore.registerLicense(license.serializedForm());
        }
    }

    private InstalledLicense readStoredLicenseStrsWithState(PublicKey publicKey, License.DeploymentScope deploymentScope, String licenseStr) {
        try {
            License license = this.readLicenseStrs(publicKey, deploymentScope, licenseStr);
            return new InstalledLicense(licenseStr, license, this.determineLicenseStatus(license));
        }
        catch (Exception e) {
            log.warn("Stored {} is invalid", (Object)this.licenseSubscriptionLabel(deploymentScope), (Object)e);
            return new InstalledLicense(licenseStr, null, LicenseStatus.INVALID, e.getMessage());
        }
        catch (Throwable e) {
            log.warn("Stored {} is invalid", (Object)this.licenseSubscriptionLabel(deploymentScope), (Object)e);
            return new InstalledLicense(licenseStr, null, LicenseStatus.INVALID, e.getMessage());
        }
    }

    private String licenseSubscriptionLabel(License.DeploymentScope deploymentScope) {
        return deploymentScope.equals((Object)License.DeploymentScope.CP_FOR_CC) ? "Enterprise license for 'CP for CC' Subscription" : "Enterprise license for CP Subscription";
    }

    private License readLicenseStrs(PublicKey publicKey, License.DeploymentScope deploymentScope, String licenseStr) throws Throwable {
        if (deploymentScope.equals((Object)License.DeploymentScope.CP_FOR_CC)) {
            JwtClaims jwtClaims = License.verify(publicKey, licenseStr);
            return this.buildLicenseFromJwtClaims(jwtClaims, licenseStr);
        }
        License license = this.readLicense(publicKey, licenseStr, true);
        License.expiration(license.jwtClaims());
        return license;
    }

    private LicenseStatus determineLicenseStatus(License license) {
        if (license != null && license.isValid()) {
            return LicenseStatus.ACTIVE;
        }
        return license.isExpired() ? LicenseStatus.EXPIRED : LicenseStatus.INVALID;
    }

    private License storeIfNewLicenseIsBetter(License newLicense, License storedLicense) {
        assert (storedLicense != null);
        if (newLicense == null) {
            return storedLicense;
        }
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
            return this.storeIFNewPrimaryLicenseIsBetter(newLicense, storedLicense);
        }
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.CSFLE)) {
            return this.storeIfNewCSFLELicenseIsBetter(newLicense, storedLicense);
        }
        return storedLicense;
    }

    private License storeIFNewPrimaryLicenseIsBetter(License newLicense, License storedLicense) {
        assert (storedLicense != null);
        if (newLicense.isFreeTier()) {
            return storedLicense;
        }
        if (storedLicense.isFreeTier()) {
            log.debug("Storing updated primary license by replacing free tier");
            this.licenseStore.registerLicense(newLicense.deploymentScope(), newLicense.serializedForm());
            return newLicense;
        }
        if (newLicense.isAddonUpdatedWithSameExpiry(storedLicense) || storedLicense.expiresBefore(newLicense)) {
            log.debug("Storing updated primary license");
            this.licenseStore.registerLicense(newLicense.deploymentScope(), newLicense.serializedForm());
            return newLicense;
        }
        return storedLicense;
    }

    private License storeIfNewCSFLELicenseIsBetter(License newLicense, License storedLicense) {
        assert (storedLicense != null);
        if (storedLicense.expiresBefore(newLicense) || newLicense.isAddonUpdatedWithSameExpiry(storedLicense)) {
            log.debug("Storing updated CSFLE license");
            this.licenseStore.registerLicense(newLicense.serializedForm());
            return newLicense;
        }
        return storedLicense;
    }

    private License getLicense(License latestLicense) {
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
            return this.getPrimaryLicense(latestLicense);
        }
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.CSFLE)) {
            return this.getCSFLELicense(latestLicense);
        }
        throw new UnsupportedOperationException("Unsupported Purpose in License Manager");
    }

    private License getPrimaryLicense(License latestLicense) {
        if (latestLicense.deploymentScope() == License.DeploymentScope.CP_FOR_CC) {
            return new CPForCCLicense(latestLicense.jwtClaims(), this.time, latestLicense.serializedForm(), this.primaryClusterClient != null ? this.primaryClusterClient.clusterId() : "-1");
        }
        return this.primaryClusterClient != null ? new License(latestLicense.jwtClaims(), this.time, latestLicense.serializedForm(), this.primaryClusterClient.clusterId()) : new License(latestLicense.jwtClaims(), this.time, latestLicense.serializedForm(), "-1");
    }

    private License getCSFLELicense(License latestLicense) {
        return new CSFLELicense(latestLicense.jwtClaims(), this.time, latestLicense.serializedForm());
    }

    protected final License readLicense(PublicKey publicKey, String licenseStr, boolean stored) throws InvalidLicenseException {
        try {
            JwtClaims givenJwtClaims = stored && this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY) ? License.verifyStored(publicKey, licenseStr) : License.verify(publicKey, licenseStr);
            return this.buildLicenseFromJwtClaims(givenJwtClaims, licenseStr);
        }
        catch (InvalidJwtSignatureException e) {
            throw new InvalidLicenseException("Invalid signature", e);
        }
        catch (InvalidJwtException e) {
            throw new InvalidLicenseException("License does not match expected form.", e);
        }
        catch (InvalidLicenseException e) {
            throw e;
        }
        catch (Throwable e) {
            throw new InvalidLicenseException("Invalid license with invalid expiration.", e);
        }
    }

    private License buildPrimaryLicenseFromJwtClaims(JwtClaims claims, String licenseStr) throws InvalidLicenseException {
        try {
            if (claims.getSubject() == null) {
                return new CPForCCLicense(claims, this.time, licenseStr, this.primaryClusterClient != null ? this.primaryClusterClient.clusterId() : "-1");
            }
        }
        catch (MalformedClaimException e) {
            throw new InvalidLicenseException("Malformed license JWT.", e);
        }
        return new License(claims, this.time, licenseStr, this.primaryClusterClient != null ? this.primaryClusterClient.clusterId() : "-1");
    }

    private License buildLicenseFromJwtClaims(JwtClaims claims, String licenseStr) throws InvalidLicenseException {
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
            return this.buildPrimaryLicenseFromJwtClaims(claims, licenseStr);
        }
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.CSFLE)) {
            return new CSFLELicense(claims, this.time, licenseStr);
        }
        throw new UnsupportedOperationException("Unsupported License Manager Purpose");
    }

    private License readAndValidateConfiguredLicense(String licenseStr) {
        if (StringUtils.isNotBlank(licenseStr)) {
            try {
                License license = this.readLicense(LicenseManager.loadPublicKey(), licenseStr, false);
                try {
                    License.expiration(license.jwtClaims());
                }
                catch (Throwable t) {
                    throw new InvalidLicenseException("Configured license has invalid expiration", t);
                }
                this.checkLicense(license, license, this.time.milliseconds(), "Configured license");
                log.debug("Found valid configured license: {}", (Object)license);
                return license;
            }
            catch (InvalidLicenseException e) {
                log.warn("Supplied license is invalid. Will attempt to use stored license.", (Throwable)((Object)e));
                this.isConfiguredLicenseInvalid = true;
                return null;
            }
        }
        return null;
    }

    protected License generateLicense(long now) throws InvalidLicenseException {
        if (!this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
            throw new UnsupportedOperationException(String.format("Cannot Generate license for %s", new Object[]{this.licenseManagerPurpose}));
        }
        if (this.licenseAutoGenerationPolicy == LicenseAutoGenerationPolicy.TRIAL_ONLY) {
            return this.generateNewTrialLicense(now);
        }
        if (this.hasOnlySingleNodeClusters()) {
            JwtClaims jwtClaims = License.baseClaims("free tier", Long.MAX_VALUE, true);
            jwtClaims.setClaim("licenseType", (Object)"free");
            String generatedLicense = LicenseManager.generateFreeLicense(jwtClaims);
            License license = new License(jwtClaims, this.time, generatedLicense);
            log.debug("All single-node cluster checks satisfied; using {}", (Object)license);
            return license;
        }
        return this.generateNewTrialLicense(now);
    }

    private License generateNewTrialLicense(long now) throws InvalidLicenseException {
        JwtClaims trialClaims = License.baseClaims("trial", now + TimeUnit.DAYS.toMillis(30L) + 1000L, true);
        trialClaims.setClaim("licenseType", (Object)"trial");
        String generatedLicense = License.generateTrialLicense(trialClaims);
        License license = new License(trialClaims, this.time, generatedLicense);
        log.debug("Creating new {}", (Object)license);
        return license;
    }

    private void checkLicense(License newLicense, License oldLicense, long now, String reason) throws InvalidLicenseException {
        if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.PRIMARY)) {
            this.checkPrimaryLicense(newLicense, oldLicense, Collections.singletonList(newLicense), now, reason);
        } else if (this.licenseManagerPurpose.equals((Object)LicenseManagerPurpose.CSFLE)) {
            this.checkCSFLEAddOnLicense(newLicense, oldLicense, now, reason);
        } else {
            throw new UnsupportedOperationException("Unsupported License Manager Type");
        }
    }

    private void checkCSFLEAddOnLicense(License newLicense, License oldLicense, long now, String reason) throws InvalidLicenseException {
        if (!newLicense.type().equals((Object)License.Type.CSFLE)) {
            reason = "License Type is not CSFLE";
            log.info(reason);
            LicenseChangedEvent event = new LicenseChangedEvent(newLicense, LicenseChanged.Type.INVALID, Collections.singletonList(newLicense), reason);
            this.notifyEventToListeners(event);
            throw new InvalidLicenseException(reason);
        }
        if (now > newLicense.expirationMillis()) {
            String msg = newLicense.toString();
            if (StringUtils.isBlank(reason)) {
                reason = msg;
            }
            this.notifyLicense(newLicense, null, Collections.singletonList(newLicense), reason);
            throw new ExpiredLicenseException(newLicense, msg);
        }
        if (newLicense.expirationMillis() < Long.MAX_VALUE) {
            String msg = newLicense.toString();
            if (newLicense.timeRemaining(TimeUnit.DAYS) < 10L) {
                log.warn(msg);
            } else {
                log.info(msg);
            }
            reason = StringUtils.isBlank(reason) ? msg : reason + " " + msg;
            this.notifyLicense(newLicense, oldLicense, Collections.singletonList(newLicense), reason);
        }
    }

    private void checkPrimaryLicense(License newLicense, License oldLicense, List<License> finalInstalledLicense, long now, String reason) throws InvalidLicenseException {
        if (newLicense.isFreeTier()) {
            reason = "License for single cluster, single node";
            log.info(reason);
            this.notifyLicense(newLicense, oldLicense, finalInstalledLicense, reason);
        } else {
            if (now > newLicense.expirationMillis()) {
                String msg = newLicense.toString();
                if (StringUtils.isBlank(reason)) {
                    reason = msg;
                }
                this.notifyLicense(newLicense, null, finalInstalledLicense, reason);
                throw new ExpiredLicenseException(newLicense, msg);
            }
            if (newLicense.expirationMillis() < Long.MAX_VALUE) {
                String msg = newLicense.toString();
                if (newLicense.timeRemaining(TimeUnit.DAYS) < 10L) {
                    log.warn(msg);
                } else {
                    log.info(msg);
                }
                reason = StringUtils.isBlank(reason) ? msg : reason + " " + msg;
                this.notifyLicense(newLicense, oldLicense, finalInstalledLicense, reason);
            }
        }
    }

    protected boolean hasOnlySingleNodeClusters() {
        return this.primaryClusterClient != null && this.primaryClusterClient.brokerCount() == 1 && this.clusterClients.values().stream().allMatch(client -> client.brokerCount() == 1);
    }

    protected final void notifyLicense(License newLicense, License oldLicense, List<License> finalInstalledLicense, String reason) {
        if (newLicense == null && oldLicense == null) {
            LicenseChangedEvent event = new LicenseChangedEvent(null, LicenseChanged.Type.INVALID, finalInstalledLicense, reason);
            this.notifyEventToListeners(event);
            return;
        }
        assert (newLicense != null);
        if (newLicense.isEquivalentTo(oldLicense)) {
            log.debug("Skipping notifying {} listeners of unchanged license: {}", (Object)this.listeners.size(), (Object)newLicense);
            return;
        }
        LicenseChanged.Type changeType = LicenseChanged.Type.UPDATED;
        if (newLicense.isExpired()) {
            changeType = LicenseChanged.Type.EXPIRED;
        } else if (newLicense.isRenewalOf(oldLicense)) {
            changeType = LicenseChanged.Type.RENEWAL;
        }
        LicenseChangedEvent event = new LicenseChangedEvent(newLicense, changeType, finalInstalledLicense, reason);
        this.notifyEventToListeners(event);
    }

    private void notifyEventToListeners(LicenseChangedEvent event) {
        log.debug("Notifying {} listeners of license change: {}", (Object)this.listeners.size(), (Object)event);
        for (Consumer<LicenseChanged> listener : this.listeners) {
            try {
                listener.accept(event);
            }
            catch (Throwable t) {
                log.error("Unexpected error when calling license manager listener:", t);
            }
        }
    }

    public static PublicKey loadPublicKey() {
        try {
            return License.loadPublicKey();
        }
        catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new IllegalStateException("Internal license validation error", e);
        }
    }

    private static String generateFreeLicense(JwtClaims claims) throws InvalidLicenseException {
        JsonWebSignature jws = new JsonWebSignature();
        jws.setAlgorithmConstraints(AlgorithmConstraints.NO_CONSTRAINTS);
        jws.setAlgorithmHeaderValue("none");
        jws.setPayload(claims.toJson());
        try {
            return jws.getCompactSerialization();
        }
        catch (JoseException e) {
            log.error("Error while attempting to start free tier: ", (Throwable)e);
            throw new InvalidLicenseException("Error creating license for trial version: ", e);
        }
    }

    public void timeUntilLicenseExpirationMs(long timeMs) {
        this.timeUntilLicenseExpirationMs = timeMs;
    }

    public long timeUntilLicenseExpirationMs() {
        return this.timeUntilLicenseExpirationMs;
    }

    public static enum LicenseStatus {
        ACTIVE("License is valid and active"),
        EXPIRED("License has expired"),
        INVALID("License is invalid or malformed");

        private final String description;

        private LicenseStatus(String description) {
            this.description = description;
        }

        public String getDescription() {
            return this.description;
        }
    }

    private class InstalledLicense {
        private final String licenseStr;
        private final License license;
        private final LicenseStatus status;
        private final String errorMessage;

        InstalledLicense(String licenseStr, License license, LicenseStatus status) {
            this(licenseStr, license, status, null);
        }

        InstalledLicense(String licenseStr, License license, LicenseStatus status, String errorMessage) {
            this.licenseStr = licenseStr;
            this.license = license;
            this.status = status;
            this.errorMessage = errorMessage;
        }

        public License getLicense() {
            return this.license;
        }

        public LicenseStatus getStatus() {
            return this.status;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public Boolean isValid() {
            return LicenseStatus.ACTIVE.equals((Object)this.status);
        }

        public Boolean isInvalid() {
            return LicenseStatus.INVALID.equals((Object)this.status);
        }

        public Boolean isExpired() {
            return LicenseStatus.EXPIRED.equals((Object)this.status);
        }
    }

    protected static class BasicClusterClient
    implements ClusterClient {
        private final Map<String, Object> adminConfig;

        BasicClusterClient(Map<String, Object> adminConfig) {
            this.adminConfig = new HashMap<String, Object>(adminConfig);
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public int brokerCount() {
            try (AdminClient admin = AdminClient.create(this.adminConfig);){
                int count = ((Collection)admin.describeCluster().nodes().get()).size();
                log.debug("Found {} brokers in Kafka cluster at {}", (Object)count, (Object)this);
                int n = count;
                return n;
            }
            catch (ExecutionException e) {
                Throwable cause = e.getCause();
                if (cause instanceof UnsupportedVersionException) {
                    log.debug("Unable to use admin client to connect to older Kafka cluster at {}", (Object)this);
                    return -2;
                }
                if (cause instanceof ClusterAuthorizationException) {
                    log.debug("Not authorized to use admin client to connect to Kafka cluster at {}", (Object)this);
                    return -3;
                }
                if (!(cause instanceof TimeoutException)) return -1;
                log.debug("Timed out waiting to connect to Kafka cluster at {}", (Object)this);
                return -4;
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                return -5;
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public String clusterId() {
            try (AdminClient admin = AdminClient.create(this.adminConfig);){
                String string = (String)admin.describeCluster().clusterId().get();
                return string;
            }
            catch (InterruptedException | ExecutionException e) {
                log.debug("Failed to fetch cLusterId of Kafka cluster at {}", (Object)this);
                return null;
            }
        }

        public String toString() {
            Object servers = this.adminConfig.get("bootstrap.servers");
            return servers == null ? "<unknown>" : servers.toString();
        }
    }

    protected static interface ClusterClient {
        public int brokerCount();

        public String clusterId();
    }

    protected static class LicenseChangedEvent
    implements LicenseChanged {
        private final License license;
        private final List<License> existingLicenses;
        private final LicenseChanged.Type type;
        private final String description;

        LicenseChangedEvent(License license, LicenseChanged.Type type, List<License> existingLicenses, String description) {
            Objects.nonNull((Object)type);
            Objects.nonNull(description);
            this.license = license;
            this.type = type;
            this.description = description;
            this.existingLicenses = existingLicenses;
        }

        @Override
        public License license() {
            return this.license;
        }

        @Override
        public List<License> existingLicenses() {
            return this.existingLicenses;
        }

        @Override
        public LicenseChanged.Type type() {
            return this.type;
        }

        @Override
        public String description() {
            return this.description;
        }

        public int hashCode() {
            return this.type().hashCode();
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj instanceof LicenseChangedEvent) {
                LicenseChangedEvent that = (LicenseChangedEvent)obj;
                return this.type() == that.type() && Objects.equals(this.license(), that.license());
            }
            return false;
        }

        public String toString() {
            return (Object)((Object)this.type) + " " + this.license + " (" + this.description + ")";
        }
    }

    static enum TrialLicenceGenerationMode {
        NEVER,
        IF_NO_CP_LICENSE,
        IF_NO_PRIMARY_LICENSE;

    }
}

