/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.model;

import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.DiskStats;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.model.SortedReplicas;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

public class Disk
implements Comparable<Disk> {
    private static final double DEAD_DISK_CAPACITY = -1.0;
    private final String logDir;
    private double capacity;
    private final Set<Replica> replicas;
    private State state;
    private final Broker broker;
    private double utilization;
    private final Map<String, SortedReplicas> sortedReplicas;

    Disk(String logDir, Broker broker, double diskCapacity) {
        this.logDir = logDir;
        this.broker = broker;
        this.replicas = new HashSet<Replica>();
        this.utilization = 0.0;
        this.sortedReplicas = new HashMap<String, SortedReplicas>();
        if (diskCapacity < 0.0) {
            this.capacity = -1.0;
            this.state = State.DEAD;
        } else {
            this.capacity = diskCapacity;
            this.state = State.ALIVE;
        }
    }

    public String logDir() {
        return this.logDir;
    }

    public double capacity() {
        return this.capacity;
    }

    public State state() {
        return this.state;
    }

    public boolean isAlive() {
        return this.state == State.ALIVE;
    }

    public Set<Replica> replicas() {
        return Collections.unmodifiableSet(this.replicas);
    }

    public Set<Replica> leaderReplicas() {
        return Collections.unmodifiableSet(this.replicas.stream().filter(Replica::isLeader).collect(Collectors.toSet()));
    }

    public Broker broker() {
        return this.broker;
    }

    public double utilization() {
        return this.utilization;
    }

    public void setState(State newState) {
        this.state = newState;
        if (this.state == State.DEAD) {
            this.capacity = -1.0;
        }
    }

    void addReplica(Replica replica) {
        if (this.replicas.contains(replica)) {
            throw new IllegalStateException(String.format("Disk %s already has replica %s", this.logDir, replica.topicPartition()));
        }
        this.utilization += replica.load().expectedUtilizationFor(Resource.DISK);
        this.replicas.add(replica);
        this.sortedReplicas.values().forEach(sr -> sr.add(replica));
    }

    void addReplicaLoad(Replica replica) {
        this.utilization += replica.load().expectedUtilizationFor(Resource.DISK);
    }

    void removeReplica(Replica replica) {
        if (!this.replicas.contains(replica)) {
            throw new IllegalStateException(String.format("Disk %s does not has replica %s", this.logDir, replica.topicPartition()));
        }
        this.utilization -= replica.load().expectedUtilizationFor(Resource.DISK);
        this.replicas.remove(replica);
        this.sortedReplicas.values().forEach(sr -> sr.remove(replica));
    }

    public void trackSortedReplicas(String sortName, Function<Replica, Boolean> selectionFunc, Function<Replica, Integer> priorityFunc, Function<Replica, Double> scoreFunc) {
        this.sortedReplicas.putIfAbsent(sortName, new SortedReplicas(this.broker, this, selectionFunc, priorityFunc, scoreFunc, true));
    }

    public void trackSortedReplicas(String sortName, Function<Replica, Boolean> selectionFunc, Function<Replica, Double> scoreFunc) {
        this.sortedReplicas.putIfAbsent(sortName, new SortedReplicas(this.broker, this, selectionFunc, r1 -> 0, scoreFunc, true));
    }

    public void untrackSortedReplicas(String sortName) {
        this.sortedReplicas.remove(sortName);
    }

    public SortedReplicas trackedSortedReplicas(String sortName) {
        SortedReplicas sortedReplicas = this.sortedReplicas.get(sortName);
        if (sortedReplicas == null) {
            throw new IllegalStateException("The sort name " + sortName + "  is not found. Make sure trackSortedReplicas() has been called for the sort name");
        }
        return sortedReplicas;
    }

    @Override
    public int compareTo(Disk d) {
        int result = this.broker.compareTo(d.broker());
        if (result == 0) {
            return this.logDir.compareTo(d.logDir());
        }
        return result;
    }

    public boolean equals(Object o) {
        if (!(o instanceof Disk)) {
            return false;
        }
        return this.compareTo((Disk)o) == 0;
    }

    public int hashCode() {
        return Objects.hash(this.broker, this.logDir);
    }

    public void writeTo(OutputStream out) throws IOException {
        String disk = String.format("<Disk logdir=\"%s\" state=\"%s\">%n", new Object[]{this.logDir, this.state});
        out.write(disk.getBytes(StandardCharsets.UTF_8));
        for (Replica replica : this.replicas) {
            replica.writeTo(out);
        }
        out.write("</Disk>%n".getBytes(StandardCharsets.UTF_8));
    }

    public String toString() {
        return String.format("Disk[logdir=%s,state=%s,capacity=%f,replicaCount=%d]", new Object[]{this.logDir, this.state, this.capacity, this.replicas.size()});
    }

    public DiskStats diskStats() {
        return new DiskStats((int)this.replicas.stream().filter(Replica::isLeader).count(), this.replicas.size(), this.utilization, this.capacity);
    }

    public static enum State {
        ALIVE,
        DEAD,
        DEMOTED;

    }
}

