/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.traffic;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Sets;
import io.confluent.kafka.multitenant.utils.AuthUtils;
import io.confluent.kafka.traffic.TrafficNetworkIdAllowedRoutes;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import kafka.server.BrokerSession;
import kafka.server.KafkaConfig;
import kafka.server.MetadataCache;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.Endpoint;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.internals.ConfluentConfigs;
import org.apache.kafka.common.metrics.MeasurableStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.network.PublicCredential;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.utils.ThreadUtils;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.util.Callback;
import org.apache.kafka.connect.util.KafkaBasedLog;
import org.apache.kafka.server.traffic.TrafficNetworkIdRoutes;
import org.apache.kafka.server.traffic.TrafficNetworkIdRoutesStore;
import org.apache.kafka.server.traffic.TrafficNetworkIdRoutesUpdater;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TopicBasedTrafficNetworkIdRoutesUpdater
implements TrafficNetworkIdRoutesUpdater {
    private static final Logger LOG = LoggerFactory.getLogger(TopicBasedTrafficNetworkIdRoutesUpdater.class);
    private static final List<String> SUPPORTED_SASL_MECHANISMS = Arrays.asList("PLAIN", "OAUTHBEARER");
    public static final String METRICS_GROUP = "traffic-metrics";
    private final Sensor topicLoadTimeSensor;
    private List<String> routesListenerNames = Collections.emptyList();
    final AtomicReference<State> state = new AtomicReference<State>(State.NOT_STARTED);
    private final Time time;
    private String sessionUuid;
    private String clusterNetworkId;
    private String networkIdRoutesTopic;
    private final Map<String, ?> interBrokerClientConfig;
    private final ObjectMapper objectMapper;
    protected final Metrics metrics;
    private final AtomicReference<Long> lastSequenceId = new AtomicReference();
    private KafkaBasedLog<String, String> networkIdRoutesLog;
    private final MetadataCache metadataCache;
    private ScheduledExecutorService executorService;
    private long periodicStartTaskMs;
    Future<?> storeStartTaskFuture;

    public TopicBasedTrafficNetworkIdRoutesUpdater(MetadataCache metadataCache, Map<String, ?> interBrokerClientConfig, Metrics metrics) {
        this(metadataCache, interBrokerClientConfig, metrics, Time.SYSTEM);
    }

    public TopicBasedTrafficNetworkIdRoutesUpdater(MetadataCache metadataCache, Map<String, ?> interBrokerClientConfig, Metrics metrics, Time time) {
        this.metadataCache = metadataCache;
        this.time = time;
        this.interBrokerClientConfig = interBrokerClientConfig;
        this.objectMapper = new ObjectMapper();
        this.metrics = metrics;
        this.topicLoadTimeSensor = metrics.sensor("NetworkIdRoutesTopicLoadTime");
        this.topicLoadTimeSensor.add(metrics.metricName("network-id-routes-topic-load-time", METRICS_GROUP, "The loading time for the network ID routes topic."), (MeasurableStat)new Max());
    }

    private boolean checkAndSetEnabledState(Map<String, ?> configs) {
        if (!this.state.get().equals((Object)State.NOT_STARTED)) {
            throw new IllegalStateException("checkAndSetEnabledState called in a non-starting state. Ignoring");
        }
        Boolean enabled = (Boolean)configs.get("confluent.traffic.cdc.network.id.routes.enable");
        if (enabled == null || !enabled.booleanValue()) {
            LOG.info("Loading network ID routes from the sync pipelines is disabled in config {}", (Object)"confluent.traffic.cdc.network.id.routes.enable");
            this.state.set(State.NOT_ENABLED);
            return false;
        }
        return true;
    }

    void configureInternal(KafkaBasedLog<String, String> log, String sessionUuid, String clusterNetworkId, String networkIdRoutesTopic, List<String> routesListenerNames, long periodicStartTaskMs) {
        LOG.warn("configure(KafkaBasedLog<>, ...) called, which should only happen in tests (ignore if this is one)");
        this.sessionUuid = sessionUuid;
        this.clusterNetworkId = clusterNetworkId;
        this.networkIdRoutesTopic = networkIdRoutesTopic;
        this.networkIdRoutesLog = log;
        this.routesListenerNames = routesListenerNames;
        this.periodicStartTaskMs = periodicStartTaskMs;
        LOG.info("Configured an instance for broker session {}", (Object)sessionUuid);
    }

    private String getBrokerNetworkId(Map<String, ?> configs) {
        String nid = (String)configs.get("confluent.traffic.network.id");
        if (nid == null) {
            throw new ConfigException("confluent.traffic.network.id is not set");
        }
        return nid;
    }

    public void configure(Map<String, ?> configs) {
        this.sessionUuid = AuthUtils.getBrokerSessionUuid(configs);
        this.routesListenerNames = ConfluentConfigs.listenerNames((String)"confluent.traffic.cdc.network.id.routes.listener.names", configs, null);
        LOG.info("Configured an instance for broker session {}", (Object)this.sessionUuid);
        if (!this.checkAndSetEnabledState(configs)) {
            return;
        }
        this.clusterNetworkId = this.getBrokerNetworkId(configs);
        this.networkIdRoutesTopic = this.getNetworkIdRoutesTopic(configs);
        this.networkIdRoutesLog = this.configureConsumer(configs, this.interBrokerClientConfig);
        this.periodicStartTaskMs = this.getPeriodicStartTaskMs(configs);
    }

    private KafkaBasedLog<String, String> configureConsumer(Map<String, ?> configs, Map<String, ?> interBrokerClientConfigs) {
        State s = this.state.get();
        if (s != State.NOT_STARTED) {
            throw new IllegalStateException("configureConsumer called in a state it can't start in: " + (Object)((Object)s));
        }
        String clientId = String.format("%s-%s-%s", this.networkIdRoutesTopic, ConfluentConfigs.ClientType.CONSUMER, configs.get(KafkaConfig.BrokerIdProp()));
        Long timeoutMs = (Long)configs.get("confluent.cdc.api.keys.topic.load.timeout.ms");
        if (timeoutMs == null || timeoutMs <= 0L) {
            throw new ConfigException("Value for config confluent.cdc.api.keys.topic.load.timeout.ms must be positive integer when using networkId routes");
        }
        HashSet consumerConfigNames = new HashSet(ConsumerConfig.configNames());
        consumerConfigNames.remove("metric.reporters");
        HashMap consumerProps = new HashMap(interBrokerClientConfigs);
        consumerProps.keySet().retainAll(consumerConfigNames);
        consumerProps.put("client.id", clientId);
        consumerProps.put("bootstrap.servers", interBrokerClientConfigs.get("bootstrap.servers"));
        consumerProps.put("allow.auto.create.topics", false);
        consumerProps.put("key.deserializer", StringDeserializer.class.getName());
        consumerProps.put("value.deserializer", StringDeserializer.class.getName());
        consumerProps.put("default.api.timeout.ms", (int)Math.min(timeoutMs, Integer.MAX_VALUE));
        return new KafkaBasedLog(this.networkIdRoutesTopic, null, consumerProps, () -> null, (Callback)new ConsumeCallback(), this.time, null, timeoutMs.longValue());
    }

    private Long getPeriodicStartTaskMs(Map<String, ?> configs) {
        Long timeoutMs = (Long)configs.get("confluent.traffic.cdc.network.id.routes.periodic.start.task.ms");
        if (timeoutMs == null || timeoutMs <= 0L) {
            throw new ConfigException("Value for config confluent.traffic.cdc.network.id.routes.periodic.start.task.ms must be positive integer when using networkId routes");
        }
        return timeoutMs;
    }

    private String getNetworkIdRoutesTopic(Map<String, ?> configs) {
        String topic = (String)configs.get("confluent.traffic.cdc.network.id.routes.topic.name");
        if (topic == null || topic.isEmpty()) {
            throw new ConfigException("Value for config confluent.traffic.cdc.network.id.routes.topic.name can not be empty when networkId routes are enabled");
        }
        return topic;
    }

    public void close() {
        if (this.sessionUuid == null) {
            LOG.warn("close() called without configure() being called first");
            return;
        }
        LOG.info("Closing consumer for session {}", (Object)this.sessionUuid);
        this.close(this.sessionUuid);
        this.metrics.removeSensor(this.topicLoadTimeSensor.name());
    }

    private void close(String brokerSessionUuid) {
        TrafficNetworkIdRoutesStore.removeRoutes((String)brokerSessionUuid);
        LOG.info("Removed instance for broker session {}", (Object)brokerSessionUuid);
        this.stopLog();
        this.stopPeriodicStartTask();
    }

    private void stopPeriodicStartTask() {
        if (this.executorService != null) {
            if (this.storeStartTaskFuture != null) {
                this.storeStartTaskFuture.cancel(false);
                this.storeStartTaskFuture = null;
            }
            this.executorService.shutdownNow();
            try {
                this.executorService.awaitTermination(TimeUnit.SECONDS.toMillis(30L), TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                LOG.debug("Shutting down executor was interrupted", (Throwable)e);
            }
            this.executorService = null;
        }
    }

    private void stopLog() {
        State s = this.state.get();
        if (this.networkIdRoutesLog != null) {
            try {
                this.networkIdRoutesLog.stop();
                LOG.info("Successfully closed the consumer");
            }
            catch (Exception e) {
                LOG.error("Error when shutting down the consumer", (Throwable)e);
            }
        }
        if (s == State.NOT_ENABLED || s == State.FAILED_TO_START) {
            return;
        }
        State prevState = this.state.getAndSet(State.CLOSED);
        if (prevState == State.RUNNING || prevState == State.STARTING) {
            return;
        }
        LOG.debug("Asked to close from a non-running state {}", (Object)prevState);
    }

    void consume(ConsumerRecord<String, String> record) {
        String key = (String)record.key();
        if (record.key() == null) {
            LOG.error("Missing key in network ID route message! (partition = {}, offset = {}, timestamp = {})", new Object[]{record.partition(), record.offset(), record.timestamp()});
            return;
        }
        Long sequenceId = AuthUtils.tryParseEventsSequenceId(record);
        if (sequenceId == null) {
            LOG.error("Missing sequence ID in network ID routes message! (partition = {}, offset = {}, timestamp = {})", new Object[]{record.partition(), record.offset(), record.timestamp()});
            return;
        }
        if (!this.verifySequenceId(sequenceId)) {
            return;
        }
        if (!this.verifyKey(key)) {
            return;
        }
        this.updateRoutes(record, sequenceId, key);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateRoutes(ConsumerRecord<String, String> record, long sequenceId, String key) {
        String jsonMsg = (String)record.value();
        try {
            if (jsonMsg == null) {
                this.updateRoutesWithForceDisconnect(TrafficNetworkIdRoutes.EMPTY);
                this.maybeLogRouteUpdate(key, sequenceId, jsonMsg, false);
            } else {
                TrafficNetworkIdAllowedRoutes value = (TrafficNetworkIdAllowedRoutes)this.objectMapper.readValue(jsonMsg, TrafficNetworkIdAllowedRoutes.class);
                this.updateRoutesWithForceDisconnect(new TrafficNetworkIdRoutes(value.allowedNetworkIds(), value.allowedDNSDomainSuffixes()));
                this.maybeLogRouteUpdate(key, sequenceId, jsonMsg, true);
            }
        }
        catch (Exception e) {
            LOG.error(String.format("Can't parse value in network ID routes message! (partition = %d, offset = %d, timestamp = %d, key = %s, sequence ID = %d)", record.partition(), record.offset(), record.timestamp(), key, sequenceId), (Throwable)e);
        }
        finally {
            this.lastSequenceId.set(sequenceId);
            LOG.trace("Finished reading record with sequence id: {}", (Object)sequenceId);
        }
    }

    private void maybeLogRouteUpdate(String key, long sequenceId, String jsonMsg, boolean isUpdate) {
        String logMessage = isUpdate ? "Updated routes for key {} from topic (sequence id: {}, jsonMsg:{}), new routes:{}" : "Deleted routes, read null value for key {} from topic (sequence id: {}, jsonMsg:{}), new routes:{}";
        this.maybeLogMessage(logMessage, key, sequenceId, jsonMsg, TrafficNetworkIdRoutesStore.getRoutes((String)this.sessionUuid));
    }

    private void maybeLogMessage(String message, Object ... args) {
        if (State.RUNNING == this.state.get()) {
            LOG.info(message, args);
        } else {
            LOG.debug(message, args);
        }
    }

    Long getLastSeenSequenceId() {
        return this.lastSequenceId.get();
    }

    public Map<Endpoint, CompletableFuture<Void>> start(Collection<Endpoint> endpoints) {
        CompletableFuture<Object> logStartedFuture;
        LOG.info("Starting store from state {}", this.state);
        if (this.state.get() == State.NOT_ENABLED) {
            LOG.debug("Trying to start from a non enabled state. Ignoring");
            return Collections.emptyMap();
        }
        if (!this.state.compareAndSet(State.NOT_STARTED, State.STARTING)) {
            throw new IllegalStateException("Trying to start a log from a state it can't be started in");
        }
        boolean networkIdRoutesTopicExists = this.metadataCache.contains(this.networkIdRoutesTopic);
        try {
            if (networkIdRoutesTopicExists) {
                logStartedFuture = CompletableFuture.runAsync(() -> this.startLog());
            } else {
                logStartedFuture = CompletableFuture.completedFuture(null);
                this.startPeriodicStartTask();
            }
        }
        catch (Exception e2) {
            this.state.set(State.FAILED_TO_START);
            LOG.error("Setting state as FAILED_TO_START due to:", (Throwable)e2);
            throw new IllegalStateException("Unable to create a future for startLog()", e2);
        }
        return endpoints.stream().collect(Collectors.toMap(Function.identity(), e -> e.listenerName().map(this.routesListenerNames::contains).orElse(false) != false ? logStartedFuture : CompletableFuture.completedFuture(null)));
    }

    private void startPeriodicStartTask() {
        LOG.info("Starting periodic task for checking topic {} existence and consumer start", (Object)this.networkIdRoutesTopic);
        this.executorService = Executors.newSingleThreadScheduledExecutor(ThreadUtils.createThreadFactory((String)"traffic-route-store-%d", (boolean)true));
        this.storeStartTaskFuture = this.executorService.scheduleAtFixedRate(() -> {
            LOG.trace("Running periodic task for checking topic {} existence and starting consumer", (Object)this.networkIdRoutesTopic);
            if (this.startLog()) {
                LOG.info("Successfully started the consumer on {} and loaded routes", (Object)this.networkIdRoutesTopic);
                this.storeStartTaskFuture.cancel(false);
            }
        }, this.periodicStartTaskMs, this.periodicStartTaskMs, TimeUnit.MILLISECONDS);
    }

    private boolean startLog() {
        if (this.state.get() != State.STARTING) {
            throw new IllegalStateException("Trying to start log from a non starting state");
        }
        if (!this.metadataCache.contains(this.networkIdRoutesTopic)) {
            return false;
        }
        try {
            long startNs = this.time.nanoseconds();
            this.networkIdRoutesLog.start();
            this.state.set(State.RUNNING);
            long loadTimeNs = this.time.nanoseconds() - startNs;
            this.topicLoadTimeSensor.record((double)loadTimeNs);
            LOG.info("Consumed initial set of network ID routes from topic took {} nanoseconds, routes:{}", (Object)loadTimeNs, (Object)TrafficNetworkIdRoutesStore.getRoutes((String)this.sessionUuid));
            return true;
        }
        catch (Exception e) {
            this.state.set(State.FAILED_TO_START);
            LOG.error("Setting state as FAILED_TO_START, unable to start consuming network ID routes from topic due to:", (Throwable)e);
            throw new IllegalStateException("Unable to start consuming network ID routes from topic", e);
        }
    }

    boolean verifySequenceId(long sequenceId) {
        Long previousSequenceId = this.getLastSeenSequenceId();
        if (previousSequenceId != null && sequenceId <= previousSequenceId) {
            LOG.info("Received network ID routes for with an earlier sequence id (last seen = {}, recent = {}), ignoring", (Object)previousSequenceId, (Object)sequenceId);
            return false;
        }
        return true;
    }

    boolean verifyKey(String key) {
        String[] split = key.split(":");
        if (split.length == 2 && Objects.equals(split[0], this.clusterNetworkId)) {
            return true;
        }
        LOG.warn("Received record with key = {} not matching this cluster's networkId = {}, ignoring", (Object)key, (Object)this.clusterNetworkId);
        return false;
    }

    private void updateRoutesWithForceDisconnect(TrafficNetworkIdRoutes newRoutes) {
        TrafficNetworkIdRoutes existingRoutes = TrafficNetworkIdRoutesStore.getRoutes((String)this.sessionUuid);
        TrafficNetworkIdRoutesStore.addRoutes((String)this.sessionUuid, (TrafficNetworkIdRoutes)newRoutes);
        if (existingRoutes != null) {
            Sets.SetView disallowedNetworkIds = Sets.difference((Set)existingRoutes.networkIdRoutes(), (Set)newRoutes.networkIdRoutes());
            disallowedNetworkIds.forEach(disallowedNetworkId -> this.closeConnections((String)disallowedNetworkId));
        }
    }

    private void closeConnections(String disallowedNetworkId) {
        BrokerSession session;
        BrokerSession brokerSession = session = this.sessionUuid != null ? BrokerSession.session((String)this.sessionUuid) : null;
        if (session != null) {
            this.maybeLogMessage("Forcing close of connections from disallowedNetworkId {} for broker session {}", disallowedNetworkId, this.sessionUuid);
            SUPPORTED_SASL_MECHANISMS.forEach(saslMechanism -> session.handleCredentialDelete(PublicCredential.saslNetworkIdCredential((String)disallowedNetworkId, (String)saslMechanism)));
        } else {
            LOG.warn("Ignoring close of connections from disallowedNetworkId {} because broker session {} is not available.", (Object)disallowedNetworkId, (Object)this.sessionUuid);
        }
    }

    private class ConsumeCallback
    implements Callback<ConsumerRecord<String, String>> {
        private ConsumeCallback() {
        }

        public void onCompletion(Throwable error, ConsumerRecord<String, String> record) {
            if (error != null) {
                LOG.error("Unexpected error in ConsumeCallback", error);
                return;
            }
            TopicBasedTrafficNetworkIdRoutesUpdater.this.consume(record);
        }
    }

    public static enum State {
        NOT_STARTED,
        NOT_ENABLED,
        STARTING,
        RUNNING,
        CLOSED,
        FAILED_TO_START;

    }
}

