/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.metadata.placement;

import io.confluent.kafka.multitenant.MultiTenantPrincipal;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cell;
import org.apache.kafka.common.CellLoad;
import org.apache.kafka.common.CellState;
import org.apache.kafka.common.PartitionPlacementStrategy;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.Time;

public class CellAssignor {
    public static final Set<CellState> PROHIBITED_TARGET_STATES = Collections.unmodifiableSet(new HashSet<CellState>(Arrays.asList(CellState.QUARANTINED, CellState.EXCLUDED)));
    public static final Set<CellState> PROHIBITED_SOURCE_STATES = Collections.singleton(CellState.QUARANTINED);
    public static final long CELL_LOAD_FRESH_DURATION_MS = TimeUnit.MILLISECONDS.convert(25L, TimeUnit.MINUTES);
    private static final CellLoad DEFAULT_CELL_LOAD = new CellLoad(-1, Double.MAX_VALUE);
    private static final double CELL_OVERLOADED_THRESHOLD = 0.9;
    private static final Set<CellState> USABLE_CELL_STATES = Collections.singleton(CellState.READY);
    private final Time time;
    private final Random random;
    private final AtomicInteger roundRobinIndex;
    private volatile Map<Integer, CellLoad> cellIdToCellLoad;
    private volatile long lastCellLoadUpdateTimeMs;

    public CellAssignor(Random random, Time time) {
        this.random = random;
        this.time = time;
        this.roundRobinIndex = new AtomicInteger(random.nextInt(100));
        this.cellIdToCellLoad = new HashMap<Integer, CellLoad>();
        this.lastCellLoadUpdateTimeMs = 0L;
    }

    public Optional<Cell> computeUsableCell(Set<Integer> usableBrokers, Collection<Cell> allCells, int minimumBrokerCount) {
        Optional<Cell> cell;
        if (this.isCellLoadDataFresh() && (cell = this.powerOfTwoRandomCell(this.sortedUsableCells(usableBrokers, allCells, minimumBrokerCount, 0.9))).isPresent()) {
            return cell;
        }
        return this.roundRobinNextCell(this.sortedUsableCells(usableBrokers, allCells, minimumBrokerCount, Double.MAX_VALUE));
    }

    public void fullUpdateCellLoadCache(Set<CellLoad> cellIdToCellLoad, long updateTimeMs) {
        this.cellIdToCellLoad = cellIdToCellLoad.stream().collect(Collectors.toMap(CellLoad::cellId, Function.identity()));
        if (!cellIdToCellLoad.isEmpty()) {
            this.lastCellLoadUpdateTimeMs = updateTimeMs;
        }
    }

    public static void checkCellMetadata(Cell newCell, Cell oldCell, Set<Integer> usableBrokers, short replicationFactor, int numTenants) {
        if (newCell.minSize() <= 0) {
            throw new InvalidRequestException(String.format("Cells must have positive minSize. Attempted to set cell %s with %s", newCell.cellId(), newCell.minSize()));
        }
        if (newCell.maxSize() <= 0) {
            throw new InvalidRequestException(String.format("Cells must have positive maxSize. Attempted to set cell %s with %s", newCell.cellId(), newCell.maxSize()));
        }
        if (newCell.minSize() > newCell.maxSize()) {
            throw new InvalidRequestException(String.format("Cells must have higher maxSize than minSize. Attempted to set cell %s with minSize %s, maxSize %s", newCell.cellId(), newCell.minSize(), newCell.maxSize()));
        }
        if (newCell.brokers().size() > newCell.maxSize()) {
            throw new InvalidRequestException(String.format("Cells' number of brokers cannot exceed maxSize. Attempted to set cell %s with maxSize %s, %s number of brokers", newCell.cellId(), newCell.maxSize(), newCell.brokers().size()));
        }
        if (newCell.cellId() < 0) {
            throw new InvalidRequestException("Cell id must be non-negative");
        }
        if (!CellState.VALID_CELL_STATES.contains(newCell.state())) {
            throw new InvalidRequestException("Cells must have valid state");
        }
        boolean newlyAssignable = CellAssignor.isCellOpenForAssignment(newCell, usableBrokers, replicationFactor);
        if (numTenants > 0 && !newlyAssignable) {
            if (newCell.brokers().size() < oldCell.brokers().size()) {
                throw new InvalidRequestException(String.format("Cells cannot be drained below %s minSize number of brokers if it contains tenants. Attempted to set %s number of new brokers while having %s number of tenants", newCell.minSize(), newCell.brokers().size(), numTenants));
            }
            if (newCell.minSize() > oldCell.minSize()) {
                throw new InvalidRequestException("Cells cannot increase minSiz if it contains tenants and has fewer number of brokers than minSize");
            }
        }
    }

    public static void checkBrokerAssignment(Set<Integer> brokerIds, int cellSize, boolean force) {
        if (!force) {
            Set uniqueSourceCellIds = brokerIds.stream().map(brokerId -> brokerId / cellSize).collect(Collectors.toSet());
            if (brokerIds.size() != cellSize || uniqueSourceCellIds.size() > 1) {
                throw new InvalidRequestException(String.format("Cell assignment and unassignment must be done in increments of cellSize %s, and the source brokerIds must be from one cell", cellSize));
            }
        }
    }

    public static void confirmInitialCellStateValid(short defaultCellMinSize, short defaultCellMaxSize, short defaultCellSize) {
        if (defaultCellMinSize <= 0) {
            throw new RuntimeException(String.format("defaultCellMinSize must be positive, received %s", defaultCellMinSize));
        }
        if (defaultCellSize < defaultCellMinSize) {
            throw new RuntimeException(String.format("defaultCellSize %s must be at least defaultCellMinSize %s ", defaultCellSize, defaultCellMinSize));
        }
        if (defaultCellMaxSize < defaultCellMinSize) {
            throw new RuntimeException(String.format("defaultCellMaxSize %s must be at least defaultCellMinSize %s", defaultCellMaxSize, defaultCellMinSize));
        }
    }

    public static boolean isCellOpenForAssignment(Cell cell, Set<Integer> usableBrokers, int minBrokers) {
        return CellAssignor.intersect(usableBrokers, cell.brokers()).size() >= minBrokers && cell.brokers().size() >= cell.minSize();
    }

    public static boolean isDefaultCellOpenForAssignment(Set<Integer> unassignedBrokers, int minBrokers) {
        return unassignedBrokers.size() >= minBrokers;
    }

    public static PartitionPlacementStrategy calculatePartitionPlacementStrategy(Optional<KafkaPrincipal> principalOpt, PartitionPlacementStrategy defaultPartitionPlacementStrategy) {
        KafkaPrincipal principal = principalOpt.orElse(null);
        boolean isMultitenant = principal instanceof MultiTenantPrincipal;
        boolean isHealthcheckTenant = false;
        if (isMultitenant) {
            MultiTenantPrincipal multiTenantPrincipal = (MultiTenantPrincipal)principal;
            isHealthcheckTenant = multiTenantPrincipal.tenantMetadata().isHealthcheckTenant;
        }
        if ((!isMultitenant || isHealthcheckTenant) && defaultPartitionPlacementStrategy == PartitionPlacementStrategy.TENANT_IN_CELL) {
            return PartitionPlacementStrategy.PARTITION_IN_CELL;
        }
        return defaultPartitionPlacementStrategy;
    }

    public static boolean isTenantCellPlacementEnabled(Optional<KafkaPrincipal> kafkaPrincipalOpt, PartitionPlacementStrategy defaultPartitionPlacementStrategy) {
        PartitionPlacementStrategy strategy = CellAssignor.calculatePartitionPlacementStrategy(kafkaPrincipalOpt, defaultPartitionPlacementStrategy);
        return strategy == PartitionPlacementStrategy.TENANT_IN_CELL;
    }

    private boolean isCellLoadDataFresh() {
        return this.time.milliseconds() - this.lastCellLoadUpdateTimeMs <= CELL_LOAD_FRESH_DURATION_MS;
    }

    private Optional<Cell> powerOfTwoRandomCell(List<Cell> assignableCells) {
        List<Cell> pool;
        if (assignableCells.size() <= 1) {
            pool = assignableCells;
        } else {
            int firstIdx = this.random.nextInt(assignableCells.size());
            int secondIdxAfterRemoval = this.random.nextInt(assignableCells.size() - 1);
            int secondIdx = secondIdxAfterRemoval < firstIdx ? secondIdxAfterRemoval : secondIdxAfterRemoval + 1;
            pool = Arrays.asList(assignableCells.get(firstIdx), assignableCells.get(secondIdx));
        }
        return pool.stream().min(Comparator.comparingDouble(c -> this.cellIdToCellLoad.getOrDefault(c.cellId(), DEFAULT_CELL_LOAD).load()).thenComparingInt(Cell::cellId));
    }

    private Optional<Cell> roundRobinNextCell(List<Cell> assignableCells) {
        if (assignableCells.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(assignableCells.get(Math.floorMod(this.roundRobinIndex.getAndIncrement(), assignableCells.size())));
    }

    private List<Cell> sortedUsableCells(Set<Integer> usableBrokers, Collection<Cell> cells, int minimumBrokerCount, double maxCellLoad) {
        return cells.stream().filter(cell -> USABLE_CELL_STATES.contains(cell.state())).filter(cell -> CellAssignor.isCellOpenForAssignment(cell, usableBrokers, minimumBrokerCount)).filter(cell -> this.cellIdToCellLoad.getOrDefault(cell.cellId(), DEFAULT_CELL_LOAD).load() <= maxCellLoad).sorted(Comparator.comparingInt(Cell::cellId)).collect(Collectors.toList());
    }

    private static Set<Integer> intersect(Collection<Integer> a, Collection<Integer> b) {
        HashSet<Integer> result = new HashSet<Integer>(a);
        result.retainAll(b);
        return result;
    }
}

