/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.ListConsumerGroupOffsetsOptions;
import org.apache.kafka.clients.admin.ListOffsetsOptions;
import org.apache.kafka.clients.admin.ListOffsetsResult;
import org.apache.kafka.clients.admin.OffsetSpec;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.InvalidOffsetException;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.IsolationLevel;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskCorruptedException;
import org.apache.kafka.streams.processor.StandbyUpdateListener;
import org.apache.kafka.streams.processor.StateRestoreListener;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.ChangelogReader;
import org.apache.kafka.streams.processor.internals.ProcessorStateManager;
import org.apache.kafka.streams.processor.internals.Task;
import org.slf4j.Logger;

public class StoreChangelogReader
implements ChangelogReader {
    private static final long RESTORE_LOG_INTERVAL_MS = 10000L;
    private long lastRestoreLogTime = 0L;
    private static final long DEFAULT_OFFSET_UPDATE_MS = Duration.ofMinutes(5L).toMillis();
    private ChangelogReaderState state;
    private final Time time;
    private final Logger log;
    private final Duration pollTime;
    private final long updateOffsetIntervalMs;
    private final Consumer<byte[], byte[]> restoreConsumer;
    private final StateRestoreListener stateRestoreListener;
    private final boolean stateUpdaterEnabled;
    private final Map<TopicPartition, ChangelogMetadata> changelogs;
    private final String groupId;
    private final Admin adminClient;
    private long lastUpdateOffsetTime;
    private final StandbyUpdateListener standbyUpdateListener;

    public StoreChangelogReader(Time time, StreamsConfig config, LogContext logContext, Admin adminClient, Consumer<byte[], byte[]> restoreConsumer, StateRestoreListener stateRestoreListener, StandbyUpdateListener standbyUpdateListener) {
        this.time = time;
        this.log = logContext.logger(StoreChangelogReader.class);
        this.state = ChangelogReaderState.ACTIVE_RESTORING;
        this.adminClient = adminClient;
        this.restoreConsumer = restoreConsumer;
        this.stateRestoreListener = stateRestoreListener;
        this.standbyUpdateListener = standbyUpdateListener;
        this.stateUpdaterEnabled = StreamsConfig.InternalConfig.getStateUpdaterEnabled(config.originals());
        this.groupId = config.getString("application.id");
        this.pollTime = Duration.ofMillis(config.getLong("poll.ms"));
        this.updateOffsetIntervalMs = config.getLong("commit.interval.ms") == Long.MAX_VALUE ? DEFAULT_OFFSET_UPDATE_MS : config.getLong("commit.interval.ms");
        this.lastUpdateOffsetTime = 0L;
        this.changelogs = new HashMap<TopicPartition, ChangelogMetadata>();
    }

    private static String recordEndOffset(Long endOffset) {
        return endOffset == null ? "UNKNOWN (since it is for standby task)" : endOffset.toString();
    }

    private boolean hasRestoredToEnd(ChangelogMetadata metadata) {
        Long endOffset = metadata.restoreEndOffset;
        if (endOffset == null) {
            throw new IllegalStateException("End offset for changelog " + metadata + " is unknown when deciding if it has completed restoration, this should never happen.");
        }
        if (endOffset == 0L) {
            return true;
        }
        if (metadata.bufferedRecords.isEmpty()) {
            TopicPartition partition = metadata.storeMetadata.changelogPartition();
            try {
                return this.restoreConsumer.position(partition) >= endOffset;
            }
            catch (TimeoutException timeoutException) {
                throw timeoutException;
            }
            catch (KafkaException e) {
                throw new StreamsException("Restore consumer get unexpected error trying to get the position  of " + partition, e);
            }
        }
        return ((ConsumerRecord)metadata.bufferedRecords.get(0)).offset() >= endOffset;
    }

    @Override
    public void enforceRestoreActive() {
        if (this.state != ChangelogReaderState.ACTIVE_RESTORING) {
            this.log.debug("Transiting to restore active tasks: {}", this.changelogs);
            this.lastRestoreLogTime = 0L;
            this.pauseChangelogsFromRestoreConsumer(this.standbyRestoringChangelogs());
            this.state = ChangelogReaderState.ACTIVE_RESTORING;
        }
    }

    @Override
    public void transitToUpdateStandby() {
        if (this.state != ChangelogReaderState.ACTIVE_RESTORING) {
            throw new IllegalStateException("The changelog reader is not restoring active tasks (is " + (Object)((Object)this.state) + ") while trying to transit to update standby tasks: " + this.changelogs);
        }
        this.log.debug("Transiting to update standby tasks: {}", this.changelogs);
        this.resumeChangelogsFromRestoreConsumer(this.standbyRestoringChangelogs());
        this.state = ChangelogReaderState.STANDBY_UPDATING;
    }

    @Override
    public boolean isRestoringActive() {
        return this.state == ChangelogReaderState.ACTIVE_RESTORING;
    }

    @Override
    public void register(TopicPartition partition, ProcessorStateManager stateManager) {
        ProcessorStateManager.StateStoreMetadata storeMetadata = stateManager.storeMetadata(partition);
        if (storeMetadata == null) {
            throw new IllegalStateException("Cannot find the corresponding state store metadata for changelog " + partition);
        }
        ChangelogMetadata changelogMetadata = new ChangelogMetadata(storeMetadata, stateManager);
        if (stateManager.taskType() == Task.TaskType.STANDBY && stateManager.changelogAsSource(partition)) {
            changelogMetadata.restoreEndOffset = 0L;
        }
        if (this.changelogs.putIfAbsent(partition, changelogMetadata) != null) {
            throw new IllegalStateException("There is already a changelog registered for " + partition + ", this should not happen: " + this.changelogs);
        }
    }

    @Override
    public void register(Set<TopicPartition> changelogPartitions, ProcessorStateManager stateManager) {
        for (TopicPartition changelogPartition : changelogPartitions) {
            this.register(changelogPartition, stateManager);
        }
    }

    private ChangelogMetadata restoringChangelogByPartition(TopicPartition partition) {
        ChangelogMetadata changelogMetadata = this.changelogs.get(partition);
        if (changelogMetadata == null) {
            throw new IllegalStateException("The corresponding changelog restorer for " + partition + " does not exist, this should not happen.");
        }
        if (changelogMetadata.changelogState != ChangelogState.RESTORING) {
            throw new IllegalStateException("The corresponding changelog restorer for " + partition + " has already transited to completed state, this should not happen.");
        }
        return changelogMetadata;
    }

    private Set<ChangelogMetadata> registeredChangelogs() {
        return this.changelogs.values().stream().filter(metadata -> ((ChangelogMetadata)metadata).changelogState == ChangelogState.REGISTERED).collect(Collectors.toSet());
    }

    private Set<TopicPartition> restoringChangelogs() {
        return this.changelogs.values().stream().filter(metadata -> ((ChangelogMetadata)metadata).changelogState == ChangelogState.RESTORING).map(metadata -> ((ChangelogMetadata)metadata).storeMetadata.changelogPartition()).collect(Collectors.toSet());
    }

    private Set<TopicPartition> activeRestoringChangelogs() {
        return this.changelogs.values().stream().filter(metadata -> ((ChangelogMetadata)metadata).changelogState == ChangelogState.RESTORING && ((ChangelogMetadata)metadata).stateManager.taskType() == Task.TaskType.ACTIVE).map(metadata -> ((ChangelogMetadata)metadata).storeMetadata.changelogPartition()).collect(Collectors.toSet());
    }

    private Set<TopicPartition> standbyRestoringChangelogs() {
        return this.changelogs.values().stream().filter(metadata -> ((ChangelogMetadata)metadata).changelogState == ChangelogState.RESTORING && ((ChangelogMetadata)metadata).stateManager.taskType() == Task.TaskType.STANDBY).map(metadata -> ((ChangelogMetadata)metadata).storeMetadata.changelogPartition()).collect(Collectors.toSet());
    }

    @Override
    public boolean allChangelogsCompleted() {
        return this.changelogs.values().stream().allMatch(metadata -> ((ChangelogMetadata)metadata).changelogState == ChangelogState.COMPLETED);
    }

    @Override
    public Set<TopicPartition> completedChangelogs() {
        return this.changelogs.values().stream().filter(metadata -> ((ChangelogMetadata)metadata).changelogState == ChangelogState.COMPLETED).map(metadata -> ((ChangelogMetadata)metadata).storeMetadata.changelogPartition()).collect(Collectors.toSet());
    }

    @Override
    public long restore(Map<TaskId, Task> tasks) {
        this.initializeChangelogs(tasks, this.registeredChangelogs());
        if (!this.activeRestoringChangelogs().isEmpty() && this.state == ChangelogReaderState.STANDBY_UPDATING) {
            throw new IllegalStateException("Should not be in standby updating state if there are still un-completed active changelogs");
        }
        long totalRestored = 0L;
        if (this.allChangelogsCompleted()) {
            this.log.debug("Finished restoring all changelogs {}", this.changelogs.keySet());
            return totalRestored;
        }
        Set<TopicPartition> restoringChangelogs = this.restoringChangelogs();
        if (!restoringChangelogs.isEmpty()) {
            ConsumerRecords<byte[], byte[]> polledRecords = this.pollRecordsFromRestoreConsumer(tasks, restoringChangelogs);
            for (TopicPartition partition : polledRecords.partitions()) {
                this.bufferChangelogRecords(this.restoringChangelogByPartition(partition), polledRecords.records(partition));
            }
            for (TopicPartition partition : restoringChangelogs) {
                TaskId taskId = this.changelogs.get(partition).stateManager.taskId();
                Task task = tasks.get(taskId);
                try {
                    ChangelogMetadata changelogMetadata = this.changelogs.get(partition);
                    totalRestored += (long)this.restoreChangelog(task, changelogMetadata);
                }
                catch (TimeoutException timeoutException) {
                    tasks.get(taskId).maybeInitTaskTimeoutOrThrow(this.time.milliseconds(), (Exception)((Object)timeoutException));
                }
            }
            this.maybeUpdateLimitOffsetsForStandbyChangelogs(tasks);
            this.maybeLogRestorationProgress();
        }
        return totalRestored;
    }

    private ConsumerRecords<byte[], byte[]> pollRecordsFromRestoreConsumer(Map<TaskId, Task> tasks, Set<TopicPartition> restoringChangelogs) {
        ConsumerRecords polledRecords;
        boolean useNonBlockingPoll = this.state == ChangelogReaderState.STANDBY_UPDATING && !this.stateUpdaterEnabled;
        try {
            this.pauseResumePartitions(tasks, restoringChangelogs);
            polledRecords = this.restoreConsumer.poll(useNonBlockingPoll ? Duration.ZERO : this.pollTime);
        }
        catch (InvalidOffsetException e) {
            this.log.warn("Encountered " + ((Object)((Object)e)).getClass().getName() + " fetching records from restore consumer for partitions " + e.partitions() + ", it is likely that the consumer's position has fallen out of the topic partition offset range because the topic was truncated or compacted on the broker, marking the corresponding tasks as corrupted and re-initializing it later.", (Throwable)e);
            HashSet<TaskId> corruptedTasks = new HashSet<TaskId>();
            e.partitions().forEach(partition -> corruptedTasks.add(this.changelogs.get(partition).stateManager.taskId()));
            throw new TaskCorruptedException(corruptedTasks, e);
        }
        catch (InterruptException interruptException) {
            throw interruptException;
        }
        catch (KafkaException e) {
            throw new StreamsException("Restore consumer get unexpected error polling records.", e);
        }
        return polledRecords;
    }

    private void pauseResumePartitions(Map<TaskId, Task> tasks, Set<TopicPartition> restoringChangelogs) {
        if (this.state == ChangelogReaderState.ACTIVE_RESTORING) {
            this.updatePartitionsByType(tasks, restoringChangelogs, Task.TaskType.ACTIVE);
        }
        if (this.state == ChangelogReaderState.STANDBY_UPDATING) {
            this.updatePartitionsByType(tasks, restoringChangelogs, Task.TaskType.STANDBY);
        }
    }

    private void updatePartitionsByType(Map<TaskId, Task> tasks, Set<TopicPartition> restoringChangelogs, Task.TaskType taskType) {
        Collection toResume = restoringChangelogs.stream().filter(t -> this.shouldResume(tasks, (TopicPartition)t, taskType)).collect(Collectors.toList());
        Collection toPause = restoringChangelogs.stream().filter(t -> this.shouldPause(tasks, (TopicPartition)t, taskType)).collect(Collectors.toList());
        this.restoreConsumer.resume(toResume);
        this.restoreConsumer.pause(toPause);
    }

    private boolean shouldResume(Map<TaskId, Task> tasks, TopicPartition partition, Task.TaskType taskType) {
        ProcessorStateManager manager = this.changelogs.get(partition).stateManager;
        TaskId taskId = manager.taskId();
        Task task = tasks.get(taskId);
        if (manager.taskType() == taskType) {
            return task != null;
        }
        return false;
    }

    private boolean shouldPause(Map<TaskId, Task> tasks, TopicPartition partition, Task.TaskType taskType) {
        ProcessorStateManager manager = this.changelogs.get(partition).stateManager;
        TaskId taskId = manager.taskId();
        Task task = tasks.get(taskId);
        if (manager.taskType() == taskType) {
            return task == null;
        }
        return false;
    }

    private void maybeLogRestorationProgress() {
        if (this.state == ChangelogReaderState.ACTIVE_RESTORING) {
            Set<TopicPartition> topicPartitions;
            if (this.time.milliseconds() - this.lastRestoreLogTime > 10000L && !(topicPartitions = this.activeRestoringChangelogs()).isEmpty()) {
                StringBuilder builder = new StringBuilder().append("Restoration in progress for ").append(topicPartitions.size()).append(" partitions.");
                for (TopicPartition partition : topicPartitions) {
                    ChangelogMetadata changelogMetadata = this.restoringChangelogByPartition(partition);
                    builder.append(" {").append(partition).append(": ").append("position=").append(StoreChangelogReader.getPositionString(partition, changelogMetadata)).append(", end=").append(changelogMetadata.restoreEndOffset).append(", totalRestored=").append(changelogMetadata.totalRestored).append("}");
                }
                this.log.info(builder.toString());
                this.lastRestoreLogTime = this.time.milliseconds();
            }
        } else {
            this.lastRestoreLogTime = 0L;
        }
    }

    private static String getPositionString(TopicPartition partition, ChangelogMetadata changelogMetadata) {
        ProcessorStateManager stateManager = changelogMetadata.stateManager;
        Long offsets = stateManager.changelogOffsets().get(partition);
        return offsets == null ? "unknown" : String.valueOf(offsets);
    }

    private void maybeUpdateLimitOffsetsForStandbyChangelogs(Map<TaskId, Task> tasks) {
        if (this.state == ChangelogReaderState.STANDBY_UPDATING && this.updateOffsetIntervalMs < this.time.milliseconds() - this.lastUpdateOffsetTime) {
            Set<TopicPartition> changelogsWithLimitOffsets = this.changelogs.entrySet().stream().filter(entry -> ((ChangelogMetadata)entry.getValue()).stateManager.taskType() == Task.TaskType.STANDBY && ((ChangelogMetadata)entry.getValue()).stateManager.changelogAsSource((TopicPartition)entry.getKey())).map(Map.Entry::getKey).collect(Collectors.toSet());
            for (TopicPartition partition : changelogsWithLimitOffsets) {
                if (this.changelogs.get(partition).bufferedRecords().isEmpty()) continue;
                this.updateLimitOffsetsForStandbyChangelogs(this.committedOffsetForChangelogs(tasks, changelogsWithLimitOffsets));
                this.lastUpdateOffsetTime = this.time.milliseconds();
                break;
            }
        }
    }

    private void bufferChangelogRecords(ChangelogMetadata changelogMetadata, List<ConsumerRecord<byte[], byte[]>> records) {
        for (ConsumerRecord<byte[], byte[]> record : records) {
            if (record.key() == null) {
                this.log.warn("Read changelog record with null key from changelog {} at offset {}, skipping it for restoration", (Object)changelogMetadata.storeMetadata.changelogPartition(), (Object)record.offset());
                continue;
            }
            changelogMetadata.bufferedRecords.add(record);
            long offset = record.offset();
            if (changelogMetadata.restoreEndOffset != null && offset >= changelogMetadata.restoreEndOffset) continue;
            changelogMetadata.bufferedLimitIndex = changelogMetadata.bufferedRecords.size();
        }
    }

    private int restoreChangelog(Task task, ChangelogMetadata changelogMetadata) {
        ProcessorStateManager stateManager = changelogMetadata.stateManager;
        ProcessorStateManager.StateStoreMetadata storeMetadata = changelogMetadata.storeMetadata;
        TopicPartition partition = storeMetadata.changelogPartition();
        String storeName = storeMetadata.store().name();
        int numRecords = changelogMetadata.bufferedLimitIndex;
        if (numRecords != 0) {
            List<ConsumerRecord<byte[], byte[]>> records = changelogMetadata.bufferedRecords.subList(0, numRecords);
            OptionalLong optionalLag = this.restoreConsumer.currentLag(partition);
            stateManager.restore(storeMetadata, records, optionalLag);
            if (numRecords < changelogMetadata.bufferedRecords.size()) {
                records.clear();
            } else {
                changelogMetadata.bufferedRecords.clear();
            }
            task.recordRestoration(this.time, numRecords, false);
            Long currentOffset = storeMetadata.offset();
            this.log.trace("Restored {} records from changelog {} to store {}, end offset is {}, current offset is {}", new Object[]{numRecords, partition, storeName, StoreChangelogReader.recordEndOffset(changelogMetadata.restoreEndOffset), currentOffset});
            changelogMetadata.bufferedLimitIndex = 0;
            changelogMetadata.totalRestored = changelogMetadata.totalRestored + (long)numRecords;
            if (changelogMetadata.stateManager.taskType() == Task.TaskType.ACTIVE) {
                try {
                    this.stateRestoreListener.onBatchRestored(partition, storeName, currentOffset, numRecords);
                }
                catch (Exception e) {
                    throw new StreamsException("State restore listener failed on batch restored", e);
                }
            }
            if (changelogMetadata.stateManager.taskType() == Task.TaskType.STANDBY) {
                try {
                    this.standbyUpdateListener.onBatchLoaded(partition, storeName, stateManager.taskId(), currentOffset, numRecords, storeMetadata.endOffset());
                }
                catch (Exception e) {
                    throw new StreamsException("Standby updater listener failed on batch loaded", e);
                }
            }
        }
        if (changelogMetadata.stateManager.taskType() == Task.TaskType.ACTIVE && this.hasRestoredToEnd(changelogMetadata)) {
            this.log.info("Finished restoring changelog {} to store {} with a total number of {} records", new Object[]{partition, storeName, changelogMetadata.totalRestored});
            changelogMetadata.transitTo(ChangelogState.COMPLETED);
            this.pauseChangelogsFromRestoreConsumer(Collections.singleton(partition));
            try {
                this.stateRestoreListener.onRestoreEnd(partition, storeName, changelogMetadata.totalRestored);
            }
            catch (Exception e) {
                throw new StreamsException("State restore listener failed on restore completed", e);
            }
        }
        if (numRecords > 0 || changelogMetadata.state().equals((Object)ChangelogState.COMPLETED)) {
            task.clearTaskTimeout();
        }
        return numRecords;
    }

    private Set<Task> getTasksFromPartitions(Map<TaskId, Task> tasks, Set<TopicPartition> partitions) {
        HashSet<Task> result = new HashSet<Task>();
        for (TopicPartition partition : partitions) {
            result.add(tasks.get(this.changelogs.get(partition).stateManager.taskId()));
        }
        return result;
    }

    private void clearTaskTimeout(Set<Task> tasks) {
        tasks.forEach(t -> {
            if (t != null) {
                t.clearTaskTimeout();
            }
        });
    }

    private void maybeInitTaskTimeoutOrThrow(Set<Task> tasks, Exception cause) {
        long now = this.time.milliseconds();
        tasks.forEach(t -> t.maybeInitTaskTimeoutOrThrow(now, cause));
    }

    private Map<TopicPartition, Long> committedOffsetForChangelogs(Map<TaskId, Task> tasks, Set<TopicPartition> partitions) {
        if (partitions.isEmpty()) {
            return Collections.emptyMap();
        }
        try {
            ListConsumerGroupOffsetsOptions options = new ListConsumerGroupOffsetsOptions();
            options.topicPartitions(new ArrayList<TopicPartition>(partitions));
            options.requireStable(true);
            Map<TopicPartition, Long> committedOffsets = ((Map)this.adminClient.listConsumerGroupOffsets(this.groupId, options).partitionsToOffsetAndMetadata().get()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue() == null ? 0L : ((OffsetAndMetadata)e.getValue()).offset()));
            this.clearTaskTimeout(this.getTasksFromPartitions(tasks, partitions));
            return committedOffsets;
        }
        catch (InterruptedException | ExecutionException | TimeoutException retriableException) {
            this.log.debug("Could not retrieve the committed offsets for partitions {} due to {}, will retry in the next run loop", partitions, (Object)retriableException.toString());
            this.maybeInitTaskTimeoutOrThrow(this.getTasksFromPartitions(tasks, partitions), (Exception)retriableException);
            return Collections.emptyMap();
        }
        catch (KafkaException e2) {
            throw new StreamsException(String.format("Failed to retrieve committed offsets for %s", partitions), e2);
        }
    }

    private void filterNewPartitionsToRestore(Map<TaskId, Task> tasks, Set<ChangelogMetadata> newPartitionsToRestore) {
        newPartitionsToRestore.removeIf(changelogMetadata -> !tasks.containsKey(this.changelogs.get(((ChangelogMetadata)changelogMetadata).storeMetadata.changelogPartition()).stateManager.taskId()));
    }

    private Map<TopicPartition, Long> endOffsetForChangelogs(Map<TaskId, Task> tasks, Set<TopicPartition> partitions) {
        if (partitions.isEmpty()) {
            return Collections.emptyMap();
        }
        try {
            ListOffsetsOptions options = new ListOffsetsOptions(IsolationLevel.READ_UNCOMMITTED);
            Map offsetSpecs = partitions.stream().collect(Collectors.toMap(Function.identity(), tp -> OffsetSpec.latest()));
            Map<TopicPartition, Long> logEndOffsets = ((Map)this.adminClient.listOffsets(offsetSpecs, options).all().get()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((ListOffsetsResult.ListOffsetsResultInfo)entry.getValue()).offset()));
            this.clearTaskTimeout(this.getTasksFromPartitions(tasks, partitions));
            return logEndOffsets;
        }
        catch (InterruptedException | ExecutionException | TimeoutException retriableException) {
            this.log.debug("Could not fetch all end offsets for {} due to {}, will retry in the next run loop", partitions, (Object)retriableException.toString());
            this.maybeInitTaskTimeoutOrThrow(this.getTasksFromPartitions(tasks, partitions), (Exception)retriableException);
            return Collections.emptyMap();
        }
        catch (KafkaException e) {
            throw new StreamsException(String.format("Failed to retrieve end offsets for %s", partitions), e);
        }
    }

    private void updateLimitOffsetsForStandbyChangelogs(Map<TopicPartition, Long> committedOffsets) {
        for (ChangelogMetadata metadata : this.changelogs.values()) {
            TopicPartition partition = metadata.storeMetadata.changelogPartition();
            if (metadata.stateManager.taskType() != Task.TaskType.STANDBY || !metadata.stateManager.changelogAsSource(partition) || !committedOffsets.containsKey(partition)) continue;
            Long newLimit = committedOffsets.get(partition);
            Long previousLimit = metadata.restoreEndOffset;
            if (previousLimit != null && previousLimit > newLimit) {
                throw new IllegalStateException("Offset limit should monotonically increase, but was reduced for partition " + partition + ". New limit: " + newLimit + ". Previous limit: " + previousLimit);
            }
            metadata.restoreEndOffset = newLimit;
            while (metadata.bufferedLimitIndex < metadata.bufferedRecords.size() && ((ConsumerRecord)metadata.bufferedRecords.get(metadata.bufferedLimitIndex)).offset() < metadata.restoreEndOffset) {
                metadata.bufferedLimitIndex++;
            }
        }
    }

    private void initializeChangelogs(Map<TaskId, Task> tasks, Set<ChangelogMetadata> newPartitionsToRestore) {
        this.filterNewPartitionsToRestore(tasks, newPartitionsToRestore);
        if (newPartitionsToRestore.isEmpty()) {
            return;
        }
        HashSet<TopicPartition> newPartitionsToFindEndOffset = new HashSet<TopicPartition>();
        HashSet<TopicPartition> newPartitionsToFindCommittedOffset = new HashSet<TopicPartition>();
        for (ChangelogMetadata metadata2 : newPartitionsToRestore) {
            TopicPartition partition = metadata2.storeMetadata.changelogPartition();
            if (metadata2.stateManager.taskType() == Task.TaskType.ACTIVE) {
                newPartitionsToFindEndOffset.add(partition);
            }
            if (!metadata2.stateManager.changelogAsSource(partition)) continue;
            newPartitionsToFindCommittedOffset.add(partition);
        }
        Map<TopicPartition, Long> endOffsets = this.endOffsetForChangelogs(tasks, newPartitionsToFindEndOffset);
        Map<TopicPartition, Long> committedOffsets = this.committedOffsetForChangelogs(tasks, newPartitionsToFindCommittedOffset);
        for (TopicPartition partition : newPartitionsToFindEndOffset) {
            Long committedOffset;
            ChangelogMetadata changelogMetadata = this.changelogs.get(partition);
            Long endOffset = endOffsets.get(partition);
            Long l = committedOffset = newPartitionsToFindCommittedOffset.contains(partition) ? committedOffsets.get(partition) : Long.valueOf(Long.MAX_VALUE);
            if (endOffset != null && committedOffset != null) {
                if (changelogMetadata.restoreEndOffset != null) {
                    throw new IllegalStateException("End offset for " + partition + " should only be initialized once. Existing value: " + changelogMetadata.restoreEndOffset + ", new value: (" + endOffset + ", " + committedOffset + ")");
                }
                changelogMetadata.restoreEndOffset = Math.min(endOffset, committedOffset);
                this.log.info("End offset for changelog {} initialized as {}.", (Object)partition, (Object)changelogMetadata.restoreEndOffset);
                continue;
            }
            if (!newPartitionsToRestore.remove(changelogMetadata)) {
                throw new IllegalStateException("New changelogs to restore " + newPartitionsToRestore + " does not contain the one looking for end offset: " + partition + ", this should not happen.");
            }
            this.log.info("End offset for changelog {} cannot be found; will retry in the next time.", (Object)partition);
        }
        if (!committedOffsets.isEmpty()) {
            this.updateLimitOffsetsForStandbyChangelogs(committedOffsets);
        }
        this.addChangelogsToRestoreConsumer(newPartitionsToRestore.stream().map(metadata -> ((ChangelogMetadata)metadata).storeMetadata.changelogPartition()).collect(Collectors.toSet()));
        newPartitionsToRestore.forEach(metadata -> ((ChangelogMetadata)metadata).transitTo(ChangelogState.RESTORING));
        if (this.state == ChangelogReaderState.ACTIVE_RESTORING) {
            this.pauseChangelogsFromRestoreConsumer(this.standbyRestoringChangelogs());
        }
        this.prepareChangelogs(tasks, newPartitionsToRestore);
    }

    private void addChangelogsToRestoreConsumer(Set<TopicPartition> partitions) {
        if (partitions.isEmpty()) {
            return;
        }
        HashSet<TopicPartition> assignment = new HashSet<TopicPartition>(this.restoreConsumer.assignment());
        if (assignment.removeAll(partitions)) {
            throw new IllegalStateException("The current assignment " + this.restoreConsumer.assignment() + " already contains some of the new partitions " + partitions);
        }
        assignment.addAll(partitions);
        this.restoreConsumer.assign(assignment);
        this.log.debug("Added partitions {} to the restore consumer, current assignment is {}", partitions, assignment);
    }

    private void pauseChangelogsFromRestoreConsumer(Collection<TopicPartition> partitions) {
        HashSet assignment = new HashSet(this.restoreConsumer.assignment());
        if (!assignment.containsAll(partitions)) {
            throw new IllegalStateException("The current assignment " + assignment + " does not contain some of the partitions " + partitions + " for pausing.");
        }
        this.restoreConsumer.pause(partitions);
        this.log.debug("Paused partitions {} from the restore consumer", partitions);
    }

    private void removeChangelogsFromRestoreConsumer(Collection<TopicPartition> partitions) {
        if (partitions.isEmpty()) {
            return;
        }
        HashSet assignment = new HashSet(this.restoreConsumer.assignment());
        if (!assignment.containsAll(partitions)) {
            throw new IllegalStateException("The current assignment " + assignment + " does not contain some of the partitions " + partitions + " for removing.");
        }
        assignment.removeAll(partitions);
        this.restoreConsumer.assign(assignment);
    }

    private void resumeChangelogsFromRestoreConsumer(Collection<TopicPartition> partitions) {
        HashSet assignment = new HashSet(this.restoreConsumer.assignment());
        if (!assignment.containsAll(partitions)) {
            throw new IllegalStateException("The current assignment " + assignment + " does not contain some of the partitions " + partitions + " for resuming.");
        }
        this.restoreConsumer.resume(partitions);
        this.log.debug("Resumed partitions {} from the restore consumer", partitions);
    }

    private void prepareChangelogs(Map<TaskId, Task> tasks, Set<ChangelogMetadata> newPartitionsToRestore) {
        TopicPartition partition;
        ProcessorStateManager.StateStoreMetadata storeMetadata;
        HashSet<TopicPartition> newPartitionsWithoutStartOffset = new HashSet<TopicPartition>();
        for (ChangelogMetadata changelogMetadata : newPartitionsToRestore) {
            storeMetadata = changelogMetadata.storeMetadata;
            partition = storeMetadata.changelogPartition();
            Long currentOffset = storeMetadata.offset();
            Long endOffset = this.changelogs.get(partition).restoreEndOffset;
            if (currentOffset != null) {
                this.restoreConsumer.seek(partition, currentOffset + 1L);
                this.log.debug("Start restoring changelog partition {} from current offset {} to end offset {}.", new Object[]{partition, currentOffset, StoreChangelogReader.recordEndOffset(endOffset)});
                continue;
            }
            this.log.debug("Start restoring changelog partition {} from the beginning offset to end offset {} since we cannot find current offset.", (Object)partition, (Object)StoreChangelogReader.recordEndOffset(endOffset));
            newPartitionsWithoutStartOffset.add(partition);
        }
        if (!newPartitionsWithoutStartOffset.isEmpty()) {
            this.restoreConsumer.seekToBeginning(newPartitionsWithoutStartOffset);
        }
        for (ChangelogMetadata changelogMetadata : newPartitionsToRestore) {
            storeMetadata = changelogMetadata.storeMetadata;
            partition = storeMetadata.changelogPartition();
            String storeName = storeMetadata.store().name();
            long startOffset = 0L;
            try {
                startOffset = this.restoreConsumer.position(partition);
            }
            catch (TimeoutException timeoutException) {
            }
            catch (KafkaException e) {
                throw new StreamsException("Restore consumer get unexpected error trying to get the position  of " + partition, e);
            }
            if (changelogMetadata.stateManager.taskType() == Task.TaskType.ACTIVE) {
                try {
                    this.stateRestoreListener.onRestoreStart(partition, storeName, startOffset, changelogMetadata.restoreEndOffset);
                }
                catch (Exception e) {
                    throw new StreamsException("State restore listener failed on restore start", e);
                }
                TaskId taskId = changelogMetadata.stateManager.taskId();
                Task task = tasks.get(taskId);
                long recordsToRestore = Math.max(changelogMetadata.restoreEndOffset - startOffset, 0L);
                task.recordRestoration(this.time, recordsToRestore, true);
                continue;
            }
            if (changelogMetadata.stateManager.taskType() != Task.TaskType.STANDBY) continue;
            try {
                this.standbyUpdateListener.onUpdateStart(partition, storeName, startOffset);
            }
            catch (Exception e) {
                throw new StreamsException("Standby updater listener failed on update start");
            }
        }
    }

    @Override
    public void unregister(Collection<TopicPartition> revokedChangelogs) {
        ArrayList<TopicPartition> revokedInitializedChangelogs = new ArrayList<TopicPartition>();
        for (TopicPartition partition : revokedChangelogs) {
            ChangelogMetadata changelogMetadata = this.changelogs.remove(partition);
            if (changelogMetadata != null) {
                if (!changelogMetadata.state().equals((Object)ChangelogState.REGISTERED)) {
                    revokedInitializedChangelogs.add(partition);
                    if (changelogMetadata.state().equals((Object)ChangelogState.RESTORING)) {
                        String storeName = changelogMetadata.storeMetadata.store().name();
                        if (changelogMetadata.stateManager.taskType() == Task.TaskType.ACTIVE) {
                            try {
                                this.stateRestoreListener.onRestoreSuspended(partition, storeName, changelogMetadata.totalRestored);
                            }
                            catch (Exception e) {
                                throw new StreamsException("State restore listener failed on restore paused", e);
                            }
                        }
                        if (changelogMetadata.stateManager.taskType() == Task.TaskType.STANDBY) {
                            ProcessorStateManager.StateStoreMetadata storeMetadata = changelogMetadata.stateManager.storeMetadata(partition);
                            long storeOffset = storeMetadata.offset() != null ? storeMetadata.offset() : -1L;
                            long endOffset = storeMetadata.endOffset() != null ? storeMetadata.endOffset() : -1L;
                            StandbyUpdateListener.SuspendReason suspendReason = changelogMetadata.stateManager.taskState() == Task.State.RUNNING ? StandbyUpdateListener.SuspendReason.PROMOTED : StandbyUpdateListener.SuspendReason.MIGRATED;
                            try {
                                this.standbyUpdateListener.onUpdateSuspended(partition, storeName, storeOffset, endOffset, suspendReason);
                            }
                            catch (Exception e) {
                                throw new StreamsException("Standby updater listener failed on update suspended", e);
                            }
                        }
                    }
                }
                changelogMetadata.clear();
                continue;
            }
            this.log.debug("Changelog partition {} could not be found, it could be already cleaned up during the handling of task corruption and never restore again", (Object)partition);
        }
        this.removeChangelogsFromRestoreConsumer(revokedInitializedChangelogs);
        if (this.changelogs.isEmpty()) {
            this.state = ChangelogReaderState.ACTIVE_RESTORING;
        }
    }

    @Override
    public void clear() {
        for (ChangelogMetadata changelogMetadata : this.changelogs.values()) {
            changelogMetadata.clear();
        }
        this.changelogs.clear();
        this.state = ChangelogReaderState.ACTIVE_RESTORING;
        try {
            this.restoreConsumer.unsubscribe();
        }
        catch (KafkaException e) {
            throw new StreamsException("Restore consumer get unexpected error unsubscribing", e);
        }
    }

    @Override
    public boolean isEmpty() {
        return this.changelogs.isEmpty();
    }

    public String toString() {
        return "StoreChangelogReader: " + this.changelogs + "\n";
    }

    ChangelogMetadata changelogMetadata(TopicPartition partition) {
        return this.changelogs.get(partition);
    }

    ChangelogReaderState state() {
        return this.state;
    }

    static class ChangelogMetadata {
        private final ProcessorStateManager.StateStoreMetadata storeMetadata;
        private final ProcessorStateManager stateManager;
        private ChangelogState changelogState = ChangelogState.REGISTERED;
        private long totalRestored;
        private Long restoreEndOffset;
        private final List<ConsumerRecord<byte[], byte[]>> bufferedRecords;
        private int bufferedLimitIndex;

        private ChangelogMetadata(ProcessorStateManager.StateStoreMetadata storeMetadata, ProcessorStateManager stateManager) {
            this.storeMetadata = storeMetadata;
            this.stateManager = stateManager;
            this.restoreEndOffset = null;
            this.totalRestored = 0L;
            this.bufferedRecords = new ArrayList<ConsumerRecord<byte[], byte[]>>();
            this.bufferedLimitIndex = 0;
        }

        private void clear() {
            this.bufferedRecords.clear();
        }

        private void transitTo(ChangelogState newState) {
            if (!newState.prevStates.contains(this.changelogState.ordinal())) {
                throw new IllegalStateException("Invalid transition from " + (Object)((Object)this.changelogState) + " to " + (Object)((Object)newState));
            }
            this.changelogState = newState;
        }

        public String toString() {
            Long currentOffset = this.storeMetadata.offset();
            return (Object)((Object)this.changelogState) + " " + (Object)((Object)this.stateManager.taskType()) + " (currentOffset " + currentOffset + ", endOffset " + this.restoreEndOffset + ")";
        }

        ChangelogState state() {
            return this.changelogState;
        }

        long totalRestored() {
            return this.totalRestored;
        }

        Long endOffset() {
            return this.restoreEndOffset;
        }

        List<ConsumerRecord<byte[], byte[]>> bufferedRecords() {
            return this.bufferedRecords;
        }

        int bufferedLimitIndex() {
            return this.bufferedLimitIndex;
        }
    }

    static enum ChangelogReaderState {
        ACTIVE_RESTORING("ACTIVE_RESTORING"),
        STANDBY_UPDATING("STANDBY_UPDATING");

        public final String name;

        private ChangelogReaderState(String name) {
            this.name = name;
        }
    }

    static enum ChangelogState {
        REGISTERED("REGISTERED", new Integer[0]),
        RESTORING("RESTORING", 0),
        COMPLETED("COMPLETED", 1);

        public final String name;
        private final List<Integer> prevStates;

        private ChangelogState(String name, Integer ... prevStates) {
            this.name = name;
            this.prevStates = Arrays.asList(prevStates);
        }
    }
}

