/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.raft.CandidateState;
import org.apache.kafka.raft.ElectionState;
import org.apache.kafka.raft.EpochState;
import org.apache.kafka.raft.FollowerState;
import org.apache.kafka.raft.LeaderAndEpoch;
import org.apache.kafka.raft.LeaderState;
import org.apache.kafka.raft.LogOffsetMetadata;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.raft.QuorumStateStore;
import org.apache.kafka.raft.ResignedState;
import org.apache.kafka.raft.UnattachedState;
import org.apache.kafka.raft.VotedState;
import org.apache.kafka.raft.internals.BatchAccumulator;
import org.slf4j.Logger;

public class QuorumState {
    private final OptionalInt localId;
    private final Time time;
    private final Logger log;
    private final QuorumStateStore store;
    private final Set<Integer> voters;
    private final Random random;
    private final int electionTimeoutMs;
    private final int fetchTimeoutMs;
    private final LogContext logContext;
    private volatile EpochState state;

    public QuorumState(OptionalInt localId, Set<Integer> voters, int electionTimeoutMs, int fetchTimeoutMs, QuorumStateStore store, Time time, LogContext logContext, Random random) {
        this.localId = localId;
        this.voters = new HashSet<Integer>(voters);
        this.electionTimeoutMs = electionTimeoutMs;
        this.fetchTimeoutMs = fetchTimeoutMs;
        this.store = store;
        this.time = time;
        this.log = logContext.logger(QuorumState.class);
        this.random = random;
        this.logContext = logContext;
    }

    public void initialize(OffsetAndEpoch logEndOffsetAndEpoch) throws IllegalStateException {
        EpochState initialState;
        ElectionState election;
        try {
            election = this.store.readElectionState();
            if (election == null) {
                election = ElectionState.withUnknownLeader(0, this.voters);
            }
        }
        catch (UncheckedIOException e) {
            this.log.warn("Clearing local quorum state store after error loading state {}", (Object)this.store.toString(), (Object)e);
            this.store.clear();
            election = ElectionState.withUnknownLeader(0, this.voters);
        }
        if (!election.voters().isEmpty() && !this.voters.equals(election.voters())) {
            throw new IllegalStateException("Configured voter set: " + this.voters + " is different from the voter set read from the state file: " + election.voters() + ". Check if the quorum configuration is up to date, or wipe out the local state file if necessary");
        }
        if (election.hasVoted() && !this.isVoter()) {
            String localIdDescription = this.localId.isPresent() ? this.localId.getAsInt() + " is not a voter" : "is undefined";
            throw new IllegalStateException("Initialized quorum state " + election + " with a voted candidate, which indicates this node was previously  a voter, but the local id " + localIdDescription);
        }
        if (election.epoch < logEndOffsetAndEpoch.epoch()) {
            this.log.warn("Epoch from quorum-state file is {}, which is smaller than last written epoch {} in the log", (Object)election.epoch, (Object)logEndOffsetAndEpoch.epoch());
            initialState = new UnattachedState(this.time, logEndOffsetAndEpoch.epoch(), this.voters, Optional.empty(), this.randomElectionTimeoutMs(), this.logContext);
        } else {
            initialState = this.localId.isPresent() && election.isLeader(this.localId.getAsInt()) ? new ResignedState(this.time, this.localId.getAsInt(), election.epoch, this.voters, this.randomElectionTimeoutMs(), Collections.emptyList(), this.logContext) : (this.localId.isPresent() && election.isVotedCandidate(this.localId.getAsInt()) ? new CandidateState(this.time, this.localId.getAsInt(), election.epoch, this.voters, Optional.empty(), 1, this.randomElectionTimeoutMs(), this.logContext) : (election.hasVoted() ? new VotedState(this.time, election.epoch, election.votedId(), this.voters, Optional.empty(), this.randomElectionTimeoutMs(), this.logContext) : (election.hasLeader() ? new FollowerState(this.time, election.epoch, election.leaderId(), this.voters, Optional.empty(), this.fetchTimeoutMs, this.logContext) : new UnattachedState(this.time, election.epoch, this.voters, Optional.empty(), this.randomElectionTimeoutMs(), this.logContext))));
        }
        this.durableTransitionTo(initialState);
    }

    public Set<Integer> remoteVoters() {
        return this.voters.stream().filter(voterId -> voterId.intValue() != this.localIdOrSentinel()).collect(Collectors.toSet());
    }

    public int localIdOrSentinel() {
        return this.localId.orElse(-1);
    }

    public int localIdOrThrow() {
        return this.localId.orElseThrow(() -> new IllegalStateException("Required local id is not present"));
    }

    public OptionalInt localId() {
        return this.localId;
    }

    public int epoch() {
        return this.state.epoch();
    }

    public int leaderIdOrSentinel() {
        return this.state.election().leaderIdOrSentinel();
    }

    public Optional<LogOffsetMetadata> highWatermark() {
        return this.state.highWatermark();
    }

    public OptionalInt leaderId() {
        ElectionState election = this.state.election();
        if (election.hasLeader()) {
            return OptionalInt.of(this.state.election().leaderId());
        }
        return OptionalInt.empty();
    }

    public boolean hasLeader() {
        return this.leaderId().isPresent();
    }

    public boolean hasRemoteLeader() {
        return this.hasLeader() && this.leaderIdOrSentinel() != this.localIdOrSentinel();
    }

    public boolean isVoter() {
        return this.localId.isPresent() && this.voters.contains(this.localId.getAsInt());
    }

    public boolean isVoter(int nodeId) {
        return this.voters.contains(nodeId);
    }

    public boolean isObserver() {
        return !this.isVoter();
    }

    public void transitionToResigned(List<Integer> preferredSuccessors) {
        if (!this.isLeader()) {
            throw new IllegalStateException("Invalid transition to Resigned state from " + this.state);
        }
        int epoch = this.state.epoch();
        this.memoryTransitionTo(new ResignedState(this.time, this.localIdOrThrow(), epoch, this.voters, this.randomElectionTimeoutMs(), preferredSuccessors, this.logContext));
    }

    public void transitionToUnattached(int epoch) {
        int currentEpoch = this.state.epoch();
        if (epoch <= currentEpoch) {
            throw new IllegalStateException("Cannot transition to Unattached with epoch= " + epoch + " from current state " + this.state);
        }
        long electionTimeoutMs = this.isObserver() ? Long.MAX_VALUE : (this.isCandidate() ? this.candidateStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (this.isVoted() ? this.votedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (this.isUnattached() ? this.unattachedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (long)this.randomElectionTimeoutMs())));
        this.durableTransitionTo(new UnattachedState(this.time, epoch, this.voters, this.state.highWatermark(), electionTimeoutMs, this.logContext));
    }

    public void transitionToVoted(int epoch, int candidateId) {
        if (this.localId.isPresent() && candidateId == this.localId.getAsInt()) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " since it matches the local broker.id");
        }
        if (this.isObserver()) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " since the local broker.id=" + this.localId + " is not a voter");
        }
        if (!this.isVoter(candidateId)) {
            throw new IllegalStateException("Cannot transition to Voted with voterId=" + candidateId + " and epoch=" + epoch + " since it is not one of the voters " + this.voters);
        }
        int currentEpoch = this.state.epoch();
        if (epoch < currentEpoch) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " since the current epoch " + currentEpoch + " is larger");
        }
        if (epoch == currentEpoch && !this.isUnattached()) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " from the current state " + this.state);
        }
        this.durableTransitionTo(new VotedState(this.time, epoch, candidateId, this.voters, this.state.highWatermark(), this.randomElectionTimeoutMs(), this.logContext));
    }

    public void transitionToFollower(int epoch, int leaderId) {
        if (this.localId.isPresent() && leaderId == this.localId.getAsInt()) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " since it matches the local broker.id=" + this.localId);
        }
        if (!this.isVoter(leaderId)) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " since it is not one of the voters " + this.voters);
        }
        int currentEpoch = this.state.epoch();
        if (epoch < currentEpoch) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " since the current epoch " + currentEpoch + " is larger");
        }
        if (epoch == currentEpoch && (this.isFollower() || this.isLeader())) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " from state " + this.state);
        }
        this.durableTransitionTo(new FollowerState(this.time, epoch, leaderId, this.voters, this.state.highWatermark(), this.fetchTimeoutMs, this.logContext));
    }

    public void transitionToCandidate() {
        if (this.isObserver()) {
            throw new IllegalStateException("Cannot transition to Candidate since the local broker.id=" + this.localId + " is not one of the voters " + this.voters);
        }
        if (this.isLeader()) {
            throw new IllegalStateException("Cannot transition to Candidate since the local broker.id=" + this.localId + " since this node is already a Leader with state " + this.state);
        }
        int retries = this.isCandidate() ? this.candidateStateOrThrow().retries() + 1 : 1;
        int newEpoch = this.epoch() + 1;
        int electionTimeoutMs = this.randomElectionTimeoutMs();
        this.durableTransitionTo(new CandidateState(this.time, this.localIdOrThrow(), newEpoch, this.voters, this.state.highWatermark(), retries, electionTimeoutMs, this.logContext));
    }

    public <T> LeaderState<T> transitionToLeader(long epochStartOffset, BatchAccumulator<T> accumulator) {
        if (this.isObserver()) {
            throw new IllegalStateException("Cannot transition to Leader since the local broker.id=" + this.localId + " is not one of the voters " + this.voters);
        }
        if (!this.isCandidate()) {
            throw new IllegalStateException("Cannot transition to Leader from current state " + this.state);
        }
        CandidateState candidateState = this.candidateStateOrThrow();
        if (!candidateState.isVoteGranted()) {
            throw new IllegalStateException("Cannot become leader without majority votes granted");
        }
        LeaderState<T> state = new LeaderState<T>(this.time, this.localIdOrThrow(), this.epoch(), epochStartOffset, this.voters, candidateState.grantingVoters(), accumulator, this.fetchTimeoutMs, this.logContext);
        this.durableTransitionTo(state);
        return state;
    }

    private void durableTransitionTo(EpochState state) {
        if (this.state != null) {
            try {
                this.state.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException("Failed to transition from " + this.state.name() + " to " + state.name(), e);
            }
        }
        this.store.writeElectionState(state.election());
        this.memoryTransitionTo(state);
    }

    private void memoryTransitionTo(EpochState state) {
        EpochState from = this.state;
        this.state = state;
        this.log.info("Completed transition to {} from {}", (Object)state, (Object)from);
    }

    private int randomElectionTimeoutMs() {
        if (this.electionTimeoutMs == 0) {
            return 0;
        }
        return this.electionTimeoutMs + this.random.nextInt(this.electionTimeoutMs);
    }

    public boolean canGrantVote(int candidateId, boolean isLogUpToDate) {
        return this.state.canGrantVote(candidateId, isLogUpToDate);
    }

    public FollowerState followerStateOrThrow() {
        if (this.isFollower()) {
            return (FollowerState)this.state;
        }
        throw new IllegalStateException("Expected to be Follower, but the current state is " + this.state);
    }

    public VotedState votedStateOrThrow() {
        if (this.isVoted()) {
            return (VotedState)this.state;
        }
        throw new IllegalStateException("Expected to be Voted, but current state is " + this.state);
    }

    public UnattachedState unattachedStateOrThrow() {
        if (this.isUnattached()) {
            return (UnattachedState)this.state;
        }
        throw new IllegalStateException("Expected to be Unattached, but current state is " + this.state);
    }

    public <T> LeaderState<T> leaderStateOrThrow() {
        if (this.isLeader()) {
            return (LeaderState)this.state;
        }
        throw new IllegalStateException("Expected to be Leader, but current state is " + this.state);
    }

    public <T> Optional<LeaderState<T>> maybeLeaderState() {
        EpochState state = this.state;
        if (state instanceof LeaderState) {
            return Optional.of((LeaderState)state);
        }
        return Optional.empty();
    }

    public ResignedState resignedStateOrThrow() {
        if (this.isResigned()) {
            return (ResignedState)this.state;
        }
        throw new IllegalStateException("Expected to be Resigned, but current state is " + this.state);
    }

    public CandidateState candidateStateOrThrow() {
        if (this.isCandidate()) {
            return (CandidateState)this.state;
        }
        throw new IllegalStateException("Expected to be Candidate, but current state is " + this.state);
    }

    public LeaderAndEpoch leaderAndEpoch() {
        ElectionState election = this.state.election();
        return new LeaderAndEpoch(election.leaderIdOpt, election.epoch);
    }

    public boolean isFollower() {
        return this.state instanceof FollowerState;
    }

    public boolean isVoted() {
        return this.state instanceof VotedState;
    }

    public boolean isUnattached() {
        return this.state instanceof UnattachedState;
    }

    public boolean isLeader() {
        return this.state instanceof LeaderState;
    }

    public boolean isResigned() {
        return this.state instanceof ResignedState;
    }

    public boolean isCandidate() {
        return this.state instanceof CandidateState;
    }
}

