/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.rest.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import io.confluent.ksql.function.FunctionRegistry;
import io.confluent.ksql.function.InternalFunctionRegistry;
import io.confluent.ksql.function.MutableFunctionRegistry;
import io.confluent.ksql.function.UserFunctionLoader;
import io.confluent.ksql.logging.query.QueryLogger;
import io.confluent.ksql.metrics.MetricCollectors;
import io.confluent.ksql.properties.PropertiesUtil;
import io.confluent.ksql.rest.server.ConnectExecutable;
import io.confluent.ksql.rest.server.Executable;
import io.confluent.ksql.rest.server.KsqlRestApplication;
import io.confluent.ksql.rest.server.KsqlRestConfig;
import io.confluent.ksql.rest.server.MultiExecutable;
import io.confluent.ksql.rest.server.PreconditionChecker;
import io.confluent.ksql.rest.server.ServerOptions;
import io.confluent.ksql.rest.server.StandaloneExecutorFactory;
import io.confluent.ksql.rest.server.state.ServerState;
import io.confluent.ksql.serde.FormatFactory;
import io.confluent.ksql.util.KsqlConfig;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.KsqlServerException;
import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.kafka.common.config.internals.ConfluentConfigs;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.security.auth.SecurityProtocol;
import org.apache.kafka.common.security.fips.FipsValidator;
import org.apache.kafka.streams.StreamsConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KsqlServerMain {
    private static final Logger log = LoggerFactory.getLogger(KsqlServerMain.class);
    private final Executor shutdownHandler;
    private final Executable preconditionChecker;
    private final Supplier<Executable> executable;

    public static void main(String[] args) {
        try {
            ServerOptions serverOptions = ServerOptions.parse(args);
            if (serverOptions == null) {
                return;
            }
            Supplier<Map<String, String>> propertiesLoader = () -> PropertiesUtil.applyOverrides((Map)PropertiesUtil.loadProperties(serverOptions.getPropertiesFile()), (Properties)System.getProperties());
            Map properties = propertiesLoader.get();
            String installDir = properties.getOrDefault("ksql.server.install.dir", "");
            KsqlConfig ksqlConfig = new KsqlConfig(properties);
            KsqlRestConfig restConfig = new KsqlRestConfig(properties);
            KsqlServerMain.validateConfig(ksqlConfig, restConfig);
            QueryLogger.configure((KsqlConfig)ksqlConfig);
            Optional<String> queriesFile = serverOptions.getQueriesFile(properties);
            MetricCollectors metricCollectors = new MetricCollectors();
            ServerState serverState = new ServerState();
            FunctionRegistry functionRegistry = KsqlServerMain.loadFunctions(propertiesLoader, metricCollectors);
            PreconditionChecker preconditionChecker = new PreconditionChecker(propertiesLoader, serverState);
            Supplier<Executable> executableFactory = () -> KsqlServerMain.createExecutable(propertiesLoader, serverState, queriesFile, installDir, metricCollectors, functionRegistry);
            new KsqlServerMain(preconditionChecker, executableFactory, r -> Runtime.getRuntime().addShutdownHook(new Thread(r))).tryStartApp();
        }
        catch (Exception e) {
            log.error("Failed to start KSQL", (Throwable)e);
            System.exit(-1);
        }
    }

    KsqlServerMain(Executable preconditionChecker, Supplier<Executable> executableFactory, Executor shutdownHandler) {
        this.preconditionChecker = Objects.requireNonNull(preconditionChecker, "preconditionChecker");
        this.executable = Objects.requireNonNull(executableFactory, "executableFactory");
        this.shutdownHandler = Objects.requireNonNull(shutdownHandler, "shutdownHandler");
    }

    void tryStartApp() throws Exception {
        boolean shutdown = this.runExecutable(this.preconditionChecker);
        if (shutdown) {
            return;
        }
        this.runExecutable(this.executable.get());
    }

    private static FunctionRegistry loadFunctions(Supplier<Map<String, String>> propertiesLoader, MetricCollectors metricCollectors) {
        Map<String, String> properties = propertiesLoader.get();
        KsqlRestConfig restConfig = new KsqlRestConfig(properties);
        String ksqlInstallDir = restConfig.getString("ksql.server.install.dir");
        KsqlConfig ksqlConfig = new KsqlConfig(restConfig.getKsqlConfigProperties());
        InternalFunctionRegistry functionRegistry = new InternalFunctionRegistry();
        UserFunctionLoader.newInstance((KsqlConfig)ksqlConfig, (MutableFunctionRegistry)functionRegistry, (String)ksqlInstallDir, (Metrics)metricCollectors.getMetrics()).load();
        return functionRegistry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean runExecutable(Executable executable) throws Exception {
        CountDownLatch latch = new CountDownLatch(1);
        AtomicBoolean notified = new AtomicBoolean(false);
        this.shutdownHandler.execute(() -> {
            executable.notifyTerminated();
            try {
                notified.set(true);
                latch.await();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        });
        try {
            try {
                log.info("Starting server");
                executable.startAsync();
                log.info("Server up and running");
                executable.awaitTerminated();
            }
            catch (Throwable t) {
                log.error("Unhandled exception in server startup", t);
                throw t;
            }
            finally {
                log.info("Server shutting down");
                executable.shutdown();
            }
        }
        finally {
            latch.countDown();
        }
        return notified.get();
    }

    private static void validateConfig(KsqlConfig config, KsqlRestConfig restConfig) {
        KsqlServerMain.validateFips(config, restConfig);
        KsqlServerMain.validateStateDir(config);
        KsqlServerMain.validateDefaultTopicFormats(config);
    }

    private static void validateStateDir(KsqlConfig config) {
        String streamsStateDirPath = config.getKsqlStreamConfigProps().getOrDefault("state.dir", StreamsConfig.configDef().defaultValues().get("state.dir")).toString();
        KsqlServerMain.enforceStreamStateDirAvailability(new File(streamsStateDirPath));
    }

    @VisibleForTesting
    static void validateDefaultTopicFormats(KsqlConfig config) {
        KsqlServerMain.validateTopicFormat(config, "ksql.persistence.default.format.key", "key");
        KsqlServerMain.validateTopicFormat(config, "ksql.persistence.default.format.value", "value");
    }

    @VisibleForTesting
    static void validateFips(KsqlConfig config, KsqlRestConfig restConfig) {
        if (config.getBoolean("enable.fips").booleanValue()) {
            FipsValidator fipsValidator = ConfluentConfigs.buildFipsValidator();
            KsqlServerMain.validateCipherSuites(fipsValidator, restConfig);
            KsqlServerMain.validateBroker(fipsValidator, config);
            KsqlServerMain.validateSslEndpointAlgo(fipsValidator, restConfig);
            KsqlServerMain.validateSrUrl(fipsValidator, restConfig);
            KsqlServerMain.validateListeners(fipsValidator, restConfig);
            log.info("FIPS mode enabled for ksqlDB!");
        }
    }

    private static void validateCipherSuites(FipsValidator fipsValidator, KsqlRestConfig restConfig) {
        HashMap<String, List> fipsTlsMap = new HashMap<String, List>();
        List cipherSuites = restConfig.getList("ssl.cipher.suites");
        if (!cipherSuites.isEmpty()) {
            fipsTlsMap.put("ssl.cipher.suites", cipherSuites);
        }
        fipsTlsMap.put("ssl.enabled.protocols", restConfig.getList("ssl.enabled.protocols"));
        try {
            fipsValidator.validateFipsTls(fipsTlsMap);
        }
        catch (Exception e) {
            log.error(e.getMessage());
            throw new SecurityException(e.getMessage());
        }
    }

    private static void validateBroker(FipsValidator fipsValidator, KsqlConfig config) {
        HashMap<String, SecurityProtocol> securityProtocolMap = new HashMap<String, SecurityProtocol>();
        if (!config.originals().containsKey("security.protocol")) {
            String errorMsg = "The security protocol ('security.protocol') is not specified.";
            log.error("The security protocol ('security.protocol') is not specified.");
            throw new SecurityException("The security protocol ('security.protocol') is not specified.");
        }
        String brokerSecurityProtocol = config.originals().get("security.protocol").toString();
        securityProtocolMap.put("security.protocol", SecurityProtocol.forName((String)brokerSecurityProtocol));
        try {
            fipsValidator.validateFipsBrokerProtocol(securityProtocolMap);
        }
        catch (Exception e) {
            log.error(e.getMessage());
            throw new SecurityException(e.getMessage());
        }
    }

    private static void validateSslEndpointAlgo(FipsValidator fipsValidator, KsqlRestConfig restConfig) {
        if (!restConfig.originals().containsKey("ssl.endpoint.identification.algorithm")) {
            String errorMsg = "The SSL endpoint identification algorithm ('ssl.endpoint.identification.algorithm') is not specified.";
            log.error("The SSL endpoint identification algorithm ('ssl.endpoint.identification.algorithm') is not specified.");
            throw new SecurityException("The SSL endpoint identification algorithm ('ssl.endpoint.identification.algorithm') is not specified.");
        }
        try {
            fipsValidator.validateRestProtocol(restConfig.originals().get("ssl.endpoint.identification.algorithm").toString());
        }
        catch (Exception e) {
            String errorMsg = e.getMessage() + "\nInvalid rest protocol for " + "ssl.endpoint.identification.algorithm";
            log.error(errorMsg);
            throw new SecurityException(errorMsg);
        }
    }

    private static void validateSrUrl(FipsValidator fipsValidator, KsqlRestConfig restConfig) {
        if (restConfig.originals().containsKey("ksql.schema.registry.url")) {
            try {
                fipsValidator.validateRestProtocol(KsqlServerMain.determineProtocol(restConfig.originals().get("ksql.schema.registry.url").toString()));
            }
            catch (Exception e) {
                String errorMsg = e.getMessage() + "\nInvalid rest protocol for " + "ksql.schema.registry.url";
                log.error(errorMsg);
                throw new SecurityException(errorMsg);
            }
        }
    }

    private static void validateListeners(FipsValidator fipsValidator, KsqlRestConfig restConfig) {
        try {
            String advertisedListener;
            List listeners = restConfig.getList("listeners");
            for (Object listener : listeners) {
                fipsValidator.validateRestProtocol(KsqlServerMain.determineProtocol((String)listener));
            }
            List proxyListeners = restConfig.getList("listeners.proxy.protocol");
            for (String listener : proxyListeners) {
                fipsValidator.validateRestProtocol(KsqlServerMain.determineProtocol(listener));
            }
            String internalListener = restConfig.getString("ksql.internal.listener");
            if (!Strings.isNullOrEmpty((String)internalListener)) {
                fipsValidator.validateRestProtocol(KsqlServerMain.determineProtocol(internalListener));
            }
            if (!Strings.isNullOrEmpty((String)(advertisedListener = restConfig.getString("ksql.advertised.listener")))) {
                fipsValidator.validateRestProtocol(KsqlServerMain.determineProtocol(advertisedListener));
            }
        }
        catch (Exception e) {
            String errorMsg = e.getMessage() + "\nInvalid rest protocol for listeners.\nMake sure that all " + "listeners" + ", " + "listeners.proxy.protocol" + ", " + "ksql.advertised.listener" + ", and " + "ksql.internal.listener" + " follow FIPS 140-2.";
            log.error(errorMsg);
            throw new SecurityException(errorMsg);
        }
    }

    private static String determineProtocol(String url) {
        if (url.isEmpty() || url.contains(String.format("%s://", "https"))) {
            return "https";
        }
        return "http";
    }

    private static void validateTopicFormat(KsqlConfig config, String configName, String type) {
        String formatName = config.getString(configName);
        if (formatName == null) {
            return;
        }
        try {
            FormatFactory.fromName((String)formatName);
        }
        catch (KsqlException e) {
            throw new KsqlException("Invalid value for config '" + configName + "': " + formatName, (Throwable)e);
        }
    }

    private static Executable createExecutable(Supplier<Map<String, String>> propertiesLoader, ServerState serverState, Optional<String> queriesFile, String installDir, MetricCollectors metricCollectors, FunctionRegistry functionRegistry) {
        Map<String, String> properties = propertiesLoader.get();
        KsqlConfig ksqlConfig = new KsqlConfig(properties);
        if (queriesFile.isPresent()) {
            return StandaloneExecutorFactory.create(properties, queriesFile.get(), installDir, metricCollectors);
        }
        KsqlRestConfig restConfig = new KsqlRestConfig(properties);
        KsqlRestApplication restApp = KsqlRestApplication.buildApplication(restConfig, serverState, metricCollectors, functionRegistry, Instant.now());
        String connectConfigFile = ksqlConfig.getString("ksql.connect.worker.config");
        if (connectConfigFile.isEmpty()) {
            return restApp;
        }
        try {
            ConnectExecutable connect = ConnectExecutable.of(connectConfigFile);
            return MultiExecutable.of(connect, restApp);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    static void enforceStreamStateDirAvailability(File streamsStateDir) {
        boolean mkDirSuccess;
        if (!streamsStateDir.exists() && !(mkDirSuccess = streamsStateDir.mkdirs())) {
            throw new KsqlServerException("Could not create the kafka streams state directory: " + streamsStateDir.getPath() + "\n Make sure the directory exists and is writable for KSQL server \n or its parent directory is writable by KSQL server\n or change it to a writable directory by setting '" + "ksql.streams." + "state.dir" + "' config in the properties file.");
        }
        if (!streamsStateDir.isDirectory()) {
            throw new KsqlServerException(streamsStateDir.getPath() + " is not a directory.\n Make sure the directory exists and is writable for KSQL server \n or its parent directory is writable by KSQL server\n or change it to a writable directory by setting '" + "ksql.streams." + "state.dir" + "' config in the properties file.");
        }
        if (!streamsStateDir.canWrite() || !streamsStateDir.canExecute()) {
            throw new KsqlServerException("The kafka streams state directory is not writable for KSQL server: " + streamsStateDir.getPath() + "\n Make sure the directory exists and is writable for KSQL server \n or change it to a writable directory by setting '" + "ksql.streams." + "state.dir" + "' config in the properties file.");
        }
    }
}

