/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.group.runtime;

import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.CoordinatorLoadInProgressException;
import org.apache.kafka.common.errors.NotCoordinatorException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.TransactionResult;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.coordinator.group.metrics.CoordinatorMetrics;
import org.apache.kafka.coordinator.group.metrics.CoordinatorRuntimeMetrics;
import org.apache.kafka.coordinator.group.runtime.CoordinatorEvent;
import org.apache.kafka.coordinator.group.runtime.CoordinatorEventProcessor;
import org.apache.kafka.coordinator.group.runtime.CoordinatorLoader;
import org.apache.kafka.coordinator.group.runtime.CoordinatorResult;
import org.apache.kafka.coordinator.group.runtime.CoordinatorShard;
import org.apache.kafka.coordinator.group.runtime.CoordinatorShardBuilderSupplier;
import org.apache.kafka.coordinator.group.runtime.CoordinatorTimer;
import org.apache.kafka.coordinator.group.runtime.PartitionWriter;
import org.apache.kafka.coordinator.group.runtime.SnapshottableCoordinator;
import org.apache.kafka.deferred.DeferredEvent;
import org.apache.kafka.deferred.DeferredEventQueue;
import org.apache.kafka.image.MetadataDelta;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.server.util.timer.Timer;
import org.apache.kafka.server.util.timer.TimerTask;
import org.apache.kafka.storage.internals.log.VerificationGuard;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.slf4j.Logger;

public class CoordinatorRuntime<S extends CoordinatorShard<U>, U>
implements AutoCloseable {
    private final String logPrefix;
    private final LogContext logContext;
    private final Logger log;
    private final Time time;
    private final Timer timer;
    private final Duration defaultWriteTimeout;
    private final ConcurrentHashMap<TopicPartition, CoordinatorContext> coordinators;
    private final CoordinatorEventProcessor processor;
    private final PartitionWriter<U> partitionWriter;
    private final HighWatermarkListener highWatermarklistener;
    private final CoordinatorLoader<U> loader;
    private final CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier;
    private final CoordinatorRuntimeMetrics runtimeMetrics;
    private final CoordinatorMetrics coordinatorMetrics;
    private final AtomicBoolean isRunning = new AtomicBoolean(true);
    private volatile MetadataImage metadataImage = MetadataImage.EMPTY;

    private CoordinatorRuntime(String logPrefix, LogContext logContext, CoordinatorEventProcessor processor, PartitionWriter<U> partitionWriter, CoordinatorLoader<U> loader, CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier, Time time, Timer timer, Duration defaultWriteTimeout, CoordinatorRuntimeMetrics runtimeMetrics, CoordinatorMetrics coordinatorMetrics) {
        this.logPrefix = logPrefix;
        this.logContext = logContext;
        this.log = logContext.logger(CoordinatorRuntime.class);
        this.time = time;
        this.timer = timer;
        this.defaultWriteTimeout = defaultWriteTimeout;
        this.coordinators = new ConcurrentHashMap();
        this.processor = processor;
        this.partitionWriter = partitionWriter;
        this.highWatermarklistener = new HighWatermarkListener();
        this.loader = loader;
        this.coordinatorShardBuilderSupplier = coordinatorShardBuilderSupplier;
        this.runtimeMetrics = runtimeMetrics;
        this.coordinatorMetrics = coordinatorMetrics;
    }

    private void throwIfNotRunning() {
        if (!this.isRunning.get()) {
            throw Errors.NOT_COORDINATOR.exception();
        }
    }

    private void enqueue(CoordinatorEvent event) {
        try {
            this.processor.enqueue(event);
        }
        catch (RejectedExecutionException ex) {
            throw new NotCoordinatorException("Can't accept an event because the processor is closed", (Throwable)ex);
        }
    }

    CoordinatorContext maybeCreateContext(TopicPartition tp) {
        return this.coordinators.computeIfAbsent(tp, x$0 -> new CoordinatorContext((TopicPartition)x$0));
    }

    CoordinatorContext contextOrThrow(TopicPartition tp) throws NotCoordinatorException {
        CoordinatorContext context = this.coordinators.get(tp);
        if (context == null) {
            throw Errors.NOT_COORDINATOR.exception();
        }
        return context;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void withActiveContextOrThrow(TopicPartition tp, Consumer<CoordinatorContext> func) throws NotCoordinatorException, CoordinatorLoadInProgressException {
        block5: {
            CoordinatorContext context = this.contextOrThrow(tp);
            try {
                context.lock.lock();
                if (context.state == CoordinatorState.ACTIVE) {
                    func.accept(context);
                    break block5;
                }
                if (context.state == CoordinatorState.LOADING) {
                    throw Errors.COORDINATOR_LOAD_IN_PROGRESS.exception();
                }
                throw Errors.NOT_COORDINATOR.exception();
            }
            finally {
                context.lock.unlock();
            }
        }
    }

    public <T> CompletableFuture<T> scheduleWriteOperation(String name, TopicPartition tp, Duration timeout, CoordinatorWriteOperation<S, T, U> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of write operation {}.", (Object)name);
        CoordinatorWriteEvent<T> event = new CoordinatorWriteEvent<T>(name, tp, timeout, op);
        this.enqueue(event);
        return event.future;
    }

    public <T> CompletableFuture<T> scheduleTransactionalWriteOperation(String name, TopicPartition tp, String transactionalId, long producerId, short producerEpoch, Duration timeout, CoordinatorWriteOperation<S, T, U> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of transactional write operation {}.", (Object)name);
        return this.partitionWriter.maybeStartTransactionVerification(tp, transactionalId, producerId, producerEpoch).thenCompose(verificationGuard -> {
            CoordinatorWriteEvent event = new CoordinatorWriteEvent(name, tp, transactionalId, producerId, producerEpoch, (VerificationGuard)verificationGuard, timeout, op);
            this.enqueue(event);
            return event.future;
        });
    }

    public CompletableFuture<Void> scheduleTransactionCompletion(String name, TopicPartition tp, long producerId, short producerEpoch, int coordinatorEpoch, TransactionResult result, Duration timeout) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of transaction completion for {} with producer id={}, producer epoch={}, coordinator epoch={} and transaction result={}.", new Object[]{tp, producerId, producerEpoch, coordinatorEpoch, result});
        CoordinatorCompleteTransactionEvent event = new CoordinatorCompleteTransactionEvent(name, tp, producerId, producerEpoch, coordinatorEpoch, result, timeout);
        this.enqueue(event);
        return event.future;
    }

    public <T> CompletableFuture<T> scheduleReadOperation(String name, TopicPartition tp, CoordinatorReadOperation<S, T> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of read operation {}.", (Object)name);
        CoordinatorReadEvent<T> event = new CoordinatorReadEvent<T>(name, tp, op);
        this.enqueue(event);
        return event.future;
    }

    private void scheduleInternalOperation(String name, TopicPartition tp, Runnable op) {
        this.log.debug("Scheduled execution of internal operation {}.", (Object)name);
        this.enqueue(new CoordinatorInternalEvent(name, tp, op));
    }

    public Set<TopicPartition> partitions() {
        this.throwIfNotRunning();
        return new HashSet<TopicPartition>(this.coordinators.keySet());
    }

    public void scheduleLoadOperation(TopicPartition tp, int partitionEpoch) {
        this.throwIfNotRunning();
        this.log.info("Scheduling loading of metadata from {} with epoch {}", (Object)tp, (Object)partitionEpoch);
        this.maybeCreateContext(tp);
        this.scheduleInternalOperation("Load(tp=" + tp + ", epoch=" + partitionEpoch + ")", tp, () -> {
            CoordinatorContext context = this.maybeCreateContext(tp);
            context.lock.lock();
            try {
                if (context.epoch < partitionEpoch) {
                    context.epoch = partitionEpoch;
                    switch (context.state) {
                        case FAILED: 
                        case INITIAL: {
                            context.transitionTo(CoordinatorState.LOADING);
                            this.loader.load(tp, context.coordinator).whenComplete((summary, exception) -> this.scheduleInternalOperation("CompleteLoad(tp=" + tp + ", epoch=" + partitionEpoch + ")", tp, () -> {
                                CoordinatorContext ctx = this.coordinators.get(tp);
                                if (ctx != null) {
                                    if (ctx.state != CoordinatorState.LOADING) {
                                        this.log.info("Ignored load completion from {} because context is in {} state.", (Object)ctx.tp, (Object)ctx.state);
                                        return;
                                    }
                                    try {
                                        if (exception != null) {
                                            throw exception;
                                        }
                                        ctx.transitionTo(CoordinatorState.ACTIVE);
                                        if (summary != null) {
                                            this.runtimeMetrics.recordPartitionLoadSensor(summary.startTimeMs(), summary.endTimeMs());
                                            this.log.info("Finished loading of metadata from {} with epoch {} in {}ms where {}ms was spent in the scheduler. Loaded {} records which total to {} bytes.", new Object[]{tp, partitionEpoch, summary.endTimeMs() - summary.startTimeMs(), summary.schedulerQueueTimeMs(), summary.numRecords(), summary.numBytes()});
                                        }
                                    }
                                    catch (Throwable ex) {
                                        this.log.error("Failed to load metadata from {} with epoch {} due to {}.", new Object[]{tp, partitionEpoch, ex.toString()});
                                        ctx.transitionTo(CoordinatorState.FAILED);
                                    }
                                } else {
                                    this.log.debug("Failed to complete the loading of metadata for {} in epoch {} since the coordinator does not exist.", (Object)tp, (Object)partitionEpoch);
                                }
                            }));
                            break;
                        }
                        case LOADING: {
                            this.log.info("The coordinator {} is already loading metadata.", (Object)tp);
                            break;
                        }
                        case ACTIVE: {
                            this.log.info("The coordinator {} is already active.", (Object)tp);
                            break;
                        }
                        default: {
                            this.log.error("Cannot load coordinator {} in state {}.", (Object)tp, (Object)context.state);
                            break;
                        }
                    }
                } else {
                    this.log.info("Ignored loading metadata from {} since current epoch {} is larger than or equals to {}.", new Object[]{context.tp, context.epoch, partitionEpoch});
                }
            }
            finally {
                context.lock.unlock();
            }
        });
    }

    public void scheduleUnloadOperation(TopicPartition tp, OptionalInt partitionEpoch) {
        this.throwIfNotRunning();
        this.log.info("Scheduling unloading of metadata for {} with epoch {}", (Object)tp, (Object)partitionEpoch);
        this.scheduleInternalOperation("UnloadCoordinator(tp=" + tp + ", epoch=" + partitionEpoch + ")", tp, () -> {
            CoordinatorContext context = this.coordinators.get(tp);
            if (context != null) {
                context.lock.lock();
                try {
                    if (!partitionEpoch.isPresent() || context.epoch < partitionEpoch.getAsInt()) {
                        this.log.info("Started unloading metadata for {} with epoch {}.", (Object)tp, (Object)partitionEpoch);
                        context.transitionTo(CoordinatorState.CLOSED);
                        this.coordinators.remove(tp, context);
                        this.log.info("Finished unloading metadata for {} with epoch {}.", (Object)tp, (Object)partitionEpoch);
                    }
                    this.log.info("Ignored unloading metadata for {} in epoch {} since current epoch is {}.", new Object[]{tp, partitionEpoch, context.epoch});
                }
                finally {
                    context.lock.unlock();
                }
            } else {
                this.log.info("Ignored unloading metadata for {} in epoch {} since metadata was never loaded.", (Object)tp, (Object)partitionEpoch);
            }
        });
    }

    public void onNewMetadataImage(MetadataImage newImage, MetadataDelta delta) {
        this.throwIfNotRunning();
        this.log.debug("Scheduling applying of a new metadata image with offset {}.", (Object)newImage.offset());
        this.metadataImage = newImage;
        ((ConcurrentHashMap.KeySetView)this.coordinators.keySet()).forEach(tp -> this.scheduleInternalOperation("UpdateImage(tp=" + tp + ", offset=" + newImage.offset() + ")", (TopicPartition)tp, () -> {
            CoordinatorContext context = this.coordinators.get(tp);
            if (context != null) {
                context.lock.lock();
                try {
                    if (context.state == CoordinatorState.ACTIVE) {
                        this.log.debug("Applying new metadata image with offset {} to {}.", (Object)newImage.offset(), tp);
                        context.coordinator.onNewMetadataImage(newImage, delta);
                    }
                    this.log.debug("Ignored new metadata image with offset {} for {} because the coordinator is not active.", (Object)newImage.offset(), tp);
                }
                finally {
                    context.lock.unlock();
                }
            } else {
                this.log.debug("Ignored new metadata image with offset {} for {} because the coordinator does not exist.", (Object)newImage.offset(), tp);
            }
        }));
    }

    @Override
    public void close() throws Exception {
        if (!this.isRunning.compareAndSet(true, false)) {
            this.log.warn("Coordinator runtime is already shutting down.");
            return;
        }
        this.log.info("Closing coordinator runtime.");
        Utils.closeQuietly(this.loader, (String)"loader");
        Utils.closeQuietly((AutoCloseable)this.timer, (String)"timer");
        Utils.closeQuietly((AutoCloseable)this.processor, (String)"event processor");
        this.coordinators.forEach((tp, context) -> {
            context.lock.lock();
            try {
                ((CoordinatorContext)context).transitionTo(CoordinatorState.CLOSED);
            }
            finally {
                context.lock.unlock();
            }
        });
        this.coordinators.clear();
        Utils.closeQuietly((AutoCloseable)this.runtimeMetrics, (String)"runtime metrics");
        this.log.info("Coordinator runtime closed.");
    }

    class HighWatermarkListener
    implements PartitionWriter.Listener {
        HighWatermarkListener() {
        }

        @Override
        public void onHighWatermarkUpdated(TopicPartition tp, long offset) {
            CoordinatorRuntime.this.log.debug("High watermark of {} incremented to {}.", (Object)tp, (Object)offset);
            CoordinatorRuntime.this.scheduleInternalOperation("HighWatermarkUpdated(tp=" + tp + ", offset=" + offset + ")", tp, () -> {
                CoordinatorContext context = (CoordinatorContext)CoordinatorRuntime.this.coordinators.get(tp);
                if (context != null) {
                    context.lock.lock();
                    try {
                        if (context.state == CoordinatorState.ACTIVE) {
                            CoordinatorRuntime.this.log.debug("Updating high watermark of {} to {}.", (Object)tp, (Object)offset);
                            context.coordinator.updateLastCommittedOffset(offset);
                            context.deferredEventQueue.completeUpTo(offset);
                            CoordinatorRuntime.this.coordinatorMetrics.onUpdateLastCommittedOffset(tp, offset);
                        }
                        CoordinatorRuntime.this.log.debug("Ignored high watermark updated for {} to {} because the coordinator is not active.", (Object)tp, (Object)offset);
                    }
                    finally {
                        context.lock.unlock();
                    }
                } else {
                    CoordinatorRuntime.this.log.debug("Ignored high watermark updated for {} to {} because the coordinator does not exist.", (Object)tp, (Object)offset);
                }
            });
        }
    }

    class CoordinatorInternalEvent
    implements CoordinatorEvent {
        final TopicPartition tp;
        final String name;
        final Runnable op;
        private final long createdTimeMs;

        CoordinatorInternalEvent(String name, TopicPartition tp, Runnable op) {
            this.tp = tp;
            this.name = name;
            this.op = op;
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                this.op.run();
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            if (exception != null) {
                CoordinatorRuntime.this.log.error("Execution of {} failed due to {}.", new Object[]{this.name, exception.getMessage(), exception});
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "InternalEvent(name=" + this.name + ")";
        }
    }

    class CoordinatorCompleteTransactionEvent
    implements CoordinatorEvent,
    DeferredEvent {
        final TopicPartition tp;
        final String name;
        final long producerId;
        final short producerEpoch;
        final int coordinatorEpoch;
        final TransactionResult result;
        final Duration writeTimeout;
        final CompletableFuture<Void> future;
        private final long createdTimeMs;

        CoordinatorCompleteTransactionEvent(String name, TopicPartition tp, long producerId, short producerEpoch, int coordinatorEpoch, TransactionResult result, Duration writeTimeout) {
            this.name = name;
            this.tp = tp;
            this.producerId = producerId;
            this.producerEpoch = producerEpoch;
            this.coordinatorEpoch = coordinatorEpoch;
            this.result = result;
            this.writeTimeout = writeTimeout;
            this.future = new CompletableFuture();
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                CoordinatorRuntime.this.withActiveContextOrThrow(this.tp, context -> {
                    long prevLastWrittenOffset = context.coordinator.lastWrittenOffset();
                    try {
                        context.coordinator.replayEndTransactionMarker(this.producerId, this.producerEpoch, this.result);
                        long offset = CoordinatorRuntime.this.partitionWriter.appendEndTransactionMarker(this.tp, this.producerId, this.producerEpoch, this.coordinatorEpoch, this.result);
                        context.coordinator.updateLastWrittenOffset(offset);
                        if (!this.future.isDone()) {
                            context.deferredEventQueue.add(offset, (DeferredEvent)this);
                            CoordinatorRuntime.this.timer.add(new TimerTask(this.writeTimeout.toMillis()){

                                public void run() {
                                    if (!CoordinatorCompleteTransactionEvent.this.future.isDone()) {
                                        CoordinatorRuntime.this.scheduleInternalOperation("WriteTimeout(name=" + CoordinatorCompleteTransactionEvent.this.name + ", tp=" + CoordinatorCompleteTransactionEvent.this.tp + ")", CoordinatorCompleteTransactionEvent.this.tp, () -> CoordinatorCompleteTransactionEvent.this.complete((Throwable)new TimeoutException("CoordinatorCompleteTransactionEvent " + CoordinatorCompleteTransactionEvent.this.name + " timed out after " + CoordinatorCompleteTransactionEvent.this.writeTimeout.toMillis() + "ms")));
                                    }
                                }
                            });
                        } else {
                            this.complete(null);
                        }
                    }
                    catch (Throwable t) {
                        context.coordinator.revertLastWrittenOffset(prevLastWrittenOffset);
                        this.complete(t);
                    }
                });
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            if (exception == null) {
                this.future.complete(null);
            } else {
                this.future.completeExceptionally(exception);
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "CoordinatorCompleteTransactionEvent(name=" + this.name + ")";
        }
    }

    class CoordinatorReadEvent<T>
    implements CoordinatorEvent {
        final TopicPartition tp;
        final String name;
        final CoordinatorReadOperation<S, T> op;
        final CompletableFuture<T> future;
        T response;
        private final long createdTimeMs;

        CoordinatorReadEvent(String name, TopicPartition tp, CoordinatorReadOperation<S, T> op) {
            this.tp = tp;
            this.name = name;
            this.op = op;
            this.future = new CompletableFuture();
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                CoordinatorRuntime.this.withActiveContextOrThrow(this.tp, context -> {
                    this.response = this.op.generateResponse(context.coordinator.coordinator(), context.coordinator.lastCommittedOffset());
                    this.complete(null);
                });
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            if (exception == null) {
                this.future.complete(this.response);
            } else {
                this.future.completeExceptionally(exception);
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "CoordinatorReadEvent(name=" + this.name + ")";
        }
    }

    public static interface CoordinatorReadOperation<S, T> {
        public T generateResponse(S var1, long var2) throws KafkaException;
    }

    class CoordinatorWriteEvent<T>
    implements CoordinatorEvent,
    DeferredEvent {
        final TopicPartition tp;
        final String name;
        final String transactionalId;
        final long producerId;
        final short producerEpoch;
        final VerificationGuard verificationGuard;
        final CoordinatorWriteOperation<S, T, U> op;
        final CompletableFuture<T> future;
        final Duration writeTimeout;
        CoordinatorResult<T, U> result;
        private final long createdTimeMs;

        CoordinatorWriteEvent(String name, TopicPartition tp, Duration writeTimeout, CoordinatorWriteOperation<S, T, U> op) {
            this(name, tp, null, -1L, -1, VerificationGuard.SENTINEL, writeTimeout, op);
        }

        CoordinatorWriteEvent(String name, TopicPartition tp, String transactionalId, long producerId, short producerEpoch, VerificationGuard verificationGuard, Duration writeTimeout, CoordinatorWriteOperation<S, T, U> op) {
            this.tp = tp;
            this.name = name;
            this.op = op;
            this.transactionalId = transactionalId;
            this.producerId = producerId;
            this.producerEpoch = producerEpoch;
            this.verificationGuard = verificationGuard;
            this.future = new CompletableFuture();
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
            this.writeTimeout = writeTimeout;
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                CoordinatorRuntime.this.withActiveContextOrThrow(this.tp, context -> {
                    long prevLastWrittenOffset = context.coordinator.lastWrittenOffset();
                    this.result = this.op.generateRecordsAndResult(context.coordinator.coordinator());
                    if (this.result.records().isEmpty()) {
                        OptionalLong pendingOffset = context.deferredEventQueue.highestPendingOffset();
                        if (pendingOffset.isPresent()) {
                            context.deferredEventQueue.add(pendingOffset.getAsLong(), (DeferredEvent)this);
                        } else {
                            this.complete(null);
                        }
                    } else {
                        try {
                            if (this.result.replayRecords()) {
                                for (int i = 0; i < this.result.records().size(); ++i) {
                                    context.coordinator.replay(prevLastWrittenOffset + (long)i, this.producerId, this.producerEpoch, this.result.records().get(i));
                                }
                            }
                            long offset = CoordinatorRuntime.this.partitionWriter.append(this.tp, this.producerId, this.producerEpoch, this.verificationGuard, this.result.records());
                            context.coordinator.updateLastWrittenOffset(offset);
                            if (!this.future.isDone()) {
                                context.deferredEventQueue.add(offset, (DeferredEvent)this);
                                CoordinatorRuntime.this.timer.add(new TimerTask(this.writeTimeout.toMillis()){

                                    public void run() {
                                        if (!CoordinatorWriteEvent.this.future.isDone()) {
                                            CoordinatorRuntime.this.scheduleInternalOperation("WriteTimeout(name=" + CoordinatorWriteEvent.this.name + ", tp=" + CoordinatorWriteEvent.this.tp + ")", CoordinatorWriteEvent.this.tp, () -> CoordinatorWriteEvent.this.complete((Throwable)new TimeoutException("CoordinatorWriteEvent " + CoordinatorWriteEvent.this.name + " timed out after " + CoordinatorWriteEvent.this.writeTimeout.toMillis() + "ms")));
                                        }
                                    }
                                });
                            } else {
                                this.complete(null);
                            }
                        }
                        catch (Throwable t) {
                            context.coordinator.revertLastWrittenOffset(prevLastWrittenOffset);
                            this.complete(t);
                        }
                    }
                });
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            CompletableFuture<Void> appendFuture;
            CompletableFuture<Void> completableFuture = appendFuture = this.result != null ? this.result.appendFuture() : null;
            if (exception == null) {
                if (appendFuture != null) {
                    this.result.appendFuture().complete(null);
                }
                this.future.complete(this.result.response());
            } else {
                if (appendFuture != null) {
                    this.result.appendFuture().completeExceptionally(exception);
                }
                this.future.completeExceptionally(exception);
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "CoordinatorWriteEvent(name=" + this.name + ")";
        }
    }

    public static interface CoordinatorWriteOperation<S, T, U> {
        public CoordinatorResult<T, U> generateRecordsAndResult(S var1) throws KafkaException;
    }

    class CoordinatorContext {
        final ReentrantLock lock = new ReentrantLock();
        final TopicPartition tp;
        final LogContext logContext;
        final DeferredEventQueue deferredEventQueue;
        final EventBasedCoordinatorTimer timer;
        CoordinatorState state;
        int epoch;
        SnapshottableCoordinator<S, U> coordinator;

        private CoordinatorContext(TopicPartition tp) {
            this.tp = tp;
            this.logContext = new LogContext(String.format("[%s topic=%s partition=%d] ", CoordinatorRuntime.this.logPrefix, tp.topic(), tp.partition()));
            this.state = CoordinatorState.INITIAL;
            this.epoch = -1;
            this.deferredEventQueue = new DeferredEventQueue(this.logContext);
            this.timer = new EventBasedCoordinatorTimer(tp, this.logContext);
        }

        private void transitionTo(CoordinatorState newState) {
            if (!newState.canTransitionFrom(this.state)) {
                throw new IllegalStateException("Cannot transition from " + (Object)((Object)this.state) + " to " + (Object)((Object)newState));
            }
            CoordinatorState oldState = this.state;
            CoordinatorRuntime.this.log.debug("Transition from {} to {}.", (Object)this.state, (Object)newState);
            switch (newState) {
                case LOADING: {
                    this.state = CoordinatorState.LOADING;
                    SnapshotRegistry snapshotRegistry = new SnapshotRegistry(this.logContext);
                    this.coordinator = new SnapshottableCoordinator(this.logContext, snapshotRegistry, CoordinatorRuntime.this.coordinatorShardBuilderSupplier.get().withLogContext(this.logContext).withSnapshotRegistry(snapshotRegistry).withTime(CoordinatorRuntime.this.time).withTimer(this.timer).withCoordinatorMetrics(CoordinatorRuntime.this.coordinatorMetrics).withTopicPartition(this.tp).build(), this.tp);
                    break;
                }
                case ACTIVE: {
                    this.state = CoordinatorState.ACTIVE;
                    CoordinatorRuntime.this.partitionWriter.registerListener(this.tp, CoordinatorRuntime.this.highWatermarklistener);
                    this.coordinator.onLoaded(CoordinatorRuntime.this.metadataImage);
                    break;
                }
                case FAILED: {
                    this.state = CoordinatorState.FAILED;
                    this.unload();
                    break;
                }
                case CLOSED: {
                    this.state = CoordinatorState.CLOSED;
                    this.unload();
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Transitioning to " + (Object)((Object)newState) + " is not supported.");
                }
            }
            CoordinatorRuntime.this.runtimeMetrics.recordPartitionStateChange(oldState, this.state);
        }

        private void unload() {
            CoordinatorRuntime.this.partitionWriter.deregisterListener(this.tp, CoordinatorRuntime.this.highWatermarklistener);
            this.timer.cancelAll();
            this.deferredEventQueue.failAll((Exception)((Object)Errors.NOT_COORDINATOR.exception()));
            if (this.coordinator != null) {
                this.coordinator.onUnloaded();
            }
            this.coordinator = null;
        }
    }

    class EventBasedCoordinatorTimer
    implements CoordinatorTimer<Void, U> {
        final Logger log;
        final TopicPartition tp;
        final Map<String, TimerTask> tasks = new HashMap<String, TimerTask>();

        EventBasedCoordinatorTimer(TopicPartition tp, LogContext logContext) {
            this.tp = tp;
            this.log = logContext.logger(EventBasedCoordinatorTimer.class);
        }

        @Override
        public void schedule(String key, long delay, TimeUnit unit, boolean retry, CoordinatorTimer.TimeoutOperation<Void, U> operation) {
            this.schedule(key, delay, unit, retry, 500L, operation);
        }

        @Override
        public void schedule(final String key, long delay, TimeUnit unit, final boolean retry, final long retryBackoff, final CoordinatorTimer.TimeoutOperation<Void, U> operation) {
            TimerTask task = new TimerTask(unit.toMillis(delay)){

                public void run() {
                    String eventName = "Timeout(tp=" + EventBasedCoordinatorTimer.this.tp + ", key=" + key + ")";
                    CoordinatorWriteEvent event = new CoordinatorWriteEvent(eventName, EventBasedCoordinatorTimer.this.tp, CoordinatorRuntime.this.defaultWriteTimeout, coordinator -> {
                        EventBasedCoordinatorTimer.this.log.debug("Executing write event {} for timer {}.", (Object)eventName, (Object)key);
                        if (!EventBasedCoordinatorTimer.this.tasks.remove(key, (Object)this)) {
                            throw new RejectedExecutionException("Timer " + key + " was overridden or cancelled");
                        }
                        return operation.generateRecords();
                    });
                    event.future.exceptionally(ex -> {
                        if (ex instanceof RejectedExecutionException) {
                            EventBasedCoordinatorTimer.this.log.debug("The write event {} for the timer {} was not executed because it was cancelled or overridden.", (Object)event.name, (Object)key);
                            return null;
                        }
                        if (ex instanceof NotCoordinatorException || ex instanceof CoordinatorLoadInProgressException) {
                            EventBasedCoordinatorTimer.this.log.debug("The write event {} for the timer {} failed due to {}. Ignoring it because the coordinator is not active.", new Object[]{event.name, key, ex.getMessage()});
                            return null;
                        }
                        if (retry) {
                            EventBasedCoordinatorTimer.this.log.info("The write event {} for the timer {} failed due to {}. Rescheduling it. ", new Object[]{event.name, key, ex.getMessage()});
                            EventBasedCoordinatorTimer.this.schedule(key, retryBackoff, TimeUnit.MILLISECONDS, true, retryBackoff, operation);
                        } else {
                            EventBasedCoordinatorTimer.this.log.error("The write event {} for the timer {} failed due to {}. Ignoring it. ", new Object[]{event.name, key, ex.getMessage()});
                        }
                        return null;
                    });
                    EventBasedCoordinatorTimer.this.log.debug("Scheduling write event {} for timer {}.", (Object)event.name, (Object)key);
                    try {
                        CoordinatorRuntime.this.enqueue(event);
                    }
                    catch (NotCoordinatorException ex2) {
                        EventBasedCoordinatorTimer.this.log.info("Failed to enqueue write event {} for timer {} because the runtime is closed. Ignoring it.", (Object)event.name, (Object)key);
                    }
                }
            };
            this.log.debug("Registering timer {} with delay of {}ms.", (Object)key, (Object)unit.toMillis(delay));
            TimerTask prevTask = this.tasks.put(key, task);
            if (prevTask != null) {
                prevTask.cancel();
            }
            CoordinatorRuntime.this.timer.add(task);
        }

        @Override
        public void cancel(String key) {
            TimerTask prevTask = this.tasks.remove(key);
            if (prevTask != null) {
                prevTask.cancel();
            }
        }

        public void cancelAll() {
            Iterator<Map.Entry<String, TimerTask>> iterator = this.tasks.entrySet().iterator();
            while (iterator.hasNext()) {
                iterator.next().getValue().cancel();
                iterator.remove();
            }
        }

        public int size() {
            return this.tasks.size();
        }
    }

    public static enum CoordinatorState {
        INITIAL{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return false;
            }
        }
        ,
        LOADING{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return state == INITIAL || state == FAILED;
            }
        }
        ,
        ACTIVE{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return state == ACTIVE || state == LOADING;
            }
        }
        ,
        CLOSED{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return true;
            }
        }
        ,
        FAILED{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return state == LOADING;
            }
        };


        abstract boolean canTransitionFrom(CoordinatorState var1);
    }

    public static class Builder<S extends CoordinatorShard<U>, U> {
        private String logPrefix;
        private LogContext logContext;
        private CoordinatorEventProcessor eventProcessor;
        private PartitionWriter<U> partitionWriter;
        private CoordinatorLoader<U> loader;
        private CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier;
        private Time time = Time.SYSTEM;
        private Timer timer;
        private Duration defaultWriteTimeout;
        private CoordinatorRuntimeMetrics runtimeMetrics;
        private CoordinatorMetrics coordinatorMetrics;

        public Builder<S, U> withLogPrefix(String logPrefix) {
            this.logPrefix = logPrefix;
            return this;
        }

        public Builder<S, U> withLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        public Builder<S, U> withEventProcessor(CoordinatorEventProcessor eventProcessor) {
            this.eventProcessor = eventProcessor;
            return this;
        }

        public Builder<S, U> withPartitionWriter(PartitionWriter<U> partitionWriter) {
            this.partitionWriter = partitionWriter;
            return this;
        }

        public Builder<S, U> withLoader(CoordinatorLoader<U> loader) {
            this.loader = loader;
            return this;
        }

        public Builder<S, U> withCoordinatorShardBuilderSupplier(CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier) {
            this.coordinatorShardBuilderSupplier = coordinatorShardBuilderSupplier;
            return this;
        }

        public Builder<S, U> withTime(Time time) {
            this.time = time;
            return this;
        }

        public Builder<S, U> withTimer(Timer timer) {
            this.timer = timer;
            return this;
        }

        public Builder<S, U> withDefaultWriteTimeOut(Duration defaultWriteTimeout) {
            this.defaultWriteTimeout = defaultWriteTimeout;
            return this;
        }

        public Builder<S, U> withCoordinatorRuntimeMetrics(CoordinatorRuntimeMetrics runtimeMetrics) {
            this.runtimeMetrics = runtimeMetrics;
            return this;
        }

        public Builder<S, U> withCoordinatorMetrics(CoordinatorMetrics coordinatorMetrics) {
            this.coordinatorMetrics = coordinatorMetrics;
            return this;
        }

        public CoordinatorRuntime<S, U> build() {
            if (this.logPrefix == null) {
                this.logPrefix = "";
            }
            if (this.logContext == null) {
                this.logContext = new LogContext(this.logPrefix);
            }
            if (this.eventProcessor == null) {
                throw new IllegalArgumentException("Event processor must be set.");
            }
            if (this.partitionWriter == null) {
                throw new IllegalArgumentException("Partition write must be set.");
            }
            if (this.loader == null) {
                throw new IllegalArgumentException("Loader must be set.");
            }
            if (this.coordinatorShardBuilderSupplier == null) {
                throw new IllegalArgumentException("State machine supplier must be set.");
            }
            if (this.time == null) {
                throw new IllegalArgumentException("Time must be set.");
            }
            if (this.timer == null) {
                throw new IllegalArgumentException("Timer must be set.");
            }
            if (this.runtimeMetrics == null) {
                throw new IllegalArgumentException("CoordinatorRuntimeMetrics must be set.");
            }
            if (this.coordinatorMetrics == null) {
                throw new IllegalArgumentException("CoordinatorMetrics must be set.");
            }
            return new CoordinatorRuntime(this.logPrefix, this.logContext, this.eventProcessor, this.partitionWriter, this.loader, this.coordinatorShardBuilderSupplier, this.time, this.timer, this.defaultWriteTimeout, this.runtimeMetrics, this.coordinatorMetrics);
        }
    }
}

