/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.analyzer.goals;

import com.linkedin.kafka.cruisecontrol.analyzer.ActionAcceptance;
import com.linkedin.kafka.cruisecontrol.analyzer.ActionType;
import com.linkedin.kafka.cruisecontrol.analyzer.BalancingConstraint;
import com.linkedin.kafka.cruisecontrol.analyzer.OptimizationOptions;
import com.linkedin.kafka.cruisecontrol.analyzer.PartitionBalancingAction;
import com.linkedin.kafka.cruisecontrol.analyzer.ReplicaBalancingAction;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.AbstractGoal;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.EntityCombinator;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.Goal;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.GoalUtils;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.metrics.OptimizationMetrics;
import com.linkedin.kafka.cruisecontrol.exception.OptimizationFailureException;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.Cell;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.monitor.ModelCompletenessRequirements;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RackAwareGoal
extends AbstractGoal {
    private static final Logger LOG = LoggerFactory.getLogger(RackAwareGoal.class);
    private Set<String> racks;

    public RackAwareGoal() {
    }

    RackAwareGoal(BalancingConstraint constraint) {
        this.balancingConstraint = constraint;
    }

    @Override
    public ActionAcceptance replicaActionAcceptance(ReplicaBalancingAction action, ClusterModel clusterModel) {
        if (clusterModel.getTopicPlacement(action.topic()).isPresent()) {
            return ActionAcceptance.ACCEPT;
        }
        switch (action.balancingAction()) {
            case LEADERSHIP_MOVEMENT: {
                return ActionAcceptance.ACCEPT;
            }
            case INTER_BROKER_REPLICA_MOVEMENT: 
            case INTER_BROKER_REPLICA_SWAP: {
                if (action.balancingAction() == ActionType.INTER_BROKER_REPLICA_SWAP && action.destinationTopicPartition().equals((Object)action.topicPartition())) {
                    return ActionAcceptance.ACCEPT;
                }
                if (this.isReplicaMoveViolateRackAwareness(clusterModel, clusterModel.broker(action.sourceBrokerId()).replica(action.topicPartition()), clusterModel.broker(action.destinationBrokerId()))) {
                    return ActionAcceptance.BROKER_REJECT;
                }
                if (action.balancingAction() == ActionType.INTER_BROKER_REPLICA_SWAP && this.isReplicaMoveViolateRackAwareness(clusterModel, clusterModel.broker(action.destinationBrokerId()).replica(action.destinationTopicPartition()), clusterModel.broker(action.sourceBrokerId()))) {
                    return ActionAcceptance.REPLICA_REJECT;
                }
                return ActionAcceptance.ACCEPT;
            }
        }
        throw new IllegalArgumentException("Unsupported balancing action " + (Object)((Object)action.balancingAction()) + " is provided.");
    }

    @Override
    public ActionAcceptance partitionActionAcceptance(PartitionBalancingAction action, ClusterModel clusterModel) {
        TopicPartition topicPartition = action.topicPartition();
        Map<String, Integer> currentReplicaCounts = this.computeReplicaCounts(topicPartition, clusterModel);
        action.replicaMoves().forEach((replica, broker) -> {
            currentReplicaCounts.merge(replica.broker().rack().id(), -1, Integer::sum);
            currentReplicaCounts.merge(broker.rack().id(), 1, Integer::sum);
        });
        int minReplicaCount = Integer.MAX_VALUE;
        int maxReplicaCount = 0;
        for (Integer replicaCount : currentReplicaCounts.values()) {
            minReplicaCount = Integer.min(replicaCount, minReplicaCount);
            maxReplicaCount = Integer.max(replicaCount, maxReplicaCount);
        }
        return maxReplicaCount - minReplicaCount > 1 ? ActionAcceptance.BROKER_REJECT : ActionAcceptance.ACCEPT;
    }

    private boolean isReplicaMoveViolateRackAwareness(ClusterModel clusterModel, Replica sourceReplica, Broker destinationBroker) {
        Broker sourceBroker = sourceReplica.broker();
        if (sourceBroker.rack().id().equals(destinationBroker.rack().id())) {
            return false;
        }
        Map<String, Integer> replicaCountsPerRack = this.computeReplicaCounts(sourceReplica, clusterModel);
        return replicaCountsPerRack.get(destinationBroker.rack().id()) >= replicaCountsPerRack.get(sourceBroker.rack().id());
    }

    @Override
    public ModelCompletenessRequirements clusterModelCompletenessRequirements() {
        return new ModelCompletenessRequirements(1, 0.0, true);
    }

    @Override
    public String name() {
        return RackAwareGoal.class.getSimpleName();
    }

    @Override
    public boolean isHardGoal() {
        return true;
    }

    @Override
    public boolean replicaActionSelfSatisfied(ClusterModel clusterModel, ReplicaBalancingAction action) {
        return this.replicaActionAcceptance(action, clusterModel) == ActionAcceptance.ACCEPT;
    }

    @Override
    public boolean partitionActionSelfSatisfied(ClusterModel clusterModel, PartitionBalancingAction action) {
        return true;
    }

    @Override
    protected SortedSet<Broker> brokersToBalance(ClusterModel clusterModel) {
        return clusterModel.eligibleSourceBrokers();
    }

    @Override
    protected void initGoalState(ClusterModel clusterModel, OptimizationOptions optimizationOptions, Optional<OptimizationMetrics> optimizationMetricsOpt) {
        this.racks = clusterModel.aliveRackIds();
        Map<Integer, String> rackByBroker = clusterModel.allBrokers().stream().collect(Collectors.toMap(Broker::id, v -> v.rack() == null ? "<null>" : v.rack().id()));
        LOG.info("Initializing RackAwareGoal with racks {} (by broker {})", this.racks, rackByBroker);
    }

    @Override
    protected void updateGoalState(ClusterModel clusterModel, Set<String> excludedTopics) throws OptimizationFailureException {
        this.ensureRackAware(clusterModel, excludedTopics);
        GoalUtils.ensureNoOfflineReplicas(clusterModel, this.name());
        GoalUtils.ensureReplicasMoveOffBrokersWithBadDisks(clusterModel, this.name());
        this.finish();
    }

    @Override
    public void finish() {
        this.finished = true;
    }

    @Override
    protected void rebalanceForBroker(Broker broker, ClusterModel clusterModel, Set<Goal> optimizedGoals, OptimizationOptions optimizationOptions) throws OptimizationFailureException {
        LOG.debug("balancing broker {}, optimized goals = {}", (Object)broker, optimizedGoals);
        Set<String> excludedTopics = optimizationOptions.excludedTopics();
        TreeSet<Replica> replicas = new TreeSet<Replica>(broker.replicas());
        for (Replica replica : replicas) {
            if (RackAwareGoal.shouldExclude(replica, excludedTopics) || clusterModel.getTopicPlacement(replica.topicPartition().topic()).isPresent()) continue;
            Map<String, Integer> replicaCountsPerRack = this.computeReplicaCounts(replica, clusterModel);
            if (broker.isAlive() && !broker.currentOfflineReplicas().contains(replica) && this.satisfiedRackAwareness(replica, replicaCountsPerRack)) continue;
            LOG.debug("Broker id: {}, broker is alive: {}, replica offline: {}", new Object[]{broker.id(), broker.isAlive(), broker.currentOfflineReplicas().contains(replica)});
            LOG.debug("Rack assignment of replica {}: {}", (Object)replica, clusterModel.partition(replica.topicPartition()).partitionBrokers());
            if (this.maybeApplyBalancingAction(clusterModel, replica, this.rackAwareEligibleBrokers(replica, clusterModel, replicaCountsPerRack), ActionType.INTER_BROKER_REPLICA_MOVEMENT, optimizedGoals, optimizationOptions, Optional.empty()) != null || !clusterModel.skipCellBalancing() && this.movePartition(clusterModel, optimizedGoals, replica)) continue;
            throw new OptimizationFailureException(String.format("[%s] Violated rack-awareness requirement for broker with id %d. topic-partition %s with replicas %s", this.name(), broker.id(), replica.topicPartition(), clusterModel.partition(replica.topicPartition()).replicas()));
        }
    }

    private boolean movePartition(ClusterModel clusterModel, Set<Goal> optimizedGoals, Replica replica) {
        Cell sourceCell = replica.broker().cell();
        TreeSet<Cell> cells = new TreeSet<Cell>(Comparator.comparingInt(Cell::numReplicas));
        clusterModel.cellsById().values().stream().filter(cell -> !cell.equals(sourceCell)).forEach(cells::add);
        for (Cell cell2 : cells) {
            Iterable<List<Broker>> brokersIterable;
            List<Replica> replicasToMove = clusterModel.partition(replica.topicPartition()).replicas().stream().filter(replicaToMove -> !replicaToMove.broker().cell().equals(cell2)).collect(Collectors.toList());
            try {
                if (replicasToMove.stream().anyMatch(r -> !r.broker().isEligible())) {
                    LOG.debug("Cannot move partition {} as one or more source brokers are ineligible for movements. Cell {} is not eligible for replica placement: {}", new Object[]{replica.topicPartition(), cell2, replicasToMove});
                    continue;
                }
                Map<String, List<Broker>> alivePartitionEligibleBrokers = cell2.eligibleBrokers().stream().filter(broker -> clusterModel.partition(replica.topicPartition()).canAssignReplicaToBroker((Broker)broker)).collect(Collectors.groupingBy(broker -> broker.rack().id()));
                ArrayList brokersByRack = new ArrayList(alivePartitionEligibleBrokers.values());
                brokersIterable = EntityCombinator.multiEntityListBalancedIterable(brokersByRack, replicasToMove.size());
            }
            catch (Exception ex) {
                LOG.debug("Cell {} not eligible for replica placement: {}", new Object[]{cell2, replicasToMove, ex});
                continue;
            }
            Map<Replica, Broker> replicaMoves = GoalUtils.getPartitionMoves(clusterModel, optimizedGoals, replicasToMove, brokersIterable);
            if (replicaMoves.isEmpty()) continue;
            replicaMoves.forEach((replicaToMove, broker) -> this.relocateReplica(clusterModel, replicaToMove.topicPartition(), replicaToMove.broker().id(), broker.id()));
            return true;
        }
        return false;
    }

    private void ensureRackAware(ClusterModel clusterModel, Set<String> excludedTopics) throws OptimizationFailureException {
        for (Replica leader : clusterModel.leaderReplicas()) {
            TopicPartition tp = leader.topicPartition();
            if (excludedTopics.contains(tp.topic()) || clusterModel.getTopicPlacement(tp.topic()).isPresent()) continue;
            Map<String, Integer> replicaCountsPerRack = this.computeReplicaCounts(leader, clusterModel);
            Map.Entry maxReplicaCount = replicaCountsPerRack.entrySet().stream().max(Comparator.comparingInt(Map.Entry::getValue)).get();
            Map.Entry minReplicaCount = replicaCountsPerRack.entrySet().stream().min(Comparator.comparingInt(Map.Entry::getValue)).get();
            if ((Integer)maxReplicaCount.getValue() - (Integer)minReplicaCount.getValue() <= 1) continue;
            throw new OptimizationFailureException("Failed to optimize goal " + this.name() + ": partition " + tp + " had " + maxReplicaCount.getValue() + " replicas on rack " + (String)maxReplicaCount.getKey() + " but " + minReplicaCount.getValue() + " replicas on rack " + (String)minReplicaCount.getKey() + " after optimization");
        }
    }

    private SortedSet<Broker> rackAwareEligibleBrokers(Replica replica, ClusterModel clusterModel, Map<String, Integer> replicaCountsPerRack) {
        TreeSet<Broker> rackAwareEligibleBrokers = new TreeSet<Broker>(Comparator.comparingInt(b -> (Integer)replicaCountsPerRack.get(b.rack().id())).thenComparingInt(Broker::numReplicas).thenComparingInt(Broker::id));
        String rackId = replica.broker().rack().id();
        int numReplicasInRack = replicaCountsPerRack.get(rackId);
        Cell replicaCell = replica.broker().cell();
        int numPartitionCells = (int)clusterModel.partition(replica.topicPartition()).partitionBrokers().stream().map(Broker::cell).distinct().count();
        return replicaCountsPerRack.entrySet().stream().filter(replicaCount -> numReplicasInRack - (Integer)replicaCount.getValue() > 1 || replica.isCurrentOffline()).flatMap(replicaCount -> clusterModel.rack((String)replicaCount.getKey()).brokers().stream().filter(Broker::isEligibleDestination).filter(broker -> numPartitionCells > 1 || broker.cell().equals(replicaCell))).collect(Collectors.toCollection(() -> rackAwareEligibleBrokers));
    }

    private boolean satisfiedRackAwareness(Replica replica, Map<String, Integer> replicaCountsPerRack) {
        int replicasInRack = replicaCountsPerRack.get(replica.broker().rack().id());
        for (Integer count : replicaCountsPerRack.values()) {
            if (replicasInRack - count <= 1) continue;
            return false;
        }
        return true;
    }

    private Map<String, Integer> computeReplicaCounts(Replica replica, ClusterModel clusterModel) {
        return this.computeReplicaCounts(replica.topicPartition(), clusterModel);
    }

    Map<String, Integer> computeReplicaCounts(TopicPartition tp, ClusterModel clusterModel) {
        Map<String, Integer> replicaCountsPerRack = this.racks.stream().collect(Collectors.toMap(k -> k, v -> 0));
        clusterModel.partition(tp).partitionBrokers().forEach(b -> replicaCountsPerRack.merge(b.rack().id(), 1, Integer::sum));
        return replicaCountsPerRack;
    }
}

