/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.snapshot;

import io.confluent.kafka.storage.cloud.EpochAndSeqNumber;
import io.confluent.kafka.storage.cloud.LinkedCloudObject;
import io.confluent.kafka.storage.cloud.SequencedObject;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;
import kafka.tier.domain.TierTopicPartitionSnapshot;
import kafka.tier.snapshot.TierTopicSnapshotMetrics;
import kafka.tier.snapshot.TierTopicSnapshotObject;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreFunctionUtils;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.FragmentType;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.metadata.TierTopicSnapshotMetadata;
import kafka.tier.topic.TierTopicManager;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.KafkaThread;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TierTopicSnapshotManager
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(TierTopicSnapshotManager.class);
    private volatile boolean shutdown = false;
    private final short tierNumPartitions;
    private final Long checkpointIntervalMs;
    private final Long maxRecordsPerSnapshot;
    private final Integer retentionHours;
    private final Integer leaderEpoch;
    private volatile Consumer<byte[], byte[]> consumer;
    private final TierObjectStore objectStore;
    private final Time time;
    private final TierTopicSnapshotMetrics metrics;
    private Long lastCheckpointTimeMs = -1L;
    private EpochAndSeqNumber previousCheckpoint;
    private Long currentSeqNumber = -1L;
    private final List<ConsumerRecords<byte[], byte[]>> buffer = new ArrayList<ConsumerRecords<byte[], byte[]>>();
    private long recordsInBuffer = 0L;
    private final Thread checkpointingThread = new KafkaThread("TierTopicPartitionCheckpoint", (Runnable)this, false);
    private static final Duration POLL_DURATION_MS = Duration.ofMillis(5L);
    private static final Long PAUSE_DURATION_MS = Duration.ofSeconds(1L).toMillis();

    public TierTopicSnapshotManager(Consumer<byte[], byte[]> consumer, TierObjectStore objectStore, Integer leaderEpoch, Short tierNumPartitions, Long checkpointIntervalMs, Long maxRecordsPerSnapshot, Integer retentionHours, Time time, Metrics metrics) {
        this.consumer = consumer;
        this.objectStore = objectStore;
        this.leaderEpoch = leaderEpoch;
        this.tierNumPartitions = tierNumPartitions;
        this.checkpointIntervalMs = checkpointIntervalMs;
        this.maxRecordsPerSnapshot = maxRecordsPerSnapshot;
        this.retentionHours = retentionHours;
        this.time = time;
        this.metrics = new TierTopicSnapshotMetrics(metrics);
    }

    public Integer leaderEpoch() {
        return this.leaderEpoch;
    }

    public void start() throws IOException {
        this.checkpointingThread.start();
    }

    public void shutdown() {
        this.shutdown = true;
        this.checkpointingThread.interrupt();
        if (this.consumer != null) {
            this.consumer.wakeup();
        }
        try {
            this.checkpointingThread.join();
            this.metrics.deRegister();
        }
        catch (InterruptedException e) {
            log.error("Shutdown interrupted", (Throwable)e);
        }
    }

    public boolean isShutdown() {
        return this.shutdown;
    }

    @Override
    public void run() {
        try {
            this.metrics.markTTPSActive();
            this.initialize();
            while (!this.shutdown) {
                this.doWork();
                Thread.sleep(PAUSE_DURATION_MS);
            }
        }
        catch (InterruptedException e) {
            if (!this.shutdown) {
                log.error("Received Interrupted exception when shutdown was false", (Throwable)e);
                this.metrics.recordSnapshotManagerFailure();
            }
        }
        catch (Exception e) {
            log.error("Fatal exception in TierTopicSnapshotManager", (Throwable)e);
            this.metrics.recordSnapshotManagerFailure();
        }
        finally {
            this.metrics.markTTPSInActive();
            this.buffer.clear();
            this.close();
        }
    }

    private void close() {
        try {
            this.consumer.close();
        }
        catch (Exception e) {
            if (this.shutdown) {
                log.info("Exception caught during shutdown", (Throwable)e);
            }
            log.error("Fatal exception in TierTopicSnapshotManager", (Throwable)e);
        }
    }

    protected void initialize() throws IOException, InterruptedException {
        Set<TopicPartition> tierTopicPartitions = TierTopicManager.partitions("_confluent-tier-state", this.tierNumPartitions);
        this.consumer.assign(tierTopicPartitions);
        Optional latestSnapshot = LinkedCloudObject.latestCommittedObject(this.latestSnapshots());
        if (latestSnapshot.isPresent()) {
            SequencedObject lastRecoveredObject = (SequencedObject)latestSnapshot.get();
            this.previousCheckpoint = lastRecoveredObject.currentEpochAndSeqNumber();
            if (this.leaderEpoch.intValue() == this.previousCheckpoint.epoch()) {
                this.currentSeqNumber = this.previousCheckpoint.seqNumber();
            }
            log.info("Latest committed object: " + lastRecoveredObject);
            TierTopicPartitionSnapshot snapshot = this.snapshot(lastRecoveredObject);
            for (TopicPartition tierTopicPartition : tierTopicPartitions) {
                this.consumer.seek(tierTopicPartition, snapshot.endOffset(tierTopicPartition.partition()).longValue());
            }
        } else {
            TierTopicSnapshotObject lastRecoveredObject = new TierTopicSnapshotObject(0L, 0L, new EpochAndSeqNumber(this.leaderEpoch.intValue(), -1L), new EpochAndSeqNumber(-1, -1L));
            this.previousCheckpoint = lastRecoveredObject.currentEpochAndSeqNumber();
            log.info("Initializing committed object: " + lastRecoveredObject);
            this.consumer.seekToEnd(tierTopicPartitions);
        }
    }

    private void doWork() throws IOException, InterruptedException {
        long currentTimeMs = this.time.milliseconds();
        ConsumerRecords records = this.consumer.poll(POLL_DURATION_MS);
        if (records.count() != 0) {
            this.buffer.add((ConsumerRecords<byte[], byte[]>)records);
            this.recordsInBuffer += (long)records.count();
        }
        if (!(this.buffer.isEmpty() || currentTimeMs - this.lastCheckpointTimeMs < this.checkpointIntervalMs && this.recordsInBuffer < this.maxRecordsPerSnapshot)) {
            ArrayList<Long> offsets = new ArrayList<Long>();
            long totalLag = 0L;
            for (int i = 0; i < this.tierNumPartitions; ++i) {
                TopicPartition partition = new TopicPartition("_confluent-tier-state", i);
                offsets.add(this.consumer.position(partition, Duration.ofMinutes(5L)));
                totalLag += this.consumer.currentLag(partition).orElse(0L);
            }
            log.info("total tier topic consumer lag :  " + totalLag);
            this.metrics.recordConsumerLag(totalLag);
            TierTopicPartitionSnapshot snapshot = new TierTopicPartitionSnapshot(this.buffer, offsets);
            EpochAndSeqNumber currentCheckpoint = new EpochAndSeqNumber(this.leaderEpoch.intValue(), this.currentSeqNumber + 1L);
            TierTopicSnapshotObject snapshotObject = new TierTopicSnapshotObject(snapshot.startTimestampMs(), snapshot.endTimestampMs(), currentCheckpoint, this.previousCheckpoint);
            TierTopicSnapshotMetadata metadata = new TierTopicSnapshotMetadata(snapshotObject);
            ByteBuffer snapshotBuffer = snapshot.payloadBuffer().compact();
            snapshotBuffer.flip();
            TierObjectStoreFunctionUtils.putBuffer(this::isShutdown, this.objectStore, metadata, snapshotBuffer, ObjectType.TIER_TOPIC_SNAPSHOT);
            log.info("Uploaded tier topic snapshot: " + snapshotObject);
            this.metrics.recordSnapshotUpload();
            this.buffer.clear();
            this.recordsInBuffer = 0L;
            this.lastCheckpointTimeMs = currentTimeMs;
            this.currentSeqNumber = currentCheckpoint.seqNumber();
            this.previousCheckpoint = currentCheckpoint;
        }
    }

    protected TierTopicPartitionSnapshot snapshot(SequencedObject snapshotObject) throws IOException, InterruptedException {
        TierTopicSnapshotMetadata metadata = new TierTopicSnapshotMetadata((TierTopicSnapshotObject)snapshotObject);
        log.info("Fetching tier topic snapshot: " + metadata);
        TierObjectStoreResponse response = TierObjectStoreFunctionUtils.getObjectStoreFragment(this::isShutdown, this.objectStore, metadata, FragmentType.TIER_TOPIC_SNAPSHOT);
        return TierTopicPartitionSnapshot.read(response.getInputStream(), (Long)metadata.snapshotObject().startTimestampMs(), (Long)metadata.snapshotObject().endTimestampMs());
    }

    protected List<SequencedObject> latestSnapshots() throws InterruptedException {
        Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
        calendar.setTimeInMillis(this.time.milliseconds());
        for (int hours = this.retentionHours.intValue(); hours > 0; --hours) {
            String prefix = TierTopicSnapshotMetadata.pathPrefix("") + "/" + TierTopicSnapshotObject.getDirName(calendar.getTimeInMillis());
            Map<String, List<VersionInformation>> objects = TierObjectStoreFunctionUtils.listObject(this::isShutdown, this.objectStore, prefix, false);
            log.info("Searching for tier topic snapshots in hour window: " + prefix + " found objects: " + objects);
            if (!objects.isEmpty()) {
                return TierTopicSnapshotManager.convertObjListToSeqList(objects);
            }
            calendar.add(11, -1);
        }
        return new ArrayList<SequencedObject>();
    }

    public static List<SequencedObject> convertObjListToSeqList(Map<String, List<VersionInformation>> objects) {
        ArrayList<SequencedObject> sequencedObjects = new ArrayList<SequencedObject>();
        for (Map.Entry<String, List<VersionInformation>> object : objects.entrySet()) {
            TierTopicSnapshotMetadata snapshotMetadata = TierTopicSnapshotMetadata.fromPath(object.getKey());
            sequencedObjects.add(snapshotMetadata.snapshotObject());
        }
        return sequencedObjects;
    }
}

