/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.SdkClientException;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.PropertiesFileCredentialsProvider;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsV2Request;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.S3VersionSummary;
import com.amazonaws.services.s3.model.SSEAwsKeyManagementParams;
import com.amazonaws.services.s3.model.VersionListing;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClient;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder;
import io.confluent.kafka.storage.checksum.E2EChecksumProtectedObjectType;
import io.confluent.kafka.storage.checksum.E2EChecksumStore;
import io.confluent.kafka.storage.checksum.E2EChecksumUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import kafka.server.KafkaConfig;
import kafka.tier.exceptions.E2EChecksumInvalidException;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.AutoAbortingGenericInputStream;
import kafka.tier.store.BucketHealthResult;
import kafka.tier.store.OpaqueData;
import kafka.tier.store.S3AutoAbortingInputStream;
import kafka.tier.store.S3TierObjectStoreConfig;
import kafka.tier.store.S3VersionInformation;
import kafka.tier.store.ThrottledFileInputStream;
import kafka.tier.store.TierObjectAttribute;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.TierObjectStoreUtils;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.FragmentType;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.metadata.HealthMetadata;
import kafka.tier.store.objects.metadata.ObjectMetadata;
import kafka.tier.store.objects.metadata.ObjectStoreMetadata;
import kafka.utils.Throttler;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.kafka.common.utils.ByteBufferInputStream;
import org.apache.kafka.common.utils.SecurityUtils;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedClientOption;
import software.amazon.awssdk.core.exception.RetryableException;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.S3AsyncClientBuilder;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.sts.StsClient;
import software.amazon.awssdk.services.sts.StsClientBuilder;
import software.amazon.awssdk.services.sts.auth.StsAssumeRoleCredentialsProvider;
import software.amazon.awssdk.services.sts.model.AssumeRoleRequest;

public class S3TierObjectStore
extends TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(S3TierObjectStore.class);
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final AmazonS3 client;
    private final Optional<S3AsyncClient> asyncClientOpt;
    private final Optional<AWSCredentialsProvider> credentialsProvider;
    private final String bucket;
    private final String prefix;
    private final String sseAlgorithm;
    private final String sseCustomerEncryptionKey;
    private final int autoAbortThresholdBytes;
    private AtomicInteger credentialsRefreshRetries = new AtomicInteger();
    private final Optional<E2EChecksumStore> checksumStoreOpt;
    private static final int DEFAULT_S3_MAX_CREDENTIAL_REFRESH_RETRIES = 5;
    private static final int DEFAULT_S3_DELETE_BATCH_SIZE = 500;
    private static final String CRC32C = "CRC32C";
    private static final String CRC32C_HEADER = "x-amz-checksum-crc32c";
    private static final String ERROR_CODE_BAD_DIGEST = "BadDigest";
    static final String ERROR_CODE_EXPIRED_TOKEN = "ExpiredToken";
    private static final String ERROR_CODE_NO_SUCH_KEY = "NoSuchKey";

    public S3TierObjectStore(S3TierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this(S3TierObjectStore.client(config), config, checksumStoreOpt);
    }

    S3TierObjectStore(AmazonS3ClientAndCredentialsProvider clientAndCredentialsProvider, S3TierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this(clientAndCredentialsProvider.client(), clientAndCredentialsProvider.asyncClientOpt(), clientAndCredentialsProvider.credentialsProvider(), config, checksumStoreOpt);
    }

    S3TierObjectStore(AmazonS3 client, AWSCredentialsProvider credentialsProvider, S3TierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this(client, Optional.empty(), credentialsProvider, config, checksumStoreOpt);
    }

    S3TierObjectStore(AmazonS3 client, Optional<S3AsyncClient> asyncClientOpt, AWSCredentialsProvider credentialsProvider, S3TierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.client = client;
        this.asyncClientOpt = asyncClientOpt;
        this.credentialsProvider = Optional.ofNullable(credentialsProvider);
        this.bucket = config.s3Bucket;
        this.prefix = config.s3Prefix;
        this.sseAlgorithm = config.s3SseAlgorithm;
        this.sseCustomerEncryptionKey = config.s3SseCustomerEncryptionKey;
        this.autoAbortThresholdBytes = config.s3AutoAbortThresholdBytes;
        this.expectBucket(this.bucket, config.s3Region, config.s3EndpointOverride);
        this.checksumStoreOpt = checksumStoreOpt;
    }

    @Override
    public String keyPrefix() {
        return this.prefix;
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.S3;
    }

    private Set<String> listObjectsWithoutVersions(String keyPrefix) {
        HashSet<String> objects = new HashSet<String>();
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                ListObjectsV2Result result;
                ListObjectsV2Request req = new ListObjectsV2Request().withBucketName(this.bucket).withPrefix(keyPrefix).withMaxKeys(Integer.valueOf(1000));
                do {
                    result = this.client.listObjectsV2(req);
                    for (S3ObjectSummary objectSummary : result.getObjectSummaries()) {
                        objects.add(objectSummary.getKey());
                    }
                    String token = result.getNextContinuationToken();
                    req.setContinuationToken(token);
                } while (result.isTruncated());
                log.debug("TierObjectStore listObjects versions: false keyPrefix:  " + keyPrefix + " # of objects returned: " + objects.size());
                return null;
            });
        }
        catch (SdkClientException e) {
            throw new TierObjectStoreRetriableException("Failed to list the objects (without versions) with prefix: " + keyPrefix, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception while listing objects (without versions) with prefix: " + keyPrefix, e);
        }
        return objects;
    }

    @Override
    public Map<String, List<VersionInformation>> listObject(String keyPrefix, boolean getVersionInfo) {
        HashMap<String, List<VersionInformation>> results = new HashMap<String, List<VersionInformation>>();
        if (!getVersionInfo) {
            for (String object : this.listObjectsWithoutVersions(keyPrefix)) {
                results.put(object, new ArrayList());
            }
            return results;
        }
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                ListVersionsRequest request = new ListVersionsRequest().withBucketName(this.bucket).withPrefix(keyPrefix).withMaxResults(Integer.valueOf(1000));
                VersionListing versionListings = this.client.listVersions(request);
                while (true) {
                    for (S3VersionSummary versionSummary : versionListings.getVersionSummaries()) {
                        results.putIfAbsent(versionSummary.getKey(), new ArrayList());
                        S3VersionInformation versionInfo = new S3VersionInformation(versionSummary.getVersionId(), versionSummary.isDeleteMarker(), versionSummary.isLatest());
                        ((List)results.get(versionSummary.getKey())).add(versionInfo);
                    }
                    if (!versionListings.isTruncated()) break;
                    versionListings = this.client.listNextBatchOfVersions(versionListings);
                }
                if (log.isDebugEnabled()) {
                    StringBuilder allBlobs = new StringBuilder();
                    results.forEach((key, versions) -> allBlobs.append("[").append((String)key).append("->").append(Arrays.toString(versions.toArray())).append("] "));
                    log.debug("TierObjectStore listObjects versions: true keyPrefix: " + keyPrefix + " " + allBlobs);
                }
                return null;
            });
        }
        catch (SdkClientException e) {
            throw new TierObjectStoreRetriableException("Failed to list the objects with prefix: " + keyPrefix, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception while listing objects with prefix: " + keyPrefix, e);
        }
        return results;
    }

    @Override
    protected TierObjectStoreResponse getObject(ObjectStoreMetadata objectMetadata, ObjectType objectType, Long byteOffsetStart, Long byteOffsetEndExclusive, VersionInformation versionInformation) {
        S3Object object;
        GetObjectRequest request;
        String key = this.keyPath(objectMetadata, objectType);
        String fullPath = "s3://" + this.bucket + "/" + key;
        GetObjectRequest getObjectRequest = request = versionInformation != null ? new GetObjectRequest(this.bucket, key, versionInformation.getVersionId()) : new GetObjectRequest(this.bucket, key);
        if (byteOffsetStart != null && byteOffsetEndExclusive != null) {
            request.setRange(byteOffsetStart.longValue(), byteOffsetEndExclusive - 1L);
        } else if (byteOffsetStart != null && byteOffsetStart != 0L) {
            request.setRange(byteOffsetStart.longValue());
        } else if (byteOffsetEndExclusive != null) {
            throw new IllegalStateException(String.format("Cannot specify a byteOffsetEndExclusive=%d without specifying a byteOffsetStart", byteOffsetEndExclusive));
        }
        log.debug("Fetching object from {}, with range {} - {}", new Object[]{fullPath, byteOffsetStart, byteOffsetEndExclusive});
        try {
            object = this.checkExpiredCredentialsExceptionAndTryRefresh(() -> this.client.getObject(request));
        }
        catch (Exception e) {
            throw this.convertFetchException(e, fullPath, objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive);
        }
        S3ObjectInputStream inputStream = object.getObjectContent();
        return new S3TierObjectStoreResponse((InputStream)inputStream, this.autoAbortThresholdBytes, object.getObjectMetadata().getContentLength());
    }

    @Override
    protected CompletableFuture<TierObjectStoreResponse> getObjectAsync(ObjectStoreMetadata objectMetadata, ObjectType objectType, Long byteOffsetStart, Long byteOffsetEndExclusive, VersionInformation versionInformation) {
        CompletableFuture<TierObjectStoreResponse> future = new CompletableFuture<TierObjectStoreResponse>();
        try {
            if (this.asyncClientOpt.isPresent()) {
                String key = this.keyPath(objectMetadata, objectType);
                String fullPath = "s3://" + this.bucket + "/" + key;
                GetObjectRequest.Builder requestBuilder = software.amazon.awssdk.services.s3.model.GetObjectRequest.builder().bucket(this.bucket).key(key);
                if (versionInformation != null) {
                    requestBuilder.versionId(versionInformation.getVersionId());
                }
                if (byteOffsetStart != null && byteOffsetEndExclusive != null) {
                    requestBuilder.range(String.format("bytes=%d-%d", byteOffsetStart, byteOffsetEndExclusive - 1L));
                } else if (byteOffsetStart != null && byteOffsetStart != 0L) {
                    requestBuilder.range(String.format("bytes=%d-", byteOffsetStart));
                } else if (byteOffsetEndExclusive != null) {
                    throw new IllegalStateException("Cannot specify a byteOffsetEndExclusive without specifying a byteOffsetStart");
                }
                log.debug("Fetching object from {}, with range {} - {}", new Object[]{fullPath, byteOffsetStart, byteOffsetEndExclusive});
                this.asyncClientOpt.get().getObject((software.amazon.awssdk.services.s3.model.GetObjectRequest)requestBuilder.build(), AsyncResponseTransformer.toBytes()).handle((responseBytes, e) -> {
                    if (e != null) {
                        future.completeExceptionally(this.convertFetchException((Throwable)e, fullPath, objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive));
                    } else {
                        future.complete(new S3TierObjectStoreResponse(responseBytes.asInputStream(), this.autoAbortThresholdBytes, responseBytes.asByteArrayUnsafe().length));
                    }
                    return null;
                });
            } else {
                log.warn("Fallback to getObject because async S3 client isn't created, metadata: {}, type: {}", (Object)objectMetadata, (Object)objectType);
                future.complete(this.getObject(objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive, versionInformation));
            }
        }
        catch (Exception e2) {
            log.error("Failed to send async fetch request, metadata: {}, type: {}", new Object[]{objectMetadata, objectType, e2});
            future.completeExceptionally(e2);
        }
        return future;
    }

    private RuntimeException convertFetchException(Throwable e, String fullPath, ObjectStoreMetadata objectMetadata, ObjectType objectType, Long byteOffsetStart, Long byteOffsetEndExclusive) {
        if (e instanceof AmazonClientException || e instanceof RetryableException) {
            return new TierObjectStoreRetriableException(String.format("Failed to fetch object from %s, metadata: %s type: %s range %s-%s", new Object[]{fullPath, objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive}), e);
        }
        return new TierObjectStoreFatalException(String.format("Unknown exception when fetching object from %s, metadata: %s type: %s range %s-%s", new Object[]{fullPath, objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive}), e);
    }

    @Override
    public ByteBuffer getSnapshot(ObjectStoreMetadata metadata, FragmentType fragmentType, int estimatedBufferSize) {
        ByteBuffer buffer;
        if (fragmentType != FragmentType.TIER_PARTITION_STATE_METADATA_SNAPSHOT && fragmentType != FragmentType.TIER_STATE_SNAPSHOT && fragmentType != FragmentType.PRODUCER_STATE) {
            throw new IllegalArgumentException("getSnapshot does not support the given fragmentType: " + (Object)((Object)fragmentType));
        }
        try (TierObjectStoreResponse response = this.getObjectStoreFragment(metadata, fragmentType);){
            buffer = ByteBuffer.wrap(Utils.readFullyToArray((InputStream)response.getInputStream(), (int)estimatedBufferSize));
        }
        catch (Exception e) {
            for (Throwable cause = e.getCause(); cause != null && cause != cause.getCause(); cause = cause.getCause()) {
                if (!(cause instanceof AmazonS3Exception) || !ERROR_CODE_NO_SUCH_KEY.equals(((AmazonS3Exception)cause).getErrorCode())) continue;
                throw new TierObjectStoreFatalException("Snapshot object not found in object store.", cause);
            }
            throw new TierObjectStoreRetriableException("Encountered an exception when fetching snapshot from object store.", e);
        }
        return buffer;
    }

    @Override
    public OpaqueData prepPutSegment() throws TierObjectStoreRetriableException, IOException {
        return OpaqueData.ZEROED;
    }

    @Override
    public String putObject(ObjectStoreMetadata objectMetadata, File file, ObjectType objectType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        String key = this.keyPath(objectMetadata, objectType);
        String fullPath = "s3://" + this.bucket + "/" + key;
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                this.putFile(key, metadata, file, objectType.toE2EChecksumProtectedObjectType());
                return null;
            });
        }
        catch (AmazonClientException e) {
            AmazonS3Exception ex;
            if (e instanceof AmazonS3Exception && ERROR_CODE_BAD_DIGEST.equals((ex = (AmazonS3Exception)e).getErrorCode())) {
                throw new E2EChecksumInvalidException("Checksum mismatch during object store upload", (Throwable)ex);
            }
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object to %s with metadata %s, file %s, type %s", new Object[]{fullPath, objectMetadata, file, objectType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object to %s with metadata %s, file %s, type %s", new Object[]{fullPath, objectMetadata, file, objectType}), e);
        }
        return key;
    }

    @Override
    public String putBuffer(ObjectStoreMetadata objectMetadata, ByteBuffer buffer, ObjectType objectType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        String key = this.keyPath(objectMetadata, objectType);
        String fullPath = "s3://" + this.bucket + "/" + key;
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                this.putBuf(key, metadata, buffer);
                return null;
            });
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object to %s with metadata %s, buffer %s, type %s", new Object[]{fullPath, objectMetadata, buffer, objectType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object to %s with metadata %s, buffer %s, type %s", new Object[]{fullPath, objectMetadata, buffer, objectType}), e);
        }
        return key;
    }

    @Override
    public void restoreObjectByCopy(ObjectMetadata objectMetadata, String key, VersionInformation lastLiveVersion) {
        String lastLiveVersionId = lastLiveVersion.getVersionId();
        String fullPath = "s3://" + this.bucket + "/" + key;
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                CopyObjectRequest request = new CopyObjectRequest(this.bucket, key, lastLiveVersionId, this.bucket, key).withNewObjectMetadata(this.objectMetadata(null));
                this.setKmsParams(request);
                log.debug("restore object for {}-{}", (Object)fullPath, (Object)lastLiveVersionId);
                this.client.copyObject(request);
                return null;
            });
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to restore object %s (version: %s)", fullPath, lastLiveVersionId), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when restoring object %s (version: %s)", fullPath, lastLiveVersionId), e);
        }
    }

    private void setKmsParams(CopyObjectRequest request) {
        if (this.usesKms()) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
    }

    @Override
    public void putSegment(ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<File> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState, Optional<Throttler> throttlerOpt) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                if (throttlerOpt.isPresent()) {
                    this.putFileWithThrottling(this.keyPath(objectMetadata, ObjectType.SEGMENT), metadata, segmentData, E2EChecksumProtectedObjectType.SEGMENT, (Throttler)throttlerOpt.get());
                    this.putFileWithThrottling(this.keyPath(objectMetadata, ObjectType.OFFSET_INDEX), metadata, offsetIndexData, E2EChecksumProtectedObjectType.OFFSET_INDEX, (Throttler)throttlerOpt.get());
                    this.putFileWithThrottling(this.keyPath(objectMetadata, ObjectType.TIMESTAMP_INDEX), metadata, timestampIndexData, E2EChecksumProtectedObjectType.TIMESTAMP_INDEX, (Throttler)throttlerOpt.get());
                    if (producerStateSnapshotData.isPresent()) {
                        this.putFileWithThrottling(this.keyPath(objectMetadata, ObjectType.PRODUCER_STATE), metadata, (File)producerStateSnapshotData.get(), E2EChecksumProtectedObjectType.PRODUCER_STATE, (Throttler)throttlerOpt.get());
                    }
                } else {
                    this.putFile(this.keyPath(objectMetadata, ObjectType.SEGMENT), metadata, segmentData, E2EChecksumProtectedObjectType.SEGMENT);
                    this.putFile(this.keyPath(objectMetadata, ObjectType.OFFSET_INDEX), metadata, offsetIndexData, E2EChecksumProtectedObjectType.OFFSET_INDEX);
                    this.putFile(this.keyPath(objectMetadata, ObjectType.TIMESTAMP_INDEX), metadata, timestampIndexData, E2EChecksumProtectedObjectType.TIMESTAMP_INDEX);
                    producerStateSnapshotData.ifPresent(file -> this.putFile(this.keyPath(objectMetadata, ObjectType.PRODUCER_STATE), metadata, (File)file, E2EChecksumProtectedObjectType.PRODUCER_STATE));
                }
                transactionIndexData.ifPresent(abortedTxnsBuf -> this.putBufMaybeEncrypt(this.keyPath(objectMetadata, ObjectType.TRANSACTION_INDEX), metadata, (ByteBuffer)abortedTxnsBuf));
                epochState.ifPresent(buf -> this.putBufMaybeEncrypt(this.keyPath(objectMetadata, ObjectType.EPOCH_STATE), metadata, (ByteBuffer)buf));
                return null;
            });
        }
        catch (AmazonClientException e) {
            AmazonS3Exception ex;
            if (e instanceof AmazonS3Exception && ERROR_CODE_BAD_DIGEST.equals((ex = (AmazonS3Exception)e).getErrorCode())) {
                throw new E2EChecksumInvalidException(objectMetadata, (Throwable)ex);
            }
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
        finally {
            this.checksumStoreOpt.ifPresent(checksumStore -> TierObjectStoreUtils.postPutSegmentCleanup(checksumStore, segmentData, offsetIndexData, timestampIndexData, producerStateSnapshotData));
        }
    }

    private <T> T checkExpiredCredentialsExceptionAndTryRefresh(Callable<T> func) throws Exception {
        try {
            T result = func.call();
            this.credentialsRefreshRetries.set(0);
            return result;
        }
        catch (AmazonS3Exception ex) {
            if (ERROR_CODE_EXPIRED_TOKEN.equals(ex.getErrorCode())) {
                this.tryRefreshCredentials(ex);
            } else {
                this.credentialsRefreshRetries.set(0);
            }
            throw ex;
        }
    }

    private void tryRefreshCredentials(AmazonS3Exception ex) {
        try {
            int numRetries = this.credentialsRefreshRetries.getAndIncrement();
            if (numRetries >= 5) {
                log.warn("Hit maximum number of consecutive credential refresh attempts without seeing a successful request, skipping refresh attempt.");
                return;
            }
            this.credentialsProvider.ifPresent(provider -> {
                log.info("S3 credentials expired; attempting a credentials provider refresh.", (Throwable)ex);
                provider.refresh();
            });
        }
        catch (Exception refreshException) {
            log.error("Received exception while attempting to refresh S3 credentials.", (Throwable)refreshException);
        }
    }

    @Override
    public void putInMemorySegment(ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<ByteBuffer> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                this.putFile(this.keyPath(objectMetadata, ObjectType.SEGMENT), metadata, segmentData, E2EChecksumProtectedObjectType.SEGMENT);
                this.putFile(this.keyPath(objectMetadata, ObjectType.OFFSET_INDEX), metadata, offsetIndexData, E2EChecksumProtectedObjectType.OFFSET_INDEX);
                this.putFile(this.keyPath(objectMetadata, ObjectType.TIMESTAMP_INDEX), metadata, timestampIndexData, E2EChecksumProtectedObjectType.TIMESTAMP_INDEX);
                producerStateSnapshotData.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, ObjectType.PRODUCER_STATE), metadata, (ByteBuffer)buf));
                transactionIndexData.ifPresent(abortedTxnsBuf -> this.putBuf(this.keyPath(objectMetadata, ObjectType.TRANSACTION_INDEX), metadata, (ByteBuffer)abortedTxnsBuf));
                epochState.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, ObjectType.EPOCH_STATE), metadata, (ByteBuffer)buf));
                return null;
            });
        }
        catch (AmazonClientException e) {
            AmazonS3Exception ex;
            if (e instanceof AmazonS3Exception && ERROR_CODE_BAD_DIGEST.equals((ex = (AmazonS3Exception)((Object)e)).getErrorCode())) {
                throw new E2EChecksumInvalidException(objectMetadata, (Throwable)ex);
            }
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
        finally {
            this.checksumStoreOpt.ifPresent(checksumStore -> TierObjectStoreUtils.postPutSegmentCleanup(checksumStore, segmentData, offsetIndexData, timestampIndexData));
        }
    }

    @Override
    public void deleteSegment(ObjectMetadata objectMetadata) {
        List<DeleteObjectsRequest.KeyVersion> keys = this.keysForSegment(objectMetadata);
        DeleteObjectsRequest request = new DeleteObjectsRequest(this.bucket).withKeys(keys);
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> this.client.deleteObjects(request));
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to delete segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting segment: " + objectMetadata, e);
        }
    }

    @Override
    public TierObjectAttribute objectExists(ObjectStoreMetadata objectMetadata, ObjectType objectType) throws TierObjectStoreRetriableException {
        TierObjectAttribute result = new TierObjectAttribute(false);
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                String key = this.keyPath(objectMetadata, objectType);
                com.amazonaws.services.s3.model.ObjectMetadata metadata = this.client.getObjectMetadata(this.bucket, key);
                log.trace("objectExists at s3://{}/{} with metadata {}", new Object[]{this.bucket, key, metadata});
                result.exist = true;
                result.size = metadata.getContentLength();
                return null;
            });
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                result.exist = false;
            }
            throw new TierObjectStoreRetriableException("Failed to check object existence: " + objectMetadata + " type: " + (Object)((Object)objectType), e);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to check object existence: " + objectMetadata + " type: " + (Object)((Object)objectType), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when checking object existence: " + objectMetadata + " type: " + (Object)((Object)objectType), e);
        }
        return result;
    }

    @Override
    public void deleteVersions(List<TierObjectStore.KeyAndVersion> keys) {
        ArrayList<DeleteObjectsRequest.KeyVersion> s3Keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        for (TierObjectStore.KeyAndVersion key : keys) {
            DeleteObjectsRequest.KeyVersion keyVersion = key.versionId() == null ? new DeleteObjectsRequest.KeyVersion(key.key()) : new DeleteObjectsRequest.KeyVersion(key.key(), key.versionId());
            log.debug("Deleting object {} {}", (Object)keyVersion.getKey(), (Object)keyVersion.getVersion());
            s3Keys.add(keyVersion);
            if (s3Keys.size() < 500) continue;
            this.makeDeleteObjectsCall(s3Keys);
            s3Keys.clear();
        }
        if (!s3Keys.isEmpty()) {
            this.makeDeleteObjectsCall(s3Keys);
        }
    }

    private void makeDeleteObjectsCall(List<DeleteObjectsRequest.KeyVersion> s3Keys) {
        DeleteObjectsRequest request = new DeleteObjectsRequest(this.bucket).withKeys(s3Keys);
        try {
            this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                log.debug("Sending a batch delete request");
                return this.client.deleteObjects(request);
            });
        }
        catch (MultiObjectDeleteException e) {
            log.error("S3 reported errors while deleting the following versioned objects:");
            e.getErrors().forEach(err -> log.error("Blob Key: " + err.getKey() + " Blob VersionId: " + err.getVersionId() + " Error Code: " + err.getCode() + " Error Message: " + err.getMessage()));
            throw new TierObjectStoreRetriableException("Failed to delete " + e.getErrors().size() + " versioned objects", e);
        }
        catch (SdkClientException e) {
            log.error(Arrays.toString(e.getStackTrace()));
            throw new TierObjectStoreRetriableException("Failed to delete versioned objects", e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting versioned objects", e);
        }
    }

    @Override
    public void close() {
        this.client.shutdown();
        this.asyncClientOpt.ifPresent(c -> c.close());
    }

    private String keyPath(ObjectStoreMetadata objectMetadata, ObjectType objectType) {
        return objectMetadata.toFragmentLocation(this.prefix, objectType.getDesignatedFragmentType().get()).get().objectPath();
    }

    private com.amazonaws.services.s3.model.ObjectMetadata objectMetadata(Map<String, String> userMetadata) {
        com.amazonaws.services.s3.model.ObjectMetadata metadata = new com.amazonaws.services.s3.model.ObjectMetadata();
        if (this.sseAlgorithm != null) {
            metadata.setSSEAlgorithm(this.sseAlgorithm);
        }
        if (userMetadata != null) {
            metadata.setUserMetadata(userMetadata);
        }
        return metadata;
    }

    private void setKmsParams(PutObjectRequest request) {
        if (this.usesKms()) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
    }

    private void putFile(String key, Map<String, String> metadata, File file, E2EChecksumProtectedObjectType objectType) {
        com.amazonaws.services.s3.model.ObjectMetadata s3ObjMetadata = this.objectMetadata(metadata);
        Optional<String> crc = this.getCrcAndSetHeader(objectType, file, metadata, s3ObjMetadata);
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, file).withMetadata(s3ObjMetadata);
        this.setKmsParams(request);
        if (crc.isPresent()) {
            log.debug("Uploading object to s3://{}/{} with crc {}", new Object[]{this.bucket, key, crc.get()});
        } else {
            log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        }
        this.client.putObject(request);
    }

    private void putFileWithThrottling(String key, Map<String, String> metadata, File file, E2EChecksumProtectedObjectType objectType, Throttler throttler) throws IOException {
        com.amazonaws.services.s3.model.ObjectMetadata s3Metadata = this.objectMetadata(metadata);
        Optional<String> crc = this.getCrcAndSetHeader(objectType, file, metadata, s3Metadata);
        try (ThrottledFileInputStream fileInputStream = new ThrottledFileInputStream(file, throttler);){
            s3Metadata.setContentLength(file.length());
            PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)fileInputStream, s3Metadata);
            this.setKmsParams(request);
            if (crc.isPresent()) {
                log.debug("Uploading object to s3 with throttling://{}/{} with crc {}", new Object[]{this.bucket, key, crc.get()});
            } else {
                log.debug("Uploading object to s3 with throttling://{}/{}", (Object)this.bucket, (Object)key);
            }
            this.client.putObject(request);
        }
    }

    private Optional<String> getCrcAndSetHeader(E2EChecksumProtectedObjectType objectType, File file, Map<String, String> metadata, com.amazonaws.services.s3.model.ObjectMetadata s3ObjMetadata) {
        if (!this.checksumStoreOpt.isPresent()) {
            return Optional.empty();
        }
        E2EChecksumStore checksumStore = this.checksumStoreOpt.get();
        Optional crc = Optional.empty();
        if (checksumStore.checksumProtectionEnabled(objectType)) {
            crc = objectType.shouldCalculateBeforeUpload() ? E2EChecksumUtils.compute32BitBase64Crc32c((File)file) : E2EChecksumUtils.getBase64CrcFromStore((E2EChecksumStore)checksumStore, (File)file, metadata);
            crc.ifPresent(s -> s3ObjMetadata.setHeader(CRC32C_HEADER, s));
        }
        return crc;
    }

    public void putBuf(String key, Map<String, String> userMetadata, ByteBuffer buf) {
        com.amazonaws.services.s3.model.ObjectMetadata s3Metadata = this.objectMetadata(userMetadata);
        s3Metadata.setContentLength((long)(buf.limit() - buf.position()));
        s3Metadata.setHeader(CRC32C_HEADER, (Object)E2EChecksumUtils.compute32BitBase64Crc32c((ByteBuffer)buf));
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)new ByteBufferInputStream(buf.duplicate()), s3Metadata);
        this.setKmsParams(request);
        log.debug("Uploading buffer to {}", (Object)key);
        this.client.putObject(request);
    }

    private boolean usesKms() {
        return this.sseCustomerEncryptionKey != null && this.sseAlgorithm.equals("aws:kms");
    }

    private void putBufMaybeEncrypt(String key, Map<String, String> metadata, ByteBuffer buf) {
        com.amazonaws.services.s3.model.ObjectMetadata s3metadata = this.objectMetadata(metadata);
        s3metadata.setContentLength((long)(buf.limit() - buf.position()));
        s3metadata.setHeader(CRC32C_HEADER, (Object)E2EChecksumUtils.compute32BitBase64Crc32c((ByteBuffer)buf));
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)new ByteBufferInputStream(buf.duplicate()), s3metadata);
        this.setKmsParams(request);
        log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        this.client.putObject(request);
    }

    private List<DeleteObjectsRequest.KeyVersion> keysForSegment(ObjectMetadata objectMetadata) {
        ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        block5: for (ObjectType objectType : TierObjectStore.getObjectTypesPerSegment()) {
            switch (objectType) {
                case TRANSACTION_INDEX: {
                    if (!objectMetadata.hasAbortedTxns()) continue block5;
                    keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, objectType)));
                    continue block5;
                }
                case EPOCH_STATE: {
                    if (!objectMetadata.hasEpochState()) continue block5;
                    keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, objectType)));
                    continue block5;
                }
                case PRODUCER_STATE: {
                    if (!objectMetadata.hasProducerState()) continue block5;
                    keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, objectType)));
                    continue block5;
                }
            }
            keys.add(new DeleteObjectsRequest.KeyVersion(this.keyPath(objectMetadata, objectType)));
        }
        return keys;
    }

    @Override
    public BucketHealthResult checkBucketHealth() {
        try {
            return this.checkExpiredCredentialsExceptionAndTryRefresh(() -> {
                ByteBuffer payload = TierObjectStoreUtils.timeHealthPayload();
                HealthMetadata metadata = new HealthMetadata(this.clusterIdOpt, this.brokerIdOpt);
                String key = metadata.toFragmentLocation(this.prefix, FragmentType.HEALTH_CHECK).get().objectPath();
                this.putBufMaybeEncrypt(key, metadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt), payload);
                try (InputStream inputStream = this.getObjectStoreFragment(metadata, FragmentType.HEALTH_CHECK).getInputStream();){
                    int read;
                    while ((read = inputStream.read()) > 0) {
                        log.trace("Bucket probe read {} bytes", (Object)read);
                    }
                }
                this.client.deleteObject(this.bucket, key);
                return BucketHealthResult.HEALTHY;
            });
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 400 && e.getErrorCode() != null && e.getErrorCode().startsWith("KMS.")) {
                log.error("Bucket health checker resulted in a BYOK related error with error code: {}, status code: {}", new Object[]{e.getErrorCode(), e.getStatusCode(), e});
                return BucketHealthResult.BYOK;
            }
            if (e.getStatusCode() == 403 && Objects.equals(e.getErrorCode(), "AccessDenied")) {
                log.error("Bucket health checker resulted in a permission error for customer key: {}", (Object)(this.usesKms() ? "not enabled" : this.sseCustomerEncryptionKey), (Object)e);
                if (this.usesKms()) {
                    return BucketHealthResult.BYOK;
                }
                return BucketHealthResult.PERMISSION;
            }
            log.error("Bucket health checker returned an unclassified error for status code: {} error code: {}", new Object[]{e.getStatusCode(), e.getErrorCode(), e});
            return BucketHealthResult.UNCLASSIFIED;
        }
        catch (Exception e) {
            log.error("Bucket health checker returned unclassified error", (Throwable)e);
            return BucketHealthResult.UNCLASSIFIED;
        }
    }

    public static String validateAndGetS3RegionName(String s3Region) {
        try {
            return Regions.fromName((String)s3Region).getName();
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Configured " + KafkaConfig.TierS3RegionProp() + " '" + s3Region + "' is not known");
        }
    }

    public static AmazonS3ClientAndCredentialsProvider client(S3TierObjectStoreConfig config) throws TierObjectStoreFatalException {
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        clientConfiguration.setUserAgentPrefix(config.s3UserAgentPrefix);
        SSLConnectionSocketFactory sslConnectionSocketFactory = S3TierObjectStore.getSSLConnectionSocketFactory(config);
        if (sslConnectionSocketFactory != null) {
            clientConfiguration.getApacheHttpClientConfig().setSslSocketFactory((ConnectionSocketFactory)sslConnectionSocketFactory);
        }
        AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard();
        builder.setClientConfiguration(clientConfiguration);
        if (config.s3ForcePathStyleAccess.booleanValue()) {
            builder.setPathStyleAccessEnabled(Boolean.valueOf(true));
        }
        if (config.s3SignerOverride.isPresent() && !config.s3SignerOverride.get().isEmpty()) {
            clientConfiguration.setSignerOverride(config.s3SignerOverride.get());
        }
        if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
            String s3RegionName = S3TierObjectStore.validateAndGetS3RegionName(config.s3Region);
            builder.setEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(config.s3EndpointOverride.get(), s3RegionName));
        } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
            builder.setRegion(config.s3Region);
        }
        log.debug("AWS_METADATA_SERVICE_TIMEOUT is {} seconds", (Object)System.getenv("AWS_METADATA_SERVICE_TIMEOUT"));
        AWSCredentialsProvider provider = config.s3CredFilePath.map(PropertiesFileCredentialsProvider::new).orElse((AWSCredentialsProvider)new DefaultAWSCredentialsProviderChain());
        if (config.assumeRoleArn.isPresent()) {
            AWSSecurityTokenServiceClientBuilder stsClientBuilder = AWSSecurityTokenServiceClient.builder();
            stsClientBuilder.setCredentials(provider);
            if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
                String s3RegionName = S3TierObjectStore.validateAndGetS3RegionName(config.s3Region);
                stsClientBuilder.setRegion(s3RegionName);
            } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
                stsClientBuilder.setRegion(config.s3Region);
            }
            AWSSecurityTokenService stsClient = (AWSSecurityTokenService)stsClientBuilder.build();
            provider = new STSAssumeRoleSessionCredentialsProvider.Builder(config.assumeRoleArn.get(), "tiered-storage").withStsClient(stsClient).build();
        }
        builder.setCredentials(provider);
        return new AmazonS3ClientAndCredentialsProvider((AmazonS3)builder.build(), S3TierObjectStore.buildAsyncClient(config, sslConnectionSocketFactory), provider);
    }

    private static Optional<S3AsyncClient> buildAsyncClient(S3TierObjectStoreConfig config, SSLConnectionSocketFactory sslConnectionSocketFactory) {
        if (sslConnectionSocketFactory != null) {
            log.error("Skip building S3 async client: sslConnectionSocketFactory is specified");
            return Optional.empty();
        }
        if (config.s3SignerOverride.isPresent() && !config.s3SignerOverride.get().isEmpty()) {
            log.error("Skip building S3 async client: s3SignerOverride is specified");
            return Optional.empty();
        }
        if (config.s3CredFilePath.isPresent() && !config.s3CredFilePath.get().isEmpty()) {
            log.error("Skip building S3 async client: s3CredFilePath is specified");
            return Optional.empty();
        }
        ClientOverrideConfiguration.Builder clientOverrideConfiguration = ClientOverrideConfiguration.builder();
        clientOverrideConfiguration.putAdvancedOption(SdkAdvancedClientOption.USER_AGENT_PREFIX, (Object)config.s3UserAgentPrefix);
        S3AsyncClientBuilder builder = S3AsyncClient.builder();
        builder.overrideConfiguration((ClientOverrideConfiguration)clientOverrideConfiguration.build());
        builder.forcePathStyle(config.s3ForcePathStyleAccess);
        if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
            String s3RegionName = S3TierObjectStore.validateAndGetS3RegionName(config.s3Region);
            builder.endpointOverride(URI.create(config.s3EndpointOverride.get()));
            builder.region(Region.of((String)s3RegionName));
        } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
            builder.region(Region.of((String)config.s3Region));
        }
        log.debug("AWS_METADATA_SERVICE_TIMEOUT is {} seconds", (Object)System.getenv("AWS_METADATA_SERVICE_TIMEOUT"));
        DefaultCredentialsProvider provider = DefaultCredentialsProvider.builder().build();
        if (config.assumeRoleArn.isPresent() && !config.assumeRoleArn.get().isEmpty()) {
            StsClient stsClient = (StsClient)((StsClientBuilder)StsClient.builder().credentialsProvider((AwsCredentialsProvider)provider)).build();
            provider = ((StsAssumeRoleCredentialsProvider.Builder)StsAssumeRoleCredentialsProvider.builder().stsClient(stsClient)).refreshRequest((AssumeRoleRequest)AssumeRoleRequest.builder().roleArn(config.assumeRoleArn.get()).roleSessionName("tiered-storage").build()).build();
        }
        builder.credentialsProvider((AwsCredentialsProvider)provider);
        return Optional.of(builder.build());
    }

    private void expectBucket(String bucket, String expectedRegion, Optional<String> endpointOverride) throws TierObjectStoreFatalException {
        try {
            String actualRegion = this.client.getBucketLocation(bucket);
            if (actualRegion.equals("US") && expectedRegion.equals("us-east-1")) {
                return;
            }
            if (!expectedRegion.equals(actualRegion)) {
                log.warn("Bucket region {} does not match expected region {}", (Object)actualRegion, (Object)expectedRegion);
            }
        }
        catch (AmazonClientException ex) {
            if (endpointOverride.isPresent() && !endpointOverride.get().isEmpty()) {
                log.warn("On-prem store does not implement S3 API's GetBucketLocation. Skipping check which ensures that actual bucket region matches expected region.");
            }
            throw new TierObjectStoreFatalException("Failed to validate that bucket location for " + bucket + " matches location " + expectedRegion + "; unable to call GetBucketLocation", ex);
        }
    }

    private static SSLConnectionSocketFactory getSSLConnectionSocketFactory(S3TierObjectStoreConfig config) throws TierObjectStoreFatalException {
        boolean hasCustomKeyStore;
        if (config.s3SecurityProviders.isPresent() && !config.s3SecurityProviders.get().trim().isEmpty()) {
            HashMap<String, String> securityProviders = new HashMap<String, String>();
            securityProviders.put("security.providers", config.s3SecurityProviders.get());
            SecurityUtils.addConfiguredSecurityProviders(securityProviders);
        }
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;
        boolean hasCustomTrustStore = config.s3SslTrustStoreLocation.isPresent() && !config.s3SslTrustStoreLocation.get().isEmpty();
        boolean bl = hasCustomKeyStore = config.s3SslKeyStoreLocation.isPresent() && !config.s3SslKeyStoreLocation.get().isEmpty();
        if (hasCustomTrustStore || hasCustomKeyStore) {
            try {
                SSLContextBuilder sslContextBuilder = SSLContexts.custom();
                if (hasCustomTrustStore) {
                    KeyStore trustStore = KeyStore.getInstance(config.s3SslTrustStoreType.get());
                    FileInputStream trustKeyStoreFile = new FileInputStream(config.s3SslTrustStoreLocation.get());
                    trustStore.load(trustKeyStoreFile, config.s3SslTrustStorePassword.get().value().toCharArray());
                    sslContextBuilder.loadTrustMaterial(trustStore, null);
                }
                if (hasCustomKeyStore) {
                    KeyStore keyStore = KeyStore.getInstance(config.s3SslKeyStoreType.get());
                    FileInputStream identityKeyStoreFile = new FileInputStream(config.s3SslKeyStoreLocation.get());
                    keyStore.load(identityKeyStoreFile, config.s3SslKeyStorePassword.get().value().toCharArray());
                    sslContextBuilder.loadKeyMaterial(keyStore, config.s3SslKeyPassword.get().value().toCharArray(), (aliases, socket) -> "confluent.kafka");
                }
                sslContextBuilder.setProtocol(config.s3SslProtocol);
                if (config.s3SslProvider.isPresent() && !config.s3SslProvider.get().trim().isEmpty()) {
                    config.s3SslProvider.ifPresent(arg_0 -> ((SSLContextBuilder)sslContextBuilder).setProvider(arg_0));
                }
                SSLContext sslContext = sslContextBuilder.build();
                sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContext, config.s3SslEnabledProtocols.toArray(new String[0]), null, (HostnameVerifier)new DefaultHostnameVerifier());
            }
            catch (Exception e) {
                throw new TierObjectStoreFatalException("Failed to load keystore or trust store for tiered object store", e);
            }
        }
        return sslConnectionSocketFactory;
    }

    private static class S3TierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final AutoAbortingGenericInputStream inputStream;

        S3TierObjectStoreResponse(InputStream inputStream, long autoAbortSize, long streamSize) {
            this.inputStream = new S3AutoAbortingInputStream(inputStream, autoAbortSize, streamSize);
        }

        @Override
        public void close() {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }

    public static class AmazonS3ClientAndCredentialsProvider {
        private final AmazonS3 client;
        private final Optional<S3AsyncClient> asyncClientOpt;
        private final AWSCredentialsProvider credentialsProvider;

        public AmazonS3ClientAndCredentialsProvider(AmazonS3 client, Optional<S3AsyncClient> asyncClientOpt, AWSCredentialsProvider credentialsProvider) {
            this.client = client;
            this.asyncClientOpt = asyncClientOpt;
            this.credentialsProvider = credentialsProvider;
        }

        public AmazonS3 client() {
            return this.client;
        }

        public Optional<S3AsyncClient> asyncClientOpt() {
            return this.asyncClientOpt;
        }

        public AWSCredentialsProvider credentialsProvider() {
            return this.credentialsProvider;
        }
    }
}

